// re2zig $INPUT -o $OUTPUT

const std = @import("std");

// Maximum number of capturing groups among all rules.
%{maxnmatch %}
const none = std.math.maxInt(usize);

const SemVer = struct {
    major: u32,
    minor: u32,
    patch: u32,
};

fn s2n(str: []const u8) u32 { // convert pre-parsed string to a number
    var n: u32 = 0;
    for (str) |c| { n = n * 10 + (c - 48); }
    return n;
}

fn parse(yyinput: [:0]const u8) ?SemVer {
    var yycursor: usize = 0;
    var yymarker: usize = 0;

    // Allocate memory for capturing parentheses (twice the number of groups).
    var yynmatch: usize = 0;
    var yypmatch: [yymaxnmatch * 2]usize = undefined;

    // Intermediate tag variables used by the lexer (must be autogenerated).
    %{stags format = "var @@: usize = none;"; %}

    %{
        re2c:yyfill:enable = 0;
        re2c:posix-captures = 1;

        num = [0-9]+;

        (num) "." (num) ("." num)? [\x00] {
            // `yynmatch` is the number of capturing groups
            std.debug.assert(yynmatch == 4);

            // Even `yypmatch` values are for opening parentheses, odd values
            // are for closing parentheses, the first group is the whole match.
            return SemVer {
                .major = s2n(yyinput[yypmatch[2]..yypmatch[3]]),
                .minor = s2n(yyinput[yypmatch[4]..yypmatch[5]]),
                .patch = if (yypmatch[6] == none) 0 else s2n(yyinput[yypmatch[6] + 1..yypmatch[7]])
            };
        }
        * { return null; }
    %}
}

test {
    try std.testing.expectEqual(parse("23.34"), SemVer{.major = 23, .minor = 34, .patch = 0});
    try std.testing.expectEqual(parse("1.2.99999"), SemVer{.major = 1, .minor = 2, .patch = 99999});
    try std.testing.expectEqual(parse("1.a"), null);
}
