/*
Copyright The Helm Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package rules // import "helm.sh/helm/v4/internal/chart/v3/lint/rules"

import (
	"errors"
	"fmt"
	"os"
	"path/filepath"

	"github.com/Masterminds/semver/v3"
	"github.com/asaskevich/govalidator"
	"sigs.k8s.io/yaml"

	chart "helm.sh/helm/v4/internal/chart/v3"
	"helm.sh/helm/v4/internal/chart/v3/lint/support"
	chartutil "helm.sh/helm/v4/internal/chart/v3/util"
)

// Chartfile runs a set of linter rules related to Chart.yaml file
func Chartfile(linter *support.Linter) {
	chartFileName := "Chart.yaml"
	chartPath := filepath.Join(linter.ChartDir, chartFileName)

	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartYamlNotDirectory(chartPath))

	chartFile, err := chartutil.LoadChartfile(chartPath)
	validChartFile := linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartYamlFormat(err))

	// Guard clause. Following linter rules require a parsable ChartFile
	if !validChartFile {
		return
	}

	_, err = chartutil.StrictLoadChartfile(chartPath)
	linter.RunLinterRule(support.WarningSev, chartFileName, validateChartYamlStrictFormat(err))

	// type check for Chart.yaml . ignoring error as any parse
	// errors would already be caught in the above load function
	chartFileForTypeCheck, _ := loadChartFileForTypeCheck(chartPath)

	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartName(chartFile))

	// Chart metadata
	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartAPIVersion(chartFile))

	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartVersionType(chartFileForTypeCheck))
	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartVersion(chartFile))
	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartAppVersionType(chartFileForTypeCheck))
	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartMaintainer(chartFile))
	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartSources(chartFile))
	linter.RunLinterRule(support.InfoSev, chartFileName, validateChartIconPresence(chartFile))
	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartIconURL(chartFile))
	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartType(chartFile))
	linter.RunLinterRule(support.ErrorSev, chartFileName, validateChartDependencies(chartFile))
}

func validateChartVersionType(data map[string]interface{}) error {
	return isStringValue(data, "version")
}

func validateChartAppVersionType(data map[string]interface{}) error {
	return isStringValue(data, "appVersion")
}

func isStringValue(data map[string]interface{}, key string) error {
	value, ok := data[key]
	if !ok {
		return nil
	}
	valueType := fmt.Sprintf("%T", value)
	if valueType != "string" {
		return fmt.Errorf("%s should be of type string but it's of type %s", key, valueType)
	}
	return nil
}

func validateChartYamlNotDirectory(chartPath string) error {
	fi, err := os.Stat(chartPath)

	if err == nil && fi.IsDir() {
		return errors.New("should be a file, not a directory")
	}
	return nil
}

func validateChartYamlFormat(chartFileError error) error {
	if chartFileError != nil {
		return fmt.Errorf("unable to parse YAML\n\t%w", chartFileError)
	}
	return nil
}

func validateChartYamlStrictFormat(chartFileError error) error {
	if chartFileError != nil {
		return fmt.Errorf("failed to strictly parse chart metadata file\n\t%w", chartFileError)
	}
	return nil
}

func validateChartName(cf *chart.Metadata) error {
	if cf.Name == "" {
		return errors.New("name is required")
	}
	name := filepath.Base(cf.Name)
	if name != cf.Name {
		return fmt.Errorf("chart name %q is invalid", cf.Name)
	}
	return nil
}

func validateChartAPIVersion(cf *chart.Metadata) error {
	if cf.APIVersion == "" {
		return errors.New("apiVersion is required. The value must be \"v3\"")
	}

	if cf.APIVersion != chart.APIVersionV3 {
		return fmt.Errorf("apiVersion '%s' is not valid. The value must be \"v3\"", cf.APIVersion)
	}

	return nil
}

func validateChartVersion(cf *chart.Metadata) error {
	if cf.Version == "" {
		return errors.New("version is required")
	}

	version, err := semver.StrictNewVersion(cf.Version)
	if err != nil {
		return fmt.Errorf("version '%s' is not a valid SemVerV2", cf.Version)
	}

	c, err := semver.NewConstraint(">0.0.0-0")
	if err != nil {
		return err
	}
	valid, msg := c.Validate(version)

	if !valid && len(msg) > 0 {
		return fmt.Errorf("version %v", msg[0])
	}

	return nil
}

func validateChartMaintainer(cf *chart.Metadata) error {
	for _, maintainer := range cf.Maintainers {
		if maintainer == nil {
			return errors.New("a maintainer entry is empty")
		}
		if maintainer.Name == "" {
			return errors.New("each maintainer requires a name")
		} else if maintainer.Email != "" && !govalidator.IsEmail(maintainer.Email) {
			return fmt.Errorf("invalid email '%s' for maintainer '%s'", maintainer.Email, maintainer.Name)
		} else if maintainer.URL != "" && !govalidator.IsURL(maintainer.URL) {
			return fmt.Errorf("invalid url '%s' for maintainer '%s'", maintainer.URL, maintainer.Name)
		}
	}
	return nil
}

func validateChartSources(cf *chart.Metadata) error {
	for _, source := range cf.Sources {
		if source == "" || !govalidator.IsRequestURL(source) {
			return fmt.Errorf("invalid source URL '%s'", source)
		}
	}
	return nil
}

func validateChartIconPresence(cf *chart.Metadata) error {
	if cf.Icon == "" {
		return errors.New("icon is recommended")
	}
	return nil
}

func validateChartIconURL(cf *chart.Metadata) error {
	if cf.Icon != "" && !govalidator.IsRequestURL(cf.Icon) {
		return fmt.Errorf("invalid icon URL '%s'", cf.Icon)
	}
	return nil
}

func validateChartDependencies(cf *chart.Metadata) error {
	if len(cf.Dependencies) > 0 && cf.APIVersion != chart.APIVersionV3 {
		return fmt.Errorf("dependencies are not valid in the Chart file with apiVersion '%s'. They are valid in apiVersion '%s'", cf.APIVersion, chart.APIVersionV3)
	}
	return nil
}

func validateChartType(cf *chart.Metadata) error {
	if len(cf.Type) > 0 && cf.APIVersion != chart.APIVersionV3 {
		return fmt.Errorf("chart type is not valid in apiVersion '%s'. It is valid in apiVersion '%s'", cf.APIVersion, chart.APIVersionV3)
	}
	return nil
}

// loadChartFileForTypeCheck loads the Chart.yaml
// in a generic form of a map[string]interface{}, so that the type
// of the values can be checked
func loadChartFileForTypeCheck(filename string) (map[string]interface{}, error) {
	b, err := os.ReadFile(filename)
	if err != nil {
		return nil, err
	}
	y := make(map[string]interface{})
	err = yaml.Unmarshal(b, &y)
	return y, err
}
