/*************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/*
//
//  Purpose:
//     Cryptography Primitive.
//     EC over Prime Finite Field (P384r1 precomputed)
//
//
*/
#include "owncp.h"
#include "gfpec/pcpgfpecstuff.h"


#define OPERAND_BITSIZE (384)
#define LEN_P384        (BITS_BNU_CHUNK(OPERAND_BITSIZE))

/* P384 affine point */
typedef struct {
    BNU_CHUNK_T X[LEN_P384];
    BNU_CHUNK_T Y[LEN_P384];
} P384_POINT_AFFINE;

extern const __ALIGN64 P384_POINT_AFFINE ec_p384r1_precomputed[77][16];


#if defined(_IPP_DATA)

#if !defined(_DISABLE_ECP_384R1_HARDCODED_BP_TBL_)
/* see ippcp_baseptbl.cpp test for generation details */

const __ALIGN64 P384_POINT_AFFINE ec_p384r1_precomputed[77][16] = {
  /* digit=0 base_pwr=2^0 */
  /* clang-format off */
{
   {{LL(0x49c0b528,0x3dd07566),LL(0xa0d6ce38,0x20e378e2),LL(0x541b4d6e,0x879c3afc),LL(0x59a30eff,0x64548684),LL(0x614ede2b,0x812ff723),LL(0x299e1513,0x4d3aadc2)}, {LL(0x4b03a4fe,0x23043dad),LL(0x7bb4a9ac,0xa1bfa8bf),LL(0x2e83b050,0x8bade756),LL(0x68f4ffd9,0xc6c35219),LL(0x3969a840,0xdd800226),LL(0x5a15c5e9,0x2b78abc2)}},
   {{LL(0x783dde91,0xc8229e55),LL(0x022b53f0,0x8e6c8f2e),LL(0xff9d48a1,0x3504e6f0),LL(0xf0687f50,0xda821495),LL(0x2de4b506,0x9c90a4fd),LL(0x427460c3,0xdb93b776)}, {LL(0x3140bfda,0x42ea8463),LL(0xc2aaccd8,0xe8e8e4a8),LL(0xdc588258,0x15e4f18b),LL(0x5172bad9,0x09f1fe41),LL(0x00b0e684,0x070d4309),LL(0x123df0c2,0xe34947f7)}},
   {{LL(0xc1dc4073,0x05e4dbe6),LL(0xf04f779c,0xc54ea9ff),LL(0xa170ccf0,0x6b2034e9),LL(0xd51c6c3e,0x3a48d732),LL(0x263aa470,0xe36f7e2d),LL(0xe7c1c3ac,0xd283fe68)}, {LL(0xc04ee157,0x7e284821),LL(0x7ae0e36d,0x92d789a7),LL(0x4ef67446,0x132663c0),LL(0xd2e1d0b4,0x68012d5a),LL(0x5102b339,0xf6db68b1),LL(0x983292af,0x465465fc)}},
   {{LL(0xebb68f2c,0x0aae8477),LL(0xee0421e3,0x30594ccb),LL(0x0aecac46,0x2e4f153b),LL(0x736400ad,0x078358d4),LL(0xd685d979,0xfb40f647),LL(0x34179228,0xcfeee6dd)}, {LL(0x9b3a03b2,0x54f3e8e7),LL(0x7bfec97e,0xe74bb7f1),LL(0x4c542ad1,0x8e3e61a3),LL(0x0418c693,0x147162d3),LL(0x3820017d,0xe607b9e3),LL(0x303df319,0x50946875)}},
   {{LL(0x68f1f0df,0xbb595eba),LL(0xcc873466,0xc185c0cb),LL(0x293c703b,0x7f1eb1b5),LL(0xaacc05e6,0x60db2cf5),LL(0xe2e8e4c6,0xc676b987),LL(0x1d178ffb,0xe1bb26b1)}, {LL(0x7073fa21,0x2b694ba0),LL(0x72f34566,0x22c16e2e),LL(0x01c35b99,0x80b61b31),LL(0x982c0411,0x4b237faf),LL(0x24de236d,0xe6c59440),LL(0xe209e4a3,0x4db1c9d6)}},
   {{LL(0x7d56dad8,0x7eb5c931),LL(0x39d3413a,0xcb2454b3),LL(0x580d57f2,0xec52930f),LL(0x1bdf6015,0x2a33f666),LL(0x2b02d33b,0x4f0f6a96),LL(0xf0430c40,0xc482e189)}, {LL(0xa7b08203,0x3f62b16e),LL(0x5b3d4dce,0x739ac69d),LL(0xb79e33b0,0x8bd4bffc),LL(0x1b546f05,0x93c9e5f6),LL(0xdf21559a,0x586d8ede),LL(0xaf2a9eba,0xc9962152)}},
   {{LL(0x7d69222b,0xdf13b9d1),LL(0x874774b1,0x4ce6415f),LL(0x211faa95,0x731edcf8),LL(0x659753ed,0x5f4215d1),LL(0x9db2df55,0xf893db58),LL(0x1c89025b,0x932c9f81)}, {LL(0x7706a61e,0x0996b220),LL(0xa8641c79,0x135349d5),LL(0x50130844,0x65aad76f),LL(0x01fff780,0x0ff37c04),LL(0x693b0706,0xf57f238e),LL(0xaf6c9b3e,0xd90a16b6)}},
   {{LL(0xdd9bcbba,0x23f60a05),LL(0xae9b587a,0x9e336de5),LL(0x93d7e30f,0x1c5c2e71),LL(0x4f3ddb37,0x1d9aebd6),LL(0x16b66423,0x1c7b5fe1),LL(0x349cd9b1,0x5db4f184)}, {LL(0xe6655a44,0x0d2cfe83),LL(0xb7e55e87,0x836dbb36),LL(0x7d8686e4,0x701754bf),LL(0xa42dbba2,0xe9923263),LL(0xc48ecf0e,0x7008d943),LL(0x0d27ef61,0x3c0c6dd7)}},
   {{LL(0x2353b92f,0x2f5d200e),LL(0x3fd7e4f9,0xe35d8729),LL(0xa96d745d,0x26094833),LL(0x3cbfff3f,0xdc351dc1),LL(0xdad54d6a,0x26d464c6),LL(0x53636c6a,0x5cab1d1d)}, {LL(0xb18ec0b0,0xf2813072),LL(0xd742aa2f,0x3777e270),LL(0x033ca7c2,0x27f061c7),LL(0x68ead0d8,0xa6ecaccc),LL(0xee69a754,0x7d9429f4),LL(0x31e8f5c6,0xe7706334)}},
   {{LL(0xc8d99c02,0x845539d3),LL(0xe58d6787,0x2a15a9a6),LL(0xab225fa3,0xe9f6368e),LL(0xeb32cabe,0x54a612d7),LL(0x5c4845ec,0xc2f64602),LL(0xdb1c212e,0xa91a5280)}, {LL(0xe67b5fce,0xbb971f78),LL(0x13b9e85c,0x03a530eb),LL(0x794eabfd,0x592ac0ba),LL(0xcfd7fd1d,0x81961b8c),LL(0x47a9b8aa,0x3e03370a),LL(0xc80174e8,0x6eb995be)}},
   {{LL(0xb68b8c7d,0xc7708b19),LL(0x44377aba,0x4532077c),LL(0x6cdad64f,0x0dcc6770),LL(0x147b6602,0x01b8bf56),LL(0xf0561d79,0xf8d89885),LL(0x7ba9c437,0x9c19e9fc)}, {LL(0xbdc4ba25,0x764eb146),LL(0xac144b83,0x604fe46b),LL(0x8a77e780,0x3ce81329),LL(0xfe9e682e,0x2e070f36),LL(0x3a53287a,0x41821d0c),LL(0x3533f918,0x9aa62f9f)}},
   {{LL(0x70313de0,0x3db84772),LL(0x5d970420,0xd4258cc5),LL(0xc8edfee1,0x03aced26),LL(0x35d77d83,0xf67eb422),LL(0xcf9ab45c,0x523c40db),LL(0x9c35b26d,0x627b415f)}, {LL(0x8be55ed8,0xfacc45e4),LL(0x27aa651a,0x80d60af6),LL(0xd0e102ac,0x8c79848f),LL(0x66bed5af,0x40c64a4e),LL(0xf7942f0e,0x0329eab1),LL(0xf9c4af3d,0x0c6e430e)}},
   {{LL(0x75ccbdfb,0x9b7aeb7e),LL(0xf6749a95,0xb25e28c5),LL(0x33b7d4ae,0x8a7a8e46),LL(0xd9c1bd56,0xdb5203a8),LL(0xed22df97,0xd2657265),LL(0x8cf23c94,0xb51c56e1)}, {LL(0x6c3d812d,0xf4d39459),LL(0x87cae0c2,0xd8e88f1a),LL(0xcf4d0fe3,0x789a2a48),LL(0xfec38d60,0xb7feac2d),LL(0x3b490ec3,0x81fdbd1c),LL(0xcc6979e1,0x4617adb7)}},
   {{LL(0x8f75244c,0x5865e501),LL(0x01ec909f,0xd02225fb),LL(0xb1f85c2a,0xca6b1af8),LL(0x88957166,0x44ce05ff),LL(0x5710c0c9,0x8058994c),LL(0x32f6b1ba,0x46d227c4)}, {LL(0x03cb68e5,0xbe4b4a90),LL(0x730a99d1,0x540b8b82),LL(0xe11dbbbf,0x1ecc8585),LL(0xd9c3b691,0x72445345),LL(0x13690a74,0x647d24db),LL(0xdefbadf5,0x4429839d)}},
   {{LL(0x4709f4a9,0x446ad888),LL(0xec3dabd8,0x2b7210e2),LL(0x50e07b34,0x83ccf195),LL(0x789b3075,0x59500917),LL(0xeb085993,0x0fc01fd4),LL(0x4903026b,0xfb62d26f)}, {LL(0x6fe989bb,0x2309cc9d),LL(0x144bd586,0x61609cbd),LL(0xde06610c,0x4b23d3a0),LL(0xd898f470,0xdddc2866),LL(0x400c5797,0x8733fc41),LL(0xd0bc2716,0x5a68c6fe)}},
   {{LL(0x7c33ed91,0xda6e8a7f),LL(0x0ecdd2d8,0x992afb5b),LL(0x7917652a,0x37cf6551),LL(0x2887d5ff,0x317b63ea),LL(0x13bdc3fa,0x37065f53),LL(0x435abaa1,0xa10896aa)}, {LL(0xefabca26,0x9b21615f),LL(0x230cf00d,0xeb07ddea),LL(0x154d410f,0x914871dc),LL(0xc88ee148,0xb333bdfb),LL(0xa72d1967,0x51c305c6),LL(0x81ef2513,0x659db481)}},
},
/* digit=1 base_pwr=2^5 */
{
   {{LL(0x8c0a64a4,0xad852b87),LL(0x08f779d5,0x0d784cf2),LL(0xc651b1dd,0x1896b9fc),LL(0x12e8dc87,0xba8953d6),LL(0xa631cfb0,0x3a9865ba),LL(0x626b3d79,0x5dd2a4a0)}, {LL(0x687c20bb,0x1148bc72),LL(0xf2a52bfd,0xa372dfc2),LL(0x9448fd08,0x77315f9e),LL(0x0a2377b6,0x4bcb06f0),LL(0xb35b4ff0,0x73b42725),LL(0xaabca99f,0xc510ad93)}},
   {{LL(0x8e89b258,0xed4de4f8),LL(0x297a9a37,0x957c980f),LL(0xf8a0580d,0xe04b3d30),LL(0xca57b7bd,0xa309199d),LL(0x3be44d56,0xfc8e87cf),LL(0xd1b30e5c,0x4f5d5ab6)}, {LL(0x30a9325b,0xb213c6a0),LL(0xf091bc01,0x0fd1c52d),LL(0x1090fede,0xfe51bbbf),LL(0x301fe259,0x6d97cabc),LL(0x5ecd3fe8,0x3ee12789),LL(0x9404ca51,0x888b708b)}},
   {{LL(0x1d15bba7,0xdc529385),LL(0x148840d4,0x8fd61e31),LL(0x52ce08f6,0xee21597e),LL(0x81998af2,0x26de4c65),LL(0x9741eb42,0xcb4aa43c),LL(0x39c18b96,0x8bf3dccc)}, {LL(0xd872984e,0x66e1d5da),LL(0x7dd87c39,0x507e2405),LL(0x305ee4f0,0x8545396f),LL(0xc831254b,0xf9d19ea0),LL(0x5d5680b8,0xcee0842f),LL(0x5a4434a4,0xb257fb6c)}},
   {{LL(0x8258030f,0xe484fd9f),LL(0xf21af80a,0x0f4fa5ef),LL(0x7c1c3984,0xc0dd449e),LL(0x18eb5195,0xf3133891),LL(0x777a16d2,0x0336aab8),LL(0xc241720a,0xa6661cc4)}, {LL(0xa7efacd0,0x678db970),LL(0x6461e382,0x22896865),LL(0xa022c7a4,0x5d85a0e4),LL(0x34a02a6b,0xb01f1e04),LL(0xa5791ce3,0x2657eedd),LL(0xa277ac5b,0x239dcab2)}},
   {{LL(0x0b3db49f,0x7f74672d),LL(0xadc8418d,0x026275e9),LL(0xfe7cc4ee,0x18866523),LL(0xbbf36aca,0x3546317b),LL(0x80caf049,0x282313fa),LL(0x9a49d91c,0x08275c9c)}, {LL(0xab71c2a8,0xe41f92ec),LL(0x1dd95aae,0xb5674698),LL(0xa8c4dc57,0x82347685),LL(0xa847fd1e,0xf78b23d6),LL(0xb692f868,0x8194f070),LL(0xa5f2ad03,0xfe0dee06)}},
   {{LL(0x22abee22,0x61361c06),LL(0x2f42a470,0xe82123bc),LL(0x3c6ceff4,0x2c97dda1),LL(0xadcb0453,0xc2eea21f),LL(0xc78ec9ff,0xdf2a4407),LL(0x9625c735,0x54557c55)}, {LL(0x7c440c0a,0xc693c985),LL(0xe56622c1,0x918e7324),LL(0xa5f0900f,0x8f990d2b),LL(0xc6aeeaef,0xa5b7d193),LL(0x450456e3,0x9f053d95),LL(0xad418ef1,0xdf783b1c)}},
   {{LL(0xbe6c3221,0x08d2ce13),LL(0x9c0624ea,0x0f53824c),LL(0x5da25412,0xad3ca744),LL(0x93fea7cc,0xecb38a41),LL(0xcba623c4,0x8ae838d8),LL(0x41d85222,0xa75c8da5)}, {LL(0x290807c8,0x9f48b828),LL(0x22f4ab6a,0x4df75f09),LL(0x7cab82a9,0x02511a3e),LL(0x21da354e,0xcc052dc3),LL(0x03ca73e9,0x7db6c1f3),LL(0xcb99244e,0xca1d4279)}},
   {{LL(0x6ae54da1,0x9a7a5b42),LL(0x7040b022,0xcfcadaab),LL(0x3d9f0e61,0x7539438e),LL(0xe328c2e2,0x013c6719),LL(0xcccbf891,0x7f4a706c),LL(0x735a2d28,0xa335ab82)}, {LL(0x3d984124,0x46694ef0),LL(0xc166b337,0x0e0bdfab),LL(0x423d47e4,0x9d54ed8b),LL(0xf44c9180,0x8075a8cb),LL(0xabe9b384,0xd4f5b184),LL(0x41abdc75,0x424dd00b)}},
   {{LL(0xf2432ae5,0x36137ae5),LL(0xad5f443d,0x2d941abc),LL(0xee013f80,0xe31e96f2),LL(0x26385266,0x24127d20),LL(0xeae97f8b,0xd4004cf3),LL(0x62527c9e,0xbf3b201a)}, {LL(0xcb8a2a2f,0x0cd0fee1),LL(0x8fdceb43,0x48894469),LL(0x2250a356,0xe67565ba),LL(0x2ccfe7d1,0xbbb5f8ca),LL(0xa350fe88,0x431a268f),LL(0x4e5b64d2,0xd2eb0373)}},
   {{LL(0xf5d0255f,0x2ad7ae0d),LL(0xd0836f8b,0x4c04dee0),LL(0x4ec33606,0x15e3493e),LL(0x5f5eea30,0x590aefe6),LL(0x2e49ba54,0x09ef5a2e),LL(0x69a0d232,0xf56d447f)}, {LL(0xe2510f5e,0x045f26b6),LL(0x9941083a,0xf90e1888),LL(0xe66f6b9b,0x6e6a3bd6),LL(0x251f82ae,0x637ad390),LL(0x43cdf91a,0xc40374b6),LL(0x2d6c6d88,0x62f5f8ad)}},
   {{LL(0x11ff6e09,0x3da03590),LL(0xa685c0e8,0x41194047),LL(0x72b9c9ce,0x7c5323ff),LL(0x4f599daf,0xf84ce505),LL(0x5882ef19,0xd0dec10e),LL(0xde1e3a06,0x1ca3449d)}, {LL(0xb09af84a,0x317e4e7f),LL(0x1a46bc75,0x0ebaa2c9),LL(0xc103f200,0x42b00a72),LL(0xc3719155,0x1c30ada5),LL(0x5c1cb58c,0x565bacfa),LL(0x35fa4670,0x374f516c)}},
   {{LL(0x400f1d87,0x99a710b1),LL(0xee47be8b,0xec3ca386),LL(0x37168fee,0x3a00dcad),LL(0x1765a0df,0x1a69d674),LL(0x917c4909,0x85233afe),LL(0xe9bd6210,0xa3aa97a4)}, {LL(0x310885f3,0x01010526),LL(0xb5007b08,0x21c5de3f),LL(0xee8ddabd,0x5df0c6ef),LL(0x0f6dfdef,0x5d17d45a),LL(0x251c9f79,0xb73d831a),LL(0xcb1df19d,0x397779e6)}},
   {{LL(0xaf897b90,0xe97f7bcf),LL(0xe05af5ab,0xbaeecdda),LL(0x1b81fd5b,0x3c80856e),LL(0xeca4f983,0xea309bd0),LL(0x22ab76a2,0x534d8950),LL(0x6d620cfe,0xa8d63627)}, {LL(0x6b0cfcfc,0xa13d622c),LL(0x84d9147e,0x584a9d2e),LL(0xd200ea9c,0xf23b9565),LL(0x193d5fe0,0x5f44b61a),LL(0x64b4c78d,0x430a3d3a),LL(0xc91c2b41,0x045dc592)}},
   {{LL(0xbe13c1e4,0xafd6bf05),LL(0xe2ea8c26,0x256e2b98),LL(0x903983f4,0x06711695),LL(0x217c1577,0xbefbfbcb),LL(0x7324c85c,0xa214b26a),LL(0xdd98b968,0xfa2d4711)}, {LL(0xf5d9cc93,0xed722297),LL(0xd45f8635,0x65cec50f),LL(0xc792e6b3,0x41627e4a),LL(0x228dbb39,0xc254a08c),LL(0xdf6fb26b,0x5b28fd3f),LL(0x4bdc6969,0x1471d353)}},
   {{LL(0x7c20f79d,0x8600c734),LL(0xca3052ba,0x2bef7d87),LL(0x7b082112,0xc52d0bdf),LL(0x2bf757ca,0x0e649a4b),LL(0xb13eeb84,0x0c1f41bd),LL(0x8ecf9ace,0xd7115e8f)}, {LL(0x1698ba68,0xa62f5e31),LL(0xc862e751,0x6a06d7be),LL(0xf95eee3f,0xd875c1a4),LL(0xa0b2b331,0xb25f8da3),LL(0x1827c037,0x4e52fd11),LL(0x9d72ba3c,0xacd38c2a)}},
   {{LL(0xd7801edf,0x5e2cb07f),LL(0x3adc065a,0xf9fa2c0b),LL(0xd4de1f25,0xa296c53f),LL(0x838f7169,0xd408060c),LL(0x2e8a6ce7,0x68e19d7b),LL(0x94b58671,0x2cc6e06c)}, {LL(0xc1cb6151,0x93d02a07),LL(0x35003126,0xa10fb4cf),LL(0x1aa3bc4e,0x6aa069f5),LL(0xdd09b142,0x0e44fbf0),LL(0x832e5945,0xe264f343),LL(0x1fc166ac,0x5dca2adc)}},
},
/* digit=2 base_pwr=2^10 */
{
   {{LL(0xa0eb176a,0x56ad7582),LL(0x6e19aa00,0x085b5a36),LL(0xe2c8b036,0x85f2c6ff),LL(0xfcd7336b,0x55c6d357),LL(0x22a46acd,0xb1ecc56d),LL(0x36277ac7,0x8e0f9767)}, {LL(0x01878921,0xa4ed11e9),LL(0x7f4fb650,0xd3835566),LL(0x266158aa,0x5fdaba45),LL(0x89e0dff0,0x6f0b27fd),LL(0xeb6b02d0,0x32ef7ae2),LL(0xcc1b46e1,0x2f145871)}},
   {{LL(0xf6e9f82c,0xe65245ce),LL(0xf6da7b5a,0x9e234dff),LL(0xbad2c806,0x5677c121),LL(0x060fcf24,0xc52dec32),LL(0x5d78ccca,0x78d07067),LL(0x1bc8b6fb,0x630002ea)}, {LL(0xeb2e99ae,0xc5cb86be),LL(0x8551d16f,0xf13981e7),LL(0xc92a70d7,0xfbb7cdf2),LL(0xf53cd2a1,0x5a9ff1f1),LL(0x984f1139,0xfdbe6b7a),LL(0xa470a9f1,0x4403d046)}},
   {{LL(0x8d040c69,0xb5b16cd4),LL(0xf95a2dc4,0xfdaba315),LL(0x61ce4704,0xc9fef349),LL(0xdbb53ed6,0x5fe87a0d),LL(0x73d70f93,0x3f0ccc79),LL(0x46724a1a,0x4601d1bc)}, {LL(0x24f08565,0x5c4a15ae),LL(0xaa577320,0x5eda1e8a),LL(0xbb32d307,0xe31ebb35),LL(0xdc770a0e,0xcdc6f13b),LL(0xc434c2f8,0xbe3ae514),LL(0x3a0ef0d1,0x57c7fdf6)}},
   {{LL(0x0f547eba,0xeb06571c),LL(0x6246c0dc,0xf292c38d),LL(0x26eed224,0xa1859667),LL(0x6100e387,0x8d9e56e4),LL(0xdc6298d1,0x470506b9),LL(0xf3350ad0,0xb19e084c)}, {LL(0x12abd898,0x83eb62a8),LL(0x2222342c,0x70f152cc),LL(0xe1bd4a82,0xb089e880),LL(0xcaf3b3fc,0xd4d1e70f),LL(0x95ffd65c,0xd0b1ec63),LL(0x9b184ebb,0x79f27f3a)}},
   {{LL(0xb92a9a13,0xc2467f9a),LL(0x6add349a,0x0c3ee8eb),LL(0x45e99644,0x59250eda),LL(0xc8a2df27,0x22ce0635),LL(0x312e8698,0xec7b643e),LL(0xaebd1587,0x334ccf2e)}, {LL(0x6bca2900,0x0c1c6873),LL(0x09826cb1,0x00beb4c2),LL(0xcde6b725,0x12dbb586),LL(0x66dfed41,0x8d7cee6c),LL(0x014de4c8,0xc38deba2),LL(0x6248442d,0xa3ba6ec7)}},
   {{LL(0xd42aeb95,0x3c686079),LL(0x85e3ea0f,0xc162e5e9),LL(0x1bbb2455,0x34cf5861),LL(0x8773b064,0x7650de1d),LL(0xd2bab35d,0x3b7562c6),LL(0x33d0741b,0x83191b44)}, {LL(0x3a6bd9cf,0x4b604db0),LL(0x074aed21,0x87cd84db),LL(0xd4f91f9c,0x02a042d2),LL(0xe42c2a67,0x5a5d52e5),LL(0x1d5f216a,0x31291acd),LL(0xcd6203c8,0x9c3971bc)}},
   {{LL(0xaa1ef488,0x92e4c851),LL(0x846528a9,0x4f91fc22),LL(0x5dc13a84,0xaa2f5d2d),LL(0xf1072d4d,0xba06aa68),LL(0x5e3a2ba9,0xf4f3b17c),LL(0x5e4dde77,0xff36a535)}, {LL(0xae17dddc,0xe8ef143a),LL(0xa1fcd4ec,0xcc82631b),LL(0xc7d3963b,0x97db807e),LL(0x21d85ce2,0xe4aff045),LL(0x74667392,0x2d6480e6),LL(0xa55d0b3f,0xc8ce97f3)}},
   {{LL(0x4c14ee0e,0xd4277731),LL(0x5e881c3f,0xda8146d1),LL(0x99f1867c,0x6b0746b0),LL(0x602dd4cc,0x1ec73d72),LL(0x38081120,0x27fae515),LL(0x2f8b2f2d,0x6a677bdc)}, {LL(0xd60544e9,0xb924af64),LL(0xdcfc6b16,0x1439e183),LL(0x068565ac,0x4e88e9ae),LL(0xa9a4f146,0x8a3dbd25),LL(0x3f93f734,0xdb4a3e48),LL(0x8f1d33bc,0xb1971c05)}},
   {{LL(0x56e8d1d1,0x059a5f98),LL(0x5c228e89,0x93a2d2ac),LL(0xfea48feb,0x0bc2c885),LL(0x7b7e188c,0x27fdf3be),LL(0xd399ba6e,0xec579682),LL(0xb3fdb5d0,0x01b12e12)}, {LL(0x742f66b8,0x8e00941b),LL(0x478d8ee4,0xd79bc58f),LL(0x9ad50729,0xefdcd053),LL(0x3bf7bfea,0x8738666f),LL(0xd7afaff1,0x20192810),LL(0xd233e892,0x6c13e29e)}},
   {{LL(0xff9ae345,0xf1f5df24),LL(0x3dd75336,0x5b8038b9),LL(0x7d2c0058,0x4746c5da),LL(0x51ddd827,0x667b07bd),LL(0x8cfdfd25,0x1b3c67c2),LL(0x06dac0d6,0x058dea06)}, {LL(0x846c847d,0xd19040fd),LL(0x07d00ec6,0x4e5cbe59),LL(0xf1930296,0x2dc442fe),LL(0xfdb583be,0x12f89640),LL(0x5606f66b,0x0694f44f),LL(0xbbef8446,0xa9fcca6c)}},
   {{LL(0x4b5cb12f,0x0c229fb9),LL(0xe930ff3e,0x0486ba2d),LL(0xb80c37f6,0x00797293),LL(0xb43335e0,0x71be9538),LL(0xda0d9cb7,0x0a604693),LL(0x745cc02e,0xb248f30d)}, {LL(0xd9a3013b,0xfa889343),LL(0x53d6a4d4,0xd8642bc0),LL(0xe216af4b,0x9f89c21a),LL(0x41947320,0x86272900),LL(0xa269c49a,0x06f5f8f7),LL(0x727495ba,0xb3bc6d62)}},
   {{LL(0xd1066e5a,0xd1de964b),LL(0x749909df,0x26ccad6b),LL(0x0de9eb07,0xbc4b55af),LL(0xa60289bf,0x9f11b995),LL(0x52392c61,0x3e59000c),LL(0x5ccff4fc,0x2128c46a)}, {LL(0xea1e0d5e,0xb28b306b),LL(0xf5cfa47d,0x5f6ef5d5),LL(0xe3f824fa,0x59b0542c),LL(0x2f52b71c,0x70f097b2),LL(0x04c6d263,0x4483b726),LL(0x34d93e5b,0x48720be7)}},
   {{LL(0x95ad5fe3,0x864bbbb7),LL(0xe8113888,0x24f15dcb),LL(0x6edb61fe,0x968fcf60),LL(0xc7e1aff9,0xbf266d1c),LL(0x3f4a5d91,0xd85a1555),LL(0xf8416ead,0xd5ff238e)}, {LL(0x30d18538,0xcc0e75da),LL(0xb2b91d71,0xcf8a5435),LL(0x027a7e72,0xfae90226),LL(0x627c27f7,0x8e39918c),LL(0x18d454b0,0x5b8235c9),LL(0x44987803,0xef92e643)}},
   {{LL(0x41ff08a5,0xb39b33ce),LL(0x32450c03,0xaf6dfb28),LL(0x7b45c3f2,0xe21947c5),LL(0xbc35bfdf,0x21b529f6),LL(0x4c5be366,0xc0a26928),LL(0xd366b21e,0x73856f3a)}, {LL(0x3df444f7,0x53e26b66),LL(0x8b9c7ac1,0xb18a295e),LL(0x301f1f11,0x6b9df497),LL(0x7607f372,0x6e77d0e2),LL(0xc311fc01,0xd2e6b296),LL(0x7a396a6c,0x18a224e0)}},
   {{LL(0x9b97dff2,0x8507c707),LL(0x983d26af,0xacc0b6a5),LL(0x2b62fdb2,0xdcc30fd6),LL(0xa68ab541,0xe9b03755),LL(0x13b6c3a0,0x270bc073),LL(0x649051fc,0x952ec4b9)}, {LL(0xe70ceb83,0x4814c2c2),LL(0xc090e636,0x07632da5),LL(0x11fcdb0b,0x1c340686),LL(0x113f2ce9,0x7b4985d7),LL(0x16da3a7d,0xc3072347),LL(0x0d7d36ab,0xf0575aef)}},
   {{LL(0x3a7f43c3,0x721bb836),LL(0xc12dca0f,0xab5b3108),LL(0x34853870,0x9cc9a789),LL(0xaf598c4f,0x78b604a7),LL(0x476f27c8,0xd370375e),LL(0x0b15cba5,0x9f0415b8)}, {LL(0xd400dc1f,0x2bcfd9a4),LL(0x4bc62ddd,0x2a6fe03b),LL(0xa211b19f,0xb05a6464),LL(0x9990b504,0xce059d41),LL(0xd94951d3,0x011c5f87),LL(0x00d9c7b0,0x13cec089)}},
},
/* digit=3 base_pwr=2^15 */
{
   {{LL(0x9b280fd9,0xf365419f),LL(0x46365672,0x13e3b127),LL(0xc41880aa,0x8a91c165),LL(0xf9712fbd,0x3eb27a97),LL(0x76c55678,0xa6587aec),LL(0x02cd79cf,0x7c3a04b7)}, {LL(0xda712eb8,0xfc878f9c),LL(0x0fca3e02,0x076e6117),LL(0xcaf6df5e,0x09a184d8),LL(0xaabcde75,0xd32bf232),LL(0x03de597a,0xf601d0de),LL(0xc5da2858,0x85d2b5fc)}},
   {{LL(0xae899afa,0xb6029961),LL(0x681b6e1b,0x8eedd66e),LL(0x4df3e5b1,0x82db693b),LL(0xa0c3e357,0xb5131488),LL(0xccb2f577,0xbfb01ff3),LL(0x27a72cc3,0x28ea9470)}, {LL(0xe39e325e,0x26170928),LL(0x84f80188,0x42d4876b),LL(0x4c872d76,0x0bec6a63),LL(0xb14d9c9f,0xa0a7cc90),LL(0x4f6c7778,0x8a32d2c4),LL(0x3b889a4c,0xe7cd346e)}},
   {{LL(0x437f2efb,0xea95b5c9),LL(0x8e52ce8e,0x67d3e9b5),LL(0x88ff5455,0xfb331024),LL(0xc3101ded,0x883ec9fc),LL(0x0ca5f1f9,0x056218e2),LL(0xbccc65d8,0x4c9b2483)}, {LL(0x123db2ec,0x13199b26),LL(0x9aff90db,0xbc1d8247),LL(0x367516f6,0xc4a9311d),LL(0xcb6e90d6,0x3b40c867),LL(0x4034415c,0xff674ff8),LL(0xa1821e7e,0x2787db45)}},
   {{LL(0x609683ac,0x860ef794),LL(0x9af1c522,0xf0631ad3),LL(0x2ee7f522,0x322c8366),LL(0x54122af0,0x58ccd95f),LL(0x2bbb2d80,0x7454880c),LL(0x86d8d577,0xea173d82)}, {LL(0xea1cc801,0xd5a3057c),LL(0xbbb2a189,0xfd08d482),LL(0xc3c512f8,0x26aac99e),LL(0x556d891e,0xc3eac036),LL(0xbf9f6112,0x866c3aa7),LL(0xc144f7e9,0x7c4c8fb9)}},
   {{LL(0x8343f5dd,0xc0c1eb0f),LL(0x0126fa2d,0xa205e66a),LL(0x44fd8be4,0x37530a32),LL(0x96ab64cf,0x3c7af6f6),LL(0x76657202,0x1f043050),LL(0x828f3fb1,0x2c59d31c)}, {LL(0xe7f5926e,0xa61cba51),LL(0x91ebeb81,0xb2de4273),LL(0xbb855476,0x976f2c34),LL(0x269e4f86,0x0c6c02c3),LL(0xe2e01fff,0xc69bed8f),LL(0x480bf7b4,0x19aa421c)}},
   {{LL(0xb617206b,0xad0d24c7),LL(0x9dd13f59,0xb8be483f),LL(0x7655aa29,0xa55134ff),LL(0x7a5e217e,0xa3d10385),LL(0x3a21b295,0xfbb9eeb5),LL(0xc84a136d,0x1de71555)}, {LL(0x0b7487b3,0x7579398e),LL(0xa14dce87,0xc6ca9575),LL(0x46452257,0x99b32e1e),LL(0x479d8f2c,0x99d54955),LL(0x930b80ef,0xc2a8a6a1),LL(0x656c850b,0x58d9db81)}},
   {{LL(0xa682508c,0x6c85a0ed),LL(0xedcce6c8,0x8350b515),LL(0x6f95898d,0xa84a6765),LL(0x1e419847,0x55b0ae97),LL(0xdd6a885d,0x11115c6e),LL(0x4fb174a4,0x6f0beaf6)}, {LL(0x815af3af,0xaaae44b2),LL(0x927a2c1c,0xcf0697b9),LL(0xd7d645ee,0x37639d62),LL(0x2effec37,0x157b7eda),LL(0x9b9c66e6,0xb55e5075),LL(0x85f597ec,0x33a66a1f)}},
   {{LL(0x8f7782b6,0x171898aa),LL(0x499b3a81,0x8b7a706b),LL(0xfdb2c1ba,0xbc0e835f),LL(0x591f5aaf,0x4ee30281),LL(0xcc272c6a,0xfd71de3b),LL(0x6e93f68a,0x532800c4)}, {LL(0x8365c576,0x35ee0804),LL(0xcd4c0221,0x6c2bcc94),LL(0x957b2ff6,0x49f37ff5),LL(0x5ec029c0,0x315d8e7e),LL(0xef324c12,0x33230602),LL(0x966b2578,0xf5847f9b)}},
   {{LL(0x1453b1e1,0xc29b123e),LL(0x20059b44,0xefb07788),LL(0x9291671e,0x15554ade),LL(0x429dea37,0xeb5a1980),LL(0x6c4b867d,0xf96dacbb),LL(0xabab4d68,0x4f5563d6)}, {LL(0xcbe76297,0xb5b0ecff),LL(0x51d6bd43,0x5a22996a),LL(0xb7e5cfc6,0x0088ec95),LL(0xfe373e05,0x4863a5a1),LL(0xc244d93d,0x42b7925b),LL(0x40117113,0x85bad135)}},
   {{LL(0xf9daa551,0x86283e21),LL(0x1f696f1b,0x47fd23f8),LL(0xb9784a9a,0x7d029b1b),LL(0xa0c0acb5,0x7c7798be),LL(0x6d7c682b,0x41241c71),LL(0x1d33c2b0,0x11c6c113)}, {LL(0x3565cf32,0x5d469ca2),LL(0xbad4bdbf,0xa949f022),LL(0xa13cf4cd,0x3d054cc2),LL(0x9e3ce279,0x13bd2166),LL(0x8a4beafc,0x01bc70e6),LL(0x8aba087e,0xb39e351d)}},
   {{LL(0x82357232,0x26072a5d),LL(0x9f0fd2f1,0x3762764e),LL(0x0c16733e,0x9c5813f6),LL(0x718951d4,0xea2e0e03),LL(0x69e63818,0xae195bd4),LL(0xfa2f9a6e,0x241a4afc)}, {LL(0x9165d59a,0x0e97519f),LL(0x58e5af1b,0x416bd373),LL(0x8197b7ec,0xc4e81128),LL(0x9c6ba0d0,0x4145be2c),LL(0x7d40b98a,0xc82cb2a1),LL(0xbccfa8b8,0xc3c28487)}},
   {{LL(0xa4ee10a1,0x84aa863e),LL(0x87919ccd,0x24d805a6),LL(0xb5c399b2,0x553f3206),LL(0x3cc109bd,0x775b9217),LL(0xfe384088,0x25c01263),LL(0xd5f743cf,0xa3c4418b)}, {LL(0xdf91f1f0,0x3d69705d),LL(0x9ebddad1,0x547d4626),LL(0x2626cebd,0x0198ab1a),LL(0x85b1afe8,0xaf8320f2),LL(0xe17e6efd,0xb9c0968c),LL(0x90215bb6,0xfedc75c2)}},
   {{LL(0x143ff74d,0x8061d340),LL(0xa23aa7b3,0x59e94fc6),LL(0x914c3b81,0xf7c79a0b),LL(0x702c6ae7,0x5a836211),LL(0x718123c3,0x2570d63c),LL(0xc9f5ce3c,0x7e86d11e)}, {LL(0x17bcce3b,0x80f03f97),LL(0x465a7446,0x073975b1),LL(0xcb357ace,0x29f66de7),LL(0x04894fb0,0xe87bd12c),LL(0x51a0b5ae,0xfc501a26),LL(0x02207a3b,0x20a3170c)}},
   {{LL(0x5fe63bd4,0x3ee52d8f),LL(0x7a7da77c,0x09f8405c),LL(0x1881a757,0x35ce95c6),LL(0xc13e3707,0x0a8cf9d9),LL(0x48d2d3f6,0xe71258d5),LL(0x0bbe7c0d,0xcf4fd691)}, {LL(0xfb9479f1,0xbd6496e1),LL(0xab8cb3a2,0x711c669a),LL(0x58cfdfb4,0xcbe85013),LL(0x59275b4a,0x655c902b),LL(0x20f722ba,0x7e0ff05b),LL(0x42b17aad,0xcb00031d)}},
   {{LL(0xd54bf1a4,0x3360ff11),LL(0x5c79494b,0xbab994cb),LL(0x757d7771,0x953ad553),LL(0x68b58ed5,0xf17f14f0),LL(0x7523c422,0x22361531),LL(0x5ebf0d49,0xf0f05f96)}, {LL(0x49182267,0x33866765),LL(0xad71c3eb,0xf87eccc1),LL(0x913d8dca,0xd7708e18),LL(0xb193eef9,0x27fe27e1),LL(0xcc45e65d,0x33376365),LL(0xd700ac20,0x599b4778)}},
   {{LL(0xed068028,0xda643272),LL(0xa91fb87f,0x86b52135),LL(0x35b43943,0x23865a7c),LL(0x4606bbf2,0x6ac01588),LL(0x1559fb9a,0x9660ab72),LL(0x3ce2f1a5,0x1fcb09e7)}, {LL(0x793d2f0c,0x62af29ab),LL(0x3aee7efc,0xad5aaef5),LL(0x44c11037,0xee9f29b7),LL(0xd36c2571,0xb2a19cf1),LL(0x65b552b7,0xb87d88e2),LL(0xbeb253d4,0xd8b4f172)}},
},
/* digit=4 base_pwr=2^20 */
{
   {{LL(0x72876ae8,0x31bdb483),LL(0x961ed1bf,0xe3325d98),LL(0x9b6fc64d,0x18c04246),LL(0x15786b8c,0x0dcc15fa),LL(0x8e63da4a,0x81acdb06),LL(0xdada70fb,0xd3a4b643)}, {LL(0xdea424eb,0x46361afe),LL(0x89b92970,0xdc2d2cae),LL(0x615694e6,0xf389b61b),LL(0x872951d2,0x7036def1),LL(0xd93badc7,0x40fd3bda),LL(0x380a68d3,0x45ab6321)}},
   {{LL(0xce0b5b72,0xc5cf8997),LL(0x9d7154ba,0x350adde1),LL(0x307b254a,0x8139681e),LL(0x75cd94d7,0xcc87fb57),LL(0x78684954,0x90e70274),LL(0x95ceb991,0xc4fdf4c0)}, {LL(0x8762c84c,0x91bbc0ab),LL(0xce09e8ad,0x5e09e226),LL(0x4b93d45f,0x1cb83d70),LL(0xf541da1f,0xe2299024),LL(0x4b7ffd10,0x3eef7ce1),LL(0xb3fc1b9a,0x53ee63bb)}},
   {{LL(0x81a2703a,0x23c1f744),LL(0xb9859136,0x1a5d075c),LL(0x5afd1bfd,0xa4f82c9d),LL(0xf89d76fe,0xa3d1e9a4),LL(0x75702f80,0x964f7050),LL(0xf56c089d,0x182bf349)}, {LL(0xbe0da6e1,0xe205fa8f),LL(0x0a40f8f3,0x32905eb9),LL(0x356d4395,0x331a1004),LL(0xfdbbdfde,0x58b78901),LL(0x9ba00e71,0xa52a1597),LL(0x55497a30,0xe0092e1f)}},
   {{LL(0x03682f59,0xe5004e80),LL(0xf642ac0f,0xccdb9cb7),LL(0xbd869f77,0x405f50d1),LL(0xe7ebea2c,0xecffa54d),LL(0xd87620ba,0x3354dc22),LL(0xb1c01ff4,0x01bb2988)}, {LL(0xe16477fd,0xd9370076),LL(0x2e71ba4b,0x45303d2a),LL(0x3291e5c5,0xc0de7627),LL(0xf0a7ca55,0x5cfebd87),LL(0x9e592a30,0xde116280),LL(0xa78ebce4,0xdd26e577)}},
   {{LL(0x70ee8f39,0x5562a856),LL(0x64e52a9c,0x86b0c117),LL(0x09c75b8c,0xc19f3174),LL(0x24923f80,0x21c7cc31),LL(0x8f5b291e,0xe63fe47f),LL(0x0dc08b05,0x3d6d3c05)}, {LL(0xee0c39a1,0x58ae455e),LL(0x0ad97942,0x78bea431),LL(0x3ee3989c,0x42c7c97f),LL(0xf38759ae,0xc1b03af5),LL(0xbcf46899,0x1a673c75),LL(0x8d508c7d,0x4831b7d3)}},
   {{LL(0xff1735a8,0x1e9b23b9),LL(0x2b0e4b7b,0xc3bf3d5b),LL(0x59b7721c,0xd4cc00fe),LL(0x9e2f4ceb,0xd5c36f9c),LL(0xc90af70e,0xdeca06ba),LL(0x416ee799,0x42676f12)}, {LL(0x6f748c6f,0x0d7afe1b),LL(0x39c39d55,0x0b7a6de5),LL(0xe6eaed18,0x11e43d6e),LL(0x496087e0,0x5baf8602),LL(0xb1a3a66e,0xf833634f),LL(0x79398677,0x25098c8a)}},
   {{LL(0xc552e354,0x76512d1b),LL(0x273020fd,0x2b7eb6df),LL(0x025a5f25,0xd1c73aa8),LL(0x5cbd2a40,0x2aba1929),LL(0xc88d61c6,0xb53cadc3),LL(0x098290f3,0x7e66a95e)}, {LL(0xaf4c5073,0x72800ecb),LL(0x9dc63faf,0x81f2725e),LL(0x282ba9d1,0x14bf92a7),LL(0xbd5f1bb2,0x90629672),LL(0xa97c6c96,0x362f68eb),LL(0x7ea9d601,0xb1d3bb8b)}},
   {{LL(0xd4720770,0xe141e763),LL(0xddb3b450,0xb9739e70),LL(0x96131446,0x46e6cde4),LL(0xcb6c2ef7,0x0458a5d5),LL(0x532f9fd8,0xb7747634),LL(0x16544457,0xf62d3721)}, {LL(0xd3100854,0xbfacb4de),LL(0xb39d3f62,0x70788a31),LL(0xf22d92e4,0x9b543220),LL(0x55723258,0xaa4590f6),LL(0x01ddb8bc,0xc7b6730e),LL(0x69e1e7bd,0xae252cf8)}},
   {{LL(0xa9c94429,0x73878f7f),LL(0x456ca6d8,0xb35c3bc8),LL(0xf721923a,0xd96f0b3c),LL(0xe6d44fa1,0x28d8f06c),LL(0xd5cd671a,0x94efdcdc),LL(0x3f97d481,0x0299ab93)}, {LL(0x2fd1d324,0xb7ced6ea),LL(0x7e932ec2,0xbd683208),LL(0xcb755a6e,0x24ed31fb),LL(0xe48781d2,0xa636098e),LL(0xf0a4f297,0x8687c63c),LL(0x07478526,0xbb523440)}},
   {{LL(0xdd4b8d8d,0xc618cf0d),LL(0x40dcfbfb,0x471cda86),LL(0x08882ce6,0xba0dd7ac),LL(0x6cd336e7,0x58e5d2f5),LL(0xaf096540,0xcdda8301),LL(0x3cf31600,0xf6d26846)}, {LL(0x2197efd5,0x6150cd98),LL(0x55fb0877,0x4440fbfa),LL(0x90757f1d,0xca31871c),LL(0xbdd756c8,0xc4a1faac),LL(0xcbb8421e,0xc9d4ac1b),LL(0xb17c43be,0x3c0c2914)}},
   {{LL(0x34124b56,0x2e5f7419),LL(0x4b3f02ca,0x1f223ae1),LL(0xe8336c7e,0x6345b427),LL(0xf5d0e3d0,0x92123e16),LL(0x45e79f3a,0xdaf0d14d),LL(0x6f3bd0c6,0x6aca6765)}, {LL(0x403813f4,0xf6169fab),LL(0x334a4c59,0x31dc39c0),LL(0xd589866d,0x74c46753),LL(0x984c6a5d,0x5741511d),LL(0x97fed2d3,0xf2631287),LL(0x11614886,0x5687ca1b)}},
   {{LL(0xaa6fe9ea,0x46fdb65c),LL(0x05494cd9,0xe0d48e5e),LL(0x4afbf837,0x5adef570),LL(0x1c9e2cad,0xc96ba4b9),LL(0x054a158c,0x1e8158f7),LL(0x9e38b88d,0x47be7320)}, {LL(0x6d2993ec,0x9b99971e),LL(0xdf980ecc,0xac9b0bfa),LL(0xd96ca391,0x9da09642),LL(0x9bf4305c,0xd6710536),LL(0xa0dfafae,0x40cc1adf),LL(0xa209699b,0xe27e32f8)}},
   {{LL(0x33836d4b,0x076d902a),LL(0x24afb557,0xec6c5c43),LL(0xa0516a0f,0xa0fe2d1c),LL(0x00d22ecc,0x6fb8d737),LL(0xdaf1d7b3,0xf1de9077),LL(0xd4c0c1eb,0xe4695f77)}, {LL(0xb4375573,0x5f0fd8a8),LL(0x5e50944f,0x76238359),LL(0x635cd76f,0x65ea2f28),LL(0x25fde7b0,0x08547769),LL(0x51944304,0xb2345a2e),LL(0xa16c980d,0x86efa2f7)}},
   {{LL(0x36e87d82,0xeaaddeb8),LL(0x1ffd7210,0xc12587a7),LL(0x731f6838,0xf93d2f5c),LL(0xf7097a65,0xb96594e8),LL(0xb016e8d3,0x08d6717a),LL(0x1984d825,0x9c378de8)}, {LL(0xcb2a0c26,0x627d41e7),LL(0xc697ceb1,0x1f447501),LL(0xc760550b,0x8dc40831),LL(0x7fac97b0,0x70ad4870),LL(0x7021c170,0x5ac7f22e),LL(0x929d5931,0xa6f730e4)}},
   {{LL(0xbf4d1d63,0x4ccbe2d0),LL(0x397366d5,0x32e33401),LL(0x71bda2ce,0xc83afdde),LL(0x478ed9e6,0x8dace2ac),LL(0x763fdd9e,0x3ac6a559),LL(0xb398558f,0x0ffdb04c)}, {LL(0xafb9d6b8,0x6c1b99b2),LL(0x27f815dd,0x572ba39c),LL(0x0dbcf842,0x9de73ee7),LL(0x29267b88,0x2a3ed589),LL(0x15ebbbb3,0xd46a7fd3),LL(0xe29400c7,0xd1d01863)}},
   {{LL(0xd186cb09,0x7d27d71f),LL(0x3bc213c7,0x67cb7f4e),LL(0x6075b2cf,0x418cafeb),LL(0xd93a06f7,0xc0d691e6),LL(0x9dd001b9,0xc16a9525),LL(0x026f17b9,0xa0583230)}, {LL(0x7845900b,0x4c1041b0),LL(0x47a22aae,0x28740791),LL(0x2c1758e9,0x8d08efd6),LL(0xe6c3229a,0x9cc6f207),LL(0x082d8924,0xec69e902),LL(0xf331dfe7,0x9cfa1dea)}},
},
/* digit=5 base_pwr=2^25 */
{
   {{LL(0xabe7c60e,0x8ec4335f),LL(0x0a6a9fb5,0x01f198c1),LL(0x01141ab6,0x3ff96de0),LL(0x2eca98a1,0xb21acc2c),LL(0x10fdf648,0x61548490),LL(0xd1403e8b,0x2c01a99c)}, {LL(0x6fa509d7,0xf1a35f30),LL(0xe3f08e9f,0xf7715fe3),LL(0x7fc9a752,0x89c26c07),LL(0x420d48a0,0x8d2535fe),LL(0x52fe2e73,0x80ec5ddd),LL(0x71704f39,0x041b8df0)}},
   {{LL(0xc65fafce,0x498808ed),LL(0x4e806bc3,0x3676a7cc),LL(0x76c6b964,0x796e25f1),LL(0xac474261,0x1aced64b),LL(0x29a460c1,0xa62470fc),LL(0x5e751e48,0x77501dce)}, {LL(0x6d9e3641,0xcc00053b),LL(0x9a3f5a0b,0x2b5bc4ae),LL(0x3f9ca178,0xddaccc2e),LL(0x0b80d1b0,0xad33f34a),LL(0x64642225,0x6a76df93),LL(0x778e761b,0xc145f36f)}},
   {{LL(0xc10b43ae,0xdffb03ae),LL(0x9433a54b,0x39b1266e),LL(0xb19fe0db,0x4c262521),LL(0x3d5c7fee,0x0ec1e54f),LL(0x05e68e1e,0x2856510b),LL(0xdc80b8a6,0x49382c1e)}, {LL(0x2471bdd5,0x80a50931),LL(0x81974aa9,0xe8cde185),LL(0x28235c52,0xca6112ee),LL(0x301f9653,0xd28a0eb8),LL(0xe11fcdf8,0x22b11e26),LL(0xe4d735f3,0x97e6fc5d)}},
   {{LL(0xecc5f6da,0x9c66a32b),LL(0x1719ba2c,0xe4ff4043),LL(0x21e716ef,0x8c6cfab7),LL(0x96ed74e6,0x32c8fccb),LL(0x0b110c83,0x475890dd),LL(0x5cb4eefe,0xdfada95f)}, {LL(0x93240fe1,0x9d7b89a6),LL(0x210b776f,0x6afdb2d0),LL(0xca7a7d52,0xc3f0b55b),LL(0x55d04585,0xa6e56a06),LL(0x4257acc5,0x818e221c),LL(0xfcb8d39e,0x05207b63)}},
   {{LL(0xac993097,0x00fc2cfe),LL(0x76b4508c,0x4d3532b1),LL(0xa24c9add,0x0887cb17),LL(0x1a9972dd,0x2698a3e0),LL(0x26352346,0xa6f0258b),LL(0x3132792c,0xf6791599)}, {LL(0x92f39495,0x2c3d2ebb),LL(0x3f9e744c,0xf8405894),LL(0x54833968,0xba220301),LL(0xb7936d27,0xa0c5454e),LL(0xb0ca30ec,0xdd9078e1),LL(0x5a80ec29,0xfa5e78ae)}},
   {{LL(0xe68b9c0c,0xc948333b),LL(0x265cb56b,0x96e30042),LL(0x3c12bd22,0x957516bd),LL(0x0033e495,0x0d328473),LL(0xc9e2117d,0x23ff9704),LL(0xe3d28a72,0x81693f98)}, {LL(0xf8066b56,0x7a0ba6c9),LL(0x332bdbe8,0x0e2e2242),LL(0xe9f4ce83,0x885d3c00),LL(0x8ae96eae,0x3145e5ec),LL(0x91e3d14c,0x8d4b26d4),LL(0xa9ad6fe2,0x17b5782a)}},
   {{LL(0xcabd9b94,0x39062094),LL(0xc68dc31e,0x20b3910b),LL(0x0b708450,0x7d78a043),LL(0x522bd0d5,0xe113da29),LL(0xf7f27be8,0xc46fa10a),LL(0x295393a0,0x8258b2ce)}, {LL(0x28d958d8,0xf69dc622),LL(0x991261e9,0x97b796c0),LL(0x36482f74,0xaedea55e),LL(0xeec962cd,0x6e4f2e3c),LL(0x2a06b626,0x782f495a),LL(0xfe23dd4b,0x41e32a4d)}},
   {{LL(0xd8ada6cc,0x0a74da82),LL(0xbea55457,0xc6b98a3c),LL(0x57c2f5ac,0x896c26bb),LL(0x845d45e4,0x981e2f72),LL(0x7e9a7d36,0xca152b87),LL(0x7b582e8f,0x49666d45)}, {LL(0x49fc20b9,0xea3b9bda),LL(0x7c71f153,0x5bcbc84a),LL(0x5748a654,0xd346fc5d),LL(0x622665ee,0x7ac2f217),LL(0xb6f16e27,0xbb5efe7f),LL(0x644c9dc8,0xb1810a70)}},
   {{LL(0x2e4b40f5,0xe0c7946c),LL(0x7dd0f6de,0x7ef21fe5),LL(0xf81f21f3,0xc2a8cdd8),LL(0xc7ebe7a3,0x6ad73e42),LL(0x5d42a7be,0x3e8099c3),LL(0x6df16d22,0x151c6de8)}, {LL(0x5293c104,0x214a649b),LL(0x56b4d1d3,0x769763d1),LL(0xe3bebab4,0x78b33355),LL(0xc78a804e,0xe10a70ce),LL(0x211f735e,0x985a16fe),LL(0x56605fc0,0xfd50c868)}},
   {{LL(0x8f646cdb,0x16bcf31c),LL(0x059f65cc,0xe9b4b8a9),LL(0xf8971dbd,0x34a47964),LL(0xcfe021f8,0xd76f597f),LL(0x573891a6,0x23bab19c),LL(0x2645a244,0xdf670285)}, {LL(0x4347674f,0xcc338b2c),LL(0x4c07ab3e,0x1ddd83da),LL(0x31bf757d,0xee6bb102),LL(0x032ec673,0x3f2828c5),LL(0x0cd7c8f8,0xcf0daf5f),LL(0x56eba0e0,0x1b5d1126)}},
   {{LL(0xdcf6b460,0xd9b864d9),LL(0x5141107c,0xf453164e),LL(0xb019575e,0xac36ccb9),LL(0xa0040dea,0x996c97bb),LL(0x86b7899e,0xf684e539),LL(0x6b00d106,0xb11bb533)}, {LL(0xbe831360,0x86af64aa),LL(0xb21650cc,0xbda68250),LL(0xffd7b0c6,0xc080ad14),LL(0x1a3844b9,0xa7cd582f),LL(0x4531ed01,0x3656ad08),LL(0x0af8481d,0xe12b761e)}},
   {{LL(0x8532b8e4,0x94a35f64),LL(0x2227ce87,0xc2cf81f8),LL(0x7f8ce7c5,0x2c0a163f),LL(0xba5e1907,0xe9de5d0f),LL(0xc1c14999,0x0a58d572),LL(0xd934e441,0x406a3ecc)}, {LL(0xc315d861,0xf9ef9a75),LL(0x864a28bd,0xa18abeae),LL(0x60831100,0x9e65f3c3),LL(0x2d4749cd,0xf757107b),LL(0xe8d782bb,0x8fec38eb),LL(0xbe94b2ae,0x74b4d444)}},
   {{LL(0x59480d9b,0x796fc73b),LL(0xe0555912,0x9f33407a),LL(0xbcd9a4bb,0xbc428d52),LL(0x82ac0b57,0xffe3b15b),LL(0xab58a714,0x0f4cbd0a),LL(0x353b82bf,0xe3b57ece)}, {LL(0xf77aa72a,0x67e16b95),LL(0x112a81cc,0x7b5d3eda),LL(0x8d749470,0x6795b775),LL(0xfe097d37,0x334bb4fc),LL(0x64deebc3,0x0489c0ce),LL(0x85092a86,0x0cf28e29)}},
   {{LL(0x29843096,0x0cc8d80b),LL(0x6aaf8c7c,0x1f5107d7),LL(0xd1647601,0x6fb26e1a),LL(0x2e2ca8c0,0x66da08d5),LL(0x34d9e92a,0xac0a6fbb),LL(0x52203763,0x596cc78b)}, {LL(0x8e88a471,0x145521ab),LL(0x64bde8f6,0x9bdbd482),LL(0x01b4d2ea,0x5a4d25bd),LL(0x6e7152ae,0xef1fc362),LL(0xfba5462d,0xd188389a),LL(0xf01d497c,0x951d4667)}},
   {{LL(0xbcc5466f,0x294045fb),LL(0x039461a1,0x145c2078),LL(0x2a620ace,0x21c890c8),LL(0x896d0458,0x1ecc5928),LL(0xd3f9fb46,0x14130b9c),LL(0x461506dc,0xf001b3c2)}, {LL(0x97a1b5c2,0xd90a95e2),LL(0xdf6487be,0x9e37948e),LL(0x3b165a95,0x00ce787c),LL(0xe49c468d,0x10d4f2a6),LL(0xacd25333,0xf0f26eb8),LL(0x63f3570b,0x25b513b0)}},
   {{LL(0x9bc5917b,0x98d08981),LL(0x187fac5d,0x9f90885d),LL(0x7cfc13db,0x651b1828),LL(0x8655a658,0x2d606e4c),LL(0x63c91b71,0xba64d3c5),LL(0xb82a5090,0x36c7d7d8)}, {LL(0xcab1d598,0x2d1dff02),LL(0xbe78f90d,0xa95788d7),LL(0x0ea1fe01,0x1ac2ee6b),LL(0xd5c1273c,0xc100b60c),LL(0xeae603e7,0x4496084c),LL(0x77c2fdfb,0x7fcaaf5f)}},
},
/* digit=6 base_pwr=2^30 */
{
   {{LL(0x6abdd842,0xe8e3225a),LL(0x3b367b02,0x8c85f18f),LL(0x9f42edb9,0xf147a421),LL(0x0d411d4d,0x6d4bc00d),LL(0x70014bb4,0xa1a13a27),LL(0xfa10166f,0xb896d97b)}, {LL(0x0c302c6d,0xb2a1dfa7),LL(0x808a63a4,0x0a24bd5d),LL(0xf88c7359,0x8409a3a2),LL(0x347726a0,0x071f3838),LL(0x27507bb9,0xd18a551c),LL(0xb359b167,0xe0c4cc34)}},
   {{LL(0x80e9d9f3,0xa44f6ef8),LL(0x1d14d618,0xaa7621e9),LL(0x0eaf6671,0xcb0e4ed8),LL(0x181514a2,0x2bf485f8),LL(0x0a2927ea,0x74670e18),LL(0x12c14645,0xe1b54616)}, {LL(0x2a67ed61,0x4068c074),LL(0x739063ca,0xd10c7a57),LL(0x698b2816,0x391b651d),LL(0x6da14fa8,0xf310d169),LL(0xd8a578b1,0xa089be6b),LL(0x44389ad7,0xa314b3a8)}},
   {{LL(0x68ed0760,0x664ff053),LL(0x38fae9fe,0xc3cdc991),LL(0xf1f30a86,0x3fe057aa),LL(0x2d08c72a,0xbae99022),LL(0x6f09e13f,0x4f5faf3f),LL(0x13d26b29,0x44461a44)}, {LL(0xf95418ed,0xc2504c1b),LL(0xdb3ff26c,0x12766ea7),LL(0x07a22399,0x2f956e95),LL(0x5a00cdd3,0x2716e70f),LL(0x0e9fba99,0x80c02014),LL(0x0519875d,0xbe587ac3)}},
   {{LL(0x70128295,0xba86aec1),LL(0xc12f35ce,0x83a09b65),LL(0x89df2f80,0x8978ff07),LL(0x97a773d5,0x85750cfd),LL(0xfc3f35f6,0x806bb730),LL(0xfed868c9,0x04503422)}, {LL(0x86ffdbae,0xdc0fcde0),LL(0x1860f43b,0x8f4297e1),LL(0x8d3ad6cd,0xfefb7d02),LL(0x97293550,0x5c652b59),LL(0xed5cfbba,0x32e12942),LL(0x98800d22,0x06192aaf)}},
   {{LL(0x3c9eb0ca,0xccf0fb42),LL(0x4aa03b40,0x8703c669),LL(0x4001af07,0x44c735a7),LL(0x2e874ed1,0x9616dd93),LL(0x474ba621,0x5c2e8520),LL(0xfa93d8b4,0xddf13cd3)}, {LL(0x75df1b67,0xd68c9b45),LL(0x8f80d389,0x4cd24228),LL(0xc09f47fc,0x0f1a16bc),LL(0x9cd4842a,0xc414dc6a),LL(0x1f353c6d,0xbb0fa94f),LL(0x1950d073,0x40512455)}},
   {{LL(0x4d2edd5f,0x935b9e24),LL(0xefb287b8,0xfeb46fb5),LL(0xf5018b92,0xa51700a3),LL(0x23864e2a,0xc328beba),LL(0x995f70c1,0x113b5c9a),LL(0xda1b5d51,0xc0b11c22)}, {LL(0xf4a360cb,0x9b99b907),LL(0xadf0b094,0xf4ee9995),LL(0xf94b3f0e,0xf67c7cf2),LL(0xdcaf10cc,0x664a51a1),LL(0xe937a669,0xa3709ccc),LL(0x4862f098,0xea97bace)}},
   {{LL(0xc942a3f5,0x51758334),LL(0x32182ba6,0x7cc01e88),LL(0x74de4fe6,0x772af257),LL(0xe9667bf8,0xb1b3c448),LL(0x8079caf6,0x71cb2738),LL(0x1d823a40,0x48890c64)}, {LL(0x0e9edbda,0x47a5887b),LL(0xbe089e5a,0x916dfb0c),LL(0x1eb42ddf,0x3185090e),LL(0xb7f3af26,0x3c7eaa13),LL(0x9e9963b2,0x940ed8c7),LL(0x3426ac10,0xd85e77db)}},
   {{LL(0x2efffd95,0x162cdf34),LL(0xd59086e8,0x92111fda),LL(0x454eb977,0x4478d114),LL(0xdea38a67,0x8ce403d8),LL(0x7435728a,0xd459633b),LL(0xa63b0504,0x3a7be4e3)}, {LL(0x335dba3e,0x0c74066b),LL(0xc6ea6ee5,0x4e8fb1d7),LL(0xa99690ed,0x3398b588),LL(0x3ad77562,0x4949517c),LL(0xcbbb60ee,0xf9824f09),LL(0x85660bec,0x9fdcafdf)}},
   {{LL(0x8875d9e8,0xf26bd861),LL(0xbea9c273,0x22e2380d),LL(0x91995508,0x5f151837),LL(0x648aa1c6,0xb97f40a6),LL(0x3977d848,0x7478f5f8),LL(0x35b57de6,0x21e876ae)}, {LL(0xa93fc7f6,0xf620b180),LL(0x1b148996,0xf49bd07e),LL(0x1c4f60e1,0xfb085726),LL(0x7ad6b84d,0x6a6653af),LL(0x2e05b686,0x913a2d02),LL(0x407dda9a,0x94746629)}},
   {{LL(0xda684157,0xb8465d16),LL(0xb238faae,0xdadde1ab),LL(0xc6b9bea8,0xe2cd45e7),LL(0x5cf413d5,0x7251d4a1),LL(0xaae1765b,0x615cea8b),LL(0x13f36885,0x75aa8318)}, {LL(0xb8767cc1,0x7d5b0bf7),LL(0x8022968c,0xec38a8ff),LL(0x2a07faeb,0x034805b6),LL(0x33b7321e,0x916f9eb0),LL(0xc0c577ce,0x34963633),LL(0xabb8d3ce,0x8ee07efd)}},
   {{LL(0xa8a29345,0xa0bd6c0d),LL(0x5d7f5ef9,0xc676b6c5),LL(0x20ad7259,0x303b6d7c),LL(0xd8fe09a7,0x06542a19),LL(0xa959014a,0x5a06653c),LL(0x5bcfe0cb,0xf45fd79a)}, {LL(0x4e583468,0x29058d98),LL(0x0cd7afc0,0xf1bd25e6),LL(0xf7dbe54c,0x2a88246e),LL(0x35e0ef3d,0x680eaff8),LL(0x726e59b9,0x5942c97f),LL(0x8d5c0825,0x43e97139)}},
   {{LL(0xa406d4d7,0x2daddb11),LL(0xa2a33d81,0xb02b5da5),LL(0x21a6aa89,0xb73ce827),LL(0x467506de,0x10919587),LL(0x428d8daa,0x0927724c),LL(0x7c17adfd,0x0ede991f)}, {LL(0xbf7ddb3d,0x8518dab1),LL(0x2a54e1b8,0x04b091c4),LL(0x89e7a398,0x5943c37f),LL(0xe273f6f3,0x8e63f5e8),LL(0x83143d22,0xc6d0352b),LL(0xebd1628e,0x30e43182)}},
   {{LL(0x9991167e,0x606658a9),LL(0x72c4b43d,0xb8773e15),LL(0xe025abce,0x6cb364cd),LL(0x0c5a653e,0xafa58e9b),LL(0x134a68bf,0xa7e35a54),LL(0xba4d9db6,0xcb831d42)}, {LL(0xde83ef97,0xae37348e),LL(0x62ddd553,0x4ac64a6a),LL(0x715bb6b4,0x5feb5e0d),LL(0x043424b2,0xf876efae),LL(0xad91a9ef,0x7b56a291),LL(0x356f3ade,0x817c7053)}},
   {{LL(0x8dff7dff,0x3d6e8d6d),LL(0xd5be4ad1,0x6b6c194a),LL(0xb6fcd08b,0x57b93f2d),LL(0xf3761f23,0x99f09948),LL(0xac8b018f,0x4062f3d6),LL(0xa27af72c,0x4b58ac05)}, {LL(0x04d0cdfd,0x4abcc815),LL(0xbda4b02f,0xa50043e0),LL(0x27a9c083,0xe11297e5),LL(0x9779c5b3,0x2b2d8d52),LL(0xdfdecfed,0x3de3d330),LL(0xae7fc522,0xfe2487ca)}},
   {{LL(0x8f7a83e5,0x1c2fc508),LL(0xb9970c92,0xa7c56233),LL(0x8bafa66f,0x949c7173),LL(0x5bbb0490,0x1e299b2d),LL(0x18fcb9e8,0xb9a79e7c),LL(0x9cb5cc50,0xe6372ce6)}, {LL(0xf465c6aa,0x114fc628),LL(0x8cb797f6,0xc5539520),LL(0xa73ad211,0x7df94ed7),LL(0x8e0cd008,0x41eb8e1f),LL(0x004cbb0d,0xb028725a),LL(0x372c1656,0x1340186d)}},
   {{LL(0x978029bb,0x4074ee27),LL(0xbae0d0c0,0xa9394bda),LL(0x72cecb4b,0xaa01d539),LL(0x9a7dd9c4,0x4b0cf127),LL(0x5bc787cf,0x3e3e3f16),LL(0x942de53f,0xdf48f7e1)}, {LL(0x567b9d0e,0x0cc69719),LL(0x8d0d2750,0x631e3315),LL(0x92314a09,0x9fedc1e2),LL(0x14a1adcb,0x7547d226),LL(0x8662b86a,0x405561a4),LL(0xf5480b7d,0x149fa2b1)}},
},
/* digit=7 base_pwr=2^35 */
{
   {{LL(0xbda4aaa7,0x923d0b44),LL(0xfee29f7b,0xced14ce4),LL(0x9cf5b87d,0x1656be00),LL(0x1d61103d,0x13a37d0d),LL(0xfb652393,0x1d705880),LL(0xed712ed8,0x870a31bb)}, {LL(0xad7c21e3,0x15ad02e6),LL(0xc36c2831,0xf004e447),LL(0xba2b3ffd,0x56aa376c),LL(0x9745443c,0xc3be2b2f),LL(0xeb903660,0x47c8a870),LL(0x6c6c192d,0x976c303e)}},
   {{LL(0xf4fb80d4,0x148bd39c),LL(0xfff04e65,0x469b208c),LL(0xce548415,0xf397fbe2),LL(0x87fdde9f,0x441e5c2c),LL(0xfee9c179,0x6366b49f),LL(0x2938dc71,0x38d02bd3)}, {LL(0xc49c5444,0x26d450fa),LL(0x2b23d3d7,0x4569f95d),LL(0x298fd876,0x5f68bf4d),LL(0x544768b6,0xe86df047),LL(0xf8491267,0x40b69a32),LL(0xf917c71a,0xcbf3adf9)}},
   {{LL(0x8125489c,0x32498d4d),LL(0xa5a46ae0,0x965e8d07),LL(0xe96a7e29,0x6cea5e47),LL(0x668039ff,0xf78293a4),LL(0xf63edd32,0x62548a96),LL(0xa83e8256,0xe8e6af95)}, {LL(0x0db6263b,0x76e60c3b),LL(0x21b3d668,0xa1ee4b06),LL(0x9e49b0b5,0xa17dbf8b),LL(0x7eb366fd,0x4b29ba12),LL(0xd29b565f,0x5e0ed781),LL(0x199b36f9,0x8cb50d53)}},
   {{LL(0x29aa3150,0xa66c7035),LL(0x479e61fc,0xd038a5ab),LL(0xb5ab5410,0xdee33e96),LL(0x7c57d123,0xd068929c),LL(0xf1d6ad37,0x0839a208),LL(0x123f8178,0x8f523dab)}, {LL(0xa67d3840,0xb3e5e524),LL(0x52eb59df,0x88bda75e),LL(0x389f2dd3,0x513a0ab7),LL(0x890bba6f,0x3197a145),LL(0x6f66bf09,0x61add75b),LL(0x4eef1722,0x5c9dfc15)}},
   {{LL(0x07769b1b,0x66dc2850),LL(0x4d71fac4,0xe07fb741),LL(0xc2abbe60,0x5ae688a6),LL(0xdcbfd296,0x08ae92fa),LL(0xb43044d1,0xbc291256),LL(0x9fcdf213,0x0e1d71ed)}, {LL(0x02485685,0xf0c5b281),LL(0xe3f68f42,0x5d3f9302),LL(0xffe4f036,0xbbbfac50),LL(0x74fdba44,0xb5b5f261),LL(0x0d746760,0x4ebe1d07),LL(0xc37f04b7,0xbb0f7812)}},
   {{LL(0x6df1199d,0x810b6ab3),LL(0xb4f293b7,0xc229308b),LL(0x89897750,0x3cf838dd),LL(0x7a336c9a,0x3e391e4e),LL(0x176f89c0,0x70148337),LL(0xbc4f1e22,0x54b15bac)}, {LL(0x2c0f2885,0x32b104f9),LL(0x67034f2a,0x2c39cefa),LL(0xbf178ac1,0xb8310437),LL(0xc99370d5,0x722299f5),LL(0x332b93a8,0x0a493cf0),LL(0xa420f719,0x00e0ab41)}},
   {{LL(0x4cab24da,0xf2592543),LL(0xe7c3b9c5,0x52be9bbc),LL(0x4660d1a0,0xaab7a8b6),LL(0x9a9600f4,0x09738b81),LL(0x37de9e3c,0x58f0c866),LL(0x5db31f4f,0x0aea5cc1)}, {LL(0x499868be,0xe480406f),LL(0xf6913a44,0x0d8fc7f0),LL(0x35f2e14e,0x72823644),LL(0x45e37a93,0xb147b310),LL(0xf15c1af7,0xb1e7aa5b),LL(0xb03e7713,0xa8685068)}},
   {{LL(0x21c34c2b,0x21feb7fc),LL(0xddb0140e,0xab6a553a),LL(0x24b04e6f,0x03a65576),LL(0x342cb0ad,0x2531f186),LL(0xa24f6426,0x088c4d54),LL(0x06a873ea,0x9a0ee15c)}, {LL(0xd33bc748,0xdbe0253f),LL(0x5db8ac9e,0xdad3339f),LL(0x73e65901,0xeaaf3681),LL(0xccbfa504,0x71f1fab2),LL(0x4b0e163e,0xb7b84522),LL(0x3c779f3b,0xe0fca837)}},
   {{LL(0x46baf373,0x710988eb),LL(0xb57d5018,0x8cceb935),LL(0xa45fdf17,0x1864603f),LL(0xef48e6d0,0x3dcaae73),LL(0x590322c5,0xadd9420b),LL(0x9b135f67,0x947783e3)}, {LL(0x8bf5049f,0xfde76368),LL(0x2caa4023,0xf00e4c18),LL(0xd355b3d6,0x4d3b0f23),LL(0x20d5799d,0x181fabcc),LL(0xab2ad0af,0x29499b40),LL(0xf9a938aa,0xf6e66328)}},
   {{LL(0xbcbe922d,0xcd7b3c42),LL(0x95dd1a5c,0x2fe02b3b),LL(0x24ef5c38,0xeb66bcbd),LL(0xe579c309,0x7edcc21c),LL(0x16f6c900,0x7b19d491),LL(0xb6317c2c,0x36019ecd)}, {LL(0x91d9001c,0x554ba553),LL(0x14f31e44,0xa5e30b98),LL(0xffda4032,0x3d1fe33b),LL(0x2306675c,0x5dfec478),LL(0x000c91e7,0xbe59305e),LL(0x25a6b879,0x3c4e52a3)}},
   {{LL(0xc5ea88ac,0x02fcc14e),LL(0x56d093b6,0xca29bb6d),LL(0x0e6fe94d,0x876aeda9),LL(0xd7225a9d,0xfa11a142),LL(0x3d03fed8,0xfea3ca05),LL(0xc54d5962,0x435854c6)}, {LL(0x54a6dfd5,0xd7707374),LL(0xa3e55d02,0xb8960017),LL(0x04d65c3e,0xd4015a0c),LL(0xe98a1204,0x397f93d1),LL(0x5f3ed850,0xb0efa2e5),LL(0x8a3ec67b,0x18f24469)}},
   {{LL(0xd62cd9f9,0xa35802f5),LL(0x4148436e,0x0ca9c15d),LL(0x472b9d21,0x261a991d),LL(0xa2f8e875,0xd81a1ed6),LL(0x699b6d63,0x942f213a),LL(0x0ae57758,0x041a12fc)}, {LL(0xbd70aabb,0x61191c82),LL(0xee4c23b3,0x3776eb8b),LL(0x52511222,0xabe23e86),LL(0x30dabb91,0x66dd967d),LL(0x7ed27424,0x77650c59),LL(0xab25a050,0x08ea2ebd)}},
   {{LL(0xb6cb5a02,0xa410ba3a),LL(0xd07c5c6b,0x6eb40d15),LL(0x07dcc811,0x0de81e91),LL(0x2631b7af,0x996f46eb),LL(0x5b7a22f3,0x5a350ba7),LL(0x634159af,0xf42b24e7)}, {LL(0xc30952fd,0x07bae0ab),LL(0xd644e0b0,0x3488cda2),LL(0xe2111e12,0x23ae40d0),LL(0xc80cdb56,0x650af54e),LL(0x7d4aa2a8,0x0f33a30b),LL(0x442a00e8,0x4e8d3e98)}},
   {{LL(0x59a8bc95,0xa624ab37),LL(0x1c971228,0x4b7e3fa6),LL(0x73aa694b,0xe8229c42),LL(0x779288ab,0x0cc31029),LL(0x57575e0e,0xf8eff30f),LL(0x7d52803e,0xee5e0194)}, {LL(0x8a78f632,0x32d87e55),LL(0xe454904e,0x48a06031),LL(0x16c6e626,0xaa2cb8dd),LL(0x2c140452,0xadd098ac),LL(0x2d3031b1,0xd25f285d),LL(0x75b59543,0xfb5fbbe1)}},
   {{LL(0xd7a21503,0x2297041f),LL(0x657f03f0,0xfe7738c2),LL(0x168fa34a,0x994a8deb),LL(0xa53c4fdb,0x0c772e02),LL(0x50124cd3,0x67f835d1),LL(0x6993cbbe,0x0e0d2635)}, {LL(0x5257f11d,0x9857ed84),LL(0xac556942,0xdc23a728),LL(0xdeb32a7f,0xf0e1bb29),LL(0xee0d70f4,0xb8c3c43f),LL(0xc60ad214,0xc294b0ef),LL(0x679067ca,0xa4d438dc)}},
   {{LL(0x05c755e4,0x520b0bb9),LL(0xf89f0048,0xa2c2c59b),LL(0x73c23975,0x85c1c73a),LL(0x783aabba,0x6e4dec49),LL(0xb0463155,0x69f0c69b),LL(0x9c97b17b,0x61a42b94)}, {LL(0x45d331a3,0x55af24a9),LL(0xf5fe81fd,0x4b0e63f8),LL(0x708671c4,0x4034283d),LL(0x5fd9001a,0x200ddab3),LL(0x342eaf3b,0xe45f28e4),LL(0x1ba936c4,0x3e8375b4)}},
},
/* digit=8 base_pwr=2^40 */
{
   {{LL(0xec074aea,0xb083ba6a),LL(0x7f0b505b,0x46fac5ef),LL(0xfc82dc03,0x95367a21),LL(0x9d3679d8,0x227be26a),LL(0x7e9724c0,0xc70f6d6c),LL(0xf9ebec0f,0xcd68c757)}, {LL(0x8ff321b2,0x29dde03e),LL(0x031939dc,0xf84ad7bb),LL(0x0f602f4b,0xdaf590c9),LL(0x49722bc4,0x17c52888),LL(0x089b22b6,0xa8df99f0),LL(0xe59b9b90,0xc21bc5d4)}},
   {{LL(0x51c2bb65,0x6f577529),LL(0x4b874bdb,0x4a0c1c28),LL(0x78b96c6d,0x19a18427),LL(0x2f593505,0xa674f992),LL(0x6b7209d6,0x5abeeec4),LL(0x47cf5fff,0x42d15d01)}, {LL(0xb49e3b4e,0xe24509b7),LL(0x639ee6e8,0x81be939c),LL(0x5761e8e3,0x7f7daf59),LL(0xd420a288,0xed5cfcb8),LL(0x7a0ff696,0x365b29eb),LL(0x99a1ac8f,0x7d146805)}},
   {{LL(0x8a31973f,0x4936c6a0),LL(0x83b8c205,0x54d442fa),LL(0x5714f2c6,0x03aee8b4),LL(0x3f5ac25a,0x139bd692),LL(0xb5b33794,0x6a2e42ba),LL(0x3ff7bba9,0x50fa1164)}, {LL(0xf7e2c099,0xb61d8643),LL(0xbd5c6637,0x2366c993),LL(0x72eb77fa,0x62110e14),LL(0x3b99c635,0x3d5b96f1),LL(0xf674c9f2,0x956ecf64),LL(0xef2ba250,0xc56f7e51)}},
   {{LL(0xadcdaa68,0x9ee1ec3a),LL(0xdcbb6548,0xd98c498f),LL(0x88102ac0,0x32b97375),LL(0xc08527f4,0xdd296cf9),LL(0xfae3dfbe,0xb74f8145),LL(0x6cd7cc4f,0x84131eb9)}, {LL(0x927ff15b,0xa0f2fe7a),LL(0xeee1a4b4,0x6b0ade4d),LL(0x0eeb90a7,0x6e7df2d4),LL(0xbe4de684,0xe2f46e20),LL(0x3fdd06bc,0xcd28feba),LL(0xe6d6d9f6,0x8e4205ae)}},
   {{LL(0xff602c1b,0x246ffcb6),LL(0x6e1258e0,0x1e1a1d74),LL(0x250e6676,0xb4b43ae2),LL(0x924ce5fa,0x95c1b5f0),LL(0xebd8c776,0x2555795b),LL(0xacd9d9d0,0x4c1e03dc)}, {LL(0x9ce90c61,0xe1d74aa6),LL(0xa9c4b9f9,0xa88c0769),LL(0x95af56de,0xdf74df27),LL(0xb331b6f4,0x24b10c5f),LL(0x6559e137,0xb0a6df9a),LL(0xc06637f2,0x6acc1b8f)}},
   {{LL(0xf4a59550,0x9ad73168),LL(0x8409d7af,0x2a488d69),LL(0x30b3a5f4,0x9e946c2d),LL(0xc89723d5,0x7bc4ced7),LL(0xbd31f607,0x79f514a1),LL(0x3a274341,0x7d493f59)}, {LL(0xd45d55d1,0x7eb01027),LL(0x14735d1e,0x63f76909),LL(0x35be21cb,0x718d760f),LL(0x7c793331,0xba58160b),LL(0x8d30d29e,0xfbc0ce1f),LL(0xc6f4b03a,0x4645c0c2)}},
   {{LL(0x34b4e381,0xbd8c0868),LL(0x30dff271,0x278cacc7),LL(0x02459389,0x87ed12de),LL(0xdef840b6,0x3f7d98ff),LL(0x5f0b56e1,0x71eee0cb),LL(0xd8d9be87,0x462b5c9b)}, {LL(0x98094c0f,0xe6b50b5a),LL(0x508c67ce,0x26f3b274),LL(0x7cb1f992,0x418b1bd1),LL(0x4ff11827,0x607818ed),LL(0x9b042c63,0xe630d93a),LL(0x8c779ae3,0x38b9eff3)}},
   {{LL(0xf78d33ef,0x35d47426),LL(0x8440c42c,0x4af25db2),LL(0x2e91bf5e,0xbd6a15e2),LL(0xc08b6b1a,0xe366a84c),LL(0x55b97de8,0x759c122f),LL(0x08a03f29,0xecec558f)}, {LL(0xea9d2060,0xdcc9fca2),LL(0x9f361fe1,0xb3e49b8e),LL(0x9b59cd04,0xdeae3902),LL(0x6f5e5bd4,0xf532ede0),LL(0x36099f4d,0x84fbeeb9),LL(0x088d2052,0x73576b1f)}},
   {{LL(0x729c5431,0xe8767d36),LL(0xbb94642c,0xa8bd07c0),LL(0x58f2e5b2,0x0c11fc8e),LL(0x547533fe,0xd8912d48),LL(0x230d91fb,0xaae14f5e),LL(0x676dfba0,0xc122051a)}, {LL(0x5ea93078,0x9ed4501f),LL(0xbd4bee0a,0x2758515c),LL(0x94d21f52,0x97733c6c),LL(0x4ad306a2,0x139bcd6d),LL(0x298123cc,0x0aaecbdc),LL(0x1cb7c7c9,0x102b8a31)}},
   {{LL(0x139a9f83,0x2b85dd03),LL(0xf55cbff9,0x02740ec2),LL(0xdc3bd5f6,0xdca3616a),LL(0xa5da9b2c,0x52843de5),LL(0x1e22c1cb,0x7ee7890a),LL(0x4aee5c76,0xd5cf1511)}, {LL(0xc7234c24,0x86d70af0),LL(0x6b6a709f,0x1735378b),LL(0x423f32e5,0x9989ed21),LL(0xe739b1fd,0x4675a4ff),LL(0x304ce529,0x873edb62),LL(0x7f1db13e,0x60b6624a)}},
   {{LL(0xfaf46675,0x22a28e59),LL(0x10a31e7d,0x10757308),LL(0x2b4c2f4f,0xc7eeac84),LL(0xb5ef5184,0xba370148),LL(0x8732e055,0x4a5a2866),LL(0xb887c36f,0x14b8dcdc)}, {LL(0x433f093d,0xdba8c85c),LL(0x1c9a201c,0x73df549d),LL(0x70f927d8,0x69aa0d7b),LL(0xd7d2493a,0xfa3a8685),LL(0x0a7f4013,0x6f48a255),LL(0xdd393067,0xd20c8bf9)}},
   {{LL(0x6b512bc6,0xee43828c),LL(0x50b91e60,0xf73dc9f5),LL(0xf5dbde6b,0x68f23f30),LL(0xddd15e00,0xaf2fe9e3),LL(0x86578d49,0xfbf34dae),LL(0x6c130010,0x68979655)}, {LL(0x09942897,0x137a5fc3),LL(0x9959f06d,0xff1f0bfe),LL(0xbd7ee14b,0x2dd0a04a),LL(0xe54e2161,0x59c46072),LL(0xea7518ad,0xf470bdae),LL(0x40c471cd,0xce556e43)}},
   {{LL(0x81625e78,0x4ec874ea),LL(0x3fbe9267,0x8b8d8b5a),LL(0x9421ec2f,0xa3d9d164),LL(0x880ea295,0x490e92d9),LL(0xd8f3b6da,0x745d1edc),LL(0x8f18ba03,0x0116628b)}, {LL(0x834eadce,0x0ff6bce0),LL(0x000827f7,0x464697f2),LL(0x498d724e,0x08dccf84),LL(0x1e88304c,0x7896d365),LL(0x135e3622,0xe63ebcce),LL(0xdc007521,0xfb942e8e)}},
   {{LL(0x829ee0c7,0x6b97dae4),LL(0xf274bc7d,0xcaaad7e3),LL(0x512f7d86,0x87bee424),LL(0x3f7b0dde,0xb5a03e5b),LL(0xf9fca5a8,0xc29db9c1),LL(0xede64af8,0x538a6f8b)}, {LL(0x8b3c715f,0x0e518ba8),LL(0x3cb5b861,0xc1cb4c31),LL(0xc76dbfa0,0x5cd61604),LL(0xffdac22e,0xc557e50d),LL(0xf67b53e8,0x7892fc51),LL(0xea7c141c,0xa42f85c1)}},
   {{LL(0xa3688621,0xbb155a66),LL(0xf91b52a3,0xed2fd7cd),LL(0xea20cb88,0x52798f5d),LL(0x373f7dd8,0x069ce105),LL(0x8ca78f6b,0xf9392ec7),LL(0x6b335169,0xb3013e25)}, {LL(0x6b11715c,0x1d92f800),LL(0xff9dc464,0xadd4050e),LL(0x8465b84a,0x2ac22659),LL(0x465b2bd6,0x2729d646),LL(0xe4eff9dd,0x6202344a),LL(0xcd9b90b9,0x51f3198f)}},
   {{LL(0x22b25f2e,0x2ba8c790),LL(0x1af0f4a8,0xf4670a51),LL(0x2fc2451e,0x6842f36a),LL(0xbb91e1e3,0xfc5c9558),LL(0xc3ead762,0x035d1dfc),LL(0x031e5556,0x3d0721cb)}, {LL(0x3af18a2e,0x3af0cc81),LL(0xbd11a363,0x7888cee2),LL(0x6ade1d12,0x80c3de0a),LL(0x93b2dcb5,0xe8c3a5bd),LL(0x90a2214d,0xe3adbd7c),LL(0x1192948d,0xfe8646d5)}},
},
/* digit=9 base_pwr=2^45 */
{
   {{LL(0xba958101,0x65026297),LL(0x8eef151d,0xbeb4adf9),LL(0x60c8bbf7,0x623763a4),LL(0x8b2a7120,0xfa8f5ad7),LL(0x085497e7,0xfd744bdf),LL(0x2ba35618,0xf9b6f97e)}, {LL(0xf8a15e86,0x0cebfe9d),LL(0x29576088,0x47a6d013),LL(0xfcf19627,0x655817a3),LL(0xc2f11261,0x30ab44f7),LL(0xddf2c850,0xbb001c9d),LL(0x073260c3,0xb45c7eff)}},
   {{LL(0xe2319f38,0xfa7d1236),LL(0xa551d3fe,0x9ba1a1c0),LL(0xbeb1282b,0x9ea27288),LL(0x07fee8a9,0x1c069efa),LL(0x5870fee9,0x5749c7b5),LL(0xafcec6fa,0xbedca76f)}, {LL(0x4c63c5e2,0xa3f8f1b1),LL(0x94758ac3,0xaa1bb156),LL(0xb59dc06e,0x753329a9),LL(0x98a92c38,0xfa8e5f5b),LL(0x3c2b4662,0x6b6f46fd),LL(0xec04c6c6,0x716f41a1)}},
   {{LL(0x59b11c7a,0x0430af77),LL(0xd4f47aca,0xf71cc5b1),LL(0x12e9190f,0xe1a7905f),LL(0x12db9e14,0x1c689b70),LL(0x0abaeeac,0x6bdd3dc9),LL(0x504f0319,0x97f1c244)}, {LL(0xa7a54b51,0x874afd61),LL(0xe3d979b8,0xd4604ecb),LL(0xebf4aab1,0x0d33eee1),LL(0x1aa49fe6,0xa3631cac),LL(0xf2217cfd,0x0d8340fb),LL(0x423b7e77,0xf6373284)}},
   {{LL(0x9563e3bb,0x78a53b5a),LL(0x86af355c,0x19c75eb2),LL(0x019a6f8e,0x3520f427),LL(0xdc3ad0ba,0xde6fcad6),LL(0x79745b7c,0xfec96e4f),LL(0xb133f2dc,0x5e566bbd)}, {LL(0x26561be7,0x50088a2b),LL(0xc5fddfc7,0x16275b4c),LL(0x23ae4b9d,0xf21332ff),LL(0x85246712,0x8cbc659e),LL(0xf50b515e,0x27fa9c8d),LL(0x494ac8b7,0x25ecf745)}},
   {{LL(0x641e3bd1,0x437b7224),LL(0x80a58460,0x84e39f79),LL(0x09759523,0x68e52927),LL(0xe77f5904,0x0176a3ac),LL(0xe151e242,0xde92fb15),LL(0xeb1438d4,0x79965c9a)}, {LL(0x596700b9,0x318a810a),LL(0xc2198cbe,0xa8a6ec57),LL(0xbf030fd2,0xd7709aaa),LL(0x72f5d326,0xb4320234),LL(0xb03bce50,0xc9945214),LL(0xd4ecba09,0x0bc06d9b)}},
   {{LL(0x96fffb94,0xa1972c21),LL(0x99d7633b,0xbe040930),LL(0x7e23d66e,0xb116ff40),LL(0x949a19f1,0xcb12b2bb),LL(0x79e49e91,0x75df10ee),LL(0x4890bcf4,0xa3bf9076)}, {LL(0x09a30252,0xcbaa76a6),LL(0x0ee5728e,0x17c224a9),LL(0xf4f3f4cb,0xcbc56e5c),LL(0x4fe868a5,0x8a07110f),LL(0x25e110a2,0x23289f21),LL(0xd7693c45,0x0289c12b)}},
   {{LL(0xf184c91f,0x06ba6db1),LL(0x45fd0382,0x3c0a348a),LL(0x4434b527,0x0d535b6e),LL(0x692bae0b,0x7bbfa2c8),LL(0xbe7fe51c,0x5c59a08e),LL(0x36e80cb8,0xbaa7d2be)}, {LL(0xbed3cae8,0x8a42d8d1),LL(0x15ff4962,0xd9e0bc0d),LL(0x644c75ff,0xe51fce93),LL(0xb9392d63,0x40222561),LL(0x023b4787,0x8ab1d286),LL(0xa1b3190d,0xfa85c220)}},
   {{LL(0x2b0c535b,0x29864753),LL(0x70506296,0x90dd6953),LL(0x216ab9ac,0x038cd6b4),LL(0xbe12d76a,0x3df9b7b7),LL(0x5f347bdb,0x13f4d978),LL(0x13e94489,0x222c5c9c)}, {LL(0x2680dc64,0x5f8e796f),LL(0x58352417,0x120e7cb7),LL(0xd10740b8,0x254b5d8a),LL(0x5337dee6,0xc38b8efb),LL(0x94f02247,0xf688c2e1),LL(0x6c25bc4c,0x7b5c75f3)}},
   {{LL(0x938e2480,0x1213a2fc),LL(0xed7b85a1,0x2dff0ea8),LL(0x0cc9619f,0x4f280c74),LL(0x66678817,0x3a215bc5),LL(0x327985a8,0xa51602ca),LL(0x10f02615,0xd6916920)}, {LL(0x3dc4f2b6,0xd01274aa),LL(0x3ff804d1,0x40feccd7),LL(0xa2100fa4,0xe508a27c),LL(0xd90cc776,0x37cc757e),LL(0xb74827c4,0xaec4884b),LL(0x4ac34a27,0x0c904582)}},
   {{LL(0xf761baaa,0x6ad3f9d3),LL(0xa5ae6088,0x3936d0d9),LL(0x83a0dd41,0x54d5bff0),LL(0x70cfa7fe,0x418c4b20),LL(0x892317fc,0xdc4d4742),LL(0x265359a1,0x1248e2c0)}, {LL(0x2120ad82,0xa14ee83c),LL(0x89260c26,0x048b7b27),LL(0xb5fe2775,0x637e71d5),LL(0xf3e08f62,0x849213d5),LL(0x5a9faeb1,0x8a8a2da8),LL(0xef96612f,0x57047761)}},
   {{LL(0x027a3ba4,0x7608efa7),LL(0xcf450422,0x14cb67ad),LL(0x5330ef70,0xc55a7aae),LL(0x64901358,0x3f18a78d),LL(0xfeb668df,0x264d3a5c),LL(0xbcf7c23b,0xdc0569c0)}, {LL(0x6ae18907,0x93af8a9b),LL(0x24b8e961,0x55dd3e3a),LL(0x36171692,0x1c3fcd34),LL(0x2f1f7ca5,0xfc0ab80c),LL(0x564878a7,0x0dadc166),LL(0x000d81f3,0x7d20d34b)}},
   {{LL(0xb19231b4,0x0e8d03f8),LL(0xdcda852b,0xe4498dbd),LL(0x31a8dddb,0x9963a6bb),LL(0xd949c6cf,0x80bfba33),LL(0x1b6e1ab7,0x72e48395),LL(0x82d7301d,0x35a2a802)}, {LL(0x01edd784,0xcfce9b91),LL(0xcd9bcb9c,0x794ab50e),LL(0xf465c3ca,0xe1f90e0c),LL(0x902b1b57,0xb7ded4b5),LL(0x8a8c27b4,0xe593dafc),LL(0x6f940ca0,0xa80faad4)}},
   {{LL(0x421e65d3,0x5396fd32),LL(0x15e91d37,0x6c826fe2),LL(0x3cc3d9cb,0x14c80f4a),LL(0x43028c7f,0xdb760a13),LL(0xce1a3e42,0xd7a541bd),LL(0xb81193f1,0x4726a5b7)}, {LL(0xf004d745,0x5c96916d),LL(0xad031ec3,0xc86961c0),LL(0xb8d28a4f,0xb5a73c58),LL(0x7ed16163,0x9a7bcb0e),LL(0x2fbd18f5,0x3f6871f3),LL(0x42155e84,0xebef020a)}},
   {{LL(0x4cdd5d8c,0xce6c810c),LL(0x078c9bc1,0x9fc290a4),LL(0x96d47e78,0x84debffd),LL(0x143590fe,0xf291d301),LL(0x69111f3a,0x7cc5c36d),LL(0x46311370,0x3a2cdc42)}, {LL(0x9e6f14a7,0xd3d8bb98),LL(0x481e22db,0xfeb97d41),LL(0x462748fe,0xe1e20cfb),LL(0xa2eaf3c7,0xcf8f5dd0),LL(0x302a52d4,0xb4f9acd4),LL(0x8d715c96,0x1c154dcd)}},
   {{LL(0xb13ddc9c,0x5f492236),LL(0xbabe466c,0xc930fb0e),LL(0x5c017d52,0x21daa0b9),LL(0xea9a6c9b,0xc182a975),LL(0x4fd839af,0x45d3d6c0),LL(0x3d89b744,0x9e4ce10a)}, {LL(0x3a947f16,0x906af540),LL(0xa377f4f3,0xa0f8e1b7),LL(0xe343d7c1,0x268360cc),LL(0xbddbeb1a,0x66fe9eb7),LL(0xc3bafefc,0xf5c80368),LL(0x6664d04e,0x0e183ee0)}},
   {{LL(0xfda8520e,0x36c9dbbe),LL(0x6ae3ea98,0x573507ce),LL(0x96a8f9f1,0x1ab38db6),LL(0x6b01e6bc,0xe031d235),LL(0x8afc4ada,0x10466ae6),LL(0xed9c44e4,0x3b35df41)}, {LL(0xc7bd99e8,0x61272c12),LL(0x805afd79,0x6a4ae7b4),LL(0x0ecc49eb,0xf4c47a91),LL(0xcbe84d5c,0xeb95dfec),LL(0x8ee497d7,0x43f3b71c),LL(0x4c6fece4,0x2547af52)}},
},
/* digit=10 base_pwr=2^50 */
{
   {{LL(0xced45039,0xe323ed0c),LL(0xa90aa713,0x04ce0b67),LL(0xe8d68e4e,0x9c092f06),LL(0xd0742e5d,0xd8f5555a),LL(0x00d3df92,0xe2d175bf),LL(0x4f71aeab,0x8ca55f15)}, {LL(0x642d391d,0xd1762d72),LL(0xaec466bd,0x0dfdd3c2),LL(0x6281f2a7,0x2caacb4c),LL(0x3603e53a,0x635ba470),LL(0x49fecf29,0x94a9811d),LL(0x466bf361,0x3a42cf09)}},
   {{LL(0x11bccf2f,0x0f05710b),LL(0x7aec1bc6,0x7113085a),LL(0x46b8d0e2,0x137da67a),LL(0x698b78cc,0x454b89fc),LL(0x258a9393,0xf2a6e1de),LL(0x16488e69,0x5f1804e7)}, {LL(0x15b3bf35,0x7c6c5502),LL(0xb05c2ec1,0x3b0e09a5),LL(0x92f15247,0x4b9de30e),LL(0x27e70a0a,0x09d4ca93),LL(0x0d149363,0x9c8b1634),LL(0xce642137,0x54a8287c)}},
   {{LL(0x346a764e,0x49ecd6f3),LL(0x4105e657,0xe46847f1),LL(0x7550f608,0xce9cb2b5),LL(0xf4cf062e,0x45f1a1f7),LL(0x2c27d38a,0xcdb19a11),LL(0x84e50b19,0x36d375b2)}, {LL(0x0dba6405,0xf4369154),LL(0x040354dc,0x4c9dc863),LL(0xa24d09cf,0x7229e70e),LL(0x7cf6831b,0xe72aa86c),LL(0x25392838,0x487fb684),LL(0x430b9b47,0xe88bed04)}},
   {{LL(0x666fa8a8,0xf8a8499b),LL(0x71bba84a,0xd0f94015),LL(0x515e1328,0xb85e1b1d),LL(0xa941e788,0x88a2636b),LL(0x2b5dd8d8,0xa045241d),LL(0x332f0350,0x161be476)}, {LL(0xa18fac6b,0x96c4b205),LL(0x73fc5337,0x5cbe8d5e),LL(0xd00b6029,0x6fc33fc6),LL(0x89aa3b79,0x07a914ee),LL(0xa4d4dd00,0x35353eb7),LL(0x673e8956,0xc026bdc0)}},
   {{LL(0x67103852,0xccda3721),LL(0x0c54de53,0xf78d2247),LL(0xafa44aa5,0xebd16036),LL(0x64a24ab9,0x7b880248),LL(0x1c2bc78e,0x86b38e96),LL(0x8d63b295,0xd0aa0d05)}, {LL(0xc62fcbf1,0x24912955),LL(0xb9ac435c,0x77a68156),LL(0x1b360b26,0x432401c7),LL(0x4c58ef8c,0x091f19f3),LL(0x83d46c9d,0x3a4a61f4),LL(0xe8d616cd,0xad0e5c72)}},
   {{LL(0xcdaa6831,0x33b029bc),LL(0x4c1f9ced,0x2548552d),LL(0xdece1c8c,0x35f1a002),LL(0xacc23aa5,0xc6b87fd7),LL(0xbab029a4,0x0b8bb275),LL(0x30bfb42d,0xf07bc067)}, {LL(0x1f69ce9d,0x1688ff5d),LL(0xdb10585e,0xeedb7b5e),LL(0xd432c197,0xb7a88cf0),LL(0x015a350a,0x20731bdd),LL(0x63223f5c,0x5fa18354),LL(0x8024693f,0xe392e131)}},
   {{LL(0x0154731e,0xdfaeea0a),LL(0xe15a0388,0x9e53419d),LL(0x25a992c8,0x2ad6a83c),LL(0xe125501a,0xa2ba020f),LL(0xd4dd04dc,0x894ebaf8),LL(0x50765559,0xd48cb958)}, {LL(0x80dec92b,0xf9b58d09),LL(0x9da299d7,0x2a0e1165),LL(0xefe9cb11,0x3c081853),LL(0xa511c5e0,0xb9f3b702),LL(0x70486180,0xa8f7a25f),LL(0x591b3e2f,0xc0358b25)}},
   {{LL(0xeaddf274,0xf36e8398),LL(0x6a5e4ddb,0xe41553a1),LL(0x4efc5b0c,0x36ab0746),LL(0xd316c434,0xb211e59a),LL(0x16ccf839,0x2515ec9f),LL(0x03dc6a07,0x6ecb7465)}, {LL(0xc65c1b07,0x842b7275),LL(0x35750ab6,0xf7ceeec5),LL(0xcef5255d,0x967d711c),LL(0x5108cb92,0xcd3bfb07),LL(0xec1b9740,0xe50c0d8a),LL(0x1a9e6308,0x9e8d5661)}},
   {{LL(0xccb8a36a,0x84f8ea13),LL(0xa05709a2,0x5f7aeeff),LL(0x60574f37,0x4942d04e),LL(0xe048b400,0x855b13e9),LL(0xa6b59c09,0x747e4067),LL(0x074d3990,0xc349fb05)}, {LL(0xec2c7e03,0x398e6afa),LL(0xec2d5a4c,0xce361865),LL(0xb6f57d22,0xfc04bf8e),LL(0x759ce6c6,0xf0e0b84c),LL(0x5ee7e528,0xb6514123),LL(0xf5c0f9b9,0x8ca144bd)}},
   {{LL(0xf6a58536,0xc1370dae),LL(0xc56b0ae9,0x6f2e5b37),LL(0x92f6b6a3,0x5511d682),LL(0xae575249,0x2e9e5034),LL(0x1d14bed7,0x3e5a32f8),LL(0x75efd17a,0xa346a86f)}, {LL(0x0309fd7a,0x4f2510a6),LL(0xd0b1425e,0x689ecd74),LL(0x9f771e24,0x9e9bebe1),LL(0xadc5b48c,0x20188045),LL(0xb845230b,0xe49811b6),LL(0x5a8687f8,0x420855ec)}},
   {{LL(0xbae1b94d,0xe650e49a),LL(0xe3199794,0xb6b162e6),LL(0xb4ec0480,0xdc706859),LL(0x4b1a06ca,0x28b618c2),LL(0x403acdc2,0x0929a001),LL(0x2da3aefd,0x796dfd97)}, {LL(0xef4c1673,0x16389072),LL(0xfc94a4f5,0x600d8bd7),LL(0xe5f386a9,0xf003214d),LL(0xe62cbb48,0xa7af0499),LL(0xde82bad5,0x750a3b00),LL(0x8e7dc8ee,0x6c615b83)}},
   {{LL(0xd71543a6,0x283eec26),LL(0xa7627841,0x98fa08be),LL(0x27ad302d,0x269a83b8),LL(0xbde3fdd0,0x225f2f12),LL(0x0130b3a6,0x046fcf38),LL(0xc3ed9043,0xea733c1a)}, {LL(0x70aa08d1,0xf870f14d),LL(0x34391e0b,0x643d18b8),LL(0x847be772,0xf3e1d5f4),LL(0xd0ed73a0,0xa9498223),LL(0x14b3babb,0x6933ccf0),LL(0x37f08f70,0xc2439ae4)}},
   {{LL(0xf88d049b,0xb643f4e0),LL(0x12682fcf,0x5e0ac1fb),LL(0x9f981c8f,0xeaf7874d),LL(0xb1af779d,0x9c2adfd2),LL(0xdaa8c275,0x9a7abead),LL(0x24cacec4,0x09ad5521)}, {LL(0x0ead1646,0x069cd5c4),LL(0x0a6157d1,0x5186bf19),LL(0x96503506,0xcc222a93),LL(0xbd29686e,0xfeaa7bde),LL(0xa7257c8d,0xb0d65b0d),LL(0x98aa227f,0xc31c0a88)}},
   {{LL(0x5be1d45b,0xb4b7651f),LL(0x7f0cf680,0x0425200a),LL(0x8960be95,0x200d12b4),LL(0x4945b193,0x02fdd1a1),LL(0x27d046d8,0xedd70e3e),LL(0x83f14e12,0xc1cc086a)}, {LL(0x2629396e,0x1580e72b),LL(0xf9ed73c1,0xc87439db),LL(0xa90c5128,0x5debdf30),LL(0x9fbe14ef,0x0b6c020e),LL(0x0149a0b0,0x168da56a),LL(0x79c58ac6,0xc66a4dbd)}},
   {{LL(0x330b8e2e,0x800aec84),LL(0x335837bf,0x1a2c033e),LL(0xfe6f6dd9,0xf1a91551),LL(0x1de7360a,0x326c42b2),LL(0x7b66f9d5,0x300e740b),LL(0x68ce95d4,0x53bcc700)}, {LL(0x3d80f228,0xc9e225ac),LL(0x6977dfdb,0x64b2ad4e),LL(0x01f23221,0xac863b08),LL(0xdf11e5f7,0x0517a648),LL(0x68d11050,0xbf7aedcb),LL(0x77b3029c,0x2607e337)}},
   {{LL(0x206add5d,0x1afa6aa6),LL(0x150ea4c9,0x66cfbbae),LL(0x5d36da4f,0x07fb920b),LL(0x291e774f,0x144d51f9),LL(0xf40d87a8,0x26c2c134),LL(0xa932f1a0,0xc8cf3524)}, {LL(0x5aeb0bde,0x35bb2a42),LL(0xc4be960a,0x5cfcc1da),LL(0xaa1838ed,0x5c40cabf),LL(0xe2855f1f,0xea0c05ff),LL(0xfd525934,0x931ebb02),LL(0x16246fd4,0x31a7b78f)}},
},
/* digit=11 base_pwr=2^55 */
{
   {{LL(0x6d6ae962,0xcd92906c),LL(0x9807d881,0x62835615),LL(0x1fdc1915,0x0d692978),LL(0x269d611e,0x45d01a8c),LL(0x9665b00a,0xe7bd1e70),LL(0x9bcaa388,0x08638534)}, {LL(0x2dd24299,0x8f189e88),LL(0xb82fb270,0x5f643392),LL(0xc633b111,0xca65bf16),LL(0xd6f1dac8,0xc6adc9c9),LL(0xa3c3381d,0x0df2c293),LL(0x8388cd12,0xdd6ae97d)}},
   {{LL(0x3863db02,0xffdbd0eb),LL(0x2f57e10f,0x8b825683),LL(0x35e7a3a2,0xc11acead),LL(0x67833028,0x4998cf8c),LL(0x844c7976,0x8f3a346b),LL(0xdb9b1a1c,0x0a9d872c)}, {LL(0xb98d445d,0x8735dabc),LL(0x305fa0a9,0x93790d80),LL(0xd267a01a,0x7c0add49),LL(0xffa20d11,0x2b46c913),LL(0xd8ab2d4a,0xf2acef26),LL(0x3d926080,0x71b701b9)}},
   {{LL(0xfbf536a1,0x7cae584d),LL(0x5b52faa9,0x0af06fc2),LL(0x2827b872,0x807706dc),LL(0x59903742,0xf029478f),LL(0x85cd7aed,0x0e8c393c),LL(0xa331bee7,0x2e1e6ade)}, {LL(0xcdb7f97a,0xf5d13dad),LL(0x8577c3cc,0xdb22fed1),LL(0x81e39043,0x6f453180),LL(0xa00a806c,0x41d4bb25),LL(0xe28670a7,0xe6064b92),LL(0xa7c63f88,0x733a31c8)}},
   {{LL(0x5133de8e,0xe9d2a98a),LL(0xb81b8b00,0x37083b60),LL(0xceaf86ae,0xf399325d),LL(0x8f161525,0x03b17c88),LL(0x84211b9d,0xd8ac35c9),LL(0x9050ca48,0x22083784)}, {LL(0xc9fab832,0xa818c44b),LL(0xe5aea7da,0x8882bcce),LL(0xf8715b04,0x633aaf35),LL(0x9d8829a9,0x5463e1b9),LL(0x84a820f1,0xb18df52d),LL(0xd096675d,0x9d5ef891)}},
   {{LL(0xc2b47dd2,0x394174d0),LL(0x30a87997,0x06b117a8),LL(0x894731e5,0xcfd81d41),LL(0xc84e7d05,0x6476b1af),LL(0x743b8218,0x37069fe2),LL(0xa36173d8,0x93f21550)}, {LL(0xedadef86,0xa8ae70f4),LL(0x76366073,0xdac5d021),LL(0x1c982076,0x4063ba57),LL(0x2d24d61e,0xc3471b5f),LL(0xae4d820f,0x57b69c41),LL(0xc463a391,0xf73e14a4)}},
   {{LL(0xac60496d,0xd54e2c7c),LL(0x04cd50a4,0xc06d5e5d),LL(0xe60f7f59,0xcb4105e8),LL(0x427483ad,0x705db308),LL(0xf2bff383,0xf73ba98b),LL(0x0220e6e9,0xa945611a)}, {LL(0xd957e12b,0xc01c46b8),LL(0xacb1f371,0x458897b7),LL(0xfa3403e6,0xf738dc0b),LL(0xd2202896,0x098bc687),LL(0x5f882e5e,0xec0c217a),LL(0xa1f4eb13,0x8f25af77)}},
   {{LL(0x195736e9,0x86e6a8fc),LL(0x1a20620f,0x88099bf3),LL(0x9cdc1db9,0xf3d9998b),LL(0x47042834,0xf36d583e),LL(0x25a0d589,0xcd4519d8),LL(0x1e04a91d,0x78c435ab)}, {LL(0xe44d9e31,0x0428e76e),LL(0xce821aab,0x63994993),LL(0x2625cfce,0xd167ceb7),LL(0x0ddf2b89,0x9f242245),LL(0x717ffb5e,0x182e33d6),LL(0x26de205c,0x3800290e)}},
   {{LL(0x59ee4124,0x2615c782),LL(0x76532b4b,0x4dc2824c),LL(0x1c84a04b,0x9c3b1d77),LL(0xcb9f9e34,0xb6fc203f),LL(0xc64f7846,0xbed65464),LL(0xeb004248,0x04f520a2)}, {LL(0x4a58fd22,0x5c017727),LL(0xc10d9472,0x25958482),LL(0xaceb0e3a,0xb78c6666),LL(0xfc046f0a,0x18d3c188),LL(0x1baa9595,0x7f3e2f30),LL(0x8a2844e8,0xa574f8cd)}},
   {{LL(0x4bcc5a81,0x9fd2e524),LL(0xd3f2f80f,0xc73e2598),LL(0x15dc157c,0xff838e9c),LL(0xc522319e,0x0b2491d2),LL(0x909592d6,0x2b65f042),LL(0x849f5dee,0xf113045c)}, {LL(0x9b287c3d,0x84fe31be),LL(0x80097bd0,0xd0481262),LL(0x63660f87,0xaf203cba),LL(0x0ea001b3,0x5621c3d8),LL(0x346ea29b,0x125a5eaa),LL(0x3ca79649,0x3dcec22c)}},
   {{LL(0x81c2d81f,0xc899eba3),LL(0xf3f0a431,0xb27267d6),LL(0xda55568e,0x607c8629),LL(0x2b1dc1d9,0x6b547228),LL(0xc9c001ff,0x23232311),LL(0x488f8b85,0x207a2eb2)}, {LL(0xdac37a28,0x3867ac9a),LL(0x2584a5f0,0xa36d14d3),LL(0xa74488ff,0x7398c647),LL(0xbe087640,0xf6ed920f),LL(0x6319a571,0x72beddc7),LL(0x6a244aeb,0x55c2cd82)}},
   {{LL(0x9ede92f1,0x42df5630),LL(0x76503aa7,0xe6957bc4),LL(0xfbec9577,0xe7abfe12),LL(0x274b7f6c,0x9660e762),LL(0x35f12ea2,0xf78af22c),LL(0xb51bec2c,0x73ab1bbb)}, {LL(0x1641f424,0x5e8e40e6),LL(0x0905e4be,0x1af751d9),LL(0x1c033198,0xb5130ade),LL(0xa22f5f19,0x96d47ab2),LL(0x55340d4c,0x5aff3e66),LL(0x1bed65fa,0xab4aa5a7)}},
   {{LL(0x25f8a53b,0xb7da79c6),LL(0xd331ad8e,0x6b950bdf),LL(0x4aa36d18,0x3481b7b5),LL(0xed0e3091,0x6efeaf88),LL(0xc993074c,0xeb017bdd),LL(0x529dd654,0x8431a6d6)}, {LL(0xbd069585,0xf5177231),LL(0x3ce85096,0x6d753b10),LL(0x4ca26741,0x194d82d3),LL(0xadcd1650,0xeaeffe15),LL(0xaf7758b7,0x4dcec3d9),LL(0x4cc2c819,0xf5fdf666)}},
   {{LL(0xae850261,0x7e9ad6e8),LL(0x11960968,0xbeb25ca0),LL(0xad4cfa42,0x60c5f0ee),LL(0xeca5c508,0x81714b3f),LL(0xb78a290f,0x810e71d7),LL(0x3260114c,0x03536cec)}, {LL(0x6a16cdf6,0x24d29f0f),LL(0x4a4c964c,0xec45a58a),LL(0xea344b44,0xbaaa5fb3),LL(0xfeb783f6,0xf1b7a68c),LL(0x3ddd7062,0x533c4b72),LL(0x1598b5f7,0x6a33a0dc)}},
   {{LL(0xa9810744,0x94bdc7f4),LL(0xf045d859,0x464195da),LL(0xd654cb57,0x27e2dab0),LL(0x7a491956,0x1d4e1e53),LL(0x31d5d099,0xa1ef570c),LL(0x295f3de7,0x01cd21ee)}, {LL(0xb8249038,0x8681b00d),LL(0x93781b71,0x17c31bce),LL(0x6a1b5748,0x4324e90c),LL(0x1222e554,0x44f9324c),LL(0xffd53dd0,0xe30ba10f),LL(0xb48eeef0,0x2e5817a8)}},
   {{LL(0x24b3a9a7,0x6b0ee060),LL(0xd5f6545c,0xcf78c3fe),LL(0xfe0f2ec6,0x050bedaa),LL(0xb3e6b4f8,0x0b7774b8),LL(0x2b4a853e,0x096b9e19),LL(0x0698f7ca,0xa7463661)}, {LL(0xf090c97b,0xe1d1d1be),LL(0x55517a2f,0xbc1e09d9),LL(0x5868ee3d,0xb3e24dd5),LL(0xd7b83619,0xd6103494),LL(0x3b0129b3,0xaf739559),LL(0x3ded9fb3,0xef42bb60)}},
   {{LL(0xc8fccaa9,0xa349bb66),LL(0x31a53ee7,0x7888755f),LL(0xc18d3750,0xa6e1d891),LL(0xae8d2bfb,0x9985aa4d),LL(0x31b33078,0x8baec9ae),LL(0x98750e94,0xee68295a)}, {LL(0xd6ddf305,0x0d834bf8),LL(0x9762126c,0xab33dff3),LL(0x0c51d098,0x0c22faaa),LL(0xb887a10f,0x32404042),LL(0x248bed32,0x31f6a614),LL(0x1ce0d662,0x311f8630)}},
},
/* digit=12 base_pwr=2^60 */
{
   {{LL(0xedc9ce62,0xac6dbdf6),LL(0x0f9c006e,0xa58f5b44),LL(0xdc28e1b0,0x16694de3),LL(0xa6647711,0x2d039cf2),LL(0xc5b08b4b,0xa13bbe6f),LL(0x10ebd8ce,0xe44da930)}, {LL(0x19649a16,0xcd472087),LL(0x683e5df1,0xe18f4e44),LL(0x929bfa28,0xb3f66303),LL(0x818249bf,0x7c378e43),LL(0x847f7cd9,0x76068c80),LL(0x987eba16,0xee3db6d1)}},
   {{LL(0xae658422,0x1238e097),LL(0x568df55f,0xb4631ddb),LL(0xf74c5c50,0x451254e7),LL(0x8805813b,0x238b16d2),LL(0x925e7a6e,0x23987b28),LL(0x2a1a10bc,0x93b72e2d)}, {LL(0x05e44b7f,0x944c784d),LL(0x8c8e3120,0x7d70fd09),LL(0xead45716,0x6bf1ab2a),LL(0x31c04205,0xd5f8f0e6),LL(0xa10b8881,0xac062526),LL(0xfe5505a8,0xa1a83cf0)}},
   {{LL(0xc42a2f52,0xcbbd8576),LL(0x9d2b06bb,0x9acc6f70),LL(0x2e6b72a4,0xe5cb5620),LL(0x7c024443,0x5738ea0e),LL(0xb55368f3,0x8ed06170),LL(0x1aeed44f,0xe54c99bb)}, {LL(0xe2e0d8b2,0x3d90a6b2),LL(0xcf7b2856,0x21718977),LL(0xc5612aec,0x089093dc),LL(0x99c1bacc,0xc272ef6f),LL(0xdc43eaad,0x47db3b43),LL(0x0832d891,0x730f30e4)}},
   {{LL(0x37d83369,0xac92ee15),LL(0xfecec65c,0xc968c187),LL(0x6e7a3265,0x29a7ca87),LL(0x8456c9af,0x0f2b7e7a),LL(0x9754326f,0x7471824e),LL(0x364d2ec8,0x498687bf)}, {LL(0x3c6ee351,0x86d8aacd),LL(0xf6f41e85,0x01ee6823),LL(0x1d79f7eb,0x9805fc88),LL(0x0040547d,0x377ac3a4),LL(0x61b4e90b,0xd39215d4),LL(0x4c5fd81b,0x2547416e)}},
   {{LL(0x0c7fecdb,0x9ffe5563),LL(0xf88101e5,0x55cc67b6),LL(0xcbefa3c7,0x3039f981),LL(0x667bfd64,0x2ab06883),LL(0x4340e3df,0x9007a257),LL(0x5a3a49ca,0x1ac3f3fa)}, {LL(0xc97e20fd,0x9c7be629),LL(0xa3dae003,0xf61823d3),LL(0xe7380dba,0xffe7ff39),LL(0x9facc3b8,0x620bb9b5),LL(0x31ae422c,0x2ddcb8cd),LL(0xd12c3c43,0x1de3bcfa)}},
   {{LL(0xf6e10789,0x03621d34),LL(0x410c4004,0xc72143b8),LL(0x6ca2933a,0x886d00f6),LL(0xde1494a2,0xd0a22382),LL(0x0bdcdf82,0xb8656c4d),LL(0xc4a58832,0x5199d9d3)}, {LL(0xe1c0f449,0xa982c1aa),LL(0x5a922bdb,0xe9cefb90),LL(0x31390c6d,0xfc268de4),LL(0x8c4fe595,0xea4160e1),LL(0x6790d82f,0x6c427a63),LL(0xfc519907,0xaceb0154)}},
   {{LL(0xd6e0f9a9,0x8c074946),LL(0x51c3b05b,0x662fa995),LL(0x04bb2048,0x6cdae969),LL(0xd6dc8b60,0x6dec9594),LL(0x54438bbc,0x8d265869),LL(0x1b0e95a5,0x88e983e3)}, {LL(0x60cbf838,0x8189f114),LL(0x771dc46b,0x77190697),LL(0x27f8ec1a,0x775775a2),LL(0x607e3739,0x7a125240),LL(0x4f793e4e,0xafae84e7),LL(0x5bf5baf4,0x44fa17f3)}},
   {{LL(0x07a6fbf8,0xa10292b9),LL(0x3fa6235b,0x292c1a20),LL(0x73ad7a1f,0x7a36f18f),LL(0x5897b11f,0x8b2c7b0c),LL(0xcb664c61,0xf7b9a272),LL(0x8f81e22c,0xb6d366af)}, {LL(0x8e342bba,0x3b99b211),LL(0xb06ced2b,0x03ce158b),LL(0x001db74b,0x3af1175d),LL(0x7159cb8a,0x526f0846),LL(0xebde4601,0x6a3c6e1f),LL(0x8c232eac,0xfad5963b)}},
   {{LL(0xd03ac439,0xa21e69a5),LL(0x88aa8094,0x2069c5fc),LL(0x8c08f206,0xb041eea7),LL(0x3d65b8ed,0x55b9d461),LL(0xd392c7c4,0x951ea25c),LL(0x9d166232,0x4b9a1cec)}, {LL(0xfcf931a4,0xc184fcd8),LL(0x063ad374,0xba59ad44),LL(0x1aa9796f,0x1868ad2a),LL(0xdff29832,0x38a34018),LL(0x03df8070,0x01fc8801),LL(0x48dd334a,0x1282cce0)}},
   {{LL(0x42a993b1,0x21387e8a),LL(0x68b1f2a0,0xbcb65533),LL(0xa5f7e4b9,0x3161d87a),LL(0x1bce0a84,0x927b4130),LL(0x87be0ae9,0x3aabd327),LL(0x0aed169d,0x0163c6aa)}, {LL(0x11584051,0xad1e0f0e),LL(0x74093225,0x1ba46302),LL(0x0a6a1f01,0xed045618),LL(0x6da060ed,0x15d07c63),LL(0xb7e003ae,0xa4584f72),LL(0xa02ec136,0xf6e6b7c4)}},
   {{LL(0x26d8503c,0x76aa9557),LL(0x6bc3e3d0,0xbe962b63),LL(0x97de8841,0xf5ca93e5),LL(0xaf3f2c16,0x1561b05e),LL(0xd34bff98,0x34be00aa),LL(0xd23d2925,0xea21e6e9)}, {LL(0x394c3afb,0x55713230),LL(0xd6c8beca,0xeaf0529b),LL(0x202b9a11,0xff38a743),LL(0x6d3a398b,0xa13e39fc),LL(0x86e2615a,0x8cbd644b),LL(0x191057ec,0x92063988)}},
   {{LL(0x74962b2e,0x71503219),LL(0xe1db7ec7,0x3beeaf81),LL(0x445c4e88,0x702784cc),LL(0x9baad7ed,0x7aa58315),LL(0xd044b38a,0x72b76e2f),LL(0x6a7f2101,0x35a2bb8f)}, {LL(0xa14c2bf2,0x5e6ae263),LL(0x34bb719d,0x8fb614c0),LL(0x5dcd7c65,0x7949eeca),LL(0x7670aeb1,0xb198be58),LL(0x93cabf25,0xb2238dfc),LL(0x1a422b95,0x6a6d4555)}},
   {{LL(0x13f89146,0x787835ce),LL(0x69446c3f,0x7fcd42cc),LL(0x840e679d,0x0da2aa98),LL(0x18779a1b,0x44f20523),LL(0xefbf5935,0xe3a3b34f),LL(0xb9947b70,0xa5d2cfd0)}, {LL(0x27f4e16f,0xae2af4ef),LL(0xb9d21322,0xa7fa70d2),LL(0xb3fd566b,0x68084919),LL(0xd7aad6ab,0xf04d71c8),LL(0x10bc4260,0xdbea21e4),LL(0x8d949b42,0xaa7dc665)}},
   {{LL(0xf7848707,0x3ddea6d9),LL(0x6dbcf13a,0xc11260d0),LL(0xa16508f0,0x62c5ee60),LL(0xcb56535c,0x4bdfa38f),LL(0x4fa5a6b0,0xd391e786),LL(0x2d4cd296,0x173c3d70)}, {LL(0x5fa3f53c,0x9c5c596f),LL(0x1ddd3732,0x77d39514),LL(0xa5bf4edc,0xca1bb91d),LL(0x1cd98b09,0xb23afd38),LL(0x60bdfa0f,0x198248ed),LL(0xfb69a4b1,0x559dd99c)}},
   {{LL(0x6ccb8213,0xd8e958a0),LL(0x91900b54,0x118d9db9),LL(0x85e8ced6,0x09bb9d49),LL(0x24019281,0x410e9fb5),LL(0x6d74c86e,0x3b31b4e1),LL(0x020bb77d,0x52bc0252)}, {LL(0x27092ce4,0x5616a26f),LL(0xa08f65cd,0x67774dbc),LL(0xc08bd569,0x560ad494),LL(0xad498783,0xbe26da36),LL(0x7f019c91,0x0276c8ab),LL(0x5248266e,0x09843ada)}},
   {{LL(0xb1b43eef,0xa54768da),LL(0xe14fda22,0x13e41f47),LL(0xfaef6863,0x774df203),LL(0xbd7471b3,0xf795a034),LL(0xb47de2e9,0xf0958718),LL(0xe1160cff,0xc92f7888)}, {LL(0x0146c790,0x86ded97b),LL(0x480a4b7b,0x015918f5),LL(0x424e8459,0x05588920),LL(0xeecf8b2b,0x37455914),LL(0xb968a6fa,0xe7d3df1f),LL(0xbad0719f,0x07a0ffd6)}},
},
/* digit=13 base_pwr=2^65 */
{
   {{LL(0x98d23f6f,0x2566021f),LL(0x34ca97ca,0xfb883e12),LL(0xd9f51b69,0x34e047a5),LL(0xf8efa646,0x0b50d91d),LL(0x971f584f,0xc2bbcbb2),LL(0x0907c91c,0x4136f0e4)}, {LL(0xe735cc48,0xa7ebeb0d),LL(0xe113c8fa,0xa7d1bedc),LL(0x3f5c962a,0xc04d9a07),LL(0x3ff74a2f,0x95c155e5),LL(0x3df0749d,0x923c65a5),LL(0x27ae35d0,0x10d5f812)}},
   {{LL(0xecee6e87,0xd5469c7b),LL(0x33a4c917,0x056180bc),LL(0xa16caa7a,0xf881ca21),LL(0xe6cc7f39,0x221de182),LL(0x31378723,0x10d61ab5),LL(0x520c9660,0xfb763bd9)}, {LL(0x0d6b1541,0x145214cd),LL(0xd70223e7,0xd9f7ff2d),LL(0x0cb1fe69,0x9fce59e3),LL(0x3e299fe7,0x2e6e77fa),LL(0xd5af78cf,0x3a0cf652),LL(0x3e852159,0x50cc42c5)}},
   {{LL(0xdf764716,0x0791dfa1),LL(0x8c66da07,0x31bf6876),LL(0xccedf4f3,0x49f25b77),LL(0x5d965c05,0x05170ccd),LL(0xd49e6727,0x37d9521b),LL(0x86a00176,0x15482512)}, {LL(0x6c00eb48,0xdab44493),LL(0xe00c5c5d,0x102c6b95),LL(0x4c2506ba,0x43660c3e),LL(0x5ec6f132,0xb2fb2616),LL(0x99ac7691,0xccc4221a),LL(0xa576deb5,0x05b29758)}},
   {{LL(0xad8c49b7,0x4a873076),LL(0x7146575f,0x891598ce),LL(0x427ea198,0xc1d3042f),LL(0xed259219,0xdc592111),LL(0x234850ca,0x0abdbd16),LL(0x43b6fe8d,0x26b94126)}, {LL(0x36a1cfe9,0xd3c79d17),LL(0x9a2b3baf,0x57638621),LL(0x5a98bf65,0xa736535d),LL(0xab2cdb2b,0xacb3b7dd),LL(0xdaaf89ee,0x37d3743a),LL(0x0b348532,0xf19d9aba)}},
   {{LL(0xd93c54b2,0xafad01a0),LL(0x95536c49,0x659bff96),LL(0xb9734c15,0x7b91aac2),LL(0x24e02f59,0x55c7f082),LL(0x3a26e551,0xebcb71e7),LL(0x6c6343eb,0x5b7225f7)}, {LL(0x17d5e775,0x021c48fb),LL(0xbd859c87,0x57536a42),LL(0xae2b63e9,0x24852cc0),LL(0xc0ce0ef1,0x21515020),LL(0x8fed825e,0x2ac7336c),LL(0xa0152819,0x4bc87fee)}},
   {{LL(0x4654712c,0xe7c396d6),LL(0x7a26e994,0x1fa5ea50),LL(0x09012b83,0xaa987687),LL(0xe9b17e98,0xf1ef9792),LL(0x6dc2db10,0x2c22bcb0),LL(0xed4be80e,0xae42ddf6)}, {LL(0x2e743405,0x672080f6),LL(0x5b7821d1,0xa15a7f97),LL(0x47adbf07,0x0cd912f2),LL(0xb6c4ae8f,0x6919c0dc),LL(0x14c6253c,0x62b13edd),LL(0xf8032287,0x66f35919)}},
   {{LL(0x9d628e1d,0x2713b58c),LL(0xfaa8ba1e,0x3729960a),LL(0xab53bc93,0xc3438130),LL(0xe9c165e7,0xcef9eda5),LL(0xfd02650b,0x9bacd1c3),LL(0x540ece72,0xbb300334)}, {LL(0xd9c4f0ed,0x21f5a5d3),LL(0xf7c19269,0x1ba32e4f),LL(0x2cf320fb,0x8f073beb),LL(0x5599646f,0xb49766c4),LL(0x49e4f200,0x68180d66),LL(0xeaeddb36,0x8203d8ae)}},
   {{LL(0x2caf088b,0xa68295da),LL(0x5c8709fe,0x23d6439a),LL(0xfe0c3df0,0x8deba0cf),LL(0x3cd00a1a,0x5b4d037b),LL(0xaa0f9088,0xe9edc429),LL(0x5847def7,0x6f5827e3)}, {LL(0x306ad966,0x9739d03d),LL(0xaed51d04,0x7c6b18af),LL(0x1759060a,0xdc3d34ff),LL(0xa7e94dbb,0x029e9aa9),LL(0xf7e8b7f3,0x2a3cdfa0),LL(0xbbd8f6f4,0x42f87bf0)}},
   {{LL(0xed210253,0x516f138b),LL(0x4433461a,0x5ec2fa32),LL(0xcdaf1280,0x0dbe2c66),LL(0x9fbf3318,0x086b91e5),LL(0x399a1ca6,0xfb0223ee),LL(0x0db5b20f,0xd6f86d9b)}, {LL(0x5752d618,0xec02bca2),LL(0xaf69f3f1,0x952fafca),LL(0x33c4d294,0xf304cb75),LL(0xdac65608,0x78085727),LL(0x840a4466,0x22f302ef),LL(0xc371c31f,0x33fb889d)}},
   {{LL(0x4f890542,0xaa4711f5),LL(0xeba822c4,0x862421d9),LL(0x848280fc,0x2f667179),LL(0xc201ed75,0x4de16d87),LL(0xc5e61b5d,0xd20e1399),LL(0x9ed67ec7,0x3f7114b4)}, {LL(0x9b5a88f9,0x561fd497),LL(0xd84db2c1,0xb202eb86),LL(0xc8637d3d,0x67d8fb90),LL(0x032b1853,0x3d1d78a1),LL(0xef1af9ac,0xe07bf775),LL(0xa57d6ada,0x691e1dee)}},
   {{LL(0x3801c65d,0x834701da),LL(0x955aa27d,0x5bb35c48),LL(0xdb7ad387,0x0ef0f375),LL(0x06cd1d53,0xd25e337f),LL(0x90cd91de,0x757a1f9d),LL(0xd61bbd60,0x1604f153)}, {LL(0x8bb95dc4,0x6a01e8cf),LL(0x75bbdb13,0x34b7be62),LL(0x21e9b029,0x0a96b3a1),LL(0x2946df44,0x25615c3b),LL(0x19d04842,0x5eda7d19),LL(0xfba84668,0x08317975)}},
   {{LL(0xa55a3a4d,0x12474bd8),LL(0x9e471af4,0xe326aaf1),LL(0x8caadaa6,0xf201a930),LL(0x35304341,0x546821f8),LL(0x088353e1,0x7fe452c3),LL(0xfc82566a,0x8f1ff628)}, {LL(0xd99f8967,0x49526f46),LL(0xa4009690,0xb19c80c0),LL(0x1cbc0716,0xeccf9759),LL(0xaf4cbc8b,0x2e13ae2c),LL(0xf32e29ad,0x12b0df13),LL(0x0b1565f0,0xa2005d6e)}},
   {{LL(0xd06014e0,0xdb891eed),LL(0x03e9970a,0x69685d61),LL(0x02838113,0x3a612db4),LL(0xdcdef0b0,0xc1cd7b3a),LL(0xa41d6c1e,0x612b299d),LL(0x0ed386a0,0x982161ed)}, {LL(0x3ea1bf1e,0xb36bbe2f),LL(0xceb2a5ec,0x0d8c3752),LL(0xec03bddd,0xc02cd7f6),LL(0x52631d9e,0xa87977c1),LL(0x7e398d7e,0x7b546cc3),LL(0x04845671,0x5b1218a8)}},
   {{LL(0x6cb173d1,0xfa3e43e5),LL(0x4591b5a5,0x2502258d),LL(0x8ca9682a,0xae8c4b55),LL(0xec81a288,0x8cb1ffb4),LL(0xbfc84fd1,0xd11ae888),LL(0xd774577c,0xa3b083a0)}, {LL(0x1da9afc5,0x119b41c1),LL(0x2934e22b,0x44bc7762),LL(0x7c639d6c,0xa04694f3),LL(0x32c5b8ea,0xd5e1ce57),LL(0x9749e8b6,0xd507c39b),LL(0x55255b63,0x16cc0b57)}},
   {{LL(0x614f6f37,0x71e5df78),LL(0xa0b80bee,0x5cf0e08d),LL(0x1e32051f,0x1f8dae17),LL(0x83bc233b,0x54ae365d),LL(0x1b84aaa6,0x97ea005b),LL(0x64c75139,0xf4766d92)}, {LL(0x41215701,0x9b93bbf2),LL(0x8cf8a865,0xb18f042d),LL(0x0867556f,0x5dfb96dd),LL(0x597fd6a1,0xe9fafbb8),LL(0xfe48bbc6,0x729b2f50),LL(0x7f37ff9b,0x2cf85f6b)}},
   {{LL(0x378ef62b,0xcd8c2ec9),LL(0x91a7c4b9,0x7f4a3c54),LL(0xdb83e1ab,0xe12386d4),LL(0xbb549bb5,0x9a792032),LL(0x2807c0f1,0xaf81cba6),LL(0xabf2008e,0xd4ad7d87)}, {LL(0x7e9ad6cd,0x9d7a7230),LL(0x8b517b2a,0xb30636b0),LL(0xec900516,0x47c324da),LL(0x7193eb30,0x408cd0d3),LL(0xd315c655,0x8f0bcce2),LL(0x869d6c22,0x540ad4e0)}},
},
/* digit=14 base_pwr=2^70 */
{
   {{LL(0xdae0ff8b,0x4b5e753d),LL(0xda3d97b5,0xe55c83e4),LL(0x42fa905b,0x4034d75f),LL(0xc33e462b,0x89b85eda),LL(0x058de3bb,0x31f413c3),LL(0x6ba75391,0x66c01c80)}, {LL(0x373e28de,0x3f500202),LL(0x4b9be739,0x5090b33b),LL(0xefa2addd,0x7297aa10),LL(0x1a6566aa,0x3e8ccdbe),LL(0xd7b4f214,0x4dfda07f),LL(0x7cb1cae7,0xa659bd1e)}},
   {{LL(0x87d11691,0x31796c23),LL(0xa9de506c,0x02991ea2),LL(0x6cb0c301,0x4ff0cb71),LL(0xd1702ca9,0xdd0cdbd7),LL(0xe1a02a90,0x470a26c8),LL(0xd7054625,0xb705b7ba)}, {LL(0xd4a1a268,0xfadc2e86),LL(0x68e9f923,0x0fd97646),LL(0x2951a8fe,0x042b5ebb),LL(0x64197a76,0xe4af9d03),LL(0x249c1b5d,0xdd2c6bb3),LL(0xf01932b1,0x60af89bd)}},
   {{LL(0x41afcd64,0x4374145d),LL(0x49d21198,0x98b72d60),LL(0xdfde8a41,0xc0ff394b),LL(0xee1ff7a5,0xed1112e5),LL(0xcb5036fc,0x87a920e8),LL(0x2deb225e,0x437123f6)}, {LL(0xb9ad8c58,0x37e527af),LL(0xabfaef38,0x3e3c9998),LL(0x50b2b4e2,0xb656bcc5),LL(0x3bf5699d,0xfacc8a19),LL(0xe616307a,0x98cec74f),LL(0xd3ef8bab,0x34af333d)}},
   {{LL(0x744ec273,0x9ded9b0d),LL(0x2e79e4d5,0x5bdfe547),LL(0x39393728,0x94f3aaf5),LL(0xa438413e,0x22136862),LL(0x373c7de3,0x449286da),LL(0x29aa1540,0xa709d85d)}, {LL(0x0284a4f6,0x1bd13e41),LL(0xf0799c8b,0x37b54d69),LL(0x6bcd0cb2,0xd43b558f),LL(0x9e610369,0xf5757c0e),LL(0xc16e0651,0x15c80b23),LL(0x001820aa,0xacb2cf64)}},
   {{LL(0xeec37f48,0x0629e4dd),LL(0xda7de716,0x3f7556a2),LL(0x661662bc,0xcacd8f27),LL(0xeaf01690,0x65d8bc2c),LL(0x7c39c893,0x83ac6647),LL(0xf59440e1,0x353f60db)}, {LL(0x852c575a,0x2597b0d6),LL(0x3a40c2b3,0x410885f7),LL(0x2fcc2488,0x953ab347),LL(0xef4cc6d8,0x9f753e5e),LL(0x97f69e63,0xfc32bb4d),LL(0x87e8c264,0x461c1b0c)}},
   {{LL(0xac4b62f5,0x2bea7e75),LL(0xc6297871,0xcf255dcd),LL(0x81b25c72,0xdd88db87),LL(0xb617dc04,0x77ad90b3),LL(0x0ed4a7d6,0x65ee1382),LL(0xd9644c8f,0x4c08df9c)}, {LL(0x525a023f,0x072d3784),LL(0xfce399fb,0x1aef69ce),LL(0xd7f29044,0xb07fd78e),LL(0x43043fa7,0xa3754e14),LL(0xff9fe4d9,0x97bdae92),LL(0x700fe6b8,0xad63ba6a)}},
   {{LL(0xa1ca17db,0xa571929b),LL(0x45e146b8,0xce7a12f5),LL(0xd9eb426d,0x39df1446),LL(0xdc27f268,0x1e48b3f8),LL(0xff548455,0xa2d7dfa1),LL(0x6ef1cc82,0x750068b8)}, {LL(0x667fce62,0x4d699306),LL(0x131c5412,0x98540b9d),LL(0x47c580b0,0xf8a62cd3),LL(0x73795005,0x2b55460f),LL(0x7b8db337,0x3206c025),LL(0xd0dda5a2,0x2280934b)}},
   {{LL(0x2352478c,0x10a8aabd),LL(0x1364c40f,0x599d9dfa),LL(0x076945a2,0xa009df1a),LL(0x03861f02,0xf869152c),LL(0x9f866a3c,0xc405226e),LL(0x8b41ecbb,0x93bd737d)}, {LL(0x33901eed,0xb5c9ed10),LL(0x99312b80,0x90e4ce8f),LL(0x57589279,0x1a9ef22e),LL(0x7fe2d6aa,0x83ef607d),LL(0x3473dbfa,0xf2da8454),LL(0x57879066,0x14f36d3a)}},
   {{LL(0x0ea3ce34,0x2c780f22),LL(0x4b8aac72,0x240a211e),LL(0xc2625a99,0x7a266e5d),LL(0x3b30c878,0x1cb15d3e),LL(0x0e1b21d3,0x8cd8ccab),LL(0xadc1a6b3,0x53c64279)}, {LL(0x60bf708d,0xe60d15b9),LL(0x0cb5ad4b,0x6e431c1b),LL(0x82033111,0xec874c3e),LL(0x9141eae5,0x88054a1d),LL(0xddf53a28,0x98438a5a),LL(0xfa12c657,0x168f0b0f)}},
   {{LL(0x629d7a57,0xd621ce26),LL(0xcc1f8af1,0xbf571de7),LL(0x304adaa3,0x2c5cfaf9),LL(0x3f283b49,0x950addbd),LL(0xee6d1cd1,0x622dc27c),LL(0x1f0863fb,0x26d92004)}, {LL(0xa41ec585,0xe243b2a2),LL(0x2bea6235,0x0dbd9adf),LL(0x6f0820f5,0xcb083c4d),LL(0xd5493931,0x809ecbaf),LL(0x14b7ffa8,0x96470674),LL(0x53fc2224,0x2cdfe22e)}},
   {{LL(0x7cd74a06,0xc8b1333c),LL(0x271006bf,0x7d5ac4dc),LL(0xc14e0e56,0xe9377d9f),LL(0x7a92ab1f,0xaa8651db),LL(0x5fc11fb5,0x77cee814),LL(0x27870b14,0x1e7c5ca0)}, {LL(0x6f959698,0xdeef4b3f),LL(0x5daa9bda,0x7c59f26e),LL(0xfd312368,0x4d0d5aec),LL(0xb738de68,0x5247a6f3),LL(0x90c04807,0x1c4e8ba9),LL(0xcce126ca,0x0554b41b)}},
   {{LL(0xa81cc26a,0x3df98ea9),LL(0x83b2c6f3,0x982ed568),LL(0xea6d6976,0xc9cbd1b5),LL(0x7e25ffbc,0x3f9f2319),LL(0x7da6280e,0xbca8e056),LL(0x35cda713,0x7abd3166)}, {LL(0x740ae011,0x46ef321d),LL(0xdb214a33,0xb17f6c75),LL(0x51de4044,0x37b73b4b),LL(0xccd9ba8d,0x5bccf3cc),LL(0xd0f7045b,0xa2ca080d),LL(0x68cf4dcc,0x79caf906)}},
   {{LL(0xa7b07d22,0xcc3605a9),LL(0xb4ebe4e7,0x4370eb18),LL(0x248867c1,0xbe393039),LL(0xcb1a75ff,0xc8e4851e),LL(0x39cb6da4,0x215f3fbe),LL(0xe41f9a34,0x6f2102ee)}, {LL(0xdfae7c27,0x61d484ab),LL(0x6f1260fc,0xf5143bd2),LL(0x7514bccc,0xa70b6c06),LL(0xd23506f5,0xe71ca833),LL(0xeae03a8e,0xe2f50ba8),LL(0x83c33359,0x2ac3b508)}},
   {{LL(0x3655cabd,0xe94b930f),LL(0xa342443c,0x6ef6aac4),LL(0xbae255da,0x2feb8005),LL(0xac6e2095,0x4625a15c),LL(0x0ec76c1e,0x75c6311d),LL(0x9b81c6f2,0x896a0740)}, {LL(0x073378bb,0xbad3e2f9),LL(0x29266ec0,0x2984a106),LL(0xa788010a,0xa68a5351),LL(0x017cd052,0x321aa113),LL(0x2f34db5c,0xecfb6175),LL(0xca2b51df,0xfe080ced)}},
   {{LL(0x03360c88,0xba296908),LL(0x36311812,0x3fb087c0),LL(0xd9ff6a48,0xec5fb10a),LL(0x207dd8e1,0x52f7077f),LL(0x44c02fbc,0x8e65cfb8),LL(0xfbf4bfd5,0x4f4fcde2)}, {LL(0x6cc74320,0xd1ff5415),LL(0xf68b036a,0xf989d544),LL(0x973bd9be,0xafedc2db),LL(0x4785e26a,0x7bdc3569),LL(0x751ae9ba,0x0df36796),LL(0xccdabd95,0xfebde691)}},
   {{LL(0xc77488cc,0x3499b44b),LL(0xaa8f3c10,0xde000e01),LL(0xc1f517d7,0x30140406),LL(0x82a174e8,0xd4b3c8eb),LL(0xaf2c9b3a,0xc8835b13),LL(0x443716f4,0x2cd7626e)}, {LL(0x9c22de71,0x336c4f4b),LL(0xf2529f60,0x9dd2b277),LL(0x828c34d3,0x6ffba2dc),LL(0x34d0d1f7,0x257a345c),LL(0x6ca4fbce,0xbc1eff05),LL(0x0768349f,0x05b33562)}},
},
/* digit=15 base_pwr=2^75 */
{
   {{LL(0x10fa53ce,0x5d7afe9a),LL(0x1ff49021,0xe2415b50),LL(0xc6523492,0xe39a067d),LL(0x27557f5d,0x34585275),LL(0x930e9f9d,0x756b8d86),LL(0x040d52d8,0x88df6219)}, {LL(0x5362b045,0x606eb60b),LL(0x9e383cbe,0xd179818c),LL(0xe068d293,0xa6215748),LL(0x2fce158d,0x73fbdca2),LL(0xcb183c8f,0x9cfee07d),LL(0xa5e03c98,0xc0bf2beb)}},
   {{LL(0xf14a99d9,0x86230934),LL(0x97c1c092,0x1cf9c66e),LL(0x6f595ed3,0x01e186ba),LL(0xe2284a58,0xd3291c3d),LL(0x1b9e5e25,0x03dee231),LL(0x15cc9f53,0xf2e9b4ad)}, {LL(0x9770c29d,0x4fba1567),LL(0x50c4ae2f,0xbf7d6736),LL(0x2532d015,0x86901eb9),LL(0x4e7455de,0x4396fd78),LL(0xbcf811c9,0x2fbcea8f),LL(0xae952b37,0x3981ad15)}},
   {{LL(0x4caea411,0x5563ba50),LL(0x59297ddc,0x6d06f7ff),LL(0xb2fb7a6e,0x244c00ac),LL(0x35df586e,0x25aac3d4),LL(0xd67be5f9,0x6c6777e4),LL(0x11c24c8a,0x476cbbf1)}, {LL(0x83b496fc,0x60832359),LL(0x09a871a0,0xcc13c191),LL(0x56f78170,0x7c0e049f),LL(0x6eafe939,0x19bda64b),LL(0x09aa41e3,0x04381fb6),LL(0x0b8c4500,0xe4d4dcbb)}},
   {{LL(0x4cb89afa,0x6bd2c54d),LL(0x36527751,0xe78c8bfa),LL(0xe3eee747,0x27f52654),LL(0x9598d907,0x56f20583),LL(0x27cb3712,0x5f91c2d0),LL(0xa3e33c5b,0xc501819f)}, {LL(0x4eded738,0x248490aa),LL(0x27789065,0xde7ac944),LL(0x74f7d38b,0x20138b3d),LL(0x2fb60214,0xae791f60),LL(0xbd033d4e,0x6b4fb300),LL(0xbdfd1f17,0xc69c25d9)}},
   {{LL(0x92717482,0x8a5cb7ae),LL(0xc49d53e9,0x3dc97730),LL(0x5302d584,0x07b1e2cc),LL(0x397b46cb,0x06617e8e),LL(0xe8e9f451,0x73426657),LL(0xdd517335,0x82dcb228)}, {LL(0x5cad6d10,0x39cde297),LL(0x2e4b0ebd,0x76991f56),LL(0x5020cfc6,0xf3e3b3b2),LL(0xe51bbaca,0x35a63790),LL(0x4a23f2ec,0x47b21e6f),LL(0x620e0b06,0x964c6fb5)}},
   {{LL(0xbe832a0c,0x9bd2b68d),LL(0xf1e49e28,0x0efa6b60),LL(0x89586da1,0x67e7f359),LL(0xfd0944ca,0xbc04c8aa),LL(0xf0569b5a,0xb8db5424),LL(0x0cc5546e,0xc27dd125)}, {LL(0xad67b577,0x8f0ef9df),LL(0x9824cbc2,0x1b5d4068),LL(0x4e7329b8,0x2ec1eb64),LL(0xe6896b04,0x3e0e7f37),LL(0xc118b6bf,0xad2bff11),LL(0x40f93fa6,0xe17ec8dc)}},
   {{LL(0x98692833,0xc7e68075),LL(0xe1257bf1,0x311dd043),LL(0xe6258442,0xd4356992),LL(0x05ddf6d2,0x7868deca),LL(0x95a0951b,0x61b3a961),LL(0x61c6fc20,0x0cda07bf)}, {LL(0x2d061312,0xac9aa65d),LL(0xb65f1a23,0x4d9c4f20),LL(0xb06611a8,0x862e4927),LL(0x6cccfba0,0x0d1c906e),LL(0x6a49234c,0x0c82ac34),LL(0xaadbaea7,0x70f3c70b)}},
   {{LL(0xbe21a890,0x0c3d2056),LL(0x8fcfba99,0x1c1ffbfb),LL(0x1fbf56ca,0x1b68a98b),LL(0x396e31cd,0x56fd85ff),LL(0x4382c03b,0xd2ca5844),LL(0x7d3ef917,0xc442030a)}, {LL(0x426afafa,0x4129a731),LL(0x5eaae9c6,0xacff17ff),LL(0x653f3b23,0x9e854180),LL(0x9ee066bd,0xe65a1a14),LL(0x362ea5fe,0x3420084e),LL(0xc7911e2e,0x6fe58801)}},
   {{LL(0x02a97a2b,0x4dd38c42),LL(0xfdb98b51,0x2a8d49c2),LL(0x5cb2e0ca,0x7034c753),LL(0xc4b59d32,0x87e0d6d3),LL(0x35514758,0x9544bb07),LL(0x4ff27eb0,0x9ed743da)}, {LL(0xa0163883,0x1d21a1ad),LL(0x38c57492,0xd6442498),LL(0xf2fd1b7f,0xf46e8acc),LL(0xc8005cfe,0xec9932c5),LL(0xb802c109,0x5fe24f52),LL(0xe3ee9ba7,0x5539c03b)}},
   {{LL(0x05a95afb,0x4c3f4f49),LL(0xf5db9d6b,0xd53a1d3a),LL(0x1844be86,0x0fbd9bad),LL(0x272afc14,0x6b3b80ee),LL(0xaa40a738,0xd727b0b6),LL(0x4d809a37,0x8a21bf45)}, {LL(0x70767232,0x09fb0640),LL(0x45bf5d89,0xa671a2cc),LL(0xd73a211b,0xa7e5ff8a),LL(0xd9a6b97d,0xccdbf621),LL(0xc89e8f56,0x38448e2c),LL(0x2126da09,0x07e4e9c3)}},
   {{LL(0x7adcc8b8,0x762e6584),LL(0x7924eafd,0xedef40a7),LL(0x5913e468,0xe1990688),LL(0x289bc581,0x1882db47),LL(0xe6084637,0x680cd99b),LL(0x1adacd71,0x5249f00e)}, {LL(0x933bf6cb,0x0d3b583f),LL(0x9b02a63d,0x51c86774),LL(0x1dc57813,0x0be0ada7),LL(0x1e2c5f9f,0xd4b63964),LL(0x2c77c4e5,0x810b38b0),LL(0x862013cb,0x9f06b31d)}},
   {{LL(0x6c4be6a0,0x43f0ae67),LL(0x504bffeb,0xabc6a17a),LL(0xdbb4492e,0xd5be6c25),LL(0x84bff97f,0x7efc9ee8),LL(0x062da2e2,0x54fbd9d7),LL(0xc6d2ac32,0x1befeb61)}, {LL(0xcbafef5b,0x14cf6dc0),LL(0x71d12192,0x8e640e47),LL(0xd9a16800,0xd0566543),LL(0xbeb1e28d,0x9cc2ade9),LL(0x38e65833,0xcbfeb450),LL(0xd0f5acb2,0x3852eaac)}},
   {{LL(0x3b5a25dc,0x8bbac220),LL(0x7deb0149,0x6e216404),LL(0x7ae10837,0x3d6eecea),LL(0xeb944203,0x2ab3cd3d),LL(0x5114d51a,0x8fdcb595),LL(0x92471196,0xfbf0f726)}, {LL(0xb20cdb3f,0x6523ea38),LL(0x5796c8f8,0xc08ad84a),LL(0x71b564cb,0xd7977bc2),LL(0x5d66093e,0x16a9e700),LL(0x144cd814,0x67fb9a6f),LL(0xeecab0d4,0xbf2094bb)}},
   {{LL(0x6c50f6a1,0x4f10299a),LL(0x69a1caf7,0xa51c6774),LL(0x0cbede41,0x024b18bc),LL(0x6b03ac7d,0x7883c01e),LL(0x1c5213be,0xa7b99e74),LL(0x85a97540,0x529fad7a)}, {LL(0x03125409,0x0b6fe59d),LL(0x76275b90,0xf1581679),LL(0x6ad18919,0xcaa0969d),LL(0xee7761dc,0xc47afe13),LL(0xdfb12539,0xcae681fc),LL(0x02ea65f4,0x58dabc76)}},
   {{LL(0x35a505bf,0x3f19ec08),LL(0xb0032e38,0x83186122),LL(0x2b3126ec,0xe5c2eac4),LL(0xf752524f,0x4368683f),LL(0x3ab28694,0x03de9cdd),LL(0x61d0253b,0xfc5e70df)}, {LL(0x789a16d4,0x7991742a),LL(0x226ee928,0xefdb8003),LL(0xa4e62a4f,0x0468fe4c),LL(0xd07dd729,0x3ce92ccb),LL(0xfe688c1e,0x447bed93),LL(0xcae39d89,0xd5ff355d)}},
   {{LL(0xc4650b85,0x5e930d65),LL(0x6350da54,0xbe96b2ae),LL(0xfa08bd49,0xcfac4f7e),LL(0xa6e10f64,0x277e8456),LL(0x407ac162,0x41be3067),LL(0x52a9b68b,0xcfd1d032)}, {LL(0x9c337e0b,0x8d8d216a),LL(0x4e1b9cf5,0xace044dc),LL(0xc60d54c3,0xad9a4102),LL(0x28815187,0xb09420f0),LL(0x0b3b8e59,0x881179c6),LL(0x5b09aba1,0x872685ed)}},
},
/* digit=16 base_pwr=2^80 */
{
   {{LL(0x5852b59b,0x22313dee),LL(0xb6a0b37f,0x6f56c8e8),LL(0xa76ec380,0x43d6eeae),LL(0x0275ad36,0xa1655136),LL(0xdf095bda,0xe5c1b65a),LL(0x367c44b0,0xbd1ffa8d)}, {LL(0x6b48af2b,0xe2b419c2),LL(0x3da194c8,0x57bbbd97),LL(0xa2baff05,0xb5fbe51f),LL(0x6269b5d0,0xa0594d70),LL(0x23e8d667,0x0b07b705),LL(0x63e016e7,0xae1976b5)}},
   {{LL(0x236b71dc,0x022aa09d),LL(0xa65a7640,0xb1ce6a0e),LL(0xb38b417a,0x317344c5),LL(0x436451ec,0x29a74cdb),LL(0xa8b1c876,0xd898eb6c),LL(0xb74eeffd,0xf0134f99)}, {LL(0x225d71f7,0x0d9eab64),LL(0xceb3cc2d,0x9679b453),LL(0x14dbff2f,0x37c894ce),LL(0x27065280,0x3704d349),LL(0xba29a0cd,0x9ee435d8),LL(0x09c11c4f,0x675bea14)}},
   {{LL(0xfbecaaae,0x2fde4893),LL(0x30332229,0x444346de),LL(0x09456ed5,0x157b8a5b),LL(0x25797c6c,0x73606a79),LL(0x33c14c06,0xa9d0f47c),LL(0xfaf971ca,0x7bc8962c)}, {LL(0x65909dfd,0x6e763c51),LL(0x14a9bf42,0x1bbbe41b),LL(0xc49e9efc,0xd95b7ecb),LL(0xb38f2b59,0x0c317927),LL(0xb3c397db,0x97912b53),LL(0x45c7abc7,0xcb3879aa)}},
   {{LL(0x714a10e8,0x8b3269a2),LL(0xa4a2727e,0x64cef040),LL(0xe428865c,0xbc5ac714),LL(0xfdaba094,0x531dd17f),LL(0x18d657f2,0x86d24057),LL(0x2f99dbbf,0xe807b0d9)}, {LL(0x6848ef88,0xc428a80f),LL(0xd0b73ce5,0xb3ef0709),LL(0x22a5d255,0xa7526919),LL(0x18a18586,0xbfe63923),LL(0xfcf633b3,0x28a0c772),LL(0x3f3c5298,0xad22b4ec)}},
   {{LL(0x24359b81,0xcd81bdcf),LL(0xdb4c321c,0x6fd326e2),LL(0xf8ebe39c,0x4cb0228b),LL(0xb2cdd852,0x496a9dce),LL(0xd0e9b3af,0x0f115a1a),LL(0xd8eeef8a,0xaa08bf36)}, {LL(0x06e5e739,0x5232a515),LL(0x8407a551,0x21fae9d5),LL(0x8994b4e8,0x289d18b0),LL(0x09097a52,0xb4e346a8),LL(0x324621d0,0xc641510f),LL(0x95a41ab8,0xc567fd4a)}},
   {{LL(0x7176dd90,0xc8c9b0ae),LL(0x2917d487,0xa9560454),LL(0xe62c508e,0xb03b7946),LL(0xe9fe2321,0x60425926),LL(0x80c1d136,0x73b10bba),LL(0x9d218c9c,0xc30a847d)}, {LL(0x2073859f,0x6ed0c8ef),LL(0x432dd97f,0xa176eabf),LL(0xb9e96167,0x3078096a),LL(0xc473e377,0xb28f0e6c),LL(0x683a3bc8,0xb44e4995),LL(0xd3523796,0x483512ee)}},
   {{LL(0xd57c8de9,0x261578c7),LL(0x3836c5c8,0xb9bc491f),LL(0x14c8038f,0x993266b4),LL(0xfaa7cc39,0xbacad755),LL(0xd69b7e27,0x418c4def),LL(0xae751533,0x53fdc5cd)}, {LL(0xc3eea63a,0x6f3bd329),LL(0xe53dd29e,0xa7a22091),LL(0xdc4c54ec,0xb7164f73),LL(0x44d3d74e,0xca66290d),LL(0x4c9ea511,0xf77c6242),LL(0x1f714c49,0x34337f55)}},
   {{LL(0xfa43699c,0x93a62e7c),LL(0x1bc422d9,0xdad73890),LL(0x10cc9544,0x265e3cbb),LL(0x2f37154c,0x28cceb06),LL(0x3bf2e08b,0x6b79b071),LL(0x3ab39091,0x88e025df)}, {LL(0x126522bd,0x50a8d04d),LL(0xb779bacf,0xeabbc1b7),LL(0xc21cc62e,0x3db4336a),LL(0x6fc00450,0x4747f0a3),LL(0x544b2d95,0x067cbf1c),LL(0xfd2be7a7,0x2480b7d8)}},
   {{LL(0xa64b6c4b,0x5ed2b216),LL(0x3aae640d,0x1c38794f),LL(0x8905794f,0x30bbaee0),LL(0xc8699cfb,0x0d9ee41e),LL(0xcf7b7c29,0xaf38daf2),LL(0x43e53513,0x0d6a05ca)}, {LL(0x2606ab56,0xbe96c644),LL(0xe9eb9734,0x13e7a072),LL(0x5ff50cd7,0xf9669445),LL(0x47da6f1d,0x68ef26b5),LL(0x23687cb7,0xf0028738),LL(0x6217c1ce,0x5ed9c876)}},
   {{LL(0xb6ba9a0d,0x21965f08),LL(0x06cab532,0xa744f068),LL(0x5d4b9a92,0xd15d5fa4),LL(0x955ee1f4,0xc79fc89b),LL(0x7b2775f6,0x6367e8b2),LL(0xe2294382,0x2076cd8b)}, {LL(0xc9089d2e,0xe5c20ced),LL(0xd76d6424,0xe6097605),LL(0x02b41ca6,0xa69f79eb),LL(0x63e07498,0x9738d39d),LL(0x92974776,0x35c14a6a),LL(0x7270fd02,0x3a000a51)}},
   {{LL(0x0a3a9691,0x423ba513),LL(0xb3179296,0xf421b1e7),LL(0x1a871e1b,0x6b51bcdb),LL(0x464e4300,0x6e3bb5b5),LL(0xfc6c54cc,0x24171e2e),LL(0xd3e58dc2,0xa9dfa947)}, {LL(0x9de9cfa7,0x175b3309),LL(0x2d1015da,0x707b2529),LL(0x993ea65a,0xcbb95f17),LL(0x0447450d,0x93515063),LL(0x1b2753c9,0x0f47b205),LL(0xe7d427cf,0x4a0bab14)}},
   {{LL(0x58e996c5,0x2724c535),LL(0xd7453876,0x24010322),LL(0x74818d0b,0x3532b79c),LL(0xb27b07e9,0x1984146e),LL(0x471646df,0x948b0c4f),LL(0x32ef8e08,0x551b1c61)}, {LL(0x25f892bb,0x22c08434),LL(0x4853c594,0x8fd873a8),LL(0xffdf802e,0xff0bdf02),LL(0x8239b779,0x527d6993),LL(0x27ace140,0xed66f25c),LL(0xa1a846a0,0x7389d738)}},
   {{LL(0xb5aa7ca1,0xa39def39),LL(0xc47c33df,0x591cb173),LL(0x6bbab872,0xa09dac79),LL(0x7208ba2f,0x3ef9d7cf),LL(0x7a0a34fc,0x3cc18931),LL(0xbcc3380f,0xae31c62b)}, {LL(0x0287c0b4,0xd72a6794),LL(0x68e334f1,0x3373382c),LL(0xbd20c6a6,0xd0310ca8),LL(0x42c033fd,0xa2734b87),LL(0x8dce4509,0xa5d390f1),LL(0x3e1afcb5,0xfc84e74b)}},
   {{LL(0x12e471f2,0xd605097e),LL(0x2bb4aa64,0x8cff86ab),LL(0xea65a755,0x4a878412),LL(0x2f2b55fb,0xfa790b77),LL(0x5098a08f,0xd13fb9b3),LL(0xb359117d,0x39e474d7)}, {LL(0xe8e5a3d5,0xc37499c1),LL(0xc7586167,0x5abf6859),LL(0x310fec8a,0x13143ba8),LL(0x75918fc2,0x3543979f),LL(0xcbb0bde9,0xf55a9b62),LL(0x29359b93,0xd6c1d000)}},
   {{LL(0xf2cd8a9c,0xb028334d),LL(0x570f76f6,0xb8719291),LL(0x01065a2d,0x662a386e),LL(0x53d940ae,0xdf1634cb),LL(0x8f5b41f9,0x625a7b83),LL(0xee6aa1b4,0xa033e4fe)}, {LL(0x1e42babb,0x51e9d463),LL(0x0d388468,0x660bc2e4),LL(0xfcbb114a,0x3f702189),LL(0xb414ca78,0x6b46fe35),LL(0x4a57316b,0x328f6cf2),LL(0x381ad156,0x917423b5)}},
   {{LL(0xd52eb122,0x0233e423),LL(0x1154b0c9,0xc2848352),LL(0x6349e35b,0x2ca09cef),LL(0xded2ec54,0x3b70afc3),LL(0x52dded3d,0xc813474d),LL(0x12f00ee0,0x2d3f21bf)}, {LL(0x92f215c6,0xa0908f76),LL(0x4e9c0440,0xb97d60e9),LL(0x34b6a8e0,0x84ad10c1),LL(0x6e7c163e,0x6f37fd95),LL(0xd057e0c3,0x7caae8c8),LL(0x553721a2,0x534f52c2)}},
},
/* digit=17 base_pwr=2^85 */
{
   {{LL(0x72a041b2,0xa354c1de),LL(0x29d1330a,0xe83df259),LL(0x9d532bbd,0x67661099),LL(0x52011751,0xb7c2f4cf),LL(0xf659e35e,0x6945d34f),LL(0xa1303b7b,0x6217d20b)}, {LL(0x25751bad,0xa200ddba),LL(0x01d3566d,0xa74a7290),LL(0xaa82b46f,0x3018445f),LL(0xfccedc1b,0xc3e6a3ac),LL(0x3353e29f,0xe86ae870),LL(0xfd7e8547,0x1c8085bb)}},
   {{LL(0xb4dcea78,0x306b63e3),LL(0x13636935,0x4b102092),LL(0x8bdeddea,0x36bb6889),LL(0x67a329ac,0x9331655d),LL(0xba92ccce,0x14c7fe26),LL(0xbe0519b4,0x4e7d6929)}, {LL(0x164d50e2,0x0dc39dbd),LL(0xb1679cc5,0xd4c430a0),LL(0xfa8682ba,0xc7f78818),LL(0xb60aad97,0x43396ead),LL(0xff2c64cc,0x751784d7),LL(0x866af43e,0xd37928be)}},
   {{LL(0x5a91c9aa,0xf0a69e68),LL(0x05c13197,0x0304d201),LL(0xcd14af1d,0x773a3ab7),LL(0x558d555d,0xc0b88edd),LL(0xd2e63dd6,0xeb12d197),LL(0xbcd9cdb3,0x4a8e849f)}, {LL(0x965eaa14,0x06432985),LL(0x1a5a6f43,0x453d9386),LL(0x4171b9bc,0xbd28f616),LL(0xbbfcf90a,0x37781639),LL(0x3a36084d,0x1f93898f),LL(0xdd00ca75,0x1fefd8b7)}},
   {{LL(0x2b2f1bcc,0xb9207dbc),LL(0xa3e83ef7,0x6afd6871),LL(0x34ba150b,0x49924e55),LL(0xdfec9972,0x2935ebf1),LL(0xb76f870d,0x34bf5e94),LL(0x4c20385c,0x22d0f32b)}, {LL(0x8ccc8e72,0xc78ac172),LL(0x0ccecb0a,0x7b45b822),LL(0xcfb4b8ba,0x76c67ee4),LL(0xcd8724b6,0xecfaefb2),LL(0x340bc1ef,0xe9bc3d67),LL(0xca5541b5,0xed40b2b9)}},
   {{LL(0x069e6432,0x9b8a3eb3),LL(0x721397f7,0x43aaa7fc),LL(0xa7e83a71,0x46e23c6c),LL(0x93fa3c25,0x71b261d5),LL(0x0f523a72,0x4a47a105),LL(0x8dcad752,0x31919e89)}, {LL(0x0c5dd2ad,0x4c8b06e7),LL(0x8bdc55e7,0x677ec5f3),LL(0xcb1b5828,0x4372d55d),LL(0xf04dd321,0x7bf054c1),LL(0x2e44584e,0x4e8c1a99),LL(0x51d35d78,0x68078037)}},
   {{LL(0x81d311b3,0x25ae3499),LL(0x3b16037b,0x8640f52a),LL(0x1d947065,0xac083994),LL(0xe2e693d2,0x3723c75e),LL(0xb66f429e,0x65040a51),LL(0x035a3a53,0x7f582b0b)}, {LL(0x0a166da6,0x20eca9e1),LL(0x2c4cc565,0x45b37e20),LL(0x7a8a96e3,0xeab88295),LL(0xb60a1a1b,0x99e771da),LL(0x23b03965,0x2cdd778c),LL(0x91052478,0x8d4d7a72)}},
   {{LL(0x7e8f73b5,0xa52d3c2d),LL(0x51842657,0x86d80633),LL(0xb3949eba,0x58f01253),LL(0xe79367d0,0x97689f15),LL(0x0d820328,0x918bf9a3),LL(0x41c959df,0x2d4bc994)}, {LL(0x8c16ee54,0x37392f6e),LL(0xe6f0849b,0x9f726d58),LL(0xb8208f08,0x497de1e4),LL(0xd51a29b3,0x60c51233),LL(0xc9e1d465,0x0f61fb03),LL(0xfbe2613c,0x09494bd0)}},
   {{LL(0x154d0f99,0x4f293478),LL(0xd07a24b3,0x1b82320d),LL(0x64d55f6f,0x1bf7c94f),LL(0x725c5125,0x4489b57d),LL(0xb1b6a091,0x3aa4d43a),LL(0xcf7a60fa,0x054842bd)}, {LL(0x2aeb4cb6,0xaa918a4d),LL(0xac7d317b,0xcbdaff99),LL(0x6812a03c,0xed0e00a1),LL(0x0b0a1e4b,0xb09acf27),LL(0xac28386b,0xc73a41f7),LL(0xf4cd1321,0x43134dbd)}},
   {{LL(0xd8c33924,0x08336ffe),LL(0x5140b253,0x15b56cbf),LL(0x306caedb,0x38dcd310),LL(0x47944afd,0x04ecd496),LL(0x68a48f95,0x1280d23f),LL(0x34363c6e,0xf4142204)}, {LL(0xcaa8717f,0xd0a397ea),LL(0xc3994b80,0xb51a1669),LL(0x6c56808b,0xa02eed91),LL(0x83545c3c,0xc3ab55c5),LL(0xfd26114a,0x8b835820),LL(0xffff324c,0xe0cfa4a6)}},
   {{LL(0x51c0d95e,0xcf63d279),LL(0xac86a014,0x3b170a0b),LL(0x881095e1,0xc21eaaa9),LL(0x6069a3eb,0xed2fda11),LL(0xbd2f1c5a,0x536264b9),LL(0xde312c2c,0x819e1cff)}, {LL(0xdfd6ce38,0x6c30f983),LL(0x980b439e,0x2f32cc4c),LL(0x3b9c03b2,0x9fab10b6),LL(0x011ab74a,0xdfebe34e),LL(0xb80963f6,0x587360e3),LL(0x8692e352,0x3db1f610)}},
   {{LL(0xfe3d070b,0x63a094c5),LL(0x88515eb1,0xf769b919),LL(0x50d1131d,0xafe86e14),LL(0x6774d3d4,0x6bf27788),LL(0xffd805d0,0x7231d699),LL(0x6304116e,0x05132e5b)}, {LL(0xe34ce5bc,0x3d5e255b),LL(0xc95e3089,0xfd9c3bd0),LL(0xb83cbac9,0x22a24023),LL(0xb0b3b98a,0xfb6d2b6f),LL(0xf7e36fcd,0x74af1115),LL(0xf9da3bf0,0xcfe15eaf)}},
   {{LL(0xbb2fcdae,0x266d2c1c),LL(0x52be93f2,0xb538d4a2),LL(0x73bd0094,0x774c88ba),LL(0x81a7e042,0x65283a9b),LL(0xd0381625,0xe1438bbf),LL(0x4d0db206,0x450e1f64)}, {LL(0x4e60fc4a,0xb38ae9ef),LL(0x12719817,0x14ce87e1),LL(0x570303f0,0x831d41ec),LL(0x28850444,0x71729170),LL(0xccd609f5,0x2077ea32),LL(0xcd273fdc,0x091d1166)}},
   {{LL(0x4b2ca517,0xdf216dd8),LL(0xc6b74c4d,0xb3eec4b9),LL(0x1c14e77b,0xf564e6c8),LL(0x2c2c9395,0xcde25f1c),LL(0x049fcc83,0x7e31f7a5),LL(0x9284c753,0x6913707b)}, {LL(0x58e6eb5f,0xb92a6f24),LL(0x95148292,0x85b0cab5),LL(0x7449be92,0xeaad036d),LL(0xeb94a702,0x2f6a2888),LL(0x47d59fb0,0xd7d8773d),LL(0x03c0bf25,0x612d2573)}},
   {{LL(0x84d2c3bb,0xb872a105),LL(0x39196026,0x44bca571),LL(0x4e352e5d,0x857327d8),LL(0xd925f99f,0xa6c6004a),LL(0xbab79ead,0x48aaf266),LL(0xadab2a3f,0x213ad923)}, {LL(0xf371cc48,0x3be29b6d),LL(0x2385c9f4,0xe732b906),LL(0x23f0a84e,0x562e0be1),LL(0x28c4b0db,0xbb6b0172),LL(0xf4c6d8be,0x71a93ae5),LL(0x551f1fe9,0x76b8bb16)}},
   {{LL(0xc39926a8,0xbbfaaa94),LL(0x60a138aa,0xb9a59fdb),LL(0x947e30e9,0x217a1aa2),LL(0xc52c9fff,0xcac988c9),LL(0x3bae3c39,0x5676473a),LL(0x857f04c9,0x7d84b353)}, {LL(0xdd324e24,0xdeded30c),LL(0x9c242899,0xf07c678a),LL(0x8cb64f3b,0x956d0553),LL(0x502cb2b0,0x9d34e2f5),LL(0x51dd03b3,0x99e1054a),LL(0x4d60a593,0x86b8bfa5)}},
   {{LL(0x754d15e9,0x10a53b90),LL(0x5f4c7218,0x6cde9a0c),LL(0xabef2b96,0x740d513f),LL(0xd3f802fd,0xff6cc47c),LL(0xeb0627af,0x1be6825b),LL(0x5886c2dc,0xdb21ede5)}, {LL(0xf5daaed7,0xb6cfb2c6),LL(0xfae29a9c,0x68b61aa8),LL(0x3a5a485b,0x7a1e16f5),LL(0xe7b2223e,0x16b60b92),LL(0x36a13a9b,0x332f33d8),LL(0x876cd1a2,0x4567c313)}},
},
/* digit=18 base_pwr=2^90 */
{
   {{LL(0x4aea3fa2,0xa896d28e),LL(0x6db06ee9,0xc6137a45),LL(0x06fb15cc,0x1bbafe8c),LL(0x1cdffdad,0x2daab296),LL(0xe1119b3a,0x984defc8),LL(0xde2a25a3,0x9cd44c3c)}, {LL(0x54ed6d73,0xa7f54ece),LL(0x50907054,0xd283017f),LL(0x6a3b9442,0x69130efc),LL(0x6785163b,0x5d17f127),LL(0x172b1d0a,0xc019911b),LL(0x7e3e093c,0xa19c745f)}},
   {{LL(0xab83d932,0xe185bdc2),LL(0xd7c4e754,0x0a75845d),LL(0xc3fe5695,0x1f6f3397),LL(0x61f6a04f,0x6c9f3a5f),LL(0xb390a92b,0x3c0f9d4b),LL(0x4793b454,0x9e3336b7)}, {LL(0x1472f06b,0x91ad0c34),LL(0x892cbdd7,0x4110047a),LL(0x65d53c83,0xfa24d905),LL(0x4176007d,0xd63e5833),LL(0x2cd1623c,0x741089fd),LL(0x2685d345,0x6b3d9202)}},
   {{LL(0x3586c802,0x1ac37f31),LL(0xa9cb8662,0x70daca37),LL(0x8d9c7bf6,0x6e57d6a7),LL(0xe97eec9b,0x27069e2c),LL(0x4e877a82,0x16284ceb),LL(0x774f6288,0x30856e26)}, {LL(0xfa33820f,0x88e53ea7),LL(0x715ced07,0xbfe3b89f),LL(0x743b11c6,0x6cf55589),LL(0x2edd201e,0x020f8a09),LL(0xb15c27e0,0xc80afc2c),LL(0xcc53751a,0x56557371)}},
   {{LL(0xc9cb7f6d,0x1d510157),LL(0x46ab7372,0x532a0773),LL(0xc6dde9e2,0x2ea07e2f),LL(0x37d5bb1f,0xceed9ad9),LL(0x98cc6e28,0x3121994b),LL(0x67ad8fc4,0x67d2fbb5)}, {LL(0xdc9f195d,0x34707fb3),LL(0x1fd5a013,0x6a601f48),LL(0x81ef6cb5,0xfe939b8d),LL(0x1223a9a1,0x5c51e8ab),LL(0xdb74cf37,0x8f6d7993),LL(0x972808e1,0x0b81c5b7)}},
   {{LL(0xe26e1212,0xcc54c384),LL(0xfee6836b,0xf4c6a3cb),LL(0x91cccc5d,0x62589405),LL(0x135a7e68,0x5ed3f3ce),LL(0xe1994768,0x11cc139f),LL(0x77c07147,0x6386c5c4)}, {LL(0x75fb9b2d,0x444230ae),LL(0x8029bfeb,0xa5972fd9),LL(0xdb5f8291,0x46e687ec),LL(0x0de6d5f7,0xf00bf689),LL(0x36af6be3,0xbf946c50),LL(0xd39dee1d,0xd6d9aaef)}},
   {{LL(0x3bf921af,0xcb4e8512),LL(0x532e81d3,0x28fc6332),LL(0xf69f907d,0x682d8637),LL(0x5f759a16,0xbd9fa8f4),LL(0x51f03716,0x091ea9fa),LL(0x32c630e9,0xd685a141)}, {LL(0x3d249cf4,0x7600c9ac),LL(0x002cd2b5,0x687e2022),LL(0x55334058,0x7ec205ab),LL(0x3ecf1368,0x9d0d86b1),LL(0xfc7baf6d,0xb3fc17a7),LL(0x361c91cd,0x57939961)}},
   {{LL(0x37318b6d,0xeb74751b),LL(0x5abe7213,0xb967a3cf),LL(0x02ab79cc,0x80741539),LL(0xafbc3e08,0x11b647ae),LL(0x9e949616,0x3e34458e),LL(0x72591fad,0x6d714d62)}, {LL(0x131b7659,0x813ba91c),LL(0xfd0e295e,0x7f836d36),LL(0xc5d7be99,0x192531f8),LL(0x901e480e,0xce83f561),LL(0xf1bba4dd,0x7b187da4),LL(0xf5c82a33,0x090754d0)}},
   {{LL(0x010c0754,0x0db33228),LL(0x8eca7c59,0x10635ffa),LL(0x0e8a38fa,0x6efd8538),LL(0x769360d8,0xc1812ea5),LL(0x76f27ef5,0x505723dc),LL(0xf35af2e8,0xd0358e02)}, {LL(0xd99419ee,0x9f7bb7fe),LL(0x430a0e2d,0x87c66e83),LL(0x773eaf7f,0x01187549),LL(0x89d51bda,0x05bbbba4),LL(0x640ccde6,0x52cabb06),LL(0xe7ff387d,0x0d5cb557)}},
   {{LL(0x526f59f3,0xa338a9f7),LL(0x216004f8,0x9b866285),LL(0xeb6fcc58,0xe94cfd05),LL(0x2432ba24,0xc60b12ea),LL(0xef1227f5,0x3bb7acf1),LL(0x5ec503dd,0x176e0ef7)}, {LL(0x236f9707,0x9c2337f7),LL(0xe029aa27,0xdc2b0f77),LL(0xa5f85372,0xb795424a),LL(0x1485af98,0x2eb6d54b),LL(0x7e3f46c7,0xd2ce87dd),LL(0x830f743c,0xc5eaf6ad)}},
   {{LL(0x10e06f1a,0x709d61ca),LL(0x78eba75c,0xaa1e9fc5),LL(0x914b2cfc,0xf85d062f),LL(0x9089d85c,0xe73b3baf),LL(0xc4a284b9,0x4ac05fea),LL(0x3acb7268,0x92c78a43)}, {LL(0xee45bb4d,0x7b5586f8),LL(0x6ac0a9e7,0xc39a0d0e),LL(0x4d6f9ab8,0xe4bbe3d5),LL(0x1fd46a08,0x1489463f),LL(0x29dba364,0x3ba31825),LL(0x8138511b,0x94f000d6)}},
   {{LL(0xee9b6d83,0x9f8eafec),LL(0xf4eac676,0x069d5ad9),LL(0xb5687ebb,0xbf9ccc6e),LL(0x9d213f1d,0x5d0f44be),LL(0x0d0d6e70,0x372b5296),LL(0xf8bb90f3,0xef1466ef)}, {LL(0x949789b2,0x2ec1daae),LL(0xb40be288,0x5c7d9b6d),LL(0xc351e87f,0x48319460),LL(0x5db0fde9,0xfeaa721c),LL(0xd16dc699,0xb33d58e9),LL(0x3ca8d3d6,0xe3f296f2)}},
   {{LL(0xc39c1cef,0x70187dfb),LL(0x0c50c71e,0xa785216e),LL(0x6a6c0d60,0x30188b81),LL(0x6a27e97b,0xeaeda67d),LL(0x6ba389aa,0x4a519282),LL(0xa2bf1273,0xb96c7c7e)}, {LL(0x267fe714,0x8ff10657),LL(0x996d91b7,0xdff4a271),LL(0x1dc7aed4,0xe34ba3e1),LL(0x38853d61,0xc457048b),LL(0x1ccbf658,0xe89825db),LL(0x6b255edd,0x68c7b455)}},
   {{LL(0x49d38208,0x2f6f6a0f),LL(0x11ee2340,0x38fbcb5b),LL(0x865a4304,0x06e43b73),LL(0x06d728ad,0x08c696e6),LL(0x961c650b,0xdb82930d),LL(0xc20c5b68,0xf1de8cb5)}, {LL(0xf8e1f9b9,0xd8e3cf7b),LL(0xf3a304ed,0xd6e081ab),LL(0xcd3cb8bb,0xf08d8624),LL(0xb26aa9cb,0xa3fe7742),LL(0x0de295f6,0xa3a9d220),LL(0xb9579b7f,0xfd465046)}},
   {{LL(0x74871e18,0xdc14cb2a),LL(0xfcb8974a,0x017b1340),LL(0x6e93c20f,0xea5cb054),LL(0xa9e2ad1f,0xa7c078ad),LL(0xbeb26838,0xa37207d4),LL(0xde7ee8ee,0xcd8b3b25)}, {LL(0x2801a7ff,0xdca6606a),LL(0x0f8af3fa,0xad2fedcf),LL(0x9b530c05,0xf27d30b4),LL(0x6b2a4613,0x071fc1c3),LL(0xb72cea9f,0x363aaa99),LL(0x3d350374,0x7a33ed8f)}},
   {{LL(0xf2f3e9ea,0xac02833f),LL(0x956fb86e,0x43bcbde0),LL(0x95735c62,0xf888696b),LL(0xe7b2fd9e,0xefa03aae),LL(0xe75b2684,0x0e563e92),LL(0xd7f335bd,0xa49d2f31)}, {LL(0x8ac31404,0x08664171),LL(0x07ccf31f,0x35866eab),LL(0x04122373,0x6bca6111),LL(0xcb21398b,0x730c92e5),LL(0xb93102d5,0x283791a8),LL(0x50dc1b38,0x4b41d94d)}},
   {{LL(0xbb20fabf,0xc377b373),LL(0xf986b847,0x68d3aa52),LL(0xf39b6894,0xd9c2f2ad),LL(0xbd6da22e,0x1bbff106),LL(0x7e09678e,0x3f7e5b8e),LL(0x7ed3ee78,0xad6a8789)}, {LL(0xaf9807b9,0x689e6b31),LL(0xbd1f6ef2,0xeca87778),LL(0xdda78c54,0x17d3277e),LL(0xefb65cb7,0xe686cacc),LL(0x19a30f0c,0x758aa1ab),LL(0xb11f071e,0xb40df97f)}},
},
/* digit=19 base_pwr=2^95 */
{
   {{LL(0x953b9223,0x16f3708b),LL(0x770e7cf3,0x0d3780f8),LL(0x27bb71a8,0x97a615b2),LL(0x162f8b55,0xa8b9a864),LL(0xd91e3fb9,0x80ee8362),LL(0xf83a4ff6,0xb2009a09)}, {LL(0xc1696281,0x07a7873a),LL(0x23095ddc,0x17ff00c2),LL(0x860d60ba,0x427f683d),LL(0x1f87d32a,0xea995927),LL(0xa050319d,0xb2ac69fa),LL(0xd2d0b9ce,0x30c362b9)}},
   {{LL(0xf26feef9,0x24480c57),LL(0x3a0e1240,0xc31a2694),LL(0x273e2bc7,0x735002c3),LL(0x3ef1ed4c,0x8c42e9c5),LL(0x7f4948e8,0x028babf6),LL(0x8a978632,0x6a502f43)}, {LL(0xb74536fe,0xf5f13a46),LL(0xd8a9f0eb,0x1d218bab),LL(0x37232768,0x30f36bcc),LL(0x576e8c18,0xc5317b31),LL(0x9bbcb766,0xef1d57a6),LL(0xb3e3d4dc,0x917c4930)}},
   {{LL(0xaae4bfd4,0xff41d51f),LL(0x17c44fac,0xcf50b141),LL(0x657a1ea4,0x078b808e),LL(0x93c00c55,0xc5aac1a8),LL(0xcc4d1c0f,0xcb99cfd0),LL(0x3fa123a6,0x1d104893)}, {LL(0x023ca92f,0x49646059),LL(0xf3982134,0x5833e326),LL(0xc5781cdd,0x2e0d4bc9),LL(0x8d5e75f5,0x5f7f84ed),LL(0xe1e8a383,0xb6655f1f),LL(0x296e4943,0xcc18514c)}},
   {{LL(0xb475d8f3,0x53ebbaae),LL(0xff76beda,0x3d6ea31c),LL(0x340986b4,0x3c15f25d),LL(0x3365312a,0xc5925d2e),LL(0x51641f96,0xc35d3ee2),LL(0x984128e4,0x11eb2f75)}, {LL(0x3d04bc99,0xb41a21a8),LL(0x6436c3d0,0xf2d28600),LL(0xfaf5663c,0x4ffcf4c0),LL(0x0a62c9dc,0x889d285a),LL(0xcb2d60c5,0x0908665a),LL(0x0a131be5,0xe2f19c59)}},
   {{LL(0xb1e46617,0xa93a7dbc),LL(0xd77195a7,0xd0ad4a47),LL(0xe5948165,0x020d8467),LL(0x25267b60,0xa3375bd9),LL(0x8c44e226,0x8cf54ab2),LL(0xf2bf5bfe,0xc4f7f467)}, {LL(0xc414077a,0xde336f92),LL(0xc92d7219,0x92656952),LL(0xc09ee1d9,0x31645a70),LL(0x292630eb,0x8cfe567d),LL(0x7c16c0a8,0xb835edb9),LL(0x48dee1be,0x8ac88e9c)}},
   {{LL(0x22319bfb,0x318feb4c),LL(0xa1ee9625,0xfd0a1331),LL(0x5b238661,0x1e4a786d),LL(0xa722c591,0x88e04305),LL(0xf406cb01,0x38eb062a),LL(0xe7216364,0x21caa381)}, {LL(0xf0e1f665,0x450c1d29),LL(0x207a1320,0x369af7bf),LL(0x6f6c0680,0xfe46a53a),LL(0x25eac032,0x4553199a),LL(0xffc49722,0x41fa659a),LL(0xbbcb7a29,0xfb9e0c73)}},
   {{LL(0x7e09207e,0xca53bd85),LL(0xfa171b01,0xa304cf7d),LL(0x881beffa,0xa93499d7),LL(0x95cf1295,0xc0b04ee5),LL(0x90cd1e30,0xe0cf548e),LL(0x821b4efc,0xfa82436e)}, {LL(0x979ab01d,0xbb4d7958),LL(0xd48ca82b,0x4a1c815b),LL(0x627640fd,0xbacf6a1b),LL(0xabd9a758,0x31150946),LL(0x906bea56,0x2c9d7d14),LL(0x17e06ed1,0xd2450bcf)}},
   {{LL(0x091354ff,0xc0de60f5),LL(0xa1bd1975,0xd7cc38bb),LL(0xe734e2df,0xf4122aa8),LL(0xef773db6,0x08f40f63),LL(0xce2d71c5,0x0a7e9484),LL(0x78a3f825,0xcc791733)}, {LL(0xb47beec9,0x0cac7a5b),LL(0xa3f7b5b6,0x1cbea0e4),LL(0xd3e18145,0xecf19a90),LL(0x0aadf689,0x0d1b062a),LL(0xf3f0acf7,0x42299f1f),LL(0x5ac252b9,0x63a64539)}},
   {{LL(0x8db83d7d,0xfda26247),LL(0x2ad24ac5,0x3a2d1c29),LL(0xa01d0daa,0x586b6219),LL(0x949dbc96,0xa92f773b),LL(0x5dc355d0,0x89c9668d),LL(0xb5e40672,0x0aaeecdf)}, {LL(0x2c701014,0x5945a47f),LL(0xc5590e89,0x50e4a494),LL(0x7fd21edd,0x307cbd9f),LL(0xb85543b7,0x96005378),LL(0xba214861,0x9f87fddf),LL(0xf9d9d2c7,0xcdd615c5)}},
   {{LL(0x5bd11a42,0xa4bfe151),LL(0x0ea6729b,0x38920da2),LL(0xa0ee708f,0x41e28260),LL(0xabc9d5f5,0xff4fdff4),LL(0xffaae99e,0x6ed92241),LL(0xc04fe4d9,0x6075ce0d)}, {LL(0x5db066f5,0xf10a173e),LL(0xe75ef129,0xa2edee12),LL(0x8ed02e85,0xd2a0823f),LL(0x2e522dc1,0xffa78cf4),LL(0x00c939fd,0x07041e46),LL(0x3a9a8bba,0x3369357f)}},
   {{LL(0x1a4e68b1,0x3b202533),LL(0xf7504686,0xa2ead2d7),LL(0xa569cb36,0xb0bb1808),LL(0xc015c68f,0x953f539c),LL(0x9a6fefe1,0x45c876dc),LL(0xbede60ec,0xe87ce8ac)}, {LL(0xf3c69642,0xecfc84dd),LL(0x3a90826b,0xabe549ed),LL(0x57cca429,0x03a29df4),LL(0xb8e72b83,0x08d4e9a1),LL(0x79e7ef57,0x88d9c51b),LL(0xdc075e38,0x48102f0e)}},
   {{LL(0xba264fa8,0xa1fc20f7),LL(0x4105e64a,0x0f625415),LL(0x8e2e8e16,0x2e5581c2),LL(0x4c456234,0x912c80fa),LL(0xe9e31ad4,0x7fd61fb9),LL(0x1c19e811,0xa5de4e7d)}, {LL(0x2961b64d,0x6a7ab2cc),LL(0x5bb2da03,0xbf06db25),LL(0xab06af4d,0x33f76a85),LL(0x4866378d,0xda387e54),LL(0x56826f8c,0xbb71deb8),LL(0xeb64ecb6,0xbbe3ce38)}},
   {{LL(0xbb63bdd0,0x2a6f3d84),LL(0xc2792d41,0x8b5becec),LL(0xec9fa26b,0xb0412202),LL(0xbc706607,0xbdaad566),LL(0xd8d361d4,0xd34e3ef1),LL(0xeb3c3f14,0xb4fe020e)}, {LL(0x82656dd4,0xc6bac95d),LL(0x88ffb328,0x8cf56ba0),LL(0xdc84969a,0x33f04dda),LL(0xa5f1d0eb,0x642d93d6),LL(0x93ce88dd,0x122e35f6),LL(0x02c0e82a,0xbcffc369)}},
   {{LL(0xfd5d084c,0x0e935934),LL(0x9121a6e0,0x7cd4992a),LL(0x8e15d863,0xab773dba),LL(0xcab64644,0x9cea4a51),LL(0x2efff061,0x516754d7),LL(0xcd3a36a4,0xd8af89da)}, {LL(0x4615774e,0xc7d352ac),LL(0x21ae0d27,0xd1bb914b),LL(0x9199938e,0x8a8aed97),LL(0xcd6f3495,0xeb06789a),LL(0x775f93ee,0xc51d7766),LL(0x0a8af851,0x7eb6909f)}},
   {{LL(0x6a82fd92,0xf6708309),LL(0xfffaccf2,0x08652b2b),LL(0xf0da4c5c,0x8ac0e67b),LL(0xb0fabc5d,0x26c5a4be),LL(0xb1dcad44,0x3e5ad350),LL(0x3aa30664,0xb539e10e)}, {LL(0xa41fef81,0x266f4163),LL(0x78ef116f,0x860401ec),LL(0x3563575b,0xb0fc5cc3),LL(0xd4c9e03a,0xc28a5cb5),LL(0xcc9bbe6c,0x93399eff),LL(0x9c9a84a2,0x50a48cb2)}},
   {{LL(0x5c1d4586,0x44bfd166),LL(0x8d1d86d6,0x46434e19),LL(0xc3936683,0xe50fcf81),LL(0x8b08680f,0xc9b4eb06),LL(0x2832aab0,0xf90882c5),LL(0xecbf5dda,0x42823cef)}, {LL(0x44ae08f0,0xfd4d51c7),LL(0xbbd21c1c,0xb54a08f1),LL(0xfb187c34,0xb72953db),LL(0xf8ed037f,0x166f7f26),LL(0x097bad45,0xd2b1077a),LL(0x790dd808,0x47794cdc)}},
},
/* digit=20 base_pwr=2^100 */
{
   {{LL(0xd1d1b007,0xa05c751c),LL(0x0213e478,0x016c213b),LL(0xf4c98fee,0x9c56e26c),LL(0xe7b3a7c7,0x6084f8b9),LL(0xdecc1646,0xa0b042f6),LL(0xfbf3a0bc,0x4a6f3c1a)}, {LL(0x51c9f909,0x94524c2c),LL(0x3a6d3748,0xf3b3ad40),LL(0x7ce1f9f5,0x18792d6e),LL(0xfc0c34fa,0x8ebc2fd7),LL(0x780a1693,0x032a9f41),LL(0x56a60019,0x34f9801e)}},
   {{LL(0x62c36887,0xa4b957d2),LL(0xfc24cff8,0xaf15a485),LL(0xa271d9e0,0x11575e80),LL(0x4b9367e5,0x0fff68d4),LL(0x2279779f,0xf55ba673),LL(0xd4d68f68,0x9d72cca6)}, {LL(0x590ffe4c,0x01474ab1),LL(0x074d634b,0xd20f44e1),LL(0x36111d25,0x63903a83),LL(0xab531cef,0x37342a5f),LL(0x702ed867,0xd3c93fe7),LL(0x6279f7e1,0x05d14369)}},
   {{LL(0xf0db3751,0xb398290c),LL(0xba42c976,0x01170580),LL(0x56560b89,0x3e71aa29),LL(0x50e6647b,0x80817aac),LL(0xa0be42da,0x35c833ad),LL(0xf1baba4e,0xfa3c6148)}, {LL(0xcd8f6253,0xc57be645),LL(0xc657ad0d,0x77cee46b),LL(0x0defd908,0x83007731),LL(0x899cba56,0x92fe9bce),LL(0xbceffb5a,0x48450ec4),LL(0xf2f5f4bf,0xe615148d)}},
   {{LL(0xcbaf4685,0xdccf68bc),LL(0x270a2bcc,0xb333e464),LL(0x254dd3e3,0xe43ae199),LL(0xddce5c84,0xe8526e26),LL(0xea0b4258,0x52bad815),LL(0x094574c4,0x67c12c1b)}, {LL(0x861545b7,0xa5362fcb),LL(0xc2b2eb62,0x3e904c35),LL(0xeeffc2cd,0x0f9312b5),LL(0x14de4e5b,0x5475657b),LL(0xf0233fa5,0x746e67d4),LL(0x35471ec2,0xb5157d7f)}},
   {{LL(0x90b86166,0xf55edabb),LL(0x075430a2,0x27f7d784),LL(0x9bf17161,0xf53e822b),LL(0xafe808dc,0x4a5b3b93),LL(0xd7272f55,0x590bbbde),LL(0xeaea79a1,0x233d63fa)}, {LL(0xfe1eba07,0xd7042bea),LL(0x10750d7e,0xd2b9aea0),LL(0x31078aa5,0xd8d1e690),LL(0x7e37bc8b,0x9e837f18),LL(0x85008975,0x9558ff4f),LL(0x421fe867,0x93edb837)}},
   {{LL(0x1f23a0d8,0x3503d937),LL(0xc321dde0,0x64c598a8),LL(0x5b52e0f0,0x67f101ef),LL(0xf955b5fa,0xb6b5b4c2),LL(0x880e0569,0xb5f03d53),LL(0xc99393ef,0x121c3ac1)}, {LL(0x57330666,0x90755bd6),LL(0xd4d71d3d,0x70ae5793),LL(0x9e9ce792,0x326ffd51),LL(0x96ccfa14,0x1b772d73),LL(0x874a22de,0x652710f4),LL(0xdb210342,0x72768469)}},
   {{LL(0x83d55b5a,0xaa6489df),LL(0x86bf27f7,0xea092e49),LL(0x5fa2efec,0x4d8943a9),LL(0x720e1a8c,0xc9baae53),LL(0x95a4f8a3,0xc055444b),LL(0xa7c1206b,0x93bd01e8)}, {LL(0x714a27df,0xd97765b6),LL(0x193f1b16,0xd622d954),LL(0xf1503b15,0x115cc35a),LL(0xa9fa21f8,0x1dd5359f),LL(0x6dfed1f1,0x197c3299),LL(0xf77f2679,0xdee8b7c9)}},
   {{LL(0x171c1439,0x58d6998e),LL(0x01feedec,0xfd4a98f4),LL(0x65739fce,0x420b2a01),LL(0x22f7a073,0x5c5db308),LL(0x05042f00,0x016c5478),LL(0xa12413d9,0x5fc73ce2)}, {LL(0xe932aa17,0x8ceb2d70),LL(0x0537afaf,0xb4d66b67),LL(0x339c146b,0x2638d012),LL(0x28ac0555,0x02fbb7b6),LL(0x62d46e63,0x7fcb0c81),LL(0x066d088e,0xeaa9ff4f)}},
   {{LL(0x394fd855,0x5405179f),LL(0x49fdfb33,0xc9d6e244),LL(0xbd903393,0x70ebcab4),LL(0xa2c56780,0x0d3a3899),LL(0x683d1a0a,0x012c7256),LL(0x80a48f3b,0xc688fc88)}, {LL(0x6f7df527,0x18095754),LL(0x71315d16,0x9e339b4b),LL(0xa956bb12,0x90560c28),LL(0xd42eee8d,0x2becea60),LL(0x50632653,0x82aeb9a7),LL(0xdfa5cd6a,0xed34353e)}},
   {{LL(0x2d189057,0x8273db99),LL(0xe1b5f8cc,0x4d1b05fc),LL(0x0a7c32d1,0x5fec7c83),LL(0xea9b4d45,0x28ddaf28),LL(0xa2fc58be,0xb6bb62ac),LL(0x4a41852d,0xfc65b7aa)}, {LL(0x1c9e6045,0x6e765194),LL(0xfc116257,0x3acabf28),LL(0x4b5a4ba8,0xc9d5e805),LL(0xcbdcf1eb,0x9a072259),LL(0x439fc8fc,0xc67cf643),LL(0xb4333aa8,0x917ef6f8)}},
   {{LL(0x91aecce4,0x82154d2c),LL(0x5041887f,0x312c6070),LL(0xfb9fbd71,0xecf589f3),LL(0xb524bde4,0x67660a7d),LL(0x724acf23,0xe99b029d),LL(0x6d1cd891,0xdf06e4af)}, {LL(0x80ee304d,0x07806cb5),LL(0x7443a8f8,0x0c70bb9f),LL(0x08b0830a,0x01ec3414),LL(0x5a81510b,0xfd7b63c3),LL(0x453b5f93,0xe90a0a39),LL(0x9bc71725,0xab700f8f)}},
   {{LL(0xfc37efed,0x311ebba2),LL(0x60cfd6bc,0x8a6a42d6),LL(0xf2a4871e,0xb4051b3a),LL(0xc2f0ebf0,0x66ce77b8),LL(0x0ad28477,0x84abc948),LL(0x63d9d11a,0xc82e5c62)}, {LL(0x007dcf93,0x99ffc70c),LL(0xd964c822,0x5e974edf),LL(0x513085e3,0x0fee3572),LL(0x46ce8444,0xbe67a880),LL(0x06d17129,0x136ceeb8),LL(0x662d86fd,0x0da512ae)}},
   {{LL(0xb9f00793,0x9401aec2),LL(0xb997f0bf,0x064ec4f4),LL(0x849240c8,0xdc0cc1fd),LL(0xb6e92d72,0x39a75f37),LL(0x0224a4ab,0xaa43ca5d),LL(0x54614c47,0x9c4d6325)}, {LL(0xc6709da3,0x1767366f),LL(0x23479232,0xa6b482d1),LL(0x84d63e85,0x54dc6ddc),LL(0xc99d3b9e,0x0accb5ad),LL(0xe8aa3abf,0x211716bb),LL(0x69ec6406,0xd0fe25ad)}},
   {{LL(0x57e0105d,0xe7acaa84),LL(0x3851fd57,0x3c06d3bd),LL(0x2a9c631b,0x23cf3c61),LL(0x33863bf8,0x13888aaa),LL(0x717783ee,0xf2396355),LL(0x36b300e1,0xf21e1a48)}, {LL(0x9d27b4cb,0xa734cb3b),LL(0x796e34b6,0x0a7effed),LL(0x3615cc7a,0xfc586477),LL(0x88844a21,0x1f98ed77),LL(0x7ad4c7bd,0xd6e28940),LL(0xe9331c7e,0xa00d64eb)}},
   {{LL(0xdf85c705,0x0d5c1769),LL(0xa409dcd1,0x7086c93d),LL(0x0e8d75d8,0x9710839d),LL(0xebdd4177,0x17b7db75),LL(0xf649a809,0xaf69eb58),LL(0x8a84e220,0x6ef19ea2)}, {LL(0x65c278b2,0x36eb5c66),LL(0x81ea9d65,0xd2a15128),LL(0x769300ad,0x4fcba840),LL(0xc8e536e5,0xc2052ccd),LL(0xac263b8f,0x9caee014),LL(0xf9239663,0x56f7ed7a)}},
   {{LL(0x713b8541,0x17e3d0b8),LL(0xc6b5e839,0xf372b048),LL(0xd0bb1848,0xf8ef0261),LL(0xc71a3bbe,0x9b804cee),LL(0x542a88ae,0x00b7d171),LL(0xe9097b9e,0xf2b8ed10)}, {LL(0x2c0a009a,0xdbad9f12),LL(0x205fb1bf,0x245fc1e9),LL(0xb83debf5,0xa8a4834f),LL(0x637e449b,0xc3ee226d),LL(0xcab82664,0xe3070d93),LL(0xb37320e8,0x24b8094d)}},
},
/* digit=21 base_pwr=2^105 */
{
   {{LL(0x25059699,0xb506b7b9),LL(0x01ab02e5,0x349fd83f),LL(0x3789281e,0x64b729ad),LL(0xf9af4561,0x69ae8f81),LL(0x56f91860,0x007befe1),LL(0xedc250fb,0xb578c566)}, {LL(0x67ae4801,0x1c16d75a),LL(0xa1d3f592,0x04c35a7e),LL(0x36881f89,0x5dc97da9),LL(0x03a5b1ea,0xaddb6031),LL(0x3e153a0f,0x7eb515b1),LL(0x2b4a1ee2,0xdc3a9219)}},
   {{LL(0x512cc92a,0xe7365f9e),LL(0x5172a654,0x9efdcf8b),LL(0x8e611fc3,0xbfb389ac),LL(0x699c227d,0xce778fd5),LL(0x3ff2ef17,0xdc1f47b6),LL(0x02672ed4,0x2ae0f683)}, {LL(0xa8e879cb,0x51c63806),LL(0x3909f526,0xd3dfecf0),LL(0xe00e12a2,0x375b3d13),LL(0xbf8df325,0x91f9f750),LL(0x1df5f21a,0xf1ea0e42),LL(0x32c60584,0xaed73e7f)}},
   {{LL(0xe0728e6d,0x03b07fb5),LL(0x1012234e,0x9e046920),LL(0x35637644,0x272e6449),LL(0xa55bcadf,0x2b6ad1c2),LL(0x5c71c6af,0x86c52776),LL(0xc1678806,0xa25bd60b)}, {LL(0x76bb32f8,0x0cae8294),LL(0x33e03cb2,0x389ce4e6),LL(0x7513dfb6,0x504df833),LL(0x1b351dda,0x4260ee8e),LL(0xdbaf7cd0,0xa473c5d9),LL(0x71e390fd,0x22cb7cb4)}},
   {{LL(0x26caebd8,0x1d9aa9fa),LL(0x26b7a673,0x6b646869),LL(0x7f167b47,0x7ebed6a1),LL(0x5bd9153d,0x324c13f8),LL(0x4c682ba6,0xe9ea5b73),LL(0x7e3ff6e2,0x2961da7d)}, {LL(0x01a83dad,0x1ed2b050),LL(0xc4a2f60a,0xb232951d),LL(0xd68b8ec6,0xafcea5d3),LL(0x1c6ce0d2,0x21dc058d),LL(0xe719410c,0x0043de75),LL(0xe15cf534,0x4edd792c)}},
   {{LL(0x3babe09e,0x0f45245f),LL(0xa9f2fac5,0x0959326f),LL(0x5cc136e5,0x7629e7fb),LL(0xe48b7eb0,0x208bd5a6),LL(0xb75a85cb,0x637891d6),LL(0x9f27b57e,0xf0ad9d8d)}, {LL(0xe0454b05,0x437b6944),LL(0x02ed3592,0x022c51d7),LL(0x0dc0a769,0x0f79e2bd),LL(0xd9b81f9b,0x54ace1fd),LL(0xf95ea8dc,0x38611d66),LL(0xf0e6147b,0x52443ca8)}},
   {{LL(0x8bc272d0,0x857d6855),LL(0xb5be2485,0x4583eeee),LL(0xfe0152ec,0xb83586da),LL(0xe830294a,0x8b0eb223),LL(0xa5b0e880,0x757582b6),LL(0x9cca7fff,0x5140c016)}, {LL(0xe9228f12,0x07a00782),LL(0xd4973080,0xb96e2b5d),LL(0xe88efbe6,0x3cceb9a6),LL(0x73fcdd25,0x9955b630),LL(0x2805d470,0x04f26ab0),LL(0x424da086,0x90b38299)}},
   {{LL(0xf4f6c5b4,0x73f1ae48),LL(0x4a477f01,0xee5af13d),LL(0xddb93d52,0x274614a2),LL(0xc320aaf5,0x90b0c563),LL(0xef990b0b,0xee2303c8),LL(0x3061f140,0x00d028e7)}, {LL(0xcb3d8eaf,0xff705011),LL(0x62594f4c,0xae1d9908),LL(0xdafea438,0x22a27cec),LL(0xc5962ea9,0xa78e12d5),LL(0x8e65f9cf,0x5bbe9d87),LL(0xf47cefa6,0xa222580c)}},
   {{LL(0x959abb9e,0xf7aaa732),LL(0x2ebf80b9,0x1222ad0a),LL(0x2e0c286e,0xa1a41737),LL(0x5da3472d,0x3b668502),LL(0x7576f2a2,0xbc0d116b),LL(0xa36a27d4,0xfdbcad95)}, {LL(0x9d54f7ee,0xcdb3f474),LL(0x8a5643a3,0xe2e0f5f9),LL(0x69d4f171,0xc70d11b9),LL(0x6cca4ef7,0xdf96d136),LL(0x2fc6afdf,0x570693db),LL(0x567504da,0x5059e67b)}},
   {{LL(0x7fe632a2,0x2c8107d4),LL(0xede7bff8,0xfc46c745),LL(0x4650025b,0x2d3b1286),LL(0xe74cd65f,0x815ef3cb),LL(0xa256f01c,0x5431b01b),LL(0x39915cfa,0xe832ff11)}, {LL(0x07d7af84,0x2c106de6),LL(0x6d4753e7,0x67303b78),LL(0x6d75c8de,0x5f886ffa),LL(0x967131cf,0x932a6c20),LL(0x70aebbb0,0x5bc94a91),LL(0xfd56e06d,0xa85b3044)}},
   {{LL(0xe7eba799,0xc904558a),LL(0xb2fa7331,0x46b6031b),LL(0x0653675c,0x6620e2b5),LL(0x7d2218f7,0xd1373a35),LL(0xaf55a5e7,0x0f4b3ca3),LL(0x714e70c2,0x50774160)}, {LL(0x69188455,0xacc63d14),LL(0x043b8b30,0x89a795fe),LL(0xe1e4b9cf,0xac2fd66c),LL(0x1bf67f26,0xac792702),LL(0x1143d437,0xb9513f0d),LL(0x811f2931,0x02198050)}},
   {{LL(0x7b480776,0x6d4acdba),LL(0x66dffeb5,0x8b518cd4),LL(0x51918859,0x8826c994),LL(0x38fad835,0xd2b6a7a0),LL(0x6929a870,0xd315417a),LL(0xc5a769e1,0x05d85252)}, {LL(0xec0d091e,0x2fa06335),LL(0x87768c88,0xb0cc3372),LL(0x58a2eb9d,0xacbda5ba),LL(0x76b7b057,0x2a404fc9),LL(0x838c6135,0x073abb71),LL(0x5cfc4f3c,0xbdf89b13)}},
   {{LL(0x3508675f,0xd00eb9c5),LL(0xa117dc95,0x92ec76a4),LL(0x334ca15c,0xf58d6f85),LL(0x9cee0544,0xeeb52216),LL(0xf21457c2,0x3eb9847f),LL(0x5524c60d,0x547908bc)}, {LL(0x5198709b,0xb5b49d22),LL(0x324abc67,0x718abce6),LL(0x4abd54ba,0xdab8ff2f),LL(0x7184d444,0x98be59e6),LL(0xbabeb4b0,0x45b74b54),LL(0xff71a5ac,0xd8d8bb30)}},
   {{LL(0x8ec13e6f,0x8aedf7e2),LL(0xd950792d,0x8b952620),LL(0x04918f59,0x36e9dac2),LL(0xd3dd47ed,0x5e49a5a2),LL(0xe863c2bd,0xb17455be),LL(0x326a0d66,0x8caac6a9)}, {LL(0x27bb72e1,0xb6c3f5e4),LL(0xee5fe09b,0x17566c9d),LL(0x5e3db64d,0xfd6bbcc2),LL(0x3189319c,0xd437d07a),LL(0xcd3166a5,0xad00dfc4),LL(0x0bd63003,0xab75927b)}},
   {{LL(0xafc43be8,0xa7672a39),LL(0xc72f97aa,0xefc49015),LL(0x0e48f2ed,0x81c63c05),LL(0x833a22cc,0x62f39f32),LL(0x72c0c0c4,0xf7a34801),LL(0xa4158538,0x4711cd41)}, {LL(0x1d15f2f3,0xa3c99a4d),LL(0x7bee1b47,0x4b82c1c1),LL(0x9d199f10,0xc7d60b48),LL(0x5f16fa95,0xd1d1f03d),LL(0x32fbeaf2,0x96c780c9),LL(0x376ff106,0x0662e250)}},
   {{LL(0x78571c8f,0x728e3346),LL(0x6cb339d3,0xd0a886b5),LL(0x0a5671ba,0xf4ea3338),LL(0xa64850a4,0x43823401),LL(0x33117b9d,0xa7729cd5),LL(0x2b78cffb,0x4dd45760)}, {LL(0x5a67d812,0xbe057111),LL(0x5105a3fc,0x7ec6cf92),LL(0x0ccafeec,0x5dbcb4bc),LL(0x803092f8,0xa7587f15),LL(0xa884efad,0x67ee61d5),LL(0xca47d9ca,0xd4ced554)}},
   {{LL(0x3b03dcbc,0x02c6b608),LL(0x3b9d868a,0x2b20149e),LL(0x4f57eb0c,0xaf5ab01d),LL(0xd750e515,0x59935b94),LL(0x8f89ad68,0x32721b40),LL(0xa7e3ceff,0x673bd755)}, {LL(0xef3b3393,0xbd462fd8),LL(0x0e59a120,0x99142264),LL(0x9263fa61,0x4162da61),LL(0xb6488eb3,0x2ed1f2de),LL(0x725680c4,0xb0bd37a8),LL(0x29ec27b0,0x17218bf0)}},
},
/* digit=22 base_pwr=2^110 */
{
   {{LL(0x4bd044ef,0xfb09e756),LL(0xbb964fb3,0xdbc9fcdf),LL(0xcdb1f4f5,0x451c5b01),LL(0xf1dd1cf0,0xb02f9068),LL(0x0c687e41,0xd4765e7c),LL(0xd1967bd3,0x89b64981)}, {LL(0xf0439d65,0x06a0e4ec),LL(0xa5abbcec,0x564c387d),LL(0xc651d806,0xc1e9d01a),LL(0x0618a96c,0x5e6ebd83),LL(0xc54ad8ce,0x9ce1aace),LL(0x9953f90f,0xe5248a08)}},
   {{LL(0xbb296c27,0xda2b0725),LL(0xd341171b,0x1f22ffa4),LL(0x5b132756,0xc721e35a),LL(0xfadb6907,0xe5695e84),LL(0xc283f546,0xbc5a3bf4),LL(0xdde128ae,0x9182cb3e)}, {LL(0x6592e05e,0x179c7fa6),LL(0xf38e8586,0x1e604790),LL(0xa16bad55,0xaf7e83be),LL(0x9137ecd8,0x6f41231e),LL(0x8f30d1ab,0xac87543d),LL(0xb1ee0ee8,0x630a9d87)}},
   {{LL(0x0850b471,0xd8ccf550),LL(0xf7ebfcd8,0xea8f73d2),LL(0x7f138136,0xfb374f99),LL(0x6cd70e73,0x6b3a1ab8),LL(0x42f40008,0xc4577c13),LL(0xa0517e75,0xb23800f9)}, {LL(0x7088a19b,0x79deaa45),LL(0x702183ed,0xa8410f51),LL(0x2c9f6594,0xe2761fa1),LL(0x52868276,0xed1d0112),LL(0xc22157a4,0x9e9805d7),LL(0xac7653e9,0xbe4aa213)}},
   {{LL(0x5e4f1914,0x359cbfa0),LL(0xd922176a,0x6992cd48),LL(0x630cbcb5,0x2fd5e90f),LL(0x6ddbf238,0x2b0f6130),LL(0x3af753e6,0x5070970a),LL(0x41727016,0x433d0094)}, {LL(0x9dca97e2,0x2eb15b3b),LL(0x00d96875,0x3a7379cd),LL(0xe930a75c,0x0d4374ae),LL(0x849e7727,0x3cd36046),LL(0x336a19ff,0x9ac3117e),LL(0x433ddc04,0x2d1373c9)}},
   {{LL(0x13a2a0f2,0xa60026f0),LL(0xd6d8a91c,0x173b08a8),LL(0x2b0ff7fb,0xfa6b3eea),LL(0x4201b05d,0xd5417c7e),LL(0x598f4c56,0x91dacda7),LL(0x0a1513a5,0x00323d9a)}, {LL(0x38bf2619,0x079a4ef1),LL(0xee3512cc,0xfe91faae),LL(0xbc03468f,0xbf9aabdf),LL(0x01d3ca1f,0x0d46d3a2),LL(0x3e677e61,0x6bfad511),LL(0x6dabf925,0x219a70aa)}},
   {{LL(0x8faa3eca,0x8e145869),LL(0x511c5e09,0x811c0ab1),LL(0xc39fd20b,0x08a52f04),LL(0x5e5f514e,0xa7c22263),LL(0x8507e5c6,0x365701af),LL(0x495177cc,0xba1c1ce2)}, {LL(0xbd3d7f93,0xed128050),LL(0x05af7fa4,0xf5b3a508),LL(0x7eb1e384,0xe27a017c),LL(0x1fbca0a7,0xaa95d99a),LL(0xbaa36562,0x61a459cf),LL(0xb7f845c1,0xc1ee4d7b)}},
   {{LL(0xa4945d9f,0x164a4fa1),LL(0x8ce7f161,0xf9fb0327),LL(0x9891eb36,0x5284c704),LL(0x9bd23713,0x9ca1ee6b),LL(0xbdc043e0,0xb364fe43),LL(0xae82eb35,0xcf9c9e70)}, {LL(0xbcde9b87,0xa375b9d5),LL(0x6d5ada37,0x55f437d7),LL(0x85bf9126,0x49fc8d72),LL(0xbd83c1bc,0xf13cbab7),LL(0xf847972a,0xc3d75306),LL(0xb1da55aa,0x8c27482a)}},
   {{LL(0xe4362d67,0x985dcb43),LL(0xa939bea4,0xecb860c2),LL(0x55fbf1d5,0x40597f30),LL(0x43fcd98a,0xb6d166bf),LL(0x15ec99ca,0x59325709),LL(0xc5bdd370,0xe05ae3b9)}, {LL(0x0c7b943b,0xc18f7827),LL(0x4dd572cc,0x84bde9c6),LL(0xf478e56b,0x5d50a89d),LL(0x64d29053,0x242c2f48),LL(0x61cf7e0e,0xcda12c61),LL(0xac8d1d40,0xf8b6890e)}},
   {{LL(0x37d03584,0x3f2c311e),LL(0xbb91574c,0xb387b3cc),LL(0x8d7aa7af,0xa64914d1),LL(0xea0fb673,0x7586999a),LL(0x14e5014b,0x25ee7a70),LL(0x557fce0f,0x1015b142)}, {LL(0x10cc9d92,0xbb839712),LL(0xabbb3b2a,0x31118426),LL(0xa29866e5,0xc3a4fa6f),LL(0x66310a76,0xbdb3a495),LL(0xa11055e7,0xd397ceb6),LL(0x0240bd0d,0xc8d47e10)}},
   {{LL(0xfc8d5f6c,0x76624aad),LL(0x0144a0ff,0xb7dbb41a),LL(0x30df5bea,0xd7be190f),LL(0x9ed58b6a,0xb61e3e97),LL(0x14bf09f6,0xce192eb3),LL(0x27ad3fc4,0x3d6632af)}, {LL(0xcdd7c93b,0x51f180e8),LL(0x932c89f0,0x257df031),LL(0xb680f621,0xed1ffbcb),LL(0x34ef3db2,0x6d8be601),LL(0x044b7f8a,0xc32869e3),LL(0x09958ada,0x8db40995)}},
   {{LL(0x7ebfc0d7,0x004888a9),LL(0x0f90df8a,0x5546c1b1),LL(0xddfb6a92,0x9c73aa4e),LL(0x2782e2a4,0xb492a723),LL(0x2b9b9390,0x5e79ce87),LL(0x2564d95a,0xb8165f79)}, {LL(0xa8f207cb,0x235665e2),LL(0x1087fea6,0xdef16762),LL(0x7b9d891a,0xf4275ea2),LL(0x1f98a6d9,0x70b2662a),LL(0x5fd15926,0x2990c521),LL(0xc30089d5,0x2734975d)}},
   {{LL(0xb6b11af8,0x7a9a9dd7),LL(0x16a42f8f,0xac4c50dd),LL(0x6726c324,0x1550267b),LL(0x7f841afb,0x12cfb2fd),LL(0x2f046430,0xa41b1905),LL(0xb7d786cb,0x8f5f5373)}, {LL(0x729e09fb,0x7b61b39c),LL(0x2626da4c,0x07a5415f),LL(0x4a348905,0x65d6efb8),LL(0xf2e864b9,0xf377862c),LL(0xab96cdb2,0x3c94000d),LL(0x8efef0f6,0x2176321a)}},
   {{LL(0xaffedcf4,0xfbd2489f),LL(0x41feb03e,0xe41dfcac),LL(0xe9a86cd1,0xf13f6579),LL(0xb7ff732f,0xfe3c2a91),LL(0x82f008fc,0xd447c728),LL(0x0e1924b1,0xe4e9c054)}, {LL(0x048bb350,0xe0d6bfc3),LL(0x2fb5fc5b,0x5fe1c204),LL(0xaa50972b,0x70799d49),LL(0xd2c42b50,0x7c79c24e),LL(0x5405cd06,0x4ffa2c53),LL(0xc7ba6f8b,0xd890bea7)}},
   {{LL(0x94747d5c,0xee68456d),LL(0xef40112f,0x5c0312fe),LL(0x0fcb2565,0x57a480bc),LL(0x7c529c68,0x549b31ad),LL(0x402bdb03,0xcfdc5d33),LL(0xbf45407e,0x955bfd5d)}, {LL(0x1fdb20fd,0x8381a1ec),LL(0x4fe0b38e,0x89dafbdc),LL(0x68f59249,0xf33c8706),LL(0x68be67d2,0xb6cc7618),LL(0xea4845b8,0xa7897c7c),LL(0x93f2f9a9,0x6773f08b)}},
   {{LL(0x14a03fc5,0x642405c1),LL(0x4cf34763,0x5b973584),LL(0x3331c03d,0x9f24c2cc),LL(0x6f9bf4f5,0x56428a37),LL(0x703e5e73,0x2e83d070),LL(0x573c3cf4,0xb8e4e61a)}, {LL(0x8f3e3efe,0x7a18ec58),LL(0xb812e5a7,0xbab02660),LL(0x8289de63,0xee826e36),LL(0xc1a8eb16,0x86d263f9),LL(0xd7b85103,0xee8f3e72),LL(0xe567c787,0xa3627d9b)}},
   {{LL(0xc1c1bc68,0x8c558000),LL(0x83fd6ca6,0x9e48a67a),LL(0xeb7a35cb,0xacf0d75c),LL(0xf0a93110,0x0fbdce4c),LL(0x9cc50c85,0x82b2d13a),LL(0xcef70d6f,0x696fd259)}, {LL(0x457b88c2,0x1cc9be2e),LL(0x1f04c0bc,0x0d58b34b),LL(0x195a532b,0x52bd479f),LL(0x1ab3605c,0x769fe6ca),LL(0x8a24c1e0,0xba6a63e4),LL(0x99da5d7a,0x86dea462)}},
},
/* digit=23 base_pwr=2^115 */
{
   {{LL(0x65682820,0x509abccb),LL(0x4ff86137,0xfbfa1d09),LL(0x640bc2f6,0x1ae371bd),LL(0x8f546c68,0xa155c297),LL(0xc08b8cbf,0x8858cadc),LL(0x1d96948b,0xafac5b0a)}, {LL(0x82e25016,0x919cb226),LL(0xb064ffc3,0xd147df4a),LL(0xb4abe560,0x25dd0f1a),LL(0x9cb75bd1,0xc6bbe636),LL(0x47a778e4,0xbb367cf9),LL(0xbde524b3,0x5714aa4d)}},
   {{LL(0x166915df,0xc6307399),LL(0x5da8a26e,0xb35545bc),LL(0xe3a99321,0x8e0126aa),LL(0xda9308d1,0x0fbfdf76),LL(0x6168e505,0x2163ed6b),LL(0x7500d8bd,0x71f3d008)}, {LL(0x5ac13f65,0xf5715960),LL(0x55172d5b,0xc1cd9a67),LL(0x53d84c65,0x6b225f7e),LL(0x025029da,0x9c031269),LL(0x17d89aed,0x54c1edfa),LL(0x6b435150,0x5b023878)}},
   {{LL(0x1e94d949,0x1e7ae160),LL(0xe78e6221,0x177dc53e),LL(0x7aeb9882,0x8af29d8f),LL(0x9e3f3906,0x2d9a60fd),LL(0xdf962156,0x6979fcd6),LL(0x7e1e54b8,0xdd2fe588)}, {LL(0x76643453,0x9cccf310),LL(0x4e0643d0,0x94ece1a8),LL(0xc111d8cf,0x745449cb),LL(0xe6cfbd97,0x872afa4a),LL(0x10dfb34e,0x5c27b7ca),LL(0x533480fe,0x505e62bc)}},
   {{LL(0xae238fa1,0x009ef80a),LL(0x486af6b5,0xb41d9b24),LL(0x685add95,0xab4455ed),LL(0x72c7dac8,0x18f323f6),LL(0xe372f168,0xe7009790),LL(0x067bea99,0x4d5bcba6)}, {LL(0x51a2a9a9,0xf15bdbcc),LL(0x2fac9169,0xde7e4f74),LL(0x4bdbec36,0x2fd62c30),LL(0x1b3ac6c7,0x8b3ea93d),LL(0x9c293889,0xce1c8e5c),LL(0x11564f8b,0x19664dda)}},
   {{LL(0x175418b2,0x5f266f13),LL(0x03a626f9,0x6bd7a869),LL(0x4a6f11ca,0xc7b53230),LL(0xbfc8cced,0xa216b056),LL(0xb288cb7c,0xa274d5d8),LL(0x924897ad,0x6fc4a35d)}, {LL(0x81fc6940,0x1ea532eb),LL(0x2acbbc45,0x2fcd817e),LL(0x67814fa3,0x45eee93f),LL(0x1229e035,0x3b3da48c),LL(0xefd8e3d7,0xd049a976),LL(0xbf81f314,0x8087dff7)}},
   {{LL(0xbd366155,0x77faac22),LL(0x282f11b5,0x13cc4038),LL(0x5fbd35ab,0x31ad1dd4),LL(0x45d6d40e,0x7e0de9da),LL(0x39749ef6,0xa16c5f19),LL(0x85691cf2,0x761cd6cf)}, {LL(0x4d59b802,0x156536ad),LL(0x87c4b11d,0xee98dc41),LL(0xd35088fc,0x165a1eac),LL(0x38fb995c,0xce8a7335),LL(0x3293b3a5,0x34d0d331),LL(0x8b570e79,0xfcf548ca)}},
   {{LL(0xa85af34a,0x7be5946e),LL(0xda6fb0e0,0x420593c9),LL(0x987f9246,0x40b83c00),LL(0xa15d192b,0xac35f4e9),LL(0x776a678c,0x1979bd33),LL(0x8f6068d3,0x0a7d973e)}, {LL(0x7e6298fe,0x71d322e8),LL(0x36af9b65,0xbb23a299),LL(0x6644c50c,0x14e2b970),LL(0x73570bd3,0x5f7f2073),LL(0x9055538b,0x40215c56),LL(0x365500c9,0x91372e64)}},
   {{LL(0x9af0a75c,0x6a3a2327),LL(0x9f1f250d,0xf832a815),LL(0x22a82d3f,0x17030c33),LL(0x14cbc835,0x24bf18ea),LL(0xb2da2727,0x319dc4ca),LL(0x6d020d4a,0x481df360)}, {LL(0x7fc22ba5,0xaeebdd8a),LL(0xa91e28ab,0xbd0515c6),LL(0x595f361d,0xfc8a2978),LL(0x1ae8fa3c,0xe60dd96c),LL(0xa5341575,0x19c2109a),LL(0x06a0ee48,0xfd6e92bb)}},
   {{LL(0x77b5c7c5,0x650008f9),LL(0x02a6d087,0x4bcf6002),LL(0x82234273,0x391ebfa3),LL(0x86cd884e,0x9dcb05d3),LL(0x8753b3fa,0x1b5e7afa),LL(0x2d1e513c,0xd453e9b4)}, {LL(0x4b3a74b4,0x3b1a0dea),LL(0x5bebd592,0xf8989aac),LL(0x61dc640b,0xac3ec9f5),LL(0x6c4b301b,0x4ba9dd0a),LL(0x04e48df6,0xf686a5fe),LL(0x0631d1e6,0x8b0d2d76)}},
   {{LL(0x03c87a20,0x443deaab),LL(0xbb817740,0x57dbd224),LL(0xd51d6acd,0xf88918de),LL(0x7d4ec9be,0x48c9b2f1),LL(0xdfc48a69,0x78b41104),LL(0x78827f54,0x44a81443)}, {LL(0x16a22495,0x74ed7949),LL(0xea92d3e6,0x6b9e9128),LL(0x0d9fe252,0x6fe6449b),LL(0xc14e825a,0x31743d06),LL(0x08690f00,0xd9d4ad8c),LL(0x0e65f748,0x562e2f60)}},
   {{LL(0xb1befb0a,0xe9e01117),LL(0xd25cb1ae,0x7ce74721),LL(0x2d4437ad,0x946216c6),LL(0xea8b2d85,0x83b03131),LL(0x2c4fa895,0x3614b15d),LL(0xc23c6bf7,0x393317d9)}, {LL(0xdcd80eb4,0xda73495a),LL(0x6338db53,0x850dfb46),LL(0xc9b8943a,0x3329c498),LL(0x0765d94a,0x1ee615b5),LL(0xd67c17b3,0x2d16a6cf),LL(0x6f8a9ed5,0x050d6bfe)}},
   {{LL(0x8b7f8cb2,0x92501ff8),LL(0x578a4cfd,0xcdaa6c93),LL(0xe5303846,0x95b99a05),LL(0x2543d1a9,0x30a72fdc),LL(0x11808771,0x6648126b),LL(0xeefb8145,0x6980dd99)}, {LL(0xf1949aad,0xd9a87f83),LL(0xf7b8ab95,0x3b4f208d),LL(0xc40122e4,0x3341e118),LL(0x853b291e,0xe513567f),LL(0x3b565bad,0xd4f1bbb7),LL(0xa7658f77,0x86207b43)}},
   {{LL(0xf87cd72c,0xe5737dce),LL(0x7a1c9f7f,0xdf41230a),LL(0x34e0d519,0x7fe5cfcf),LL(0x26033c9d,0x83cc1da5),LL(0xe16dffd0,0x4195ef1a),LL(0x8af91671,0x7c388f6e)}, {LL(0x62283e40,0xeb1002b8),LL(0xa96be04c,0x08122e57),LL(0xdf9fac7c,0x48cb2d0c),LL(0x41f0946d,0xa3c007ae),LL(0xcec27088,0x4d6fa5c5),LL(0xdcd5fcfa,0xe1fbc03e)}},
   {{LL(0x9bece059,0xbab37a63),LL(0x7fa28fc3,0x5a60ec87),LL(0x1d4a8f5b,0x69e1086c),LL(0x65814282,0x3a873844),LL(0x2676cc7d,0xb642dc16),LL(0x97e423a0,0x8d152bb9)}, {LL(0x160ff40b,0x66ac1b37),LL(0xf657547a,0x7410ba67),LL(0xadcc6823,0x8f47debb),LL(0xb1536fa1,0x44e9bf64),LL(0x939e5461,0x1cfdd504),LL(0x43bb3b90,0xba5729e1)}},
   {{LL(0xdd63f6d4,0xb5c26108),LL(0x60229177,0xefb8dfc7),LL(0x23e4cc7d,0x9d980889),LL(0xc211c92b,0x59579ee8),LL(0x35fe17b4,0x6f146300),LL(0x0ca61e4a,0x6c1e7075)}, {LL(0x19d582e1,0x8671156d),LL(0x07230605,0x36a3d858),LL(0xa352dcdc,0xd9314e61),LL(0x4c6bc2b1,0x7a9d0de7),LL(0xc894ab77,0xb57d3f71),LL(0xd4e5a04e,0xb3564bc7)}},
   {{LL(0xcde5e785,0x2e32f896),LL(0xb9db8f31,0xcd55ae7a),LL(0x8f832885,0x278db1ad),LL(0xadcbd933,0x271d9078),LL(0x4a64f863,0x2208fae3),LL(0x39c89365,0x974046e0)}, {LL(0xb3cd0cd3,0xcb46f272),LL(0x74e59edc,0x31f34e1a),LL(0xedd50418,0x3421d316),LL(0xcabe36ed,0xb1d8a064),LL(0x362efcda,0xdb13e560),LL(0xc71eb3ee,0x567c2b6c)}},
},
/* digit=24 base_pwr=2^120 */
{
   {{LL(0x70d4d7bc,0x2af8ed81),LL(0xb632435c,0xabc3e15f),LL(0x78219356,0x4c0e726f),LL(0xb87254c4,0x8c1962a1),LL(0xc9e7691a,0x30796a71),LL(0xa75a12ee,0xd453ef19)}, {LL(0x13ae4964,0x535f42c2),LL(0x0da9586a,0x86831c3c),LL(0xe39a7a58,0xb7f1ef35),LL(0xd459b91a,0xa2789ae2),LL(0x02fd429d,0xeadbca7f),LL(0x65290f57,0x94f215d4)}},
   {{LL(0x4b950889,0xc0855002),LL(0x8ce24da0,0xee99dbfe),LL(0x4318e860,0xdda71d96),LL(0x04fe9b85,0x01d3d396),LL(0xe25e7e20,0xda4bc065),LL(0xe076c81c,0xd3a50b87)}, {LL(0x31e5f494,0x5b9f8219),LL(0xa6a1b821,0x6a140527),LL(0xd8dd159b,0xf52683e4),LL(0x20b18043,0xca9c8887),LL(0x08a0d8f5,0x73c040fa),LL(0x179525c4,0x92e482e8)}},
   {{LL(0x1cfb79ac,0x37ed2be5),LL(0xe7af84c3,0x801946f3),LL(0xe77c2f00,0xb061ad8a),LL(0x44de16a8,0xe87e1a9a),LL(0x7ee490ff,0xdf4f57c8),LL(0x005993ed,0x4e793b49)}, {LL(0xbccb593f,0xe1036387),LL(0x95e09b80,0xf1749411),LL(0x5ab42f91,0x59cb20d1),LL(0xac0ff033,0xa738a18d),LL(0x2ac1e7f4,0xda501a2e),LL(0x84d8a6e0,0x1b67eda0)}},
   {{LL(0xa4d2313e,0x3a828904),LL(0x92e66888,0xbf4946b1),LL(0xe5fa19d2,0xc574898a),LL(0x5e1c5fa4,0x0b13dbb6),LL(0x7c390fc2,0xf11343ba),LL(0xd7d32187,0x35b1418f)}, {LL(0x83e7fe7b,0xc92cb1bb),LL(0xd78365c4,0x0b969455),LL(0x672f2af7,0xda69dfe5),LL(0x30932441,0x9c62d7b4),LL(0x94af02d6,0x165672ad),LL(0xcde81c22,0xd2cc734d)}},
   {{LL(0x1080e90b,0x1d27efce),LL(0x3fd01dc6,0xa2815246),LL(0xcaa26d18,0x99a3fb83),LL(0xb82babbe,0xd27e6133),LL(0xd783dd60,0x61030dfd),LL(0x73c78cb8,0x295a2913)}, {LL(0x68be6a92,0x8707a2cf),LL(0xeeb3474a,0xc9c2fb98),LL(0xa2b176b8,0x7c3fd412),LL(0xc7202101,0xd5b52e2f),LL(0xf0a6d536,0x24a63030),LL(0x04648ec0,0x05842de3)}},
   {{LL(0x33f4d416,0xd45e3501),LL(0x4bf9131e,0xbb40233a),LL(0xe302483a,0x1532a088),LL(0x2c2485c0,0x3475e8b8),LL(0x969cdbe6,0x08f9ea56),LL(0x253cd738,0x31928645)}, {LL(0xac9836be,0x1cf323a4),LL(0x02b6e4de,0xdf647ccf),LL(0xc06f3d09,0x9a31e84f),LL(0x39efe6d9,0xd326b86e),LL(0x14ac4dec,0x77e3e1df),LL(0xf3e0c582,0xf2d5917a)}},
   {{LL(0x30577ac9,0x67477cdc),LL(0x244f92a8,0x51dd9775),LL(0x917eec66,0x31fd60b9),LL(0xd66c5c1d,0xacd95bd4),LL(0xbf9508ba,0x2e0551f3),LL(0x688cb243,0x121168e1)}, {LL(0x4540d230,0x8c039740),LL(0x009ecdf9,0xc4ed3cf6),LL(0x44db62af,0x191825e1),LL(0xc4a030da,0x3ee8acab),LL(0x94081504,0x8ab154a8),LL(0x486c9cd0,0x1fe09e4b)}},
   {{LL(0x02cf37fd,0xe92b56c0),LL(0xf71b34de,0xa75bbcb0),LL(0x50f5c482,0x7754d0ef),LL(0x11fa89fe,0x850a9ef6),LL(0xba4ea7d8,0x97d74b1b),LL(0xaab7ba2e,0xfc757c25)}, {LL(0xf2a67fdd,0x06f30ab0),LL(0x12e72af8,0xb10aba14),LL(0x7a2e053d,0x47580bca),LL(0xdcf0e14c,0x85795598),LL(0xd6f55310,0xc3596781),LL(0x4c9b7e18,0x8ab251b7)}},
   {{LL(0xd113450b,0x512f82f9),LL(0x2dbc9197,0x5878c901),LL(0xe13f355b,0xdb87412b),LL(0x935b8a5e,0x0a0a4a9b),LL(0xf25a5351,0x818587bd),LL(0x31e3d9c7,0xe8079310)}, {LL(0x611bc1b1,0x8b1d47c7),LL(0x72a823f2,0x51722b58),LL(0x53b36b3e,0x6f97ee8a),LL(0x946dd453,0x6e085aac),LL(0xe65e6533,0x2ec5057d),LL(0x4bb18801,0xf82d9d71)}},
   {{LL(0x4dbb8798,0x4ab13850),LL(0x72d04cd2,0x0e7980d7),LL(0x0b3271c6,0x1755c566),LL(0x9d9d1468,0x8414efb0),LL(0x1795ce66,0x61a58630),LL(0x232924a1,0xb6a8b393)}, {LL(0xae031bd6,0xa992f0ce),LL(0x2915acc1,0x6747fb5f),LL(0x93e9c0d2,0x03daa266),LL(0x5400d554,0xc18fa364),LL(0x9497e895,0xaf04ff8d),LL(0x50b6b339,0x86c3cfc2)}},
   {{LL(0x8ba5aa8e,0xad81fa93),LL(0x8f7aa69e,0x723e628e),LL(0xef35937c,0x0ba7c2de),LL(0x6decfb40,0x83a43ec5),LL(0xe60c4f2d,0xf520f849),LL(0x457e3b5e,0x8260e8ae)}, {LL(0xbf1d9ed7,0x7ce874f0),LL(0x7f1a5466,0x5fde3553),LL(0x0c162dbb,0x5a63777c),LL(0xdad87289,0x0fd04f8c),LL(0x640761d5,0xca2d9e0e),LL(0x38501adb,0x4615cff8)}},
   {{LL(0x04e1e6e3,0x376b2a7f),LL(0xa31774b4,0xea0dcb70),LL(0x5cbdec2e,0xfc7fe4cc),LL(0xf03f459e,0x8568499d),LL(0x8b78900e,0xe9fd8fb2),LL(0xe431bf97,0xd33c6e30)}, {LL(0xc896e766,0xd904b8f5),LL(0x82748cef,0xa8f577cf),LL(0x87e044b3,0x93dd921b),LL(0xf76eebe9,0x23d79837),LL(0xe569feeb,0x5e0a7493),LL(0x414dddb6,0xd0797549)}},
   {{LL(0x110b4a25,0x9422789b),LL(0x70ad8cc1,0x5c26779f),LL(0xec4f1e14,0x4ee6a748),LL(0x5c7ab5e0,0xfb584a0d),LL(0xfb21ee66,0xed1dcb0b),LL(0x11c6863c,0xdbed1f00)}, {LL(0xb1b1d187,0xd2969269),LL(0xafe964e6,0xf7d0c3f2),LL(0x12bb865e,0xe05ee93f),LL(0xed79118e,0x1afb7bee),LL(0x0f0fe453,0x220af138),LL(0x52782ab9,0x1463aa1a)}},
   {{LL(0xbfe5b1a7,0xfd9e3542),LL(0x75938cea,0xb42d2a41),LL(0x3befb760,0x74688a15),LL(0x2e33dbe7,0x8daeeaa2),LL(0x3e677801,0xc9c1ea08),LL(0x34effe1e,0x68ecf4e4)}, {LL(0xd294c321,0x927700cc),LL(0xe940afc5,0x9e2e723d),LL(0x7cf6cd43,0xbcfac07a),LL(0xd1006bc3,0xa009ef94),LL(0x373d13e3,0xa02016b0),LL(0xabae5822,0x4e097adb)}},
   {{LL(0xd7dbe5f9,0x7c139d56),LL(0x0b83685b,0xfc16e611),LL(0x9018463c,0xfa723c02),LL(0x840bf5d7,0xc472458c),LL(0x0af07591,0x4d809359),LL(0x3308dfd9,0x418d8830)}, {LL(0x0c365ae3,0x9b381e04),LL(0xf8190fd1,0x3780bf33),LL(0xdd03e854,0x45397418),LL(0x4e51e491,0xa95d030f),LL(0xe3286cea,0x87c8c686),LL(0x900b5f83,0x01c773bf)}},
   {{LL(0xc898b8bc,0x8db8b78c),LL(0x502940cd,0x686896da),LL(0x2dde2e3c,0x67e50f02),LL(0x8cbf406c,0x2e2461f3),LL(0xe1f7ff60,0x32182781),LL(0xe30e2688,0x26934b05)}, {LL(0xfc4494f6,0x95adc204),LL(0x161b7499,0x4c7f30c5),LL(0xb7341737,0xd5caf060),LL(0xd128d46c,0xed93187f),LL(0x20fc1e04,0x3f2819cb),LL(0x2b7f70a1,0x48c4086f)}},
},
/* digit=25 base_pwr=2^125 */
{
   {{LL(0x0ece13ae,0xd74d09c1),LL(0x57a6bd95,0x5e59d9e0),LL(0xe132b940,0xdb1ccfdc),LL(0x843d3c66,0xa0e5309c),LL(0xf9cb3ef4,0x1fbd03a5),LL(0x00ea5177,0xcdc9ef0a)}, {LL(0xcb784a6b,0x1ebf5a15),LL(0x8a0d109a,0xa67382af),LL(0xa0d34d15,0x3256c37a),LL(0x0fca43af,0xee40efa5),LL(0xb9841bde,0xc299bbd4),LL(0x3bef4a0b,0x6df68f60)}},
   {{LL(0xd9d7c50a,0xe01295fd),LL(0x67f8ef0d,0xaf31b4ea),LL(0x9eaf8eb7,0x2ec9689f),LL(0xc622acc5,0x327b96c5),LL(0xb2757f2a,0xae918f81),LL(0x4fd6606e,0x74927d68)}, {LL(0x18574215,0x09bb7fce),LL(0xe8e68b72,0xfea383bc),LL(0x5fb47511,0xdf2a6f12),LL(0x8e399520,0xbe88faa1),LL(0x3fb1c3a2,0x0166d57e),LL(0xe525f81f,0x5907ef2f)}},
   {{LL(0x57797e5c,0xf24fab9b),LL(0x3add682c,0xd8500987),LL(0x33ac4fb5,0x75e370e7),LL(0xc8a69765,0xd8d61642),LL(0x5e30aa0e,0xf057f10a),LL(0x4f1e8637,0xaf953087)}, {LL(0x701cd1a0,0x887f06b3),LL(0x8a200082,0xd7c6a9c6),LL(0x6e6483a0,0x4319f7eb),LL(0x475d4bfd,0x44f1aaff),LL(0x0f811fc7,0xa8ee73b5),LL(0x82916f8b,0xbb67bcdc)}},
   {{LL(0x8a37f660,0xdefe3a7b),LL(0x858f5765,0x7898db8c),LL(0x73d1f9b4,0x7366c26a),LL(0x237ae8b7,0x35d5d718),LL(0xb4478259,0x3efb20fe),LL(0xaa545ee3,0xccd0fed7)}, {LL(0xed22d152,0x750edd05),LL(0xee20d4c6,0x4f8020f9),LL(0x0a9e29dc,0x16e60f37),LL(0xbfbec7f6,0x9cf0a136),LL(0x2e47e143,0xb430a34b),LL(0xc6cdd1a9,0x2e2560bb)}},
   {{LL(0x2a9af85a,0x2cb5625c),LL(0xfea96b60,0x9ba19601),LL(0xd7549809,0x25dd6aca),LL(0x5b4c31e9,0x9c02e613),LL(0xe09dbb63,0x40c4baea),LL(0xf49beece,0x38455db7)}, {LL(0x5b848716,0xe9846d0d),LL(0xd2ebce35,0x9c63043d),LL(0xbc26a79e,0xedee8e86),LL(0x0869c85f,0x70c46f94),LL(0x7bc49e34,0x34b05934),LL(0xa80d7d8e,0x737fe693)}},
   {{LL(0xe4161a65,0x799352da),LL(0x56253ce6,0xe5cf7ad8),LL(0x6de32775,0xf606bf79),LL(0x57fce8db,0xddc0f3a3),LL(0x16cf4a47,0x1075fc23),LL(0xb27c5ad8,0x078f0e04)}, {LL(0x3f7100aa,0x9fc47795),LL(0x4673ffa2,0x3ac48925),LL(0xf9cd8348,0xb8263f42),LL(0x68cc92d5,0x5bdfde30),LL(0x1ac37f9c,0x2250927b),LL(0xb33da359,0x26ec8328)}},
   {{LL(0x236eaa28,0x76923635),LL(0x657d0640,0x5acc650b),LL(0xc20a58ab,0x2652ec7e),LL(0x1a5459f9,0x93385f9f),LL(0x306e2091,0x45952d39),LL(0x34cef2f2,0xb31f5a42)}, {LL(0x1f3c055f,0x0d9b0484),LL(0xae3b1f4d,0x6c5ef8ee),LL(0xbd24becd,0xd3fdb464),LL(0xa2210148,0x49b11a10),LL(0x1472893a,0x73de2855),LL(0x6ee531f7,0x99435b57)}},
   {{LL(0xc88d568a,0xf186d6bc),LL(0x528535dd,0x872bc4c7),LL(0xdfe64dc3,0xc9e7432e),LL(0xd795ea57,0xd9fc4832),LL(0xc845af2b,0xf4ffdb81),LL(0x2b670517,0x66d7e788)}, {LL(0xd7b7a1c6,0xa7c1be04),LL(0xd5b2a249,0xbed88479),LL(0x03f2ef6d,0x62ff8aba),LL(0x20dc701d,0x60ecaac4),LL(0x4ff10119,0x9f4b559f),LL(0x3cd54fd0,0x0582c931)}},
   {{LL(0x3940d5dd,0x271f5d70),LL(0x192fb2b6,0x7707d109),LL(0xda210560,0xc6131828),LL(0x0bc534c0,0x21750048),LL(0x23487a61,0xa96593d1),LL(0x0b9c6e92,0xdab1698b)}, {LL(0x1f7abe02,0x8a7951a5),LL(0xecd49e72,0xa9394a4b),LL(0x2f6f53d5,0x95ba141d),LL(0x6cdb90cc,0x03809379),LL(0x0586a3ef,0xa5e0ebf3),LL(0x70606681,0x75f006dd)}},
   {{LL(0x12bbaeb6,0xea9da8f0),LL(0x8c9f8360,0x3fba06b1),LL(0xb28c0ac3,0xc11bd7ab),LL(0xaa8a01bd,0x1e05af2f),LL(0xf000b1c3,0xae1e99c5),LL(0x53d79930,0x93ee8064)}, {LL(0x4c4f5513,0x5728089e),LL(0xb1f70b76,0x755351f3),LL(0x675f77ef,0x187ac651),LL(0x53067d84,0x5cf7bfb5),LL(0x8174b5c0,0x62929083),LL(0x8d5be74d,0x720e2079)}},
   {{LL(0x15c26385,0xcdb0e974),LL(0x40c9a381,0x7110dd68),LL(0x8180d82b,0x1fd85657),LL(0xb97cf516,0xa42cad51),LL(0xc8e096e2,0xd0e97506),LL(0xac077c1c,0x9372ac5c)}, {LL(0x528c6d23,0x080cac77),LL(0xc192bee8,0xa62b7b39),LL(0x596c03b4,0xf1b2be68),LL(0x65cdde1a,0x9e367c67),LL(0xa97f58bb,0xdb0a51db),LL(0xc29dbc0d,0xa9b52c4d)}},
   {{LL(0xf5eb9ad1,0xab8cc09f),LL(0x132edbfe,0x97a4de76),LL(0x8baf6347,0xa2e11c54),LL(0x683cfcf6,0xcee54229),LL(0xdcfc6555,0xe1e993b8),LL(0xbe9df066,0x333bf16a)}, {LL(0x060d62df,0x5207e093),LL(0x69b0f5fa,0xfa32324d),LL(0xd3243d2d,0xef16fbcf),LL(0xf04f8e45,0x540a2e59),LL(0x48317bba,0xb5e70f9c),LL(0x5b35baa1,0x00dbe9b2)}},
   {{LL(0x04a6bdf1,0xa7c2e951),LL(0x45310414,0xe7b50010),LL(0x250deed3,0x0ad7ac85),LL(0x07fd3b65,0xeb7ad465),LL(0x6bdc321a,0xcac35e8c),LL(0xe992ce8e,0xf773a3e9)}, {LL(0x0f682437,0x99cbc651),LL(0x7ff5f4b3,0x62f7b15d),LL(0x6d131441,0xf3c08dfd),LL(0xfbb0fd4b,0x8b998754),LL(0xe061d5b7,0x88dae889),LL(0x2a346488,0x9f6e8dca)}},
   {{LL(0xebb512ee,0x0eaed675),LL(0x058efbd5,0x347e0756),LL(0x296d3d47,0xadf792ca),LL(0x4654d012,0x57f00c0a),LL(0xbccc5803,0xa1e08a04),LL(0x5b2f11d1,0x610677f0)}, {LL(0xb81acfd2,0x0d9393d7),LL(0x0587c219,0xb258e157),LL(0xb4ceba47,0x372a1857),LL(0x3ecc1c5d,0xe1ce8bb5),LL(0x922cecd0,0x7efdf301),LL(0x0d8aa653,0xcab8cb17)}},
   {{LL(0xd85124dd,0xc60f118a),LL(0xa77febb9,0x4a045272),LL(0x45555221,0xbae0edf5),LL(0xab65b900,0x1fb21695),LL(0xd56b9882,0x9179d546),LL(0x006bc680,0xead5f78e)}, {LL(0xed9d8095,0x6c3ef1c1),LL(0xd024f8cd,0xe22832c1),LL(0x157c5363,0x8c783b9d),LL(0x000d3603,0x25c1bd7e),LL(0xe8ff7a81,0x4e76a723),LL(0x4a955196,0xa55cfcb4)}},
   {{LL(0x79d05497,0x31954a56),LL(0xfe76d4d8,0xc12520b6),LL(0xe37ef1d2,0x8c433ec5),LL(0x75bc3b66,0xcd0f2035),LL(0x249cd98b,0x3723f145),LL(0xea3b42a3,0x1356e0d2)}, {LL(0xf174c7b5,0xf607fee0),LL(0x0127be39,0x318afc5e),LL(0xcea5417f,0xd47b5d74),LL(0x10fca22b,0x6891940a),LL(0x2b635e8b,0x5cea4133),LL(0xb5934fef,0x93db2ed6)}},
},
/* digit=26 base_pwr=2^130 */
{
   {{LL(0x41b959b5,0xb1f4fead),LL(0xe71890c0,0x6edb53a9),LL(0x2e28aa2a,0x48b47efe),LL(0xb3151d67,0x70dad2e9),LL(0x436a3460,0x87a8178b),LL(0x801f7af7,0x0f86f9f5)}, {LL(0xa982fc14,0xfab462e3),LL(0xcb03e978,0xe29126ba),LL(0xe6681282,0xb4696b3f),LL(0x6a3fdc1d,0x3bd9910a),LL(0x49e37dac,0x44091284),LL(0xcf605fb3,0x3b4bfabc)}},
   {{LL(0xdf9a9f18,0x57edf71e),LL(0x627a0b79,0xbf834240),LL(0xa6934160,0xb37aba1a),LL(0x5e639a54,0xd45b3d2c),LL(0x70bce957,0x62c6b9ad),LL(0x5d7e87f3,0x16bc35a7)}, {LL(0x66b4a982,0xb0216982),LL(0x0e51c9bc,0xb56050dd),LL(0x478e4b91,0x15aa692b),LL(0xbe3fe25a,0xdd67cf29),LL(0x06bdd4a8,0xf1ef75b0),LL(0x41df627a,0xf71a285b)}},
   {{LL(0xb7a3ce87,0x3176a43a),LL(0x5f130e73,0x9fa09e97),LL(0x9368e156,0x971cc37b),LL(0xb8981792,0x2cabf535),LL(0x4d0f0bc0,0xaec2862e),LL(0x3ce8c100,0xa1a48c18)}, {LL(0x4af2eae9,0x288f4e69),LL(0x1f9339bd,0x778845f2),LL(0x17dfaa6a,0x1ef5fdfd),LL(0x3483a6fc,0xc784117e),LL(0xf3c5c19e,0xe8c82f05),LL(0x1da87ab6,0xf39b3c1d)}},
   {{LL(0x8a541be6,0xa58a27c5),LL(0x54fd7683,0xaf669499),LL(0x00079a25,0x24318266),LL(0x2606caf5,0x113f6fcf),LL(0x16cb28c8,0xf6ff2be3),LL(0x3c17caa6,0x8f7fc60e)}, {LL(0x7d35e26c,0x8ea577e0),LL(0xf0628903,0xc3e744c0),LL(0x592a57ee,0x4b28eff4),LL(0x5e3f67b2,0x76e1f87c),LL(0xfb008902,0x40d7a676),LL(0x4b6e6b7e,0x68a9dc76)}},
   {{LL(0xe4fbe1d8,0xfc14716f),LL(0x88d5fc05,0x711c4f60),LL(0x8ab2552f,0x53df2711),LL(0xb3039434,0xeb4a3587),LL(0x825eba03,0x84223085),LL(0xf69569cf,0x1aff30b7)}, {LL(0x30d38055,0x752a2b25),LL(0x7dc7727e,0x8fb8ef5e),LL(0x679f84b9,0x8f0b8bea),LL(0x004815a9,0x076e7b05),LL(0x65debb2f,0x93275037),LL(0x83a686b0,0x24f92c79)}},
   {{LL(0xdff082f7,0xf9597802),LL(0xeb44042c,0xb47fcfee),LL(0x8ac1e9a9,0x0e2570c9),LL(0x9168391b,0x144f5763),LL(0xfe49b3b2,0x7c5fe9e6),LL(0xe1202056,0x30dd5a41)}, {LL(0xe5132272,0xb0866f7c),LL(0x199d029c,0xaa2d7c60),LL(0x2bd29ff9,0xb37c3232),LL(0x29bc5c36,0x6c0e511b),LL(0xbaa69a42,0xb64e16b6),LL(0x28545caf,0x9fd7d79b)}},
   {{LL(0x0c3576d3,0x0620003c),LL(0x31b24876,0xa9608ecd),LL(0x2d83f362,0x7dcb576a),LL(0x7df98941,0x88a3a38a),LL(0xb3596afe,0xdd182e8a),LL(0xe02a5357,0x9c740d01)}, {LL(0xb75d5d3b,0x2fed2fcc),LL(0x8d34ca34,0xd0ea6d50),LL(0x8f4bd1dd,0xfe4b8743),LL(0xb130c5c5,0x73c5acd3),LL(0x2dadbb20,0x9f8b0817),LL(0xc5a59dfc,0x34b74294)}},
   {{LL(0x12575913,0x76e2751a),LL(0x4a5f8c4a,0x2c605991),LL(0x71fba662,0x58322dfb),LL(0x5e0886af,0x228aec08),LL(0x6aee544c,0x8d83b627),LL(0xe29f9639,0x338f5fb6)}, {LL(0xbf5e19fa,0x1ba4cfe0),LL(0xb9e4f8f6,0x2eea84c5),LL(0xcee95d92,0x7e0eed58),LL(0xbe535540,0x2d29282a),LL(0x07a9a1f4,0x866638b6),LL(0x6ab8dc82,0x91599977)}},
   {{LL(0xa2144cce,0x6741d444),LL(0x771c7c6c,0x06ce4c04),LL(0xae0ff352,0x3d5bc2d2),LL(0x0e05ca8a,0xa0ae03ba),LL(0x833183d8,0xbf718851),LL(0xccdc956f,0x7095f4d4)}, {LL(0xb3ab0c80,0x23268e4a),LL(0xc16c6dad,0x984cf1eb),LL(0x67120ec8,0xc2333145),LL(0x9cd9a03f,0xc615f60e),LL(0xb3d16871,0xe9e35fd3),LL(0x15e76b83,0xc9b9853d)}},
   {{LL(0xbb7fefd2,0xd8e314ab),LL(0xaa61ea98,0x74d946ba),LL(0x00fd45fb,0x10cfd8b5),LL(0x357f8e42,0x87dfda88),LL(0x27bf7a3a,0x6dab4a2f),LL(0x03b613f5,0x4dd49a15)}, {LL(0x74f2c1a8,0x1f5491dc),LL(0xcc8ee02e,0x98d17370),LL(0x3f141b03,0x01ef39fe),LL(0x6984d4c3,0xa5b04673),LL(0x4a926cdb,0x10c40183),LL(0xe2af1758,0x3e7bbe18)}},
   {{LL(0xc2faa336,0xda178c95),LL(0x68d20d12,0xb4ae744b),LL(0x5e066620,0x5b75f4e9),LL(0x41e95800,0x4138315f),LL(0x010c2c6a,0xf5c97da9),LL(0xe07bbf8d,0xbc4ff40f)}, {LL(0x21709050,0xbdd6bc9f),LL(0xc5f7bb78,0xf3f4dcf2),LL(0x9b1da924,0x584ce030),LL(0xec3d9507,0x44b81c48),LL(0x73e1961e,0x1dfab021),LL(0xa77458a8,0x9212120e)}},
   {{LL(0xf9612455,0x599ece96),LL(0xc725cba4,0xe1c6eb81),LL(0x6373ccee,0x2afc5ba8),LL(0xc64f7261,0xe4556833),LL(0x414ce1ab,0xf29ad540),LL(0x3ee82a77,0x7094e067)}, {LL(0xdc9716fa,0xfa26eaa0),LL(0x9ceb36af,0x340da4be),LL(0x4e1c94bd,0x9570ade5),LL(0x20b0ddbe,0xf4a86d8e),LL(0xdc6eff6f,0x967ab653),LL(0xe8bab7a4,0x48890be5)}},
   {{LL(0xc96437e8,0x74ee2fe8),LL(0x0bfd2bb3,0xeafe095d),LL(0x103d0a33,0xc75f0e53),LL(0x70f13700,0x22190801),LL(0x101d133f,0x96684d5b),LL(0xf21b224b,0x6c98178f)}, {LL(0x1d59dd67,0xe3003316),LL(0x734e7699,0x933de68f),LL(0xa305ea84,0xa3883631),LL(0x6d66c90f,0x21afb4d0),LL(0x330568ce,0xebeecc72),LL(0x62b5a3ae,0x6b65b757)}},
   {{LL(0xf637841c,0x4069700f),LL(0xb1c94076,0x6ee84605),LL(0x829e1bc4,0xcd64ecd7),LL(0x45cdda86,0x606944a8),LL(0x58f036ae,0xb0f2d5fe),LL(0xec18a23f,0x0391590a)}, {LL(0x1a82e6c2,0xe29e3891),LL(0x6bd4a55b,0x5489a628),LL(0x7ae6380c,0xf3e7a351),LL(0x6ae0ae07,0xb3272f23),LL(0xe66cd800,0xb605a039),LL(0x2aac1b66,0xd4e522ea)}},
   {{LL(0x8b153f87,0xbb9d0b1c),LL(0x518a921d,0xa3d52cbe),LL(0x2110cd42,0x232a29d2),LL(0x8a40d59b,0x9083e225),LL(0xaa50f55d,0x3b00382e),LL(0xc02cf900,0x4ad64680)}, {LL(0x36cad5d1,0x2f25bd26),LL(0x64eb9c45,0x9c693e85),LL(0xccf60708,0xe4ea93e5),LL(0xf9270541,0xb932b958),LL(0x5eb454ce,0xee93f51f),LL(0x38840e07,0x1e82e354)}},
   {{LL(0xaf4d260b,0xa16c79cf),LL(0xfab3c3c8,0xfe853f6c),LL(0xc2f47e68,0xb8bd6aa0),LL(0x2c9b4914,0x277d590f),LL(0x097242a8,0xb6d1c810),LL(0x45f75512,0xcf2f3d8e)}, {LL(0x74a20c3b,0x2176162b),LL(0x2b2bcdda,0xeee8bcb8),LL(0xa503aee7,0xfcf8c0d1),LL(0x7af4dd78,0x5d1f94a5),LL(0x2ab43be4,0x8f0bc1a6),LL(0xba9e071b,0xd22dbf16)}},
},
/* digit=27 base_pwr=2^135 */
{
   {{LL(0xb4e814b3,0x705bfe37),LL(0x702013c6,0x22f0de61),LL(0xbc456797,0x811e77a9),LL(0x17081a2f,0x4f52c4e6),LL(0x9fe1640e,0x87405d81),LL(0x707711d7,0x53fa82b7)}, {LL(0x0ee4aea6,0xdc6fff83),LL(0xfd60373d,0x8413e22f),LL(0xa9cf3ead,0x0ecb66be),LL(0x87139b8b,0x7418372e),LL(0x5e42b4d7,0x6aaccf29),LL(0x31fc932e,0xb6dc5925)}},
   {{LL(0xb88ee8f9,0xfa3b4c8e),LL(0xb521ab57,0x1f288e60),LL(0x2e8c4d8c,0x06aa3956),LL(0xcf89935b,0x4981c3e5),LL(0x45fa071e,0xbdbd0c47),LL(0x496073be,0xa78f831c)}, {LL(0xa4e5c001,0x09a72986),LL(0x709cb728,0xac527731),LL(0x988f2781,0x9a64b5b3),LL(0x73b1719d,0x6ac9440d),LL(0xe3d2e807,0x58ad54c7),LL(0x8f06742b,0x1c157448)}},
   {{LL(0x75c953b2,0x6b5b7b2a),LL(0xa7f1cd5b,0x927ed77c),LL(0x4cba0e5e,0x2e8c5399),LL(0x3f4a941b,0x03aeb14a),LL(0xa1385c8a,0xedbad9a0),LL(0x67fd2258,0x925a49c1)}, {LL(0x3365ffed,0xe7e368ee),LL(0xd106eb87,0xcc4aad2d),LL(0xa980b53b,0x4ce908da),LL(0x16929ac8,0xd3f49540),LL(0xd5c05c32,0x613c804d),LL(0xd7973344,0xa42290cc)}},
   {{LL(0xa98cf218,0x33952177),LL(0x579ee53a,0x841d9e1f),LL(0x0a285bd5,0x1084d61e),LL(0x71171b1c,0x3935a84e),LL(0xf29b29f9,0x8ac2433c),LL(0x6dd1e9bd,0x5dd868b5)}, {LL(0x8d102390,0x88da0478),LL(0x657400d1,0x1140735a),LL(0x9d5b19e1,0xa792a25f),LL(0x6a27fa79,0x9ee015cb),LL(0x7ba16a8e,0xea3bf8b5),LL(0xc15fde67,0xc5f0cc26)}},
   {{LL(0xee2c3290,0x2e152d95),LL(0x4a9ceda4,0x8437df2e),LL(0x3c7ebfd1,0x4151754e),LL(0x88f80aea,0x556c59a8),LL(0x8de44dbc,0x8d099c5d),LL(0x77abeecc,0x9ecce7fc)}, {LL(0x3aa311cf,0x5e0a0f38),LL(0xb8f2bff5,0x99ff1eec),LL(0xb5dcf488,0x5ae0b483),LL(0x91483a02,0x11212c45),LL(0x312134a1,0x99fe0738),LL(0xa72745ef,0x3b855db0)}},
   {{LL(0x892c8eec,0x37f50e43),LL(0x7d85a7e2,0xf06a2f04),LL(0xe1d11150,0x3916af85),LL(0x6785ae1c,0xf56e852f),LL(0xae6ada8c,0xbf8c72ad),LL(0xe13285b2,0x1fcd53e3)}, {LL(0xbd56d348,0x5327920c),LL(0x445658a8,0x82a394fb),LL(0x3caf3792,0xa7132857),LL(0x550ffe1c,0xb15ab34b),LL(0x6a5d4e4f,0x81898066),LL(0x2f854f9d,0x0bda153b)}},
   {{LL(0x722730fe,0x77a31009),LL(0xd5cdd297,0x93707ac4),LL(0xd3811e8c,0xa290be39),LL(0x92a5cdb7,0x831a9b95),LL(0xe7342270,0xc74cda84),LL(0x3f48affc,0x96466190)}, {LL(0x5520b0f0,0xb0496cca),LL(0xbae930ff,0xc8742cd9),LL(0xeaea703a,0x3a30737a),LL(0xfb758854,0x0a8e6fb7),LL(0x6796f4d1,0x9ab9523e),LL(0xfdf7140f,0x36e6c05d)}},
   {{LL(0x64ef6a95,0x3b623150),LL(0xaaa5b792,0x97645381),LL(0x56471100,0x4bc2c31c),LL(0x1bae8d2a,0x4a0e73bb),LL(0x8df1f76a,0xbfc0770a),LL(0xa7bb16ca,0x5089916f)}, {LL(0xf31fe82e,0x2afe5b1c),LL(0xf0119977,0x0b06831d),LL(0xa1af2a82,0x97caa333),LL(0xdafed6cd,0x93cb92c5),LL(0x92c3b2e3,0x09553e7e),LL(0x61af2956,0x3d9c4b7d)}},
   {{LL(0x08f84746,0xd83f574a),LL(0xca07f5f8,0x48fc9715),LL(0xdcc51638,0xb3d5d0d2),LL(0x6153bdcd,0xc2a5e335),LL(0x8aa4ef74,0x8242cd9a),LL(0x0bdaa0d0,0xe71ba25b)}, {LL(0xa4ff172d,0x4342d4bb),LL(0xfc1341a2,0x81db10df),LL(0x7dacb140,0xdd93dd87),LL(0xd12d347f,0x6f8a4e81),LL(0x1bc369be,0x0d4e7e46),LL(0x1fafd0c5,0x3ce10a77)}},
   {{LL(0xe67145b6,0x5559dd31),LL(0x5b2427e7,0xf2d905b4),LL(0xcaf57d0c,0x0d840fab),LL(0x78742ab6,0x96258665),LL(0x409c1c8e,0xc85482ad),LL(0xadaa6167,0xdca2a058)}, {LL(0x0c8885fd,0xec26ad9a),LL(0x2a600cb2,0x1b93b8a2),LL(0x2539986b,0x340aa7fc),LL(0xa23dee41,0xd7674876),LL(0x2e1a9837,0xa948a929),LL(0x71438da9,0x9ae67d2a)}},
   {{LL(0xd56bdf1f,0xeac6f447),LL(0xc2b502ff,0xb22e8425),LL(0xfca5a501,0xe1cc9d3d),LL(0xb64baf39,0x8192bc29),LL(0x52ce849e,0xeb2c901a),LL(0x1dd506f1,0x7f5f38b1)}, {LL(0x0f0a1d68,0xfb3684b1),LL(0xe9240ff8,0x16c4aacd),LL(0x5a4d8995,0xffa68243),LL(0x54e4c95d,0x27264ab5),LL(0x4f34ffaa,0x9aa40cdc),LL(0x5fd818ee,0xcb8a30a3)}},
   {{LL(0xf7f35053,0x39038863),LL(0x328787d2,0x421a17f3),LL(0xf3d8310f,0x38aa682e),LL(0xf4123153,0xb52d41e8),LL(0x7026310b,0x4fbef3dd),LL(0xf6ff5692,0x0c6bd7ad)}, {LL(0xa9be5d0c,0x3831c6b2),LL(0xe8d328b8,0xb5c9ae85),LL(0x6516bba4,0x76d26abc),LL(0x446d35a8,0xc237f9a5),LL(0xf012a8d0,0xb2b16c0f),LL(0x0ee0315b,0xddf2b7fe)}},
   {{LL(0x056ad6c2,0xbb85b640),LL(0xac074372,0x7c51ef96),LL(0xf10b43fc,0x1c7ce31c),LL(0x26f4d3a4,0x08e4101b),LL(0x3968459f,0xd18511c4),LL(0xd6d07839,0x00e20c3f)}, {LL(0xe4fcdc11,0xd5bcd598),LL(0xc877f6a2,0x99e9a4d0),LL(0xbd491646,0x9c5dd9d0),LL(0x9bfd7a1a,0x83918f60),LL(0x7e2b95a3,0x4bc130cd),LL(0xfbc31c83,0x668825fb)}},
   {{LL(0x06a9ad54,0x817d77b1),LL(0x89a25eca,0x3a999d7d),LL(0xda68b768,0xd3ac4107),LL(0xbebc4c4d,0x6904bcdd),LL(0xa53d39e9,0xb0d2103c),LL(0x30a5e950,0xdba86bd2)}, {LL(0x4f52208e,0xb0925680),LL(0x28495b2c,0x37c3156a),LL(0xc15855ae,0x2389ab34),LL(0x3017194f,0xc14dfd96),LL(0x1146b838,0x420e0719),LL(0x8fb4b6fc,0x1a9f909b)}},
   {{LL(0xdd63404d,0x2926ef17),LL(0x1399cc68,0x0e89c4d4),LL(0xf7ec20b8,0x6507fede),LL(0x88c751d6,0x1ac084ff),LL(0xdefe29e6,0x31bc08be),LL(0x4f0692c5,0xd4219971)}, {LL(0x36069bc0,0x4d6ee742),LL(0xff80f3d7,0x3868ef6a),LL(0x5a9c6f4b,0x6df02d7c),LL(0x101abf69,0x2c3096bb),LL(0x8eaacaeb,0x0c2b01ec),LL(0xeb2e687a,0x65914c20)}},
   {{LL(0x79ed523a,0x13722ab0),LL(0x249d5624,0x33b29bec),LL(0xf76fdaf7,0xd3d0f467),LL(0x12ddfd9a,0x7ce072f9),LL(0x47bdefd3,0xce918a57),LL(0x750e5315,0x14d38ab4)}, {LL(0x3346f647,0x08bbb20e),LL(0x05b26894,0x428b917f),LL(0xca865ba6,0xc8fb5c21),LL(0x2e6e8e6f,0xee6e41e0),LL(0x4c608b60,0xd00ae621),LL(0x6ff685cd,0x65975639)}},
},
/* digit=28 base_pwr=2^140 */
{
   {{LL(0xa368eff6,0xbbccce39),LL(0x8ceb5c43,0xd8caabdf),LL(0xd2252fda,0x9eae35a5),LL(0x54e7dd49,0xa8f4f209),LL(0x295100fd,0xa56d72a6),LL(0x56767727,0x20fc1fe8)}, {LL(0x0bbaa5ab,0xbf60b248),LL(0x313911f2,0xa4f3ce5a),LL(0xb93dab9c,0xc2a67ad4),LL(0x22d71f39,0x18cd0ed0),LL(0x5f304db2,0x04380c42),LL(0x6729c821,0x26420cbb)}},
   {{LL(0x0eb008c8,0xca07923c),LL(0x9985912e,0xab79402d),LL(0x3cb02510,0x41e379e8),LL(0xbeb383ef,0xfabac005),LL(0x1076dd0d,0x24d12d9a),LL(0xb208f127,0x95afd46f)}, {LL(0xb1031e46,0x9cc38a60),LL(0x7009f6bc,0x93e21e97),LL(0x8ac219ef,0x6f6360d9),LL(0xaf284c80,0x1edaab3f),LL(0x019e366a,0x9c3b5281),LL(0xbc9e9726,0x6475c579)}},
   {{LL(0xbdfbcae8,0x26bd07d6),LL(0xdf01a80a,0x10b5173f),LL(0x6798b96c,0xd831c546),LL(0x1d3f3859,0x1d6b4108),LL(0x991b9ec7,0x501d38ec),LL(0xd78431a9,0x26319283)}, {LL(0x118b343c,0x8b85baf7),LL(0x58def7d0,0x4696cddd),LL(0x7acdcf58,0xefc7c110),LL(0x848d5842,0xd9af415c),LL(0x0ac7fdac,0x6b5a06bc),LL(0xa344319b,0x7d623e0d)}},
   {{LL(0x8d85a25a,0x8410d829),LL(0x4af81a14,0x48ee0135),LL(0x18c25348,0xae460d0d),LL(0x7eb035a3,0x5d0279a0),LL(0x9a114414,0x87e7c128),LL(0xc0744f79,0x17c08a8e)}, {LL(0x025cdbe3,0xb7b2b4f1),LL(0x82d1af60,0x9a74f15d),LL(0xb51ee685,0x124a7395),LL(0xf6122422,0xf2937c4b),LL(0x07f1a7ff,0xb4ec1332),LL(0xf886032e,0xad801112)}},
   {{LL(0x0c9d3547,0x4c0d7806),LL(0xcf2aed47,0x993f048d),LL(0xe4b57e22,0x5217c453),LL(0xf4172b28,0xb4669e35),LL(0x49f999f8,0x509a3cd0),LL(0x87c69d41,0xd19f8632)}, {LL(0x4c8fded0,0xe14d01e8),LL(0xeafd9e1c,0x342880fd),LL(0x70dc2bf0,0x0e17bff2),LL(0xc0186400,0x46560b7b),LL(0x49a4dd34,0xe28c7b9c),LL(0x0f325d06,0x18211916)}},
   {{LL(0x7bb5346e,0xdd4eb3d0),LL(0x382e7db7,0x9a46ad01),LL(0xdc1973c7,0x1200285d),LL(0xa0046b98,0xfd342bea),LL(0x1219a7fc,0xd1917349),LL(0xb7caffe5,0x5383d319)}, {LL(0x2e0fa118,0xea5a0c4e),LL(0xa5457b28,0x1cc2de3c),LL(0x6046eeea,0x5b2a16dc),LL(0xcc8e64b1,0x1755e1fe),LL(0x9e7fadda,0x51e4946e),LL(0xfcbf4ec2,0xf805422f)}},
   {{LL(0xd7e02e18,0x46d70888),LL(0xd9f11fd9,0x7c806954),LL(0x4fbea271,0xe4948fca),LL(0xbd80a9df,0x7d6c7765),LL(0xf3871c71,0x1b470ea6),LL(0x8330a570,0xd62de244)}, {LL(0xc659c3a7,0xdaecddc1),LL(0x077f7afc,0x8621e513),LL(0xcaeeef13,0x56c7cd84),LL(0xc685a356,0xc60c910f),LL(0x9dd93ddc,0xe68bc5c5),LL(0xfeb64895,0xd904e89f)}},
   {{LL(0xbd08ffaf,0xf877e8c6),LL(0xaf23012f,0x24718fef),LL(0x2b004cfe,0x19ff269f),LL(0x95450f8b,0x8adc5d77),LL(0xe2a7d458,0x688ce8bc),LL(0x97bd7fdc,0x74d7445b)}, {LL(0x41e6abad,0x1b9f4ad6),LL(0xf00e4bf5,0x6652ed05),LL(0x71d83d86,0xabee1f7e),LL(0x25ffc219,0xe693c76d),LL(0xc873f553,0x1c9a84af),LL(0x66d77a55,0x84d27187)}},
   {{LL(0x8ba7917a,0x75d874fb),LL(0xfd043bd4,0x18fa7f53),LL(0x1fc3979e,0x212a0ad7),LL(0x5d6eac0e,0x5703a7d9),LL(0x017dead5,0x222f7188),LL(0x0f6c1817,0x1ec687b7)}, {LL(0x238bacb6,0x23412fc3),LL(0x54ced154,0xb85d70e9),LL(0xbda674d0,0xd4e06722),LL(0x36f5a0c2,0x3ea5f178),LL(0xf5c6d2ca,0x7e7d79cf),LL(0x3dbb3c73,0x1fff9464)}},
   {{LL(0x7e5f7121,0xe566dc05),LL(0x2ed07bc3,0xccac74e2),LL(0xc70401b4,0xaabfdfcd),LL(0x6254e0db,0xac9fc449),LL(0x11c7de05,0x358d885f),LL(0xd60772b4,0xb8e6a4a9)}, {LL(0xcfe917ce,0x884272a5),LL(0x9a3d347a,0xdfbe9868),LL(0xc9d1bacc,0x06b90848),LL(0xdb8c6288,0xc4ccedb6),LL(0x79e5683e,0x892878b9),LL(0x243273e3,0x1b521829)}},
   {{LL(0xf163e4a8,0x916e19d0),LL(0x1489df17,0x1e6740e7),LL(0x339f3a47,0x1eaf9723),LL(0x124b8dad,0x22f0ed1a),LL(0x49c3dd04,0x39c9166c),LL(0xce1e9acc,0x628e7fd4)}, {LL(0x40031676,0x124ddf27),LL(0x1eddb9be,0x00256939),LL(0xd360b0da,0xd39e25e7),LL(0x4aa6c4c9,0x6e3015a8),LL(0x623eda09,0xc6a2f643),LL(0x50aa99fb,0xbeff2d12)}},
   {{LL(0xbf0c6fbe,0x099369c4),LL(0xfe7d5727,0x976f78b2),LL(0xd18267a9,0x32feb503),LL(0x1a7dd0fe,0x162c4150),LL(0x26b8e969,0x3141e377),LL(0x3b53a94a,0x50497a64)}, {LL(0x607b4cfc,0x96159f41),LL(0x2f111bab,0x1999b704),LL(0x760f2eae,0x3254987c),LL(0x841014fa,0x5308075b),LL(0x4e7adad8,0xc634127e),LL(0x59ffbfe6,0x32a70a60)}},
   {{LL(0x93ee8089,0x1feef7ce),LL(0x252dd7bd,0xc6b180bc),LL(0x1788f051,0xa16fb20b),LL(0xe046ed39,0xd86fd392),LL(0x9378ce1d,0xda0a3611),LL(0xa5f7a61d,0x121ef3e7)}, {LL(0x92d13cae,0x94d22061),LL(0x77c72e08,0x5076046a),LL(0x7d2308b9,0xf18bc233),LL(0x17f977b1,0x004db3c5),LL(0x0471c11d,0xd05ae399),LL(0x85cd1726,0x86a2a557)}},
   {{LL(0xa1f857e6,0x7279c369),LL(0x27fb373a,0x029d30ef),LL(0x6827358b,0xe82cbc80),LL(0xa18f57ab,0x2bfe09aa),LL(0xe5503492,0x63bf3145),LL(0xfb28ee43,0x7ea15bea)}, {LL(0x5eec91b8,0x8e6d428f),LL(0x611b1799,0x215e03e9),LL(0x61d476de,0xb9957371),LL(0xe76726a5,0x2320c764),LL(0x8e5e26f5,0xc5de8817),LL(0x9161e0b7,0x24aae069)}},
   {{LL(0x72107804,0xb8d9b286),LL(0x3303b79b,0xb5a7c413),LL(0x5fa37ded,0x927eef78),LL(0xad67daba,0xa1c5cf1e),LL(0x7360e7c7,0xaa5e3fb2),LL(0x0a0c0993,0x8354e61a)}, {LL(0x7f5458cc,0x2ec73af9),LL(0x48474325,0xde4cb488),LL(0x7209bc69,0x2dd134c7),LL(0x451a2abe,0xb70c5567),LL(0x8e293018,0x2cd1b200),LL(0xd33c0d72,0x15f8da7a)}},
   {{LL(0x893b9a2d,0x5584cbb3),LL(0x00850c5d,0x820c660b),LL(0x7df2d43d,0x4126d826),LL(0x0109e801,0xdd5bbbf0),LL(0x38172f1c,0x85b92ee3),LL(0xf31430d9,0x609d4f93)}, {LL(0xeadaf9d6,0x1e059a07),LL(0x0f125fb0,0x70e6536c),LL(0x560f20e7,0xd6220751),LL(0x7aaf3a9a,0xa59489ae),LL(0x64bae14e,0x7b70e2f6),LL(0x76d08249,0x0dd03701)}},
},
/* digit=29 base_pwr=2^145 */
{
   {{LL(0x31cb94c9,0xaff47822),LL(0x803c1af4,0xf1b5a0b7),LL(0x2ef696a9,0xbeb85f8d),LL(0x4fa94fca,0x8ce5baab),LL(0x00d41a43,0x0a32f962),LL(0x74f6e772,0x0f69ad57)}, {LL(0x6ccb5157,0xbe0221af),LL(0x2a4f91ff,0xcb83969a),LL(0xa7e49f39,0x78ff85d6),LL(0xcb5d3c63,0x63006589),LL(0x96eb65f5,0xe8e43835),LL(0xff8adbdf,0x79f59da9)}},
   {{LL(0x10eeed24,0x082ea61d),LL(0x143fd59d,0x7c9d5ade),LL(0x2e54f5cf,0x7d33df96),LL(0xe39dc6ab,0x340b0d36),LL(0x8d179b13,0xd97a8b84),LL(0x288d388c,0x88184bb0)}, {LL(0xe116ae6d,0x2237e507),LL(0x211b2cf0,0x3e97b063),LL(0x42be7459,0x645f8bcb),LL(0xde2176b6,0xce2b0f54),LL(0xd1e2f09c,0xaf570a09),LL(0x57fdc001,0x110adf56)}},
   {{LL(0x842e4246,0x9d21c740),LL(0x4ab098a5,0x30f474c4),LL(0xaae5d701,0x57f8b1a3),LL(0x91978d15,0x477e4f88),LL(0x0fb85b1e,0x2913ffb4),LL(0x58489fb4,0x80aedb22)}, {LL(0x0912d86a,0x0e1ab267),LL(0x82933f3a,0xea5e6a41),LL(0x57ab8d86,0xf578ccb6),LL(0x547f64bd,0x339fd796),LL(0x3f3e497e,0x90469394),LL(0x1d864706,0x2cde596c)}},
   {{LL(0xbcb6db29,0x158bfe27),LL(0x0054d963,0x96721241),LL(0x8e71aca1,0xf07b153b),LL(0x71b11643,0x5e676981),LL(0xd04e2f90,0x77b7dd7d),LL(0xf0dcf109,0x07814aa6)}, {LL(0xfe1d0b1e,0xd3bab2a4),LL(0xbe69e691,0x50abba31),LL(0xc6f53cd9,0x54fe99af),LL(0x628039e4,0x071f2a4f),LL(0xb183aa16,0xf1f44181),LL(0x5010f6f9,0xdf0138e0)}},
   {{LL(0xaff45947,0x5737f903),LL(0x62c5dfd1,0xe576f8ff),LL(0xe1e379a2,0x2a7d4f18),LL(0xf2f26f8c,0x8596ffaf),LL(0xfab86608,0x12f49653),LL(0x3aab1eaf,0xcc618497)}, {LL(0x865d9611,0x7b5d029f),LL(0xbf6861f5,0x87af2d58),LL(0x31b2d7cf,0xb7653646),LL(0x8be46ef9,0x60f47e05),LL(0xb1c175ce,0xa21106a9),LL(0xdeb065b6,0x21cb0e27)}},
   {{LL(0xb340f4eb,0xb287e26d),LL(0xb34c2948,0xa397cfd1),LL(0x9a7ead15,0x0d91d763),LL(0x66e88f14,0xd71901b9),LL(0xb4884ac2,0x02447ad3),LL(0xd006f448,0x5e6f7545)}, {LL(0xec5744e3,0x2af6bfee),LL(0xf55cd6ab,0x643f46b1),LL(0x87b2c127,0x13400bc4),LL(0x4629859b,0xd1dddff4),LL(0x20feecb9,0x9710cae0),LL(0xd02d4300,0x594bc27c)}},
   {{LL(0xaa331700,0xb0233c4d),LL(0x30b3a854,0x3bf63a61),LL(0x49b806e2,0xc269d68b),LL(0xda8ddca6,0x4a079274),LL(0x0db04d76,0x256bc009),LL(0x1f47dfee,0x7395c11c)}, {LL(0x13886ec2,0x51bbcc95),LL(0x244d1f0d,0x9d5377eb),LL(0xff9e6b3a,0xa6e0f054),LL(0x204e5258,0x307f9e93),LL(0x8d34d97b,0x8069c01d),LL(0x43c24997,0x30f785ce)}},
   {{LL(0x9c77ca60,0xb1e36501),LL(0x01018e14,0xfe084a23),LL(0xa4bfdcec,0xbf451d2c),LL(0xb29cdcfe,0xd210892f),LL(0x94514871,0x5b12bcd8),LL(0x1809b1e3,0xd03ca18c)}, {LL(0x5858e4ea,0x09b24311),LL(0xe57524b4,0x37b30d50),LL(0x5de334b5,0xcef0a16b),LL(0x0b116076,0xfe0bd1e2),LL(0x89ae2bf4,0x54e4b482),LL(0x68c8a937,0xfbcc5e1a)}},
   {{LL(0x91ae19fb,0xf7473902),LL(0x1bed3f8d,0xc1c228d3),LL(0x6552154e,0x763ba8ee),LL(0xe2063ce7,0xb7b60248),LL(0x178a5184,0xafaf01b8),LL(0x7901d21a,0xe193b834)}, {LL(0x260eb30f,0x1d29f7d6),LL(0xc23b4b94,0x030516c6),LL(0x7dc4a09a,0x30a8046a),LL(0xb5ee1147,0x8414133a),LL(0x4453f1a6,0x7ec8dccf),LL(0x59da8e8d,0xe380e69c)}},
   {{LL(0x22c77f4b,0x28fdf72f),LL(0x32485220,0xfa077b15),LL(0xd6325081,0xc3801faf),LL(0x891ff5df,0x8dd4e3c9),LL(0xaa73c827,0x6726c5ec),LL(0xe9b9b128,0x12af9707)}, {LL(0xff25cf07,0x5e6ae3cb),LL(0xd7cb7adc,0x0613bef8),LL(0xcef4cd47,0xfd426caa),LL(0xced0e435,0xe14ed3ef),LL(0x6000ec80,0x8323a861),LL(0x941f8071,0x58221041)}},
   {{LL(0xa83eb17c,0x6f766821),LL(0xbd7ff851,0x91fe58c1),LL(0x9063af8a,0x7ce09f04),LL(0x6c109e02,0x5230b3c6),LL(0x36274fb9,0x501adb33),LL(0x90547af3,0xcfb34bec)}, {LL(0x40f61ec9,0x622d1387),LL(0x0e3e98d1,0xdf26e4c6),LL(0x7c676ee3,0x6b3b3d62),LL(0xf0244737,0x9f841097),LL(0x4fc8dd58,0xcfaf6ead),LL(0xcd534ab4,0x4f5d463b)}},
   {{LL(0x64be0f56,0xa5023e13),LL(0x0046f45c,0x6a7310e0),LL(0xec8700d3,0xe0af09ae),LL(0xeb2d38f0,0xdea5fb7c),LL(0x859852e6,0xc038eae6),LL(0x8c34f04c,0xd515fb4c)}, {LL(0x1488c207,0x546b778e),LL(0x6258d8ba,0x8cf4f114),LL(0x5182c96c,0x474e60d8),LL(0x3dbde757,0xcd038730),LL(0x76ab01ff,0x387232f8),LL(0x28231392,0x277614f6)}},
   {{LL(0x80b86bc6,0x930c3c59),LL(0xa06d10a8,0xebc8c144),LL(0x81e55432,0x20743ddd),LL(0x006d3073,0x059521d4),LL(0x12dbc785,0x8d4fe303),LL(0xdf4901d6,0x15a4debc)}, {LL(0xb37b2d8b,0xc9b226b1),LL(0x0fe9bdbf,0x385ac3c9),LL(0xfeaa5bff,0x64df89f7),LL(0x81de42ba,0x67de8e03),LL(0x5bf63f6f,0x12bd8322),LL(0x32b4ec13,0xb499f8ff)}},
   {{LL(0x9608c827,0xba0800b8),LL(0x47dfcd84,0x560c31f5),LL(0x520dba6e,0xd52e8fdc),LL(0x854fa4ee,0x66a917c2),LL(0xc6e5d664,0xfc543b79),LL(0x32b530aa,0x5b643692)}, {LL(0x563a3933,0x427dc1ef),LL(0x88d5902f,0x4e47c929),LL(0x93eca4fb,0x9eff3a40),LL(0x59260a69,0xc5b396ee),LL(0x48e70137,0x936a5062),LL(0x21bba959,0x69eb44f6)}},
   {{LL(0xab9f58a2,0x0af34aa7),LL(0x7d150dcb,0x0ba16d98),LL(0xb7c119b4,0x444d5495),LL(0x5ecc3210,0x317a55e5),LL(0xb1aa90c0,0x8fa3b3ce),LL(0x0b2e9392,0x8e7c7539)}, {LL(0xe8921afa,0x5ee5f3e8),LL(0xa3100dd3,0x3934ea8d),LL(0x4c8ae6ce,0x955045b1),LL(0x649897fc,0xa54acb5f),LL(0xe7081246,0x4204fab6),LL(0x6dce3a55,0xec3dc968)}},
   {{LL(0x3ef5a413,0x309b1eb3),LL(0xa81f43fb,0xa7607981),LL(0xbf8a894c,0x87c2b81e),LL(0x0d293293,0x27a40bce),LL(0xe4bf3714,0x7f4c315b),LL(0x01236895,0x03fdc14e)}, {LL(0xdff053fe,0x319c88f8),LL(0xea3fa121,0x146bb448),LL(0xf0dd1380,0xfcc2a05d),LL(0x4acba9fa,0xc8d55b02),LL(0x5927313e,0x871358de),LL(0x17ce294a,0xfd1d81d3)}},
},
/* digit=30 base_pwr=2^150 */
{
   {{LL(0xff63ae69,0xd3797831),LL(0xce4c7eaf,0xa753de02),LL(0x11a4e339,0x2ff7a6a6),LL(0x5328043a,0x904f86f0),LL(0x12e9f7dd,0xe29d31c0),LL(0xc0a51904,0x8825a639)}, {LL(0xebfc2cc7,0x070c2696),LL(0xc5f7a943,0xc03ce643),LL(0x12c8a1f5,0x5b970d0c),LL(0xab352a83,0x572aaaa1),LL(0x0c5eb0c7,0x63df45a9),LL(0xd4977599,0x95c951e1)}},
   {{LL(0xbca07a42,0x2c5b0e42),LL(0x7a0dffa1,0xbe57f359),LL(0x9aa90727,0xace48595),LL(0xf658699b,0x32be886a),LL(0xda3b18e6,0xce75d6c6),LL(0x69caf667,0x9d563e4f)}, {LL(0x065eb772,0xc17c66cf),LL(0x4df9f6ef,0xfbe12381),LL(0x623db4ef,0xceb80041),LL(0xc74762e1,0xe75615b2),LL(0x8671c52f,0xade8a543),LL(0xcacaf2ec,0xb713c401)}},
   {{LL(0x467eb167,0xe6d039ef),LL(0x74696cf9,0xa7e0959d),LL(0x7078d8a0,0xf3a19b9d),LL(0x07cdc6f6,0x5d4ec99c),LL(0x8386eed8,0x4842d0f9),LL(0x545fc0d5,0x48f5ab80)}, {LL(0x6d39c2f7,0x8906fc62),LL(0x1bf5366a,0x1c050d69),LL(0x9f54d0d6,0xac506c57),LL(0xf9e4b94c,0x9a356a6e),LL(0x08a75e61,0x62632c51),LL(0xc6951dc2,0xfc1b9fa5)}},
   {{LL(0xa4886619,0xa933aaf7),LL(0x4af13c7f,0x9ec1915f),LL(0x854de496,0x25a9dff8),LL(0x247bec15,0xa8b31d9b),LL(0x4661e58d,0x468a25c8),LL(0x786a0707,0x8989c046)}, {LL(0xbb66922e,0x282db8ca),LL(0x45ca29ff,0x73bf240d),LL(0xeaeda06e,0xa2c40faa),LL(0xadd94b47,0x69632929),LL(0xb0069076,0xc72354f6),LL(0x7878e92c,0x8d197fbf)}},
   {{LL(0xa83bbb88,0x69a9ebd8),LL(0x29f98875,0xcbab0b5a),LL(0x4e7611f0,0x325e487e),LL(0xd955cc3b,0x90aa24b1),LL(0x3c264d53,0x840e70a1),LL(0xad7f4f81,0x15bcf88b)}, {LL(0x2cf0df0a,0xe47552cc),LL(0x79205ea9,0xcb999733),LL(0x10d5ca45,0x25dc58bd),LL(0x1228b978,0x0947d715),LL(0x4f2c7c4a,0x9a0204da),LL(0x4690052c,0x4377ea4a)}},
   {{LL(0xb8e79179,0x015c325e),LL(0x5b57dce6,0xf4fc6133),LL(0x78d6858f,0x27a51e5d),LL(0x4dd5f180,0x13babcab),LL(0x847e499e,0xfaa19cb1),LL(0x08aaea61,0xe2688ae6)}, {LL(0xe86100d5,0xe20d7edc),LL(0xed2fedac,0xa9b0d46b),LL(0x1d357ded,0x5e99cc0c),LL(0x723cac89,0x4c1263ab),LL(0xf15e22f4,0xad5f3e6f),LL(0xd77dae65,0xf25f3950)}},
   {{LL(0x37e8e6b2,0x3c0e2b97),LL(0x575da8b7,0xa2037913),LL(0xb925cbb2,0xeedf0a75),LL(0xc561b405,0x4f28ec1b),LL(0x2901931c,0x368fb274),LL(0x2f26221f,0x52b54eee)}, {LL(0x247812a9,0x381845b6),LL(0x9115a0df,0xf9bcc961),LL(0xcb84d25b,0xef127dfe),LL(0xfa10e0a7,0x4256afe5),LL(0x353a15eb,0x0c08a532),LL(0x6a91e61e,0xbbd15b17)}},
   {{LL(0x6150771a,0x854b0584),LL(0xd9ca9868,0x35fdd9b4),LL(0x4c32fc71,0xec829389),LL(0x9ec8f90d,0x882fad4c),LL(0xc6c7b9c0,0x2d39990d),LL(0xd71a25e5,0x7fbc201b)}, {LL(0x5166da7d,0x6b852e65),LL(0x3d8c6e36,0xc6bde23a),LL(0x5857f048,0x37001154),LL(0x1ccb9bc8,0x746621fc),LL(0x612bb853,0x97e44e63),LL(0x758da4ed,0xabc3b450)}},
   {{LL(0xbb44db8a,0x856463a8),LL(0x4baf2f64,0x18ec5abe),LL(0x92980518,0x75efa67d),LL(0x94b03911,0xabd300f8),LL(0xc1cd1cf7,0x4ea44bd4),LL(0x124ef41a,0x525c3583)}, {LL(0x6b3a701c,0x03c18c0a),LL(0xcc7b6885,0x5ef82731),LL(0xa00d0089,0xb3716386),LL(0x741c0ea9,0xed9af50c),LL(0x8a36f03f,0x31f0e49a),LL(0xca724a16,0xce993757)}},
   {{LL(0x9a90c6b9,0x2b6bf5e2),LL(0xa2c0b35f,0x9b134fc8),LL(0xa6717af8,0xd64157f8),LL(0x45dd16f0,0x7228156c),LL(0xdaa99226,0x0a9ab894),LL(0xbccb07e5,0xa7b130f7)}, {LL(0x4d9243d5,0x2b7ab41f),LL(0x88568fa4,0x570b4ed0),LL(0xc9c84d91,0x666b98be),LL(0x02124f7a,0x2a75793d),LL(0x0aa16891,0x0fb9b82e),LL(0x3cc7f2d7,0x71e94c93)}},
   {{LL(0xf47211c0,0xb7c13bfb),LL(0xe70929ae,0x738fe4a9),LL(0x423db6f3,0x4cdac99b),LL(0xe4255a4d,0x3883201a),LL(0x9730d749,0x7ffe82d3),LL(0xafbaaabb,0x033ba755)}, {LL(0xf836d62b,0xa302ba65),LL(0x143a88bb,0xc626f604),LL(0x31c16b3f,0xe1d189aa),LL(0x7d68ab17,0x4664c050),LL(0xc02eb56b,0xcffc651c),LL(0x6ab49c24,0x6c52f3c4)}},
   {{LL(0x8fb415b7,0x58775857),LL(0x9e88dca3,0xab580f15),LL(0xa62265d2,0x86a12e3d),LL(0x72d98b08,0x8ec42786),LL(0xf61e9c85,0xb9da8016),LL(0x800994ce,0xa895aedb)}, {LL(0xe38ef526,0x63d0878c),LL(0x1efb6575,0xa081d714),LL(0x6a1c1efa,0x780b9e12),LL(0xebd0497a,0xeed68d0e),LL(0x9265231e,0xbfeee3d2),LL(0x80e03127,0x46f751da)}},
   {{LL(0x03e4074c,0x806dce0b),LL(0x6973bc6d,0xd4054700),LL(0xa2897b68,0x8c4d393e),LL(0xc7f9af16,0x7d592d04),LL(0x4f895dbf,0x0625826a),LL(0x37dfc8a1,0x038bed29)}, {LL(0x4799d78a,0x981862da),LL(0x6675c1b3,0xf34c1895),LL(0x8706afd3,0x64de9a5b),LL(0xf74e6ca3,0xc80ff68e),LL(0x83ed8cf2,0x26a90422),LL(0x77a47011,0xc1f5ce53)}},
   {{LL(0x4aabf727,0x0bd15783),LL(0x54fce3ad,0xfe79d2dd),LL(0x1a77ce78,0x242b1806),LL(0xda7e489c,0x30cf6c32),LL(0xb2966f38,0x854e43f8),LL(0xa2dd4ea3,0x6e5fb045)}, {LL(0xde3c9b7a,0xb2a48aec),LL(0x3625ebd6,0x66ac77e6),LL(0x0bcbabe2,0x332a969b),LL(0x5be51225,0x19de2701),LL(0x5b9b80e6,0xf26c73fc),LL(0x91025007,0x98c9cbf7)}},
   {{LL(0xbc098b8e,0x1e41874e),LL(0x8a234773,0x758ccd5f),LL(0xc1bc847f,0x47ab76ca),LL(0xc540ceaf,0x377f32e9),LL(0x69c2df21,0xba8897c3),LL(0x233c3a02,0x6afacb01)}, {LL(0xfb54ffb1,0xf1609b45),LL(0xb8f9c150,0xb7a98e5a),LL(0x08b1977e,0x607478b0),LL(0xd48b7a90,0xa9500582),LL(0x32fa7597,0xacd841e3),LL(0xf9333957,0xdf53373d)}},
   {{LL(0x04926a41,0xd25f6508),LL(0x514045da,0x7236b475),LL(0x08b9b08b,0x0b360311),LL(0x3fe92e91,0x16477aff),LL(0x03189ddc,0x6e5f6cb1),LL(0xc698a38f,0x81ff008e)}, {LL(0xc93adb23,0x02a09218),LL(0x445d8fae,0x71fcecd3),LL(0x8fd6b76c,0x55a15eac),LL(0x11ef96b4,0x1e37ec36),LL(0x30e433b5,0xd1b3b3fc),LL(0x51d174c3,0x49518733)}},
},
/* digit=31 base_pwr=2^155 */
{
   {{LL(0xb8c9f82e,0x7914213d),LL(0xfc038e90,0x7a3e4e38),LL(0x26a34238,0x6edae5a1),LL(0x701ce8c7,0xe566bf50),LL(0x55656e02,0x3562e875),LL(0xb4e8efbf,0x48325ebf)}, {LL(0x66505ec3,0x5f10a504),LL(0x8da78aec,0xd8b9834b),LL(0xcc2f2e40,0x49d1fc25),LL(0xaf5718c1,0xe973bb1c),LL(0xd2d6b890,0x9b8825da),LL(0xe2f00f12,0x7de7885e)}},
   {{LL(0x7ef79898,0xe37211be),LL(0x21344d16,0xa8103877),LL(0xa1b9f8b4,0xfdcd7e26),LL(0x7d7f72d5,0x5641e45d),LL(0xc449c920,0x5377c1be),LL(0xefc7b2a1,0xd3edcb0c)}, {LL(0xe14b42fc,0xc657a9ff),LL(0x00831b07,0xc8f858c8),LL(0xd020eaa8,0x6bfcd1bc),LL(0x3f6860c7,0x17534b0a),LL(0x84c7c806,0x8ce57222),LL(0x2bd7456b,0xa1d40eaf)}},
   {{LL(0xc9aa57ee,0xe0c93007),LL(0x8895a604,0xebb2d47b),LL(0xc4fd6ffe,0xb8aebc49),LL(0x73f300b6,0x2c06e1e5),LL(0x81628b8b,0xa019070d),LL(0xbaf8c1ea,0x2db1690b)}, {LL(0xcc94ccd2,0xb3fce6c8),LL(0x85bcdf4f,0xf3014638),LL(0xe2f82c32,0xb1e62616),LL(0x68295a54,0x85581e24),LL(0xbf51f8fa,0x0f2e2ff5),LL(0x155c1f6f,0x940716f1)}},
   {{LL(0x4e623856,0xaed02b6b),LL(0x3e1d74cb,0x7a6d2bef),LL(0x654e7c30,0x82226ec4),LL(0xe7034bfd,0x008ac003),LL(0x7fd6b555,0xe343c540),LL(0x1b429d44,0xca1b2907)}, {LL(0x9c3ceea2,0xe0702a33),LL(0x732694c3,0x48079aa9),LL(0xd4652401,0x7e6d72f6),LL(0x35f60043,0xd92655ed),LL(0x273e8cc4,0xa0dbaac6),LL(0x3c3ffb40,0x0bb8f0f9)}},
   {{LL(0xc95cd23b,0xb41b87b6),LL(0x55e371a4,0xb99714ba),LL(0x6f571ceb,0xb138ee8f),LL(0x80146ec7,0x09c42be4),LL(0xee9aa125,0x275ee21e),LL(0x3a878b59,0x0cef4d6f)}, {LL(0xa801068d,0xd436eb1c),LL(0x762b8a80,0xe2c5448c),LL(0xf3640eca,0x243beee1),LL(0x32bbba7a,0xf979458b),LL(0xa63407d3,0x6bc26cfe),LL(0x392dd1d3,0xd3b6e132)}},
   {{LL(0x3de4ba2e,0xbc06ecab),LL(0x9e491bcd,0xf51ca063),LL(0x453c94be,0xa6fc6fa0),LL(0xed1a6731,0x5460f943),LL(0x4ec3f1fb,0xeb11656a),LL(0xff1e7d4e,0x2fcb2cab)}, {LL(0x8fea2286,0x59526467),LL(0x4e0bee38,0x838117a3),LL(0x24fd2ce5,0x7bdf5888),LL(0x9f2c2925,0x13df0c83),LL(0xdee97f30,0x1bf621e6),LL(0xebea6641,0xb43b2558)}},
   {{LL(0x8e729329,0x246c8660),LL(0xd693dac8,0x39fcc41d),LL(0xc062a6c0,0x48a65b54),LL(0x6a5a3101,0x368a5770),LL(0x47ed1988,0xd143600f),LL(0xa764ce3d,0x48466d92)}, {LL(0x5a22cb6b,0xb0500613),LL(0xedea070c,0xf1d77247),LL(0x617f2464,0xb1ddd151),LL(0x28b83fd9,0x79050698),LL(0xd70bf93e,0x021abb26),LL(0xab5a5e1e,0x590b3c42)}},
   {{LL(0x1cfb991f,0x5906a35c),LL(0x740a7744,0xb62a4f80),LL(0x36f84763,0x65c8ac91),LL(0xbe0f1dd3,0xf73b3deb),LL(0xa2d26c21,0x40358868),LL(0x76792ae7,0xd907e90a)}, {LL(0x668c3d5f,0x3ecea167),LL(0x6754b49c,0x731068f2),LL(0x0e006243,0x6db89109),LL(0xdd94681b,0xd29106e6),LL(0xa85a3de2,0xb40b8694),LL(0x936b86cc,0xc80c7bf1)}},
   {{LL(0x11913575,0x003d45d0),LL(0x87e1186b,0x866cb2dd),LL(0x46b69a22,0x692f6301),LL(0x8174c1d1,0xd296a55c),LL(0x9f17af00,0x77ef6fbe),LL(0x3aa922e1,0x6b588be9)}, {LL(0x033e6dd7,0x99ecb44f),LL(0x1d22b7cd,0x32edea2c),LL(0xba7006f3,0x3122b027),LL(0xbb6ebc5c,0x8950054b),LL(0x82dab805,0x4f6d6061),LL(0x1bae5f1b,0xc1205518)}},
   {{LL(0xe08c180f,0x28d33e79),LL(0xf6aec9ce,0x768c7794),LL(0xce683c5b,0x5a749f3b),LL(0x8371fe75,0x717629d9),LL(0x57712c1d,0x5e828fc0),LL(0x7e4c61aa,0xb46c6ed1)}, {LL(0x5bccf95c,0x5d927bad),LL(0xd72f68ec,0x55d6fc80),LL(0x98591dc2,0x560a99a3),LL(0x4836664c,0xc885fe8a),LL(0x26d79298,0xd18acd42),LL(0x185df1d7,0x05e4cd17)}},
   {{LL(0xfcc83355,0xebc60e21),LL(0xd9119b77,0xc94dbc02),LL(0x2f18ae9a,0xceb05a31),LL(0xb8f69016,0xa8462962),LL(0x8f67b5f4,0x58dde5a4),LL(0xaf3c234d,0xb8bdf9c9)}, {LL(0x80e85df8,0xe95c069f),LL(0xab3aa0e5,0x9d525e1b),LL(0x76276d8b,0x73c8a92f),LL(0x163530ef,0x7feb4abd),LL(0x5ef5ad73,0x8ca949b3),LL(0x2e3d057b,0xe129431e)}},
   {{LL(0x60dec308,0x9d8a925b),LL(0x6b3ea363,0xb72e3efa),LL(0xdfb534b8,0x4f53ca6d),LL(0x6dd78a32,0x4e64874c),LL(0xc2a146d5,0x336e5b46),LL(0x98395201,0x07c76d63)}, {LL(0x8fe3e815,0xa4c09522),LL(0x3221cc26,0x887e659d),LL(0xc36286ec,0x0ff92f64),LL(0xc3ebb08c,0x57b1b903),LL(0x65f00c30,0xc6bdc9b6),LL(0x9a46d36e,0x82624226)}},
   {{LL(0x1a0b619a,0xc782c16c),LL(0xbe316086,0x8643d42b),LL(0xc0daa421,0x49d2966b),LL(0xb7b487e0,0x080b1caf),LL(0x144de273,0x1d33bb53),LL(0x6faf7ed9,0x8bafce2d)}, {LL(0x408d4636,0xdafbe3cf),LL(0x7ee8835b,0xf10527df),LL(0xe2e75522,0xe1123f3e),LL(0xebe27d60,0xb388c64b),LL(0xe3f1f55e,0x2cb38dc1),LL(0xe34524d8,0x57ff8e43)}},
   {{LL(0x85653dc8,0xd67dc92b),LL(0x0bc93ab9,0x8e0970af),LL(0x8b87c0af,0xb6f09baa),LL(0x52760ef4,0x5a8a9030),LL(0x1047bf85,0x2e2ae756),LL(0x85bd4e74,0xd049078f)}, {LL(0x3729f708,0xced11ff8),LL(0xd91068a6,0xdd21cbeb),LL(0x24b3e911,0x83d488ff),LL(0x1afd2196,0x6e166fda),LL(0x4f0d2128,0x66a91211),LL(0x05c9f39c,0xd11078ed)}},
   {{LL(0xbdbdf0cf,0xd87003d3),LL(0x56c298f1,0xe9750b5b),LL(0xb73ad05d,0xc256c3a2),LL(0x2ee94279,0xe0779a19),LL(0x279626a3,0x31d8b3c6),LL(0x90163bc8,0x469056bb)}, {LL(0x23755853,0xe6aeabc6),LL(0x896a6f4c,0x9fffdfe2),LL(0xa36cf41b,0x15c1ce78),LL(0xeee41941,0xd4c8c025),LL(0x7653be9d,0xf7a917ee),LL(0x59d52222,0xfa3cba96)}},
   {{LL(0x5f8ab132,0x913f9207),LL(0x5c14080f,0xd5b6792c),LL(0x787c3594,0xefab4e2c),LL(0xe7b7b7dd,0xa55d465f),LL(0x34e28e6a,0x921aaad8),LL(0x12d6a7bc,0xc4f3a35e)}, {LL(0x6115a5ae,0x109803c4),LL(0xe709f9a1,0xc023098c),LL(0x99c5bb66,0x1a8c8bdb),LL(0xbc7c2da7,0x1cd1c2b6),LL(0x5f927eef,0x50189c97),LL(0x229f9410,0x493823d1)}},
},
/* digit=32 base_pwr=2^160 */
{
   {{LL(0xf0ce2df4,0x11a8fde5),LL(0xfa8d26df,0xbc70ca3e),LL(0xc74dfe82,0x6818c275),LL(0x38373a50,0x2b0294ac),LL(0xe8e5f88f,0x584c4061),LL(0x7342383a,0x1c05c1ca)}, {LL(0x911430ec,0x263895b3),LL(0xa5171453,0xef9b0032),LL(0x84da7f0c,0x144359da),LL(0x924a09f2,0x76e3095a),LL(0xd69ad835,0x612986e3),LL(0x392122af,0x70e03ada)}},
   {{LL(0x6754f492,0x3ee0a31c),LL(0x96769ff5,0x02636c6b),LL(0xf0fbfa96,0x90a64f4f),LL(0xfafea65a,0x513f054e),LL(0x9cf4b9f9,0x796ba747),LL(0x932a9590,0x3198c068)}, {LL(0x549ee095,0x93af8a65),LL(0xa212760f,0xb8b6f72c),LL(0xc1a46c8f,0x23bc71e9),LL(0x4c9bca72,0x000643af),LL(0x848cea30,0xb6d967c7),LL(0x73312ec2,0xe06b6b4e)}},
   {{LL(0x67aad17b,0xfeb707ee),LL(0x83042995,0xbb21b287),LL(0x9a0d32ba,0x26de1645),LL(0x1ffb9266,0x9a2ff38a),LL(0x8f578b4a,0x4e5ad96d),LL(0x883e7443,0x26cc0655)}, {LL(0x2ee9367a,0x1d8eecab),LL(0x881de2f8,0x42b84337),LL(0xd758ae41,0xe49b2fae),LL(0x4a85d867,0x6a9a2290),LL(0xe68cba86,0x2fb89dce),LL(0x7f09a982,0xbc252635)}},
   {{LL(0x1d85a725,0x52ec9956),LL(0xf3208012,0x0f9be000),LL(0x6dcc7816,0xe881337c),LL(0x791f7cf1,0xe4e7b6d9),LL(0x59885a42,0xfaa717aa),LL(0xf9c01e41,0xb1bbb5c7)}, {LL(0xa0361880,0xcf208d58),LL(0x20afa350,0x24426e40),LL(0x264ce04a,0x7261871b),LL(0xcd42026a,0x66be4a86),LL(0x829f99fe,0xc5397b77),LL(0x24578e2b,0xffe4a6bc)}},
   {{LL(0x8c61aaac,0xadc79436),LL(0x5e926563,0x24c7fd13),LL(0x0406c129,0xef9faaa4),LL(0x8b658d3c,0xf4e6388c),LL(0x1e435baf,0x7262beb4),LL(0xfdaeac99,0x3bf622cc)}, {LL(0x4e1aeddc,0xd359f7d8),LL(0xd78c17b7,0x05dc4f8c),LL(0x29498ba5,0xb18cf032),LL(0x85bf35ad,0xc67388ca),LL(0x62aa4bc8,0x8a7a6aa2),LL(0x72f4627a,0x0b8f458e)}},
   {{LL(0xf822d5f9,0x0733667a),LL(0x18339700,0xd7f81b9e),LL(0xa7bc265f,0x7ca29b27),LL(0xeb4f0c7a,0x9fefa698),LL(0x01f27630,0x7b6f3513),LL(0xfcfb1133,0x72f0f152)}, {LL(0x5c81eb14,0x9928d9d0),LL(0xed8ff6cb,0xa16ac36b),LL(0xe041bef3,0x7fbd1acb),LL(0xf8d99854,0x7d25159a),LL(0xdb5a0dc5,0x2ec3a7d8),LL(0x87e3e933,0xd86fc4cc)}},
   {{LL(0xc68e4488,0x3fb812ee),LL(0x60ef7281,0x53c5eaa4),LL(0x8fbefbe4,0xe5724183),LL(0xa4b24a05,0x2b7d49f4),LL(0x710c0a43,0x23b138d0),LL(0xa85ec1db,0x16a5b4c1)}, {LL(0x305feb02,0x7cc1f3d7),LL(0x5b6c1b54,0x52f7947d),LL(0x8f56981c,0x1bda2312),LL(0xb4080a01,0x68663eae),LL(0x9f999b7f,0x8dd7ba7e),LL(0xb686580c,0xd8768d19)}},
   {{LL(0x4c20e15f,0xba8418f3),LL(0xfb54404e,0x7eed2494),LL(0xbce1e82d,0x4e6438d7),LL(0xb397915b,0x9e489b3e),LL(0xfb4cf659,0xa9baea9f),LL(0x42ef4aff,0x8bc5b2ba)}, {LL(0x7e62a188,0xae3fb533),LL(0x496e8e35,0xcd648493),LL(0xdefe047b,0x89728e28),LL(0xd24e60fe,0x63a8c679),LL(0x470f710c,0xadacbf92),LL(0x5e198d3c,0xd470aeb9)}},
   {{LL(0x7afdda94,0xbcd0e0ad),LL(0x34a30687,0x95a0dbbe),LL(0x8c5e2665,0xbbe3c3df),LL(0xebf2bc16,0x742becd8),LL(0x3fa163a6,0x300ceb48),LL(0x4663354b,0x0c5d02ee)}, {LL(0xb5e606a4,0xe4fb9ad6),LL(0xcf49ff95,0x93f507b8),LL(0x585c193b,0x9406a90c),LL(0x4ecf9517,0xad1440c1),LL(0x9cea53f1,0x184cb475),LL(0x8ef11302,0x6855c474)}},
   {{LL(0x7a3e874a,0x8e3807dd),LL(0x89ac3a99,0xc4edb45b),LL(0x4bfd77d2,0x9ba9cdaf),LL(0xb540fffc,0x31d33f59),LL(0x0c60028b,0x404c8779),LL(0x89688c81,0x7f89da71)}, {LL(0x504b862b,0xdd3390e5),LL(0xe937efe3,0xdf1e721b),LL(0x63e6036f,0x5833d0df),LL(0x385fbab4,0x7712527a),LL(0xd210c0d4,0x6347236b),LL(0x8d238e2d,0x12d7733c)}},
   {{LL(0xedcafa52,0x00ecb523),LL(0x086f69d3,0x0da0ae0e),LL(0xc242f347,0xc384de15),LL(0x848c12b7,0xfb050e6e),LL(0x64e015ce,0x22f67654),LL(0x7ca122f2,0xcbdc2a48)}, {LL(0x445fb02c,0xa940d973),LL(0x3767d89d,0x00f31e78),LL(0x613dabdd,0x2b65a237),LL(0xc875ae09,0x2be0ab05),LL(0xba204f8e,0xb22e54fd),LL(0x0f7687b9,0x65e2029d)}},
   {{LL(0x302e943f,0x0ecb0723),LL(0x4a443e78,0xd180ca1e),LL(0x23dd2c9e,0x39e78911),LL(0x01fe50bb,0xfa2a4404),LL(0x154d39d1,0x4678e7ed),LL(0xaf513e01,0x64ddaee1)}, {LL(0x634904da,0x6d4c615a),LL(0xba5c900c,0x937c6326),LL(0xeb6c8582,0x70658f5f),LL(0xf3d65166,0x2a04fd51),LL(0xb676eb47,0xcefe7472),LL(0xf597d887,0xd3565a71)}},
   {{LL(0x1855a71c,0xffd82538),LL(0x438bd8d8,0x26a330b3),LL(0xf9d8c5f9,0x89628311),LL(0x953738a0,0x8d5fb9cf),LL(0xedfcd4e5,0xcb7159c9),LL(0x2064c7c2,0xd64e5230)}, {LL(0x689f3cfe,0xf858ed80),LL(0x56128b67,0x4830e309),LL(0xe0e90688,0x2e1692da),LL(0xca9cc232,0xab818913),LL(0xa5d229a6,0xe2e30c23),LL(0x0e740e23,0xa544e8b1)}},
   {{LL(0xe5dcba80,0x299520f4),LL(0x2b758045,0x522ad4b5),LL(0x193b36d4,0x54eabe27),LL(0x45e9e442,0xda4d3bff),LL(0x637311f3,0x44cb9252),LL(0x71338ebf,0x4cd620a9)}, {LL(0xcc9524fb,0xec908157),LL(0xa8c955d7,0x2731a11b),LL(0x5cb94009,0x72a5e054),LL(0x9126cfe8,0x7eee8f3b),LL(0x3dd5d5ce,0xc71e2920),LL(0x22069494,0xe886f91a)}},
   {{LL(0xdc61e6cc,0x1c15e569),LL(0x58fc7800,0x8fd72967),LL(0x37a9dfc5,0xe61e7db7),LL(0x5afd7822,0x3f34a9c6),LL(0x19e80773,0x0a112742),LL(0x4760fc58,0xa353460c)}, {LL(0xb3124c71,0x2fb7deeb),LL(0x2d4009cc,0x48463627),LL(0xc3a10370,0x399d1933),LL(0x54388dbd,0x7eb19450),LL(0x7c2a006a,0x8ecce639),LL(0x55c932a0,0x3d565daf)}},
   {{LL(0x0db962c0,0x294d2955),LL(0x6d523ab0,0xd6994ef4),LL(0x58f95037,0xfa1a7f91),LL(0x64420c94,0xb1379811),LL(0x093caea8,0x2b686e1e),LL(0xf9e1c340,0xdef10944)}, {LL(0x611d9bf5,0xcd1beecf),LL(0xa1b5267b,0x34696c50),LL(0x2dfc2b16,0xcecbc719),LL(0xcee7e854,0x2cdb955d),LL(0xf2635cc8,0x9fefc321),LL(0x2936f7d3,0x276d2e4f)}},
},
/* digit=33 base_pwr=2^165 */
{
   {{LL(0x67141724,0x4962c021),LL(0xabe7762f,0x5f81eabe),LL(0xdd189c3f,0x78549a79),LL(0x6ce517a7,0x47675cdd),LL(0x32d6bb97,0x5102294e),LL(0x6ed1a029,0xb19500c6)}, {LL(0xb16a206c,0x3efb54e8),LL(0x0dc135b8,0x7dbdcc25),LL(0x8967fb04,0x955bc294),LL(0xbe04e909,0x373615c9),LL(0x111efad6,0xf1fcf820),LL(0x6fd2e97a,0x8530f97d)}},
   {{LL(0xb4805410,0x3f2b5bd4),LL(0xf96c5ee7,0x201ca7a9),LL(0x94256fe1,0x532ef2db),LL(0x318ddb03,0xacbfc459),LL(0x5f24c8e1,0x2375f9fd),LL(0x370783db,0xd27c479b)}, {LL(0x56541ae6,0x1bd461e8),LL(0x7f7ea49a,0x78f054a7),LL(0x8845f315,0xc9f8777d),LL(0x97fc92c7,0x81aed296),LL(0x49929540,0x9f2f8d79),LL(0xff5ebfe0,0x7531e78b)}},
   {{LL(0x543b3e41,0xbd9a66d6),LL(0x2ae73774,0x2948c0a6),LL(0xef38e9b3,0xa75151df),LL(0x754fb3fb,0xa3348ae5),LL(0x13069b72,0x1218fa8f),LL(0x0835dfaf,0x532bb051)}, {LL(0xdf2be3c6,0x2121a98e),LL(0x9e5199bc,0x85980de6),LL(0x1a1eb6ee,0x1b23a4be),LL(0xadeb3ae5,0xb5c48b92),LL(0xedea2b45,0xeebd305d),LL(0xc37198ea,0x20543f04)}},
   {{LL(0x0fab968c,0xd0960bd8),LL(0xae028db0,0x6899e4fa),LL(0xa9850916,0x975ccc77),LL(0xe5f81554,0xb41bd531),LL(0xc8cff2c8,0xbdf8ab57),LL(0xf5822be3,0xea306a01)}, {LL(0xbefbdbbe,0x1f0ac0e7),LL(0x60519f87,0x72f4b0e9),LL(0xe3cc86ab,0x22bd8b82),LL(0x2b2beaee,0xc43bde8d),LL(0x412617ff,0x8168781e),LL(0xb7ee7096,0xc5610627)}},
   {{LL(0x40671bd2,0xc4085538),LL(0x599f8eba,0xccb74902),LL(0x8d65e832,0x477a4a09),LL(0xf1241626,0xc620c314),LL(0xe7344054,0x05f2f152),LL(0x6d83320d,0x2483a2d9)}, {LL(0x0e344da8,0x167439d7),LL(0x1002fb36,0xa0c02b1b),LL(0x30afbda2,0x46cb4a70),LL(0x229d4efe,0xe74e3488),LL(0x1cef3aec,0xd371dce2),LL(0x7c3f2521,0x96e0592c)}},
   {{LL(0xcc5f4a6e,0xdec0e091),LL(0x201108d0,0x055ff295),LL(0xdc202800,0x2b371998),LL(0x0d5f5f9d,0x1b650d83),LL(0xa0226262,0x29be5503),LL(0x185bf3f1,0x0f3681ac)}, {LL(0x8c3e2c4f,0x4f5c44b4),LL(0x717814fd,0xcfe74a51),LL(0x7b52a561,0xb0ce4183),LL(0xee634895,0xde143e3a),LL(0x48d46b2f,0x08f83100),LL(0x6386486d,0x34b79d55)}},
   {{LL(0xee1388ba,0xa60659c4),LL(0x7eca29dd,0x475a4d06),LL(0x66470e08,0xf1ef88c1),LL(0x687c716a,0x71cac87a),LL(0x86f43fff,0x994fb1ee),LL(0x43384658,0x53ded2ef)}, {LL(0x56c41587,0x317d9024),LL(0x9dea0f26,0x6807f0ad),LL(0xa0b0d53b,0xfee8beaa),LL(0xb0288c94,0x15c06a1f),LL(0x47028bce,0x4b9eab03),LL(0x1446bc6e,0x6d214435)}},
   {{LL(0x01a8eb44,0x0869457a),LL(0x7a7bedd7,0x52223985),LL(0x00057505,0x2c04b0c6),LL(0x0b09adeb,0x468be6e8),LL(0x6f81474f,0x2f3bf32b),LL(0xa712ccce,0xf54f949d)}, {LL(0x4cdd8f2a,0x292cee42),LL(0x9c221be1,0x3d9fdf6b),LL(0x56f47b2a,0xe54da661),LL(0x840b5d1b,0x2ca76835),LL(0x8a6e8cf6,0xb52adb6a),LL(0xdade153e,0x8b416a6b)}},
   {{LL(0x3fd55732,0xb8c3a138),LL(0x2fd63498,0x4b348243),LL(0xf52c44d3,0x60cc7a0f),LL(0x9422c87e,0x9787f356),LL(0x60ad9f57,0x0fae6177),LL(0x82e307c4,0xb9ac12d8)}, {LL(0x79416b16,0x52316618),LL(0x91611196,0xe603cb9a),LL(0xe34c3a39,0x0ce22a84),LL(0x5b2dca8b,0xcb191237),LL(0x9e5b750a,0xcb87c227),LL(0x13a02f00,0x7feccba0)}},
   {{LL(0xa3cd6034,0xd3a66685),LL(0x3a283c3a,0x74ad77ad),LL(0x66644cd0,0x0c903afb),LL(0xc1f90202,0x34e18018),LL(0x0008aa04,0x8fa168e7),LL(0xbe8e35ae,0x871573df)}, {LL(0xf4561f77,0x8841591e),LL(0xc6cd0f01,0x55e3033d),LL(0xfe1fa6ff,0x44327c97),LL(0xdddfcba8,0xed4e69d6),LL(0x52d907ec,0xe82e1e42),LL(0x79064b8d,0x8a5d9a71)}},
   {{LL(0xc26f1000,0xa47a884c),LL(0x44a4b7cb,0x9f047634),LL(0x89813d5f,0xa5a2bf4e),LL(0xd8e9318d,0x83ce02c9),LL(0x2c86c874,0xf2731254),LL(0x02a0813f,0xb76a3096)}, {LL(0x0c8b505d,0x1fabaa2d),LL(0x4895425b,0xad693de0),LL(0xa55ff754,0xaf7f5848),LL(0xeadfa222,0x098a328f),LL(0x470c5898,0x4c20adfb),LL(0x5d199733,0xe4b23ead)}},
   {{LL(0x9a605b1a,0x37b29ca7),LL(0x30caed34,0x2ca364f6),LL(0x3c2ff02f,0x1e538aae),LL(0x42c6a320,0xc10c493c),LL(0xa2b9e130,0x92f18f28),LL(0xdd0ed08f,0xaf249c41)}, {LL(0x70285f2f,0x42039a52),LL(0xcfd3ebb0,0x4e502f2e),LL(0x6e39b30f,0x42821914),LL(0x70e820ca,0xdbc809f1),LL(0xd2781e34,0xbccb5567),LL(0xbe3d05dd,0x8c1e3bfc)}},
   {{LL(0x43eeed0a,0xa29f0b65),LL(0xfd8b3697,0x0b96b6a1),LL(0x88f8755a,0xc2454ada),LL(0xaf85d4ac,0x57202337),LL(0xa7aaef40,0xaddfc388),LL(0xb156a5fc,0xc1495163)}, {LL(0x1f1a8775,0x67abbbcb),LL(0x5958bb8b,0xb83e3dd6),LL(0x36dbf23c,0x17119d02),LL(0xb954337a,0xeb336460),LL(0x6f46dca7,0x9b101001),LL(0xbc991dbe,0xef862ae8)}},
   {{LL(0x66befae3,0xaa878db5),LL(0x05110c6f,0x28bb9c9e),LL(0x4caa069a,0xc3a57a8c),LL(0x594a2753,0xbb5b550e),LL(0x5187afb8,0x01ab8056),LL(0xb9255f65,0x1f7c9ed3)}, {LL(0x146f6635,0xcd669e2b),LL(0xcb9457a3,0xa8f2d4b6),LL(0x849dba46,0x0f0541aa),LL(0x2cc9c7e9,0x2537bf02),LL(0xd79ac77e,0x872b4f59),LL(0xe0a44aa9,0x705c2219)}},
   {{LL(0x3849051d,0x03b42dbb),LL(0xf27a63a6,0xe2efcfe3),LL(0xceb478c2,0xf709a5ff),LL(0xc4f7feda,0x2cc86b82),LL(0x066a1c08,0xefa834e4),LL(0x309fd644,0x153b64ef)}, {LL(0xe62168ff,0x8cff4eb0),LL(0x095d9f3b,0x0d7781db),LL(0x10bce338,0x01f8e1af),LL(0x139d8f2b,0x14aa9a02),LL(0x259ec819,0x1985d844),LL(0x3a072e8e,0xd7758b21)}},
   {{LL(0x8565afc9,0x65f7d2c1),LL(0x70fa7b82,0x764c8971),LL(0x986436f2,0xe268634c),LL(0x33356165,0x6334d8d1),LL(0x9ec7957d,0xf1716426),LL(0xb8093983,0xae834331)}, {LL(0xd2dfcce7,0xedb1fe5c),LL(0x68463e5c,0x6195b863),LL(0xa691b665,0x746e5f4d),LL(0xe1e2727e,0x61171291),LL(0x6f27b029,0xbb4aa8f1),LL(0x7f42c197,0x1037657d)}},
},
/* digit=34 base_pwr=2^170 */
{
   {{LL(0x0d5b855a,0xfe19901b),LL(0x2f745022,0x5facb955),LL(0x56c4ce5c,0x92fd0125),LL(0x938c89ab,0x23172d65),LL(0xaaa587b1,0xa71f8a33),LL(0xb55c9c50,0x511a3745)}, {LL(0x7185086e,0xec005f6a),LL(0xf894c6ab,0x6dfc2761),LL(0x9e26361f,0x98a4d67f),LL(0x21389c25,0x7f0a2b23),LL(0x95ffbcee,0xd1588207),LL(0x9f36a888,0x4d6b29ab)}},
   {{LL(0xd0a701a5,0xeccb421e),LL(0xb60cd286,0xad4cb9a5),LL(0x05a53972,0xd344da9e),LL(0x7bc99fea,0x3a8035e0),LL(0xc0f77bf5,0xe0214485),LL(0xe54df78a,0x50ada30e)}, {LL(0x4ec2d576,0xdef45af6),LL(0x5f9a8678,0xa05d6184),LL(0xc337e017,0xa9b17db1),LL(0xb84671d5,0x026a4f66),LL(0x3b7d696d,0x60614234),LL(0x81cfd22a,0x71ed9aaf)}},
   {{LL(0x439ada39,0x1b76a3c5),LL(0x89236ae5,0x818829cf),LL(0x750f8129,0x2277cb7a),LL(0x4d46502b,0x44aa462a),LL(0x64f06dc8,0x7a12e1e1),LL(0xba5630cf,0xb9a3300d)}, {LL(0x55b05f4b,0xd2cc8d9c),LL(0xa700be7a,0x6d0b0b88),LL(0x9617500c,0xa7be9969),LL(0xc03f8a50,0x2b5b8dea),LL(0x785b3dfd,0x712f703e),LL(0xccf93950,0x96a5a60a)}},
   {{LL(0xee828fba,0xfeb984b3),LL(0xe2bd188c,0x8273f830),LL(0x3ca0a99f,0x177ef97e),LL(0xacc000ac,0x76d4796d),LL(0xb140f51a,0xbad0fa6e),LL(0x06ebc810,0xb2756567)}, {LL(0xa18cb32f,0xf89eb78f),LL(0xa65285b0,0xcfc37eae),LL(0xb25e9d1b,0xe2b29cfb),LL(0xb4e7aef3,0x9388ea8f),LL(0xe267e845,0xee606c12),LL(0x9f5806d7,0x6b103c54)}},
   {{LL(0x3766f2ae,0xf418e3f6),LL(0x053ef1c1,0x4a3ad3c8),LL(0x560db262,0xd01e5b5b),LL(0xc02bf4c3,0xa583edc7),LL(0x52f318d3,0x7c9f7060),LL(0x1f5e1ffe,0x0852556f)}, {LL(0xfeb0e63c,0xe1c70aa7),LL(0x89a8c058,0x59f0a3f9),LL(0x1ffc0ade,0x4aa4cf02),LL(0x38a78632,0xbb880e41),LL(0x6f28f096,0x35b0f759),LL(0xd5757d7d,0xf9c4fe17)}},
   {{LL(0xfd6376eb,0x78b8879c),LL(0xc01e1edb,0x22a76461),LL(0x369cf0c4,0x6a44be39),LL(0x5ae54539,0x6653670d),LL(0x6fb43ad0,0x257bd751),LL(0x12baffdd,0xb3ac3715)}, {LL(0x7548eabb,0x48659d61),LL(0x0cd468cb,0xd8f931f8),LL(0x49e3b531,0x98f02415),LL(0x70df011f,0x90b0d716),LL(0xab98f066,0x26d73c54),LL(0x88475d5e,0x06591ec9)}},
   {{LL(0x412b84c3,0x627f6328),LL(0xa04545d2,0xd427e977),LL(0x104f25c2,0x5b0145bc),LL(0x2ac7ad62,0xa6931c4f),LL(0x1f8d42f5,0x40761143),LL(0xe7f8a0b3,0xfda5a76b)}, {LL(0xfe0946b9,0x4f1ca5cf),LL(0xbeb2d427,0x6def7b9f),LL(0xc9a0d136,0x984bd4bb),LL(0x5b3af1c1,0xb9a77823),LL(0x38ac2087,0x04ee66ae),LL(0x26d9dbb9,0x63374ed9)}},
   {{LL(0x44ba39e6,0xcf947c06),LL(0xedfe78d7,0xf5d5216c),LL(0x5f1835c9,0xd00115c0),LL(0xd8c79d90,0xdf084152),LL(0x6db5f791,0xc0c3a684),LL(0x749b18cd,0x40514451)}, {LL(0x734df3f1,0xd314b7d5),LL(0x7f541415,0xbccdd3f0),LL(0x6855a942,0x97ed5af0),LL(0xe9d02ab9,0xea84ae9e),LL(0x3238a5d0,0xb87e9034),LL(0x650a0eab,0xd12d25c3)}},
   {{LL(0x86a515a9,0x1473b55b),LL(0x3b337c64,0xa9e3230a),LL(0x9db668f1,0x7e8bf904),LL(0xf27f9fc9,0x1db2c25e),LL(0x2d9e467e,0x0c108607),LL(0xa3f00d52,0x4505579a)}, {LL(0x240400a7,0xe2ad661b),LL(0x11af4874,0x8022294c),LL(0x78bba8e8,0x29e90370),LL(0xf6baca04,0xbf0fbf08),LL(0x4101fab0,0x2e46d2b7),LL(0xc61089e6,0x66065490)}},
   {{LL(0x09f8a1cb,0x2131ce5b),LL(0x8ab129e2,0x7b373ed2),LL(0x77c1292a,0x463cc8d6),LL(0x94ffe9c5,0xa9b7cf65),LL(0xb99bfc4f,0x129125ce),LL(0x9820d323,0x819b4284)}, {LL(0x76541a41,0x3f709763),LL(0xe32c7a7b,0xfd679ae5),LL(0xf65b6b3c,0xc39a208d),LL(0x50002745,0x1c22ebc0),LL(0xe2bcd202,0x268f19dd),LL(0x9c3d4266,0xfeac809c)}},
   {{LL(0x87dcdbdb,0x07467837),LL(0x320493d8,0x4cce33be),LL(0x713c7746,0x9ab08cbe),LL(0x9c6dc5cb,0xd6f0c1de),LL(0x2ac03761,0x19400538),LL(0xd0547be5,0x3fc11f38)}, {LL(0x819fe3fd,0x66b378ce),LL(0x3700fe7e,0x6a590acc),LL(0x8924b396,0x4c976a72),LL(0x70b9b250,0xa5006d8d),LL(0x12b85f9c,0x2fdce1b2),LL(0x495f8f1c,0x5858f7ce)}},
   {{LL(0x903ff177,0x357540ab),LL(0x276af514,0x225280b8),LL(0x14d7fed3,0x33d273ac),LL(0xd186ee3d,0xfef6b9ff),LL(0x01a7b1d9,0xa94c2071),LL(0x50bc8bc2,0x4ea36274)}, {LL(0xfa98a918,0xc68959c9),LL(0xc7bdc262,0x8f5eccee),LL(0xe6861310,0x7a73a4fc),LL(0xc828330f,0x19bcac90),LL(0x7ef74fdb,0x73e3b66f),LL(0x52d8f2f4,0x60f76983)}},
   {{LL(0x7e03a14e,0xab357804),LL(0x8caf673f,0x0f4f2868),LL(0x66530425,0x919e661e),LL(0x91ba47c5,0x28da445c),LL(0x66c394fe,0xd6d05375),LL(0x02e8ae91,0xfe1864a3)}, {LL(0xa753aec4,0xd34baca2),LL(0xa2c8d292,0x43b7ffe7),LL(0x04efb8f1,0x496659eb),LL(0xe0252dfd,0x310ec2a9),LL(0x9168a80e,0x98173d2f),LL(0x31497255,0xa3e018d6)}},
   {{LL(0xd9d9284a,0x3266c887),LL(0x73646ab7,0x690f818b),LL(0xaf7fc33b,0x67315ec6),LL(0xc30b1ccb,0x181e61ab),LL(0x105a9e1c,0x1b81e6cd),LL(0x5078b9bb,0x62a15daf)}, {LL(0x6fa8cc65,0x74f9840f),LL(0x43388573,0x356b7774),LL(0x06b3fd46,0xba0f7d05),LL(0x92b4fdad,0xb0ac864c),LL(0xef192cde,0xcdeac253),LL(0xc313b4a7,0x0c24810b)}},
   {{LL(0x1adb09b7,0xef8c40bf),LL(0x0b74992a,0x2efeb49c),LL(0x2b79957f,0x3f0f8a41),LL(0x87a06873,0x08927bfe),LL(0x9288cb9a,0x1f63a410),LL(0xdf2b373a,0x8c66fb70)}, {LL(0x980facae,0x98da4712),LL(0xd819d026,0x15ce5b17),LL(0x749a671a,0x097571a5),LL(0x894dd269,0x85a40804),LL(0x34cb6797,0x3e89c13c),LL(0xd07119a4,0x2d19d5e4)}},
   {{LL(0xecbafb80,0x58225208),LL(0xaa73d6de,0x4f212035),LL(0x62fe86db,0x1224e455),LL(0x2dc5b2f1,0xa8c8a478),LL(0xc3096555,0x8a957b8d),LL(0xb1591452,0x6a3248b0)}, {LL(0xcb604c18,0x1e563c58),LL(0x9bf1045e,0x32808cb5),LL(0x9462e7a2,0xf8f62de9),LL(0xc2489214,0x6b3dfe91),LL(0x2174639c,0x6c1d8fc4),LL(0xef88d4b5,0xdfca11b8)}},
},
/* digit=35 base_pwr=2^175 */
{
   {{LL(0x18690ad0,0x5a4a5ce4),LL(0xfe27f51a,0xd0f788e0),LL(0x4efe9a30,0xd459388e),LL(0xef9d074b,0x3a45c11a),LL(0x93ab9cb0,0xf68ab50b),LL(0xecd9a566,0x62fbc397)}, {LL(0xcc587a7e,0xbfb79b7f),LL(0x92870bae,0xfcf4d66f),LL(0x877390f0,0x4f31aa21),LL(0xe314cfb5,0x2de0c645),LL(0x238eab12,0x56d904f6),LL(0xccb4d4f6,0x4d104a42)}},
   {{LL(0x29358cd3,0x3eb83a87),LL(0xb9c6d430,0xad741295),LL(0x53abe4e9,0x57b8c77a),LL(0xbb9feb82,0x0a14673e),LL(0xf26f922e,0xc0a6cbf7),LL(0xa32e526c,0x213de299)}, {LL(0x7b6ca858,0xca417e67),LL(0xfc2e0900,0x8d6ae0f7),LL(0x62e135dd,0x2bae0e7a),LL(0xa7ee82c7,0x962bdcae),LL(0xe5776c74,0x573d7f6a),LL(0x6ffbefeb,0x9c4de649)}},
   {{LL(0x09335d38,0x8c962fc8),LL(0xeb38d176,0x26d1bc81),LL(0xc47711ed,0xe1aeb295),LL(0x6cbe3e4e,0x0812b992),LL(0x0ab9805d,0xeecacaf9),LL(0x3521a0ad,0x82fefbaa)}, {LL(0xe2c31b9d,0x3a6948c0),LL(0xe82daf2b,0xb7d3905b),LL(0x25a34c37,0xbd3ac90e),LL(0x61453063,0x55afd99b),LL(0x90b99303,0x56d87cd1),LL(0x97ddb0a3,0xc9bf82dd)}},
   {{LL(0x68916917,0xcbc0bb19),LL(0x1094bf88,0x0bbb9f92),LL(0xd3806442,0xf62cb350),LL(0x397a7602,0xe4d2f1cc),LL(0x43987d82,0xa54bd48e),LL(0x4f0a19fa,0x77b6f831)}, {LL(0x6e766443,0xfa0c9a45),LL(0xf51ba70b,0x995ae0ff),LL(0x9cbd8d33,0x8e242c5b),LL(0x13d97956,0x1671eb08),LL(0x40da55fa,0xccae388f),LL(0xf376dce5,0x97cc48fa)}},
   {{LL(0xe8c91718,0x1c2919bb),LL(0x5097bde3,0x9dbb727a),LL(0xf8ea2fb2,0x23f87ae7),LL(0xba310121,0xe1bfffdc),LL(0x75329669,0x5938c50c),LL(0x0549855c,0x716c63e0)}, {LL(0x654814f0,0xe091b0c9),LL(0x0e43daee,0xa20535d1),LL(0x593ddd04,0x16ce68b2),LL(0xf59900bd,0x7813a49a),LL(0xd3e5d232,0xef0d3eec),LL(0x0ee3fd4d,0xe7d12cc4)}},
   {{LL(0xef01fc5f,0xe54d92cd),LL(0xdda2e25a,0xc46c2ab8),LL(0x849f6142,0x7c907fd2),LL(0xacd0202d,0xbb11dd2d),LL(0x1d92d19a,0xa4913a70),LL(0xcf610677,0xe9a26ae0)}, {LL(0x538943c5,0xfff1e1d5),LL(0xa47b2204,0x5943dcc4),LL(0x92cabf71,0xcafcf33a),LL(0xe329d1ad,0xd571e13c),LL(0x7a9a0e4c,0x7626ad23),LL(0x130d7f86,0xf0aa0d9f)}},
   {{LL(0x19e6aa7e,0x09df3a44),LL(0x5841b1cd,0xe27ad047),LL(0xbde75934,0x02d2a69f),LL(0xfd9ba435,0xb0e05e53),LL(0xe008c16e,0x4732d88a),LL(0xea72110d,0xdebc4777)}, {LL(0x2e3143ad,0xccb7d993),LL(0xea8cd06a,0x674f3753),LL(0x051562cf,0x56012a7a),LL(0x25f74cd6,0x961df684),LL(0x214d8a95,0x26630e71),LL(0x65d92f84,0x584e8d63)}},
   {{LL(0xebc5557c,0x8a89daef),LL(0x275e1649,0x7ca71403),LL(0x5b80bb4a,0x48d92377),LL(0xa45b3626,0x0a587c52),LL(0xc75bfe91,0xdaff503c),LL(0x116d07d7,0xd845d3e6)}, {LL(0xa51eeca2,0x6b5a4715),LL(0x74481991,0x34ac02bd),LL(0x595abf8d,0x8f076cfc),LL(0xed0391ce,0xc9de4ce9),LL(0xe1fcabd3,0xaaaad03a),LL(0x87b199ed,0x8d48ec00)}},
   {{LL(0xae5dd482,0xbd0f2653),LL(0x060032dc,0x59f968dd),LL(0x67283310,0x6bea33e0),LL(0x012aa50c,0xccce88cc),LL(0x66838f46,0xbb6d7f2f),LL(0x05ec9bcb,0xb764c95f)}, {LL(0x51477cca,0xd097b604),LL(0x82b20a85,0xc2fbda7b),LL(0x24e9ca8d,0x75fe07a4),LL(0x0cc40d01,0xfc4fa824),LL(0x0c0e95f7,0x0b17d5f9),LL(0x6e1e46dc,0x285e6e8a)}},
   {{LL(0xbc9b2654,0xb0641d09),LL(0x8aa8fa35,0xf9fcc2e6),LL(0x00d5ec6e,0xd12a5b4b),LL(0x5569d89a,0x9be1a111),LL(0xffac7208,0x9c0566de),LL(0x7034edf1,0x7a9fd4ff)}, {LL(0x9571c375,0x636aeb6b),LL(0x55cdf187,0x60d05aec),LL(0x734e9d2f,0xf4e2f898),LL(0x5ccdc6bc,0xdaf74219),LL(0x608a4f28,0x9d39249f),LL(0x8820e2c3,0xb5f1bb5e)}},
   {{LL(0xd9589548,0xd02e9936),LL(0x5341402f,0x8f1bf575),LL(0x057300aa,0x1535a443),LL(0x65d29324,0x3062478e),LL(0xc656a3f3,0x4203351f),LL(0x6569c4ff,0xbeb21b51)}, {LL(0xe1f0f263,0x8113ce70),LL(0x03f9320c,0x59d12939),LL(0xd08f8936,0x95061255),LL(0x97d4b705,0x8be3c0f9),LL(0x827837c2,0x0259742e),LL(0x95c65cda,0xf55ea28d)}},
   {{LL(0x603dc3dc,0x62024812),LL(0xefd67b57,0x25dc5337),LL(0xd7f033fd,0x86b3eb38),LL(0x32646d6f,0xee3226b2),LL(0xf1dae596,0x8c4825f6),LL(0xa5bcb8e5,0xd2303055)}, {LL(0x3c0baa76,0x904a5349),LL(0xe08646a7,0xe60f6125),LL(0x21d45f89,0xaf6a329f),LL(0x06605546,0xf20ad88a),LL(0x19a93d14,0xcf7a0e96),LL(0x91c97174,0xf1eabcc8)}},
   {{LL(0x8f02af51,0x72b76e9e),LL(0xd580f95a,0xac94cbf1),LL(0x01d854a4,0x2e9cd748),LL(0x1f08a1bc,0x4ed4e906),LL(0x9d2bd936,0x0a2b2841),LL(0x51c89dda,0xbf863500)}, {LL(0xe3f00bf5,0x9407b0e7),LL(0x28b57ac1,0x6b1f71ff),LL(0xcd28801f,0xc1dfe03f),LL(0xafa55309,0xf3d83d64),LL(0x8af8f76f,0x47aafba2),LL(0x6604b2e9,0x54eed45f)}},
   {{LL(0x0f3e541f,0x59edd264),LL(0x82b76ba8,0x318674b5),LL(0x4e7f0716,0xbf4a0d30),LL(0x19b88200,0x36fc0e41),LL(0x40da801e,0x91db5602),LL(0x2c72c2c7,0x638371ad)}, {LL(0xd5822da7,0xfe960c25),LL(0x4a7415e1,0x7a7571d1),LL(0xbccc1576,0x5a6480fe),LL(0xc3c88f47,0x72f4e5e5),LL(0x9a7bd8ec,0x224e7e74),LL(0x7631455e,0x3ebbf52c)}},
   {{LL(0x8608ab37,0xae3c2bc0),LL(0x39f336b6,0x35e3da8c),LL(0x81f44511,0x74136642),LL(0x1d8506e7,0x21ce7c51),LL(0x846165f1,0x9b6718b3),LL(0xf5cabf6a,0x9e455007)}, {LL(0x02611073,0xec582a0e),LL(0x83bf042e,0x269aa18d),LL(0x86306757,0x7c54fb7c),LL(0x1b948faf,0x45333602),LL(0xb7025d73,0xd3a5c508),LL(0x428471e4,0xcd6e555b)}},
   {{LL(0x11a224e6,0x42c9fad5),LL(0x69b2ac26,0x6b6aeb8b),LL(0xb149854b,0x0cf4c7fd),LL(0x2fc359eb,0x4a7d9000),LL(0x29ec8603,0x9ff0c3ea),LL(0x9b24ee14,0x157ae785)}, {LL(0x8979e9bb,0x638c809a),LL(0x7869d8c5,0x347dfb2e),LL(0xa07ea547,0x2fb1e0f8),LL(0xaecdec3f,0x1e580d32),LL(0x0f74025b,0xbbf89573),LL(0xdd529164,0xeb94d71b)}},
},
/* digit=36 base_pwr=2^180 */
{
   {{LL(0x8d688e31,0xfa2db51a),LL(0xa09c88d4,0x225b696c),LL(0x6059171f,0x9f88af1d),LL(0x782a0993,0x1c5fea5e),LL(0x4ec710d3,0xe0fb1588),LL(0xd32ce365,0xfaf372e5)}, {LL(0x26506f45,0xd9f896ab),LL(0x8373c724,0x8d350338),LL(0xca6e7342,0x1b76992d),LL(0x6fd0c08b,0x76338fca),LL(0xa00f5c23,0xc3ea4c65),LL(0xb316b35b,0xdfab29b3)}},
   {{LL(0x04a8313d,0x14f962e4),LL(0x5f1f5a26,0xc6e3e7c4),LL(0x79e777be,0x2c0e11c0),LL(0x4657c31b,0xa1705efb),LL(0x3c494de3,0x02688fd2),LL(0x412a8718,0x75664a84)}, {LL(0x7a422f8a,0x878fc7ad),LL(0x7419bd0a,0xe5d581df),LL(0x704b70c0,0x7c813c4c),LL(0x7323c008,0x98553da8),LL(0x63089f1a,0x4f63cec6),LL(0x9655d291,0x9626d6fa)}},
   {{LL(0x483aebf9,0x84e5541f),LL(0x49165772,0x8adff7dc),LL(0x9beaad3c,0xe0a43ad6),LL(0xf51c2714,0x97dd1820),LL(0x57ea5b0c,0xac2b4cb4),LL(0xd11767ca,0x87dbd011)}, {LL(0xbfc7957a,0x18ccf36c),LL(0x1bc79227,0xd4a08841),LL(0xd8d292a8,0x9811ce43),LL(0xd58c4ee7,0x72c5fc68),LL(0xd35c65a7,0x5bc0f0be),LL(0xcbbf9669,0x0b446dbc)}},
   {{LL(0x507f8b27,0x10586ea7),LL(0xa261f7d7,0x1510deb9),LL(0xdfbfa352,0xa42fc4d7),LL(0x1e1c2291,0xbf38c382),LL(0x0e11760a,0x46e40ef6),LL(0xdcb974d7,0xc24f6061)}, {LL(0xa7619027,0x755b105b),LL(0xb8ffa759,0x8004bf09),LL(0x0945db60,0xa630d0b0),LL(0xf2809e1c,0xa160ac9c),LL(0xdc6c95c5,0x38fc1113),LL(0x5d52574f,0x01f54098)}},
   {{LL(0x9cee9bce,0x7eba3da6),LL(0xd5377750,0x3e2c1248),LL(0x2b93d8b2,0x8c917d98),LL(0x7cad1f75,0xca8fc6ac),LL(0xa0ff150a,0x5f581f19),LL(0xe08327fa,0x872cc14a)}, {LL(0xe9333188,0xc774f187),LL(0x497af7e8,0x528ed4ac),LL(0x8ad72b10,0xce036e9b),LL(0x917986cf,0x463f9ebb),LL(0x1325cf9b,0xbe516328),LL(0xdd7e5fea,0xd28d5c50)}},
   {{LL(0x0b237b7b,0xa4e9f29f),LL(0x4270ee2d,0x8ed65b09),LL(0x2993359d,0x0e2184e4),LL(0x224d5aa3,0x4f96ce7f),LL(0x3a132c48,0x0862e200),LL(0x0f015f5f,0x5bbc6ad8)}, {LL(0xd7162f5c,0xab9d5149),LL(0x1267e5d2,0xfe657729),LL(0xc1fd96e2,0x4865e671),LL(0x71a703b2,0x7baf4dbe),LL(0x142add10,0x83dd6cf9),LL(0x98461d30,0xab4fc1aa)}},
   {{LL(0xdd58bbe3,0x714c1d1b),LL(0x039afd0f,0x85ba01ae),LL(0x6951ac80,0x7f23ea3a),LL(0xac00c837,0x5c599290),LL(0xbf24cc1b,0xf6efa2b3),LL(0x1e84462b,0x393d8e42)}, {LL(0xf8b89453,0x9bda627d),LL(0xb23e0d1b,0xe66fff2e),LL(0xc3b94ec2,0xd1ee7089),LL(0x3031699a,0xf75dba6e),LL(0x242b2453,0x8ff75f79),LL(0x289bfed4,0xe721edeb)}},
   {{LL(0x698ee21c,0xcda68a7e),LL(0x6a5e725e,0xc7414d19),LL(0xdce20b91,0x483be2da),LL(0xfc69dca6,0x7de1601c),LL(0xac4f9891,0x4bec17aa),LL(0x8d479a56,0xe8741dd1)}, {LL(0xac23a286,0xc623cb8d),LL(0x166133f0,0xe20a96b5),LL(0x30dcde61,0xda9bb7c0),LL(0x3a1733fd,0xf84ea327),LL(0xe82fac31,0xd7afb6c3),LL(0xd3897449,0x37ea7d35)}},
   {{LL(0xc1390fa8,0x083215a1),LL(0x6dce8ce0,0x901d686a),LL(0x837073ff,0x4ab1ba62),LL(0x34beaba5,0x10c287aa),LL(0x46985239,0xb4931af4),LL(0xb053c4dc,0x07639899)}, {LL(0xe721eecd,0x29e7f44d),LL(0x57b3ff48,0x65817182),LL(0x5054e2e0,0x198542e2),LL(0x84616de8,0x923c9e15),LL(0xad465bb9,0x2a9c15e1),LL(0x16319245,0xd8d4efc7)}},
   {{LL(0xfd028642,0xed85257e),LL(0xb96a2068,0x93657f45),LL(0xa13ac381,0xfef64eda),LL(0x56c557a3,0x108f6ff2),LL(0x9204e3f7,0xe690d92b),LL(0x03ef8640,0x902a3e38)}, {LL(0x6416f50e,0xaefd4922),LL(0xb7eae8f0,0x9b272152),LL(0x29d93d8d,0xa911921f),LL(0x5eeeea56,0x7c6bc499),LL(0x3ca7c720,0xbd3439d8),LL(0xc39b208d,0x2f8cf2e3)}},
   {{LL(0x9961a674,0x72dc7943),LL(0xa0e13668,0x839a0a52),LL(0x334945ea,0xd7a53fa9),LL(0xe7aa25db,0xdb21db77),LL(0x66e96da3,0xb6675a7d),LL(0xe66f33c0,0x2c31c406)}, {LL(0x6ec7b9cb,0x45020b62),LL(0x0391f267,0xff46e9cd),LL(0x0fa2f221,0x7dabd744),LL(0x9d4a2a3e,0x9a32364b),LL(0x52d2e47a,0xf0f84ae8),LL(0x888f488a,0xd0b872bb)}},
   {{LL(0x0370327b,0x120649b2),LL(0xcd48cdc6,0x0e76555a),LL(0xca01db03,0x4ed54dec),LL(0xac601d22,0x7be21319),LL(0x01b6576e,0xf7116619),LL(0x4e73537f,0x7839fa06)}, {LL(0xe46e860a,0x169d43ac),LL(0x3078eed9,0xde6d658c),LL(0x5032142b,0x8df73139),LL(0x9b3c76c7,0x6be199b0),LL(0xf8bbffe5,0xc2f385f6),LL(0xd5ffd28c,0x848df7f3)}},
   {{LL(0xc9790eef,0x531e4cef),LL(0x2b8d1a58,0xf7b5735e),LL(0xef568511,0xb8882f1e),LL(0x86a86db3,0xafb08d1c),LL(0xf54de8c7,0x88cb9df2),LL(0x9a683282,0xa44234f1)}, {LL(0xa6e9ab2e,0xbc1b3d3a),LL(0x87fc99ee,0xefa071fb),LL(0xa102dc0f,0xfa3c737d),LL(0xd6a0cbd2,0xdf3248a6),LL(0x1ecc1bf4,0x6e62a4ff),LL(0xc8f1bc17,0xf718f940)}},
   {{LL(0xf874628f,0xa342bf65),LL(0xc71a57bd,0x18de3f8a),LL(0x71fc321b,0xb4d12a17),LL(0xa25ebf10,0x96716602),LL(0x1a286d80,0x744f6820),LL(0xe3cf63b8,0xd60bad1d)}, {LL(0x4368da09,0x0c0b1ac4),LL(0x73a6d3be,0x53afeae3),LL(0xa90af331,0x4d2e6ce2),LL(0x88bc0638,0xd797224f),LL(0x9396d893,0x10c60b2d),LL(0xfe45e1a8,0xae3b0c11)}},
   {{LL(0x4f63f026,0x2c8b0aad),LL(0x50b253cc,0x2aff6238),LL(0x10c4d122,0xcab3e942),LL(0x07cd2816,0x52b59f04),LL(0x982c41fc,0x22322803),LL(0x8cf50b19,0x38844e66)}, {LL(0xbe3264cd,0x42a959f7),LL(0x6c983524,0xbddc24bd),LL(0x462b8640,0xa489eb0c),LL(0x98029be7,0xb7c05092),LL(0xa1addc64,0xd5546b5f),LL(0xa0c655af,0xe7cac1fc)}},
   {{LL(0xc6a6d6d1,0xa189f30f),LL(0x69665ab8,0xdd674d36),LL(0x7d8da76d,0x307c9ec3),LL(0xc1ea7c10,0xb3e1d006),LL(0xb88c62d4,0xc15e20b3),LL(0x0bff3b3a,0xb0000ec5)}, {LL(0x9ff9aa5c,0x9e330eb1),LL(0xdf578877,0x8663f9fd),LL(0x02e1eb2a,0x157d3cb0),LL(0xf525e4d4,0x638f297b),LL(0x34a3dff1,0xa20f8332),LL(0x45a9c051,0x748ea86b)}},
},
/* digit=37 base_pwr=2^185 */
{
   {{LL(0xbc266ee3,0xfe9fdde8),LL(0xba18e6c7,0x91668688),LL(0xddde6f6e,0xa65349ac),LL(0x7e54356c,0xc53c29c9),LL(0x5709f73c,0xee15ad94),LL(0xe5429277,0x033b3940)}, {LL(0xd0c3475a,0xf52035cd),LL(0x93f1f1f0,0x9c5bef4d),LL(0xca319bd4,0x26e0b0ce),LL(0x6951fd8d,0x4e7eb67b),LL(0x95c34d6f,0xac3a6f43),LL(0x00f60b59,0x1f2769e6)}},
   {{LL(0xfb787270,0xbd10b8bf),LL(0xe43aaab6,0x4f0b1566),LL(0xc0c90781,0x9a18be5e),LL(0x1ad167ce,0x3677f4c7),LL(0xa68c1c56,0xccb254e2),LL(0xe2c4d275,0x392493e6)}, {LL(0xd5b63617,0x44958cb1),LL(0x4caa4e7c,0x178f141a),LL(0xa2ffdbd5,0x7445a767),LL(0xb0b6c22d,0x0e789c99),LL(0x5dc92b2e,0x3ff8b656),LL(0xeca98782,0x1623e5c3)}},
   {{LL(0x78207cef,0xfadf9be9),LL(0x9cb5718e,0x97d5ba56),LL(0x2f995393,0xcbad24ec),LL(0x61203303,0x6236a268),LL(0x6589a4be,0xe4bafc33),LL(0x5e23fa82,0x6cba7718)}, {LL(0x4583e65a,0x8ccbc577),LL(0x4bc2f415,0xe5d88bca),LL(0x41df8dd1,0xe6bc2d58),LL(0x14d31fcd,0xec24e1d9),LL(0xfc26010b,0xacaaf13e),LL(0xe01b92f3,0x7e1da447)}},
   {{LL(0x899ef333,0x6f6a6104),LL(0x39067165,0x95496f6d),LL(0xb51989e5,0x42fd9a6a),LL(0x68f5b168,0x1b60ce0f),LL(0x56f7fe67,0x97324d87),LL(0x676815a8,0x443812f7)}, {LL(0x685a7260,0x265ee994),LL(0x6c6515f0,0x342c7b2f),LL(0x34b4adb0,0xe9092323),LL(0x1e5a8d18,0xddcd233e),LL(0x5f4f6456,0x3dc5b27c),LL(0x7f421d9b,0x9664533a)}},
   {{LL(0xff7543c0,0xc48bc829),LL(0x4d72bfaa,0xc0bda14c),LL(0x03be0af1,0x2f470ec7),LL(0x92d37eb4,0xc70f1e8e),LL(0x418f410f,0x08abdd98),LL(0x35386176,0xe38c74ab)}, {LL(0x8c00426c,0x9c07cfdd),LL(0xa998f1ad,0xba74c310),LL(0xb7d2dda8,0x76b45140),LL(0x4948330e,0xa52b5e58),LL(0x8d8efb26,0x9b733234),LL(0x5d19a312,0x5d176373)}},
   {{LL(0x987cf64a,0x43b58def),LL(0x3d4bcd4e,0xc95b16c6),LL(0xbcd9b923,0x5d1b1373),LL(0x522e052b,0xaf560542),LL(0x83800352,0xc2ff8f75),LL(0x7fe2a4ea,0x11723aa1)}, {LL(0xe94bd9bd,0x28de7668),LL(0x874018a5,0x0ce80e0f),LL(0x8d43e726,0x0fe3755d),LL(0xf9b075c5,0xa78296ac),LL(0x82207423,0x76d58d98),LL(0x1db99205,0x5c5bc697)}},
   {{LL(0xe3b7e746,0x583ee7df),LL(0xa4fab3a8,0x0b6659e4),LL(0x1946db5d,0x34ee0275),LL(0x1a12eeca,0x5ae3c0ba),LL(0x4ccb83e7,0x36756ed4),LL(0xa80eaf3a,0x973b0861)}, {LL(0x6982ca90,0x969e38f4),LL(0x018d01fc,0x9a9bcd10),LL(0x3272476a,0xb540e953),LL(0x75ab7002,0xcf91dd0a),LL(0x39ceb983,0x2c7d363f),LL(0x974747c7,0x4369c221)}},
   {{LL(0xf893a2ca,0xbfc40c30),LL(0x0623bad1,0xdf96980e),LL(0xc027511c,0x4fd7b54d),LL(0xcf3484ce,0xf4799284),LL(0x069beea5,0x655ab811),LL(0x7392e280,0x52588bc8)}, {LL(0x4f0c17cb,0x522e7b40),LL(0xc705e9b2,0xc0d88aca),LL(0x77f3913f,0x9cf1b958),LL(0x7dd52514,0x3e06b926),LL(0x2908cbca,0x992e920e),LL(0x6d6ed529,0x13baced2)}},
   {{LL(0x95ab4944,0x41c59b9c),LL(0xf32dcf4b,0xfacdd4fa),LL(0xef361bc3,0x6401bcb6),LL(0x697706ff,0x8d1fbbf7),LL(0x12fafc26,0xa9dcd2cd),LL(0xced1b64c,0xc1fce537)}, {LL(0x06433b06,0x760b3eb2),LL(0xf6f894d2,0x53a27b08),LL(0x851c8bce,0xb50135cd),LL(0x0e058bf5,0x9bf9a243),LL(0xcc0a78ef,0x231624a6),LL(0xff090623,0x8200be42)}},
   {{LL(0x67a470ce,0x5a0f6f6b),LL(0x7ccca885,0x7b8c2a88),LL(0x25f812da,0x4421fe13),LL(0xe0833478,0xc7a9c622),LL(0x94829d7c,0x6aff42db),LL(0x37d888fe,0x6fc2f23b)}, {LL(0x64b75c84,0x9cfb8a14),LL(0x46139fee,0x56e1b7d4),LL(0xfe72fca5,0xcc6943b8),LL(0xc3d621cd,0x12e757e1),LL(0x72c1571a,0x6d9d63b0),LL(0x7b300fc6,0xa1fc3db1)}},
   {{LL(0x393e37ca,0x7d58f7e6),LL(0x5e47e4c4,0x30290d9b),LL(0xa69dc1eb,0x831e5039),LL(0x5a758799,0xe2f42725),LL(0xc8f86525,0xe8166ff7),LL(0xaa85cc41,0x28e08f58)}, {LL(0xe5409138,0x66be9bb7),LL(0xdcbb88ca,0x7d0f8807),LL(0xe7803e98,0x870cd794),LL(0xd59d39c6,0x78fd1eae),LL(0xcc0e56c0,0x1aabe0af),LL(0x5bf0f272,0x7f8de733)}},
   {{LL(0xfdf88aba,0x36aa63d9),LL(0xcdf43217,0x87198fdc),LL(0xa9a923c9,0xd8fe6f62),LL(0xee2ae4ba,0x85c81a0e),LL(0x32dbcb0b,0x7c20dea0),LL(0x129a31c6,0x08baa938)}, {LL(0xb1d60f99,0xa47b3003),LL(0x6905192e,0x31459993),LL(0x32cf2c7c,0x67e22899),LL(0x3b3c32bc,0x4c5f4375),LL(0xd499aa00,0x8de5d14d),LL(0xa311dac5,0x54875c3f)}},
   {{LL(0x898dc5c7,0xb6475210),LL(0x080e62b4,0x0f709811),LL(0xbb3bffd9,0x8a0016a7),LL(0xafb97dc7,0xfc56d337),LL(0xca1b43d9,0xff911c89),LL(0x9187747a,0x9f8d40b7)}, {LL(0x5700a9ea,0xf6ba5214),LL(0x8c04b9b7,0x7bc1c0e3),LL(0xa2c924d2,0x2dd3de09),LL(0x717b13cf,0x12378655),LL(0x6e9d0e85,0x090f2556),LL(0x88f728c3,0xf66a337d)}},
   {{LL(0xe5140e1e,0x1f90b9b6),LL(0xea2d44e5,0x7ab77ce1),LL(0xf4878fcf,0xc8a3343e),LL(0xc544e407,0xbeb73f7c),LL(0xfec32a61,0xdb3a8266),LL(0x38db88f8,0xbe30a82e)}, {LL(0x525080cc,0xf33ad1df),LL(0x1a553e27,0xf66aa44d),LL(0xa9b7b198,0x3194733f),LL(0xb4b9b4c1,0xd87c8145),LL(0x68883c51,0xca7cd392),LL(0xca49b152,0x2d12a779)}},
   {{LL(0x2c4ffe71,0x407842b5),LL(0xc3f5593b,0xfa9a9143),LL(0x69f25d01,0x97f9f32a),LL(0x4571d150,0x78e3d5fd),LL(0x89878e86,0x8cbd1078),LL(0x633ed774,0xd4a4be53)}, {LL(0xd65d0ef1,0x2e6cf7f0),LL(0x18a2e243,0x5cefa892),LL(0x130c9ba1,0xb2e8ccb6),LL(0xde1ec2f9,0xda209fc8),LL(0xb3448d09,0x64845a36),LL(0x92896b12,0xbde9ae0a)}},
   {{LL(0xbfc6c637,0x8a46c911),LL(0xe4fa4fe6,0x8dc3d699),LL(0x5cf8e4c9,0xd4ba64f1),LL(0x01cb3488,0x01b3908a),LL(0x38bd7ded,0x69b1fa5d),LL(0x18b2eae1,0x92ad4838)}, {LL(0xb33955b6,0x619324be),LL(0x5c8a6df7,0xc7f37335),LL(0x925b3f69,0xa397f42e),LL(0x5f7e4d11,0x32169a49),LL(0x8d0d9f01,0xc0fa9a54),LL(0x89d8f2e5,0xf52a1f22)}},
},
/* digit=38 base_pwr=2^190 */
{
   {{LL(0xba8e0a52,0x9c0d5231),LL(0x93e465d7,0x94d0509e),LL(0x98515454,0x67df90dd),LL(0x8dbfb46a,0x223e8b9c),LL(0x6d757ce3,0xf39529a3),LL(0xb4648296,0xffec9175)}, {LL(0xf78aae7b,0x330749e8),LL(0x45f93cc3,0x19e55496),LL(0x94083aa8,0x8c320b34),LL(0x21e321c6,0x1161f5a3),LL(0xde3e7892,0x0980deed),LL(0x6ad76ccc,0x605aa919)}},
   {{LL(0x180660f7,0x73fa3508),LL(0x2d24936f,0x4cae013e),LL(0x58493d98,0xf64a549f),LL(0xdc79f602,0xd9ceae0a),LL(0xd1512b84,0x6569e37b),LL(0x151c9151,0x11e4c022)}, {LL(0xb55c5813,0x075678c2),LL(0x09d3cb16,0xb26cdb58),LL(0xa57fb969,0x6334dca3),LL(0x223dc3ce,0x0ed90820),LL(0xbd11e277,0x74f9c3ae),LL(0x79c0b8e2,0xaeefed36)}},
   {{LL(0x2a24c385,0x0e2fc74c),LL(0x34679278,0x836a4740),LL(0x817e2c41,0x25518f16),LL(0xb4b7d3c1,0x8b573a8e),LL(0x4ab56adf,0x012797f9),LL(0xfa2ab690,0x9e0e56d0)}, {LL(0x1c9f6f08,0x009ba1ee),LL(0x2f412e9e,0x8ebf4aac),LL(0x1cfb4e02,0xb143122a),LL(0xcbf2b783,0x988cf0ec),LL(0x57f5be97,0x44a7ed96),LL(0x51804147,0xbdcad872)}},
   {{LL(0x2f9fbe67,0x378205de),LL(0x7f728e44,0xc4afcb83),LL(0x682e00f1,0xdbcec06c),LL(0x114d5423,0xf2a145c3),LL(0x7a52463e,0xa01d9874),LL(0x7d717b0a,0xfc0935b1)}, {LL(0xd4d01f95,0x9653bc4f),LL(0x9560ad34,0x9aa83ea8),LL(0xaf8e3f3f,0xf77943dc),LL(0xe86fe16e,0x70774a10),LL(0xbf9ffdcf,0x6b62e6f1),LL(0x588745c9,0x8a72f39e)}},
   {{LL(0xb1bafbc4,0xa96955b9),LL(0x646ece39,0x8dcb55a7),LL(0xeb00e541,0x2b62784f),LL(0x2693249b,0x462f9d7d),LL(0x794c189d,0x8b264697),LL(0x63354e69,0xded6ff55)}, {LL(0xeed1089f,0x7c8ea441),LL(0x1462f461,0xe355f75c),LL(0x1210fd5b,0x87b691f6),LL(0x6983cb27,0x7291bffb),LL(0x92800095,0x9ed83afc),LL(0x1f24d923,0x307a3dc8)}},
   {{LL(0xd7804b2f,0x7cec60ea),LL(0x45c11441,0x00644643),LL(0x769cd685,0x3c6de88b),LL(0xc7f01232,0x34709186),LL(0xedd2bd0d,0xd9eef41e),LL(0xe427faa9,0x3bafcccd)}, {LL(0xc07e701a,0x33e5350e),LL(0xa87c1fd1,0x9cb2eb47),LL(0x0d5f5b28,0x9fa9a779),LL(0x07ea2e53,0xa2e7076b),LL(0x5c169cf4,0x72f4da32),LL(0x7e751588,0xb7f19294)}},
   {{LL(0x32dd7a30,0x47eb1335),LL(0xa9db654d,0x9d058169),LL(0x6e7a2b1a,0x375c59df),LL(0x7a35f29f,0x55d37c67),LL(0x493c4cde,0xc78a3678),LL(0x8d83e31b,0xe5f0e2d6)}, {LL(0xe9777bf9,0xf7927002),LL(0xa5afdfc7,0xdd559324),LL(0xb81c08cc,0x077c6c48),LL(0xaa2ef694,0xba1c98cc),LL(0x4c02dd46,0x06c6c954),LL(0x7dd3145e,0x211e50f3)}},
   {{LL(0x06616d0d,0x2a5f8ecf),LL(0xc7deb373,0xca9b1cb8),LL(0xc59c4301,0x9de31ced),LL(0x0111d998,0x1e0f40b1),LL(0x960d5b95,0xd29d229f),LL(0xd1dabab8,0x10563249)}, {LL(0xa05ecac9,0x7b225cc9),LL(0x78f3b8a0,0xb02e6896),LL(0xf5fb06b2,0x009b52a1),LL(0x842b9081,0x8a575d3f),LL(0xe9272512,0xfddb48af),LL(0x0b452cb7,0xd39b8f1d)}},
   {{LL(0x67e09987,0x0d6b9c7c),LL(0x0761ad52,0x261a564d),LL(0x9f60925b,0xec462174),LL(0x18529b03,0x83ee0c12),LL(0xfbcfff74,0x72972467),LL(0x6abc4bfb,0x37fc074c)}, {LL(0x54e65e89,0x8b6015bd),LL(0x991583cb,0xde8583eb),LL(0xb4d2c62a,0x379548e1),LL(0x9b24a5e5,0x88024a9a),LL(0xfc03abfc,0x633aa869),LL(0x8fa35283,0xa27657b9)}},
   {{LL(0x61d9e770,0xde9703b4),LL(0xef4653ef,0x02d4091d),LL(0x576eb5e1,0xefd229aa),LL(0xf77eb987,0xc0b0b243),LL(0xeefe8f71,0xb11309b2),LL(0x68478044,0xfeeacf2f)}, {LL(0x43ac3dd7,0x8dfd8e86),LL(0xb07f95c3,0xc0a24181),LL(0x24be161c,0x551ca096),LL(0xb098cdc3,0x6cb2c1d4),LL(0xe74f84f8,0xbfc74e9b),LL(0x067e3388,0xe58e14d9)}},
   {{LL(0xf025baa6,0x9eca6f94),LL(0xbcf9c741,0xb2db0741),LL(0x90bb8f56,0xf8e2aab5),LL(0x08762829,0x47729032),LL(0xe2a266c8,0x067a0c5a),LL(0x71b7d7d0,0x22b104c7)}, {LL(0x53e406db,0x4a48cd69),LL(0x24f0070b,0xb85e44d5),LL(0xe10133ff,0x6168262f),LL(0xe4874e8f,0xdfc02315),LL(0xca317e3b,0x20dba2d7),LL(0xe1d2c0c3,0x441c56d2)}},
   {{LL(0x808dc4b1,0xae10069e),LL(0x8fb3ba73,0x64df30e1),LL(0x7ebaad0b,0xbbe4caf2),LL(0x3dd6119c,0x5907bf37),LL(0x9dfceefe,0x0a723dff),LL(0xf7cffc7e,0x59bff4dd)}, {LL(0x6a6f43c2,0x7bc95fa2),LL(0x3ca0e2b3,0x9001d1d5),LL(0x27b3335b,0x316a7ecd),LL(0x7b8d7d49,0xbf08e672),LL(0xc619058f,0x4b209f93),LL(0x59d8f9ea,0x4c0ca01e)}},
   {{LL(0xcae69c3b,0x18c452c4),LL(0xef0f00fa,0xf45690ac),LL(0x4f66a5cc,0x3b363aa0),LL(0x47718c52,0x9dd41c0a),LL(0x7e5cd370,0xfa219d7e),LL(0xb2196dfb,0x5d384db7)}, {LL(0x90b4d46b,0x5e14749b),LL(0xd9db9481,0x55796656),LL(0xc8cf353e,0x3bf13d0a),LL(0xa95c485a,0xb89a28a6),LL(0x5da29783,0x568fa3d0),LL(0xd182b1a4,0x4aa008ee)}},
   {{LL(0xb09fa8f3,0xf7e1ed3b),LL(0x1da5be9e,0xbb4fe6f7),LL(0xf4d1ba21,0xcbab0e01),LL(0x76a5f326,0xb7327410),LL(0x206092af,0xd94d2349),LL(0x728e0e4d,0x739f3cd0)}, {LL(0xf81fd823,0x568644aa),LL(0x6110e2f6,0x510cff6b),LL(0x566c3598,0xef4cf1ac),LL(0x62aae69b,0x2c26f171),LL(0x8964a2a5,0x1e436046),LL(0x3e472c50,0x83c0bbf6)}},
   {{LL(0xaabd965a,0x79c04804),LL(0x43d0b660,0x9581aab4),LL(0x5ba71d23,0x59bff003),LL(0xb6a0cd80,0x212ecd58),LL(0xbf1ea5d6,0x29bdcd33),LL(0x77a002e1,0x59fd2ff4)}, {LL(0x8d9cd247,0x3c9d2130),LL(0xb1786da3,0x790e9dbc),LL(0x14464d04,0x967ee5e7),LL(0x2b5373af,0xd6f7ebbd),LL(0x39768d40,0x1c0b22d5),LL(0x913f6cc3,0xdfb54983)}},
   {{LL(0x51b3f1ce,0x167ec88a),LL(0x420024c8,0x19756ee0),LL(0x3877e634,0x10f2e244),LL(0x03462cb3,0x6321bf26),LL(0x9d3afcee,0x1dbd10ee),LL(0x2ca17dcd,0x0726f5f2)}, {LL(0x0bacf018,0x09465266),LL(0xe1feb969,0xc92a9f2d),LL(0x5e1c5912,0x0043b0f9),LL(0x757d3a63,0xa09b94d1),LL(0x9fdef1e0,0x53395652),LL(0xd4fedd41,0x9826886c)}},
},
/* digit=39 base_pwr=2^195 */
{
   {{LL(0xec65b53e,0x2e75a26e),LL(0x70552fb3,0xfeb630b2),LL(0xee7d8e4a,0x53dfd057),LL(0x8994f449,0xb959110d),LL(0xbb538367,0xb4a16596),LL(0xef82f29c,0xa70917bd)}, {LL(0x43bba6ae,0x5a764300),LL(0xcfbc194a,0xee207476),LL(0x03a4184b,0xc7eab238),LL(0x0f7fcd62,0x60c67ef2),LL(0xdfa8a0c6,0x41e05799),LL(0x04d352b0,0x5d7d05e6)}},
   {{LL(0x436b59f5,0xc97c01eb),LL(0xef1848ab,0x1d15aca7),LL(0x7fa7d3c2,0xdba1ce80),LL(0x81060874,0x69e6f96e),LL(0x4d7eeead,0x6e3e0df7),LL(0xb0c87f3d,0x8a3b5e85)}, {LL(0xc8a23914,0xc55ae3db),LL(0xbf6d27d9,0x5cdc2a92),LL(0x1de7860a,0xa6008c59),LL(0x6e546731,0x8202f8b6),LL(0xaa1e45d4,0x652f2d07),LL(0x6df5415a,0x146f214f)}},
   {{LL(0x651ed62d,0x4b01246a),LL(0x0b9eb006,0x9f6824a3),LL(0xba95697f,0x763ae8fe),LL(0xefe2182e,0xeff4f88d),LL(0x74ba79b4,0x2c2ef50c),LL(0x1d23d649,0x319df6c9)}, {LL(0xf6c273a8,0x481f9faf),LL(0x9706dc07,0xe7156457),LL(0x424d5cb9,0x06f0617a),LL(0xa8ad220a,0x6d6b5d8b),LL(0xa2bd8c40,0xa485ca14),LL(0x54cb54f6,0x7a7a15bf)}},
   {{LL(0xce5fb4d9,0x041c706c),LL(0xb22a79a7,0xddc78cb3),LL(0x839e9d5a,0x7dc4cd27),LL(0xbf3c4c06,0xdfc9db83),LL(0x38b7bd22,0x85b80941),LL(0xd0f4c2da,0x1007dea2)}, {LL(0xc633fba0,0xd7b34006),LL(0x4476e55f,0xa8880acf),LL(0x75236685,0xa340b2c2),LL(0x0113a85f,0x5ddd0551),LL(0x9cb32704,0x7dfc7ab2),LL(0xdabf22ff,0x9a334a33)}},
   {{LL(0x7e546950,0xa8261c31),LL(0x26706dc1,0x89d19cae),LL(0xedc9af36,0xf8dbf6ca),LL(0x7e446207,0xda79fe4e),LL(0x81cee38d,0x8bbc3195),LL(0xb5be9577,0x9d121e9c)}, {LL(0xcddb5a61,0xdb3fff88),LL(0x3ee86665,0x751cad15),LL(0x3d07abad,0x5c0986df),LL(0x83fe8451,0xf77489b3),LL(0xf90d3e94,0x3546c5a9),LL(0xd8a694b7,0x1ec54bd1)}},
   {{LL(0xd7d1146b,0x2fbce9be),LL(0xb3980bd2,0xcec9e5d8),LL(0x9f4cbaf0,0x48ea4593),LL(0x2574a3bd,0x56c54009),LL(0xe792c39a,0x84a39630),LL(0xeef81343,0xe5c690f8)}, {LL(0x17655bc9,0xf996760f),LL(0x6c0c191c,0x6009c215),LL(0x966d7027,0xa0ca12e6),LL(0x2e6f7c63,0x92a6d5d5),LL(0x9bd13ead,0x46809d26),LL(0x67aac681,0x3c11fa79)}},
   {{LL(0xea8edc01,0xca2876f9),LL(0x28d411c8,0x85c1a4ac),LL(0xfb0299a8,0x4d5fca4c),LL(0x17fe8b37,0xa1df6c5a),LL(0xcc062556,0x2ffb3570),LL(0x465ef78f,0x4c59773e)}, {LL(0xb3601ca5,0xea3e39ae),LL(0xb7d30864,0x9806345f),LL(0xe6600e5d,0x428fe41f),LL(0x067a59c0,0x19bd665d),LL(0x908cdb52,0xbd6b8272),LL(0x0b2707d8,0xb6025028)}},
   {{LL(0xabe2cc87,0x3fc9b3cf),LL(0x514e77fe,0xfd8d64e3),LL(0xfe1ad535,0x2003a58a),LL(0xcb39149f,0xcec4be38),LL(0xbdedf470,0x4d578c99),LL(0x3a356519,0xcd35d7a3)}, {LL(0x8b078d6b,0x7a762f27),LL(0x31ae2701,0x3b6891ed),LL(0x270c508e,0xdc0e817f),LL(0x9fdb29c8,0x5a7be204),LL(0xcb2711d3,0xfa1a0be3),LL(0x3786a0c2,0x5865f55f)}},
   {{LL(0x23bd1613,0x5f5db9af),LL(0xcfd5bc16,0x0071d2c4),LL(0x81adfd03,0x273d7ad1),LL(0xa0570ffa,0x683508f4),LL(0x611a75fc,0xa49f5c08),LL(0x8fbcfbb1,0xfca5bd12)}, {LL(0x2a4fc49c,0x7eabb339),LL(0xad6e6d32,0xdc5fae69),LL(0xe0b03dce,0x2e599c43),LL(0x7ca7250c,0xf4ad8d3e),LL(0xedeee0fe,0x626c1f7e),LL(0xaab0b3f0,0x2296376f)}},
   {{LL(0x2f641cff,0x1d0af6b5),LL(0x3648c4a0,0xa932adeb),LL(0xb1ea8fc4,0x67641951),LL(0xb1fae482,0xc0b90064),LL(0x6623477b,0x7012642f),LL(0x5bf019ce,0x1cddc024)}, {LL(0xc2c32737,0xca1f4675),LL(0x97d6b18c,0x11525a5e),LL(0xd3868de9,0x9c034ef2),LL(0x044e0c18,0x0533d921),LL(0xcb5e38c3,0xba6cf14e),LL(0x509d7053,0x438309f3)}},
   {{LL(0x790b3f68,0x0b82b506),LL(0x12c1f59d,0x75b38ef1),LL(0x4dbdd80b,0x6bc1e007),LL(0xd13bb11c,0xf81480db),LL(0x17259091,0x25131887),LL(0xc61dde4e,0xc5c8823e)}, {LL(0x9b2736d1,0x0ddb06a0),LL(0xa785e570,0x64dfb1fe),LL(0xcc593359,0x6f8f9945),LL(0xd186352c,0xe8e457a9),LL(0x7c342abc,0x389479b6),LL(0x4cc71b3b,0xc73ddfee)}},
   {{LL(0xc4ff9f0b,0xe164268e),LL(0xe6c36e63,0x6c8e9349),LL(0x78ab17f3,0x734f9794),LL(0x0179ed0e,0x46d468de),LL(0xdfa26867,0x7e68f006),LL(0xe3d0485d,0xe4d4a85d)}, {LL(0xf84c0f8f,0x0913a1d7),LL(0x25a9c9cd,0x4095c8c0),LL(0x49eadd08,0xeeb1a79e),LL(0x7dd8f954,0x433f5e41),LL(0x30bb26d0,0x70a62814),LL(0xff5e8e29,0xad94d8f6)}},
   {{LL(0x5652a3c0,0x0b614a64),LL(0x34597010,0x89279185),LL(0x810a812f,0x06c97f68),LL(0xbe358e91,0x566120f4),LL(0xc044ff3c,0xcb1f8b75),LL(0x7d1a468b,0x77b3b0b0)}, {LL(0x7d49aad9,0x37a72862),LL(0x8646efd4,0x3ada117b),LL(0x21bac6d0,0xd8626c0d),LL(0xdd2c980c,0x2263d74c),LL(0xfd1b6bda,0x8afd14c0),LL(0xcae64c0b,0x693742cb)}},
   {{LL(0x9e32c0a4,0x6a352b57),LL(0x77ec7a40,0x5274a082),LL(0x240e6dc5,0xee1f7c7a),LL(0xd313b4a9,0x85d5be62),LL(0x5c01a405,0x1522c5d2),LL(0x960afd5a,0xcfa08aab)}, {LL(0x8e8a93dc,0xa3cb77f3),LL(0x6d1c98c7,0xaacb1676),LL(0x3b93fa9d,0x84090c7d),LL(0x3c0383ad,0xc77f1ee1),LL(0x76f7a220,0x461c93b7),LL(0x04ac0bfc,0x66d63a1d)}},
   {{LL(0xd2343e0b,0x3d1904c6),LL(0x775f877f,0x8ce038f7),LL(0xd797f231,0xa9d4dce1),LL(0x16c08c2f,0xb6712aa7),LL(0x335ad61a,0x5045b87a),LL(0x7115bb4b,0x44a251fb)}, {LL(0xe66511bc,0x2363cf68),LL(0x81cc48f3,0xed8ab553),LL(0x725c6bae,0x8bf71687),LL(0xc23ab12c,0x16e0d015),LL(0x21f333c0,0xfbdcc064),LL(0x62c9f01d,0xe73df709)}},
   {{LL(0x4751207f,0x3c7d6b64),LL(0xe440c1a2,0x65e1f96a),LL(0xaa0eaa1e,0x8ed15d20),LL(0xc0eab490,0xe944ad2f),LL(0xf6d9f260,0x71525aa1),LL(0x16146ba3,0x5cd14c88)}, {LL(0x14a41275,0xf9401908),LL(0x2288618d,0x3bb7ea74),LL(0xcab1060a,0x6a4e1c37),LL(0xc8cac96f,0x357fe4d0),LL(0x6a2466ec,0x97a8b8ab),LL(0x9c01be70,0xb6e83fdb)}},
},
/* digit=40 base_pwr=2^200 */
{
   {{LL(0x2a7aeced,0x5b0b5d69),LL(0x01dc545f,0x4c03450c),LL(0x404a3458,0x72ad0a4a),LL(0x9f467b60,0x1de8e255),LL(0x90634809,0xa4b35705),LL(0x706f0178,0x76f30205)}, {LL(0x4454f0e5,0x588d21ab),LL(0x64134928,0xd22df549),LL(0x241bcd90,0xf4e7e73d),LL(0x2facc7cc,0xb8d8a1d2),LL(0x1d25d2a0,0x483c35a7),LL(0x1ef9f608,0x7f8d2545)}},
   {{LL(0x22ec7edd,0xa7040160),LL(0xcc9c8ee8,0x19124972),LL(0x2ccb9417,0x697f301f),LL(0x6f00d8aa,0x3ee87764),LL(0x8138a017,0x2b5afaf8),LL(0x832d7543,0xf152b14c)}, {LL(0x383052f9,0x27c27ce2),LL(0xe1dae11b,0x4746c5b5),LL(0x5b752008,0x92dc5ac7),LL(0xe84fe5f1,0xcf382e01),LL(0x7d5929ce,0x90e03419),LL(0x15ca3ffa,0xafee3abb)}},
   {{LL(0x54ebc926,0xcb51f039),LL(0xb8d4a7bb,0xe235d356),LL(0xb41fe1a6,0x93c8fafa),LL(0xa719f254,0x6297701d),LL(0x644f5cde,0x6e9165bc),LL(0x0c11c542,0x6506329d)}, {LL(0xa92b4250,0xa2564809),LL(0x889c2e3e,0x0e9ac173),LL(0x22b1d1be,0x286a5926),LL(0x6ecdd041,0x86a3d752),LL(0x649f9524,0x4b867e0a),LL(0x0629cb0f,0x1fe7d95a)}},
   {{LL(0x1c6d03b0,0x11dd860e),LL(0x09eec660,0x30c17008),LL(0x35c0192f,0xd4f8aff6),LL(0xe3a4a900,0x96a727b1),LL(0xde78c8ba,0x1426daff),LL(0x8d1527c4,0xfacaa9bd)}, {LL(0xcd072989,0x0c0d5234),LL(0x918550b5,0x1936c20d),LL(0x3d914fb3,0x4828bee4),LL(0xf3ba26a6,0x8324ea38),LL(0xa94eb26f,0x027590f3),LL(0xacd957bf,0xfd354295)}},
   {{LL(0xca5baf54,0xf4f66843),LL(0xefe7db78,0x298db357),LL(0x7365712f,0xf607e86e),LL(0x8a822bc0,0xd5882298),LL(0xc61299b3,0x2cfbd63a),LL(0x67167b1a,0x6f713d9b)}, {LL(0xde0b077a,0x750f673f),LL(0xee2178da,0x07482708),LL(0x69123c75,0x5e6d5bd1),LL(0xeab99b37,0x6a93d1b6),LL(0x8caec6a3,0x6ef4f7e6),LL(0xcf3ed818,0x7be411d6)}},
   {{LL(0x3dfb423c,0x959353be),LL(0x36d41cc8,0x8458e858),LL(0x99d7a4f4,0x6a4826f7),LL(0x52fe4b65,0xab146ece),LL(0x35038573,0x94fc21d7),LL(0xf4d56e84,0x26f50135)}, {LL(0x3162d92d,0xe15ca04d),LL(0x34ed4e84,0x8f652fcd),LL(0x4f21a910,0xef7e7924),LL(0xdcc76132,0x2eecb7e9),LL(0xdbe89048,0x5b484745),LL(0x6e43a2c5,0xbf8c490f)}},
   {{LL(0x63a0a7d2,0xf92b3073),LL(0x881dc8cf,0x32da431c),LL(0xc578e3a3,0xe51bd5ed),LL(0x9587fa22,0xefda70d2),LL(0x9b2eba85,0xcfec1708),LL(0xaf7ba530,0x6ab51a4b)}, {LL(0x98174812,0x5ac155ae),LL(0xccb076e3,0xcaf07a71),LL(0xc38718a7,0x280e86c2),LL(0xd63745b7,0x9d12de73),LL(0xbf8a79aa,0x0e8ea855),LL(0xbd705bf7,0x5eb2bed8)}},
   {{LL(0xda756624,0x3ede2484),LL(0x73b13062,0xb22da2ab),LL(0x962a667b,0x56069e93),LL(0x130f2cea,0xc931266b),LL(0xa7366a66,0x4bd6a6fc),LL(0xaa5ac3b1,0x23f30563)}, {LL(0xd7c2b26f,0xa025d0ef),LL(0x62129bc7,0x597ce7d8),LL(0x2b3057f2,0x4809927f),LL(0x1499f884,0xb001c10a),LL(0x30b9a653,0x309d141c),LL(0xbf659d05,0xadddce7d)}},
   {{LL(0xae16de53,0x33fe9578),LL(0x10bec902,0x3ae85eb5),LL(0x44af850e,0xc4f49658),LL(0x087dd658,0x6ea222b3),LL(0xa51f1447,0xb255e6fd),LL(0x117e3f48,0xb35e4997)}, {LL(0x05616ca1,0x562e813b),LL(0x8a61e156,0xdf5925d6),LL(0x571c728b,0xb2fa8125),LL(0xa2f2d1cf,0x00864805),LL(0x1bccb6ff,0x2dc26f41),LL(0x63ae37dd,0xebd5e093)}},
   {{LL(0x9303b5b0,0xe448127c),LL(0x7a74ec27,0x6ac65681),LL(0x86f0e7ec,0x05128a52),LL(0x7603a73c,0x9d9f32c1),LL(0x556c51ea,0xb5a799bf),LL(0xa9a9a416,0xea94f169)}, {LL(0xaebde511,0xebb3e549),LL(0xf9cccd3f,0x9037046a),LL(0xf08a8254,0x2a9343a8),LL(0xd40c7f26,0xc5cc43f6),LL(0xb39677d2,0xe1146cdc),LL(0xbe66d4b6,0x6cbfec8b)}},
   {{LL(0x0a285611,0xd2d68bb3),LL(0xdc8378f2,0x3eae7596),LL(0x6cc688a3,0x2dc6ccc6),LL(0x011f5dfb,0xc45e5713),LL(0x62d34487,0x6b9c4f6c),LL(0x1fc65551,0xfad6f077)}, {LL(0x62b23b52,0x5e3266e0),LL(0xe98f4715,0xf1daf319),LL(0x3ed0ae83,0x064d12ea),LL(0x564125cb,0x5ccf9326),LL(0xc63c1e9f,0x09057022),LL(0xdc9b5d2e,0x7171972c)}},
   {{LL(0xb0a219fd,0x10867369),LL(0x5ab56581,0xe7efac97),LL(0x7813f6f5,0xde3372e8),LL(0x1d4b8ed3,0x3bbe5977),LL(0x53376573,0xf17f61e5),LL(0x97964b90,0xf719d06d)}, {LL(0xb24d2e4a,0xb1e8c483),LL(0x184379eb,0x53709647),LL(0x8a8fcb83,0xb739eb82),LL(0x60163017,0x5d2f3a4d),LL(0xf823b4af,0xccedf4f4),LL(0xff315eae,0xa6e166a1)}},
   {{LL(0xeabd21b2,0x2364fd9a),LL(0x9174ad6d,0x3ce5f4bb),LL(0xb38688c0,0xa4d6d5d0),LL(0x6d87fd7d,0x2292a2d2),LL(0x4ca02e54,0x2a7d1b53),LL(0xb4185715,0x7bee6e7e)}, {LL(0x8fc63acd,0x73e54609),LL(0x4064e09d,0xf4d93a12),LL(0x2b92daa5,0xd20e157a),LL(0xc4b81a00,0x90d125db),LL(0x7682de13,0xcb951c9e),LL(0x27987545,0x1abe58f4)}},
   {{LL(0xe1c01fab,0x76452ac1),LL(0x6bd32f0c,0x167d7326),LL(0x4a283a42,0x72d209c1),LL(0x48ea0ba9,0xd26859b6),LL(0x6369309b,0x7b3e5c46),LL(0x93c9a0b9,0x474a9625)}, {LL(0xc76e25dc,0x676ea3e4),LL(0x71400c71,0xb0c9ccf7),LL(0x36f83518,0xc4295870),LL(0x267f0c25,0xb86c1b39),LL(0x28884a80,0x8a3e3524),LL(0x43ba8e28,0xea182c45)}},
   {{LL(0x30c70c8d,0x6d351640),LL(0xce2361b8,0x8047d811),LL(0xdf8e2c81,0x3f8b3d4f),LL(0x33fa1f6c,0x5d595477),LL(0xe29b8a91,0xf769fe5a),LL(0xd737b2a2,0x26f0e606)}, {LL(0xb8b31c6a,0x70cbfa5d),LL(0x863d3aea,0x0f883b4a),LL(0xe386ae2f,0x156a4479),LL(0xade8a684,0xa17a2fcd),LL(0xe2a7e335,0x78bdf958),LL(0x3b9e3041,0xd1b4e673)}},
   {{LL(0xaf2825c1,0xa6c32900),LL(0xd223a04e,0xb37c46c1),LL(0x063de7ea,0x691e7d39),LL(0x10daf9bd,0x998df4e7),LL(0x718b5d7a,0xc7085b9e),LL(0x16b3d4b8,0xd41abcc8)}, {LL(0xf9bc4041,0x4dfce693),LL(0x659ec7a8,0x383677ed),LL(0x4491fb34,0x2c1904bf),LL(0x4552451c,0x7c1bf111),LL(0x3c5e5e40,0x6562cc2c),LL(0xfe0e4372,0x1ecaa2a1)}},
},
/* digit=41 base_pwr=2^205 */
{
   {{LL(0xf3087bc8,0x84c3630b),LL(0x74be6e26,0x152691e6),LL(0xf61af001,0x5abd125f),LL(0xbfea3525,0x69bca56f),LL(0x00e0cb6e,0x384af199),LL(0xd00475a6,0xb0b13cfe)}, {LL(0x5e394049,0xedafde49),LL(0xdaf2add6,0xd988b558),LL(0x6c8ffcc9,0xf14cf97a),LL(0xe5a9cc5c,0x4d6cec23),LL(0x8a104e05,0xb0d678f8),LL(0x9fb527c1,0x80a7fcba)}},
   {{LL(0xc22137d2,0x0d6cadbf),LL(0x628a3298,0xb5db59d3),LL(0x4ab19507,0x3b433c73),LL(0x660086b3,0x4fc53405),LL(0xa1eb0f43,0x770ae903),LL(0xf6b5b58d,0x31b5857e)}, {LL(0xd392868e,0xe206e141),LL(0x4b31de04,0x8be6956f),LL(0x47449e07,0xcfbfca2f),LL(0x39fef8e1,0xebaef256),LL(0xc16fc80c,0x959e37b8),LL(0xe911d61b,0x8bb4bdd2)}},
   {{LL(0x4343bbf8,0xc483d4b2),LL(0x8a0fc95e,0x42aca2e1),LL(0xcbab1fb0,0x5165df6f),LL(0xf6cdfc0c,0xeb284370),LL(0x994320fc,0xab565c00),LL(0x62133e80,0xc0d157fd)}, {LL(0x5b69644a,0x7850cda5),LL(0x806ec8b8,0xe37ae76a),LL(0xc2c82edd,0xd14b805c),LL(0xcf244539,0xcb5468b6),LL(0x25dbe92d,0x97d43ee8),LL(0x89fb8f1e,0x14422436)}},
   {{LL(0xa0a85236,0xd7bf2ac6),LL(0x7194c46d,0x2921b55c),LL(0x9afa9762,0x162fabaa),LL(0xb62b36ab,0x7b7f1664),LL(0x296a84e9,0x77b9f797),LL(0x7dbd843d,0xfcc1ad65)}, {LL(0xcd77b7f6,0xc6e9c1e1),LL(0x917067c9,0x9cf0e272),LL(0x3bfa90bf,0xfa7fa93d),LL(0xd050e46a,0x55846fe9),LL(0x35c56256,0x473b9a0d),LL(0x2b656a65,0xadd29e33)}},
   {{LL(0xebc69b0b,0x926c2552),LL(0xd4c7432e,0x953a850f),LL(0xb9359035,0x0ee85e14),LL(0xbde090a5,0x8b10b01a),LL(0xec423943,0xb2878dca),LL(0xf70bde20,0x2571a178)}, {LL(0xf5ebeee5,0x24ed159a),LL(0x043f6539,0x60c202af),LL(0xc8d4ffc3,0xdaaa76f4),LL(0x06eda10f,0x2fc1f1ba),LL(0x88ded556,0xddf159ee),LL(0xe67b1ec4,0xcfa71782)}},
   {{LL(0xd5d826b0,0x31521f66),LL(0x40787844,0x0a636952),LL(0x9c8f934e,0xc0a3bd05),LL(0x2f0ce835,0x12c57dd4),LL(0x67064213,0x847f6a99),LL(0xa88bd71a,0x1c9e1a7a)}, {LL(0x171e8407,0xc4060eb2),LL(0xed106780,0xdf78d8df),LL(0x0d704729,0xa3d28ceb),LL(0x46ca3912,0x4f8e5232),LL(0x017791f4,0x09e9f852),LL(0x1e6ea97d,0x59400663)}},
   {{LL(0x83939224,0x547b0d95),LL(0x1e026769,0x3a0823ff),LL(0x25bd43ac,0x60166715),LL(0x18ba5f64,0xb6cf475e),LL(0xc8b6d09d,0xa22f9c92),LL(0x3ccf50ab,0x73055368)}, {LL(0xee6deefe,0xa6de248e),LL(0xacc3ca20,0x32aaf8b2),LL(0xad44e674,0x0e254c5b),LL(0x35f95f98,0x8aa73e65),LL(0x60a2dc1e,0xe6226001),LL(0x9109020a,0xdf948210)}},
   {{LL(0x2893f2f7,0x372798f0),LL(0x9e5030ca,0x4f62bfac),LL(0x8a1e2567,0x5e64f9a9),LL(0xe70391c8,0x5870254c),LL(0x41f02458,0x2def81a3),LL(0x1d087bed,0x25d4e4dc)}, {LL(0x4fe24a13,0x3557d07d),LL(0xdc3112bc,0x6da49186),LL(0x5f73ba50,0x08c8c567),LL(0x9c7c6706,0x5309050b),LL(0xbd985072,0x2ab67da3),LL(0xe5df4e96,0x9bafa8b1)}},
   {{LL(0x5c02f173,0x3ac66289),LL(0x76d566e5,0x6a110e38),LL(0xb9577e26,0xd9cc14e2),LL(0xfdfe617e,0x6f3d5df9),LL(0x352bb2ca,0x8fac740f),LL(0xc28e6310,0x50bc8a0c)}, {LL(0x77ac93f7,0x6e572fc4),LL(0x605bb8e9,0x56277377),LL(0x402b8c55,0xad6d0637),LL(0x4509eda7,0xdab37791),LL(0x0854e91b,0xae770abc),LL(0x742b3de8,0x523bd278)}},
   {{LL(0x17fecb90,0xc2cbd644),LL(0xb32dffdb,0x61616eb3),LL(0x9f5d2095,0xdc4485a2),LL(0x6553371b,0xf7891124),LL(0xbf9b20af,0x4f06ba18),LL(0x1a2c4df1,0x136d4f29)}, {LL(0xfb8b685f,0xc04aca34),LL(0xf2b657bb,0xeec83c20),LL(0x5925a36a,0x4da5d70a),LL(0x72ff2965,0x80608741),LL(0x9f352620,0x2e0dd9ff),LL(0x46d1a7a8,0x5f0afa67)}},
   {{LL(0xf46aace5,0xa995a95b),LL(0x9eaa630a,0x44ede537),LL(0x00336e3b,0x421f3b35),LL(0xcf47c9ed,0xbf897478),LL(0x259e0827,0xf360ae32),LL(0x2e6a9f6b,0x04e0e3e8)}, {LL(0xa9136702,0xb26eae5f),LL(0x853674b4,0xd6cb15a1),LL(0x748bcbc9,0xf81276e2),LL(0x0a4ca1d7,0x7fc02e22),LL(0xcd82f330,0xf650f48e),LL(0xabaa8859,0xf4ea7c1d)}},
   {{LL(0x95c746ec,0x1ccd44ff),LL(0x10405763,0xe18914b5),LL(0x21a3a927,0x50ed6443),LL(0x43ef8e8f,0x4f96a1b1),LL(0x77952bf8,0x7f5645e5),LL(0x66dbdf15,0x4bc5c7ab)}, {LL(0x23930a08,0xacc16126),LL(0x504cf9b6,0xbf5ed482),LL(0xd71ecbd7,0xdeb7a798),LL(0x4a4dd859,0xf62e63b1),LL(0xdaf714d9,0x668809a7),LL(0xf3a4329e,0xdd836382)}},
   {{LL(0xc48f3ad5,0x894bdbd9),LL(0x09e167f6,0x687ff8de),LL(0x30371c43,0xf06104a9),LL(0xce84dd10,0x82fd34b7),LL(0x66ce5abd,0xae122deb),LL(0xfc4a90b2,0x31f041d2)}, {LL(0x9a01c607,0x2589535c),LL(0x695bd7ab,0x231bcc85),LL(0x62e3a31d,0xc67c3062),LL(0x7af3e186,0x31be4475),LL(0x88efa7f1,0x1a2077a3),LL(0x815fad1a,0xffe53e22)}},
   {{LL(0x66229776,0xdd155913),LL(0xf7882064,0x84093730),LL(0xe50ee337,0x6dddcb14),LL(0x7a1f7e81,0xa8e6ec59),LL(0xf3738a6a,0x8467f998),LL(0xad3f1840,0x70fcc6bc)}, {LL(0x723b3f4b,0xf82eb4be),LL(0x06beec1b,0xf0f39354),LL(0x7ddcb539,0x1b181ea3),LL(0xad6a81b9,0x9c82c4fa),LL(0x5c612c2b,0xcc5ea543),LL(0xbb258d6f,0x63ce7571)}},
   {{LL(0x0b96547f,0x1b588855),LL(0x65d1a59d,0x4539c9c0),LL(0x26e15084,0xd6c95fea),LL(0x86b96242,0xf84ad9e2),LL(0x451a5486,0x92f57d6d),LL(0x06a9e87e,0x0215cfcb)}, {LL(0xf66e46f3,0xe05b10ea),LL(0x655a0642,0xe7b0e72f),LL(0x7b117f43,0x03503267),LL(0x779ea4a1,0xf5b78105),LL(0xa4adac77,0x28ee00fa),LL(0x6a93a2b1,0x1ea67d71)}},
   {{LL(0x26882c6c,0x7be81fb1),LL(0xecd25498,0xe2d5a251),LL(0x7a8d1678,0xbb3d40e2),LL(0xd520811e,0x1806c67a),LL(0x86f65d23,0xadd4bb66),LL(0xe20e23d7,0x3a62b1b3)}, {LL(0x6548b3eb,0x208b4700),LL(0xb7ec2809,0x0497f09a),LL(0x121c37e2,0xbd3964f8),LL(0xa598efbb,0xd35ef301),LL(0xc5eef966,0xbd76a276),LL(0x0af64e46,0x64700a7f)}},
},
/* digit=42 base_pwr=2^210 */
{
   {{LL(0xd3812087,0x169474a2),LL(0x6698ca7a,0x9de300da),LL(0x2ede425b,0x8589de92),LL(0x6df8a890,0x50e03fea),LL(0x4ba8b8e3,0x0d8a5c1c),LL(0x3fffb91a,0xf273aa67)}, {LL(0x75fc8236,0x21cf0544),LL(0x9799c242,0x6ceafacf),LL(0xd0962c81,0xc3237eae),LL(0x213f6004,0x43d6ac34),LL(0xd4148b6b,0x45e619b2),LL(0xea5fb80a,0xfafa18b5)}},
   {{LL(0x2f063b51,0x9a8580aa),LL(0x1c216613,0xa83c8ff7),LL(0xbe07f781,0xb4da0970),LL(0x712f7b7c,0x0ac2a260),LL(0x436a7b97,0xc9b8ee84),LL(0x11fb2f62,0xd758c20d)}, {LL(0xf170b799,0x5daabed9),LL(0xc46bc387,0x018d2fdd),LL(0xd96cfb8e,0x82d6b5b7),LL(0x44d9e843,0x4d7d0d93),LL(0x91e7da3c,0xfa2a9ea9),LL(0xd531b253,0x8230c1a3)}},
   {{LL(0x5ec31754,0x82412f52),LL(0x9d32e890,0x42f462c8),LL(0xce897ff2,0x1e7b58ce),LL(0x41164628,0xcfef7852),LL(0xd8bb22ef,0x34ee0422),LL(0x7d32f01b,0x6e0d44ac)}, {LL(0x5a3cc196,0x96825165),LL(0x99eb23d3,0xa26724dc),LL(0xa75f7252,0xdb575faf),LL(0x62a3e5b1,0x778e3330),LL(0x84cccc80,0x8689884e),LL(0xb645502d,0x9883cd19)}},
   {{LL(0x34220e26,0x4cc41f28),LL(0xa49749c4,0xb5937c6d),LL(0x0fa1ca24,0x70536664),LL(0x91e5edaf,0xeeb40f3b),LL(0xf1d3de14,0xcdf98235),LL(0xff018c43,0xa65e5b7e)}, {LL(0xacee3a6e,0xaa3228e7),LL(0xe08f4ff1,0xb63a6289),LL(0x650b2daa,0x90e90425),LL(0x6d875f17,0xe4a8cad2),LL(0x9ce8a46e,0xc212029c),LL(0x5ed7cfb5,0xce051283)}},
   {{LL(0x59b79436,0xb0df2261),LL(0xa195be26,0x82bd0daf),LL(0x3398c317,0xbc99a94b),LL(0x3c96ee31,0xbeb44c90),LL(0x664d2e46,0x3c39ad81),LL(0x0a3e0585,0x08178752)}, {LL(0x413e269a,0x9a054b6b),LL(0x98c3b62e,0xbe58891d),LL(0xe5734974,0xe7fa4c4d),LL(0xd0a846a8,0x8ac535f4),LL(0xa651339c,0xea0f95f1),LL(0xd96aa239,0xa255274f)}},
   {{LL(0x9534047e,0xe23b7b22),LL(0x3a3bd625,0xbd70aea8),LL(0x238db60b,0xf44b05fe),LL(0x0293abcf,0x9c46fb14),LL(0xbfd8875b,0x12cab5d3),LL(0x12dd0c65,0x1f38d4aa)}, {LL(0x2adf9805,0x4bed4157),LL(0x8a56609d,0x3f87da92),LL(0xda02c903,0x10b93363),LL(0x21ce4786,0x7ecc7266),LL(0x1e3da5bf,0x8ae36685),LL(0xd3edee12,0x196040ff)}},
   {{LL(0xe81508e8,0x4805841f),LL(0xa4808642,0xe2a578d3),LL(0xcd0b2555,0x6bbf10ac),LL(0xaf5cde28,0xc5071eff),LL(0x9a7124a9,0x665e7543),LL(0xc1437981,0x157c11ed)}, {LL(0x7aeddd8b,0x2019367d),LL(0x386e3b8a,0x74a1e104),LL(0xfbe09a42,0xe72d429b),LL(0x061b862e,0xaca96fd9),LL(0x122595f8,0xbb2d2bc8),LL(0xc509d644,0xc90c6503)}},
   {{LL(0xcff05ada,0xadb5966f),LL(0x5c57284e,0x8ed26c02),LL(0x44693a95,0xa76e73e2),LL(0x5982bbd3,0x14da7435),LL(0x5d2ca132,0x46e982cd),LL(0x24938e76,0x8f390740)}, {LL(0x0a89b09a,0x749206b3),LL(0x93b4a1e5,0x429653c7),LL(0x7025bb7c,0xbee3d156),LL(0x19555c9e,0xe23f0e1e),LL(0x751639ba,0x0dec3837),LL(0x05d43bd0,0xb36cb844)}},
   {{LL(0x74f90b6e,0xae76a96e),LL(0x24c6789c,0x5fa8e948),LL(0x03abbb81,0x2b3584bb),LL(0x5c451f72,0xe19ce47c),LL(0xd619ac7a,0x35792fba),LL(0x50059bf4,0xfa0282a2)}, {LL(0xdabe692f,0x562bfd14),LL(0x47eeb6c2,0x1aaf542c),LL(0x045d0360,0x392d5bba),LL(0xd80fe998,0x4e7bb31b),LL(0x1111e14d,0x08f62ef3),LL(0x4e9ee1b8,0x4de917b0)}},
   {{LL(0x67166271,0x8b9d2d58),LL(0x142bab7c,0x658db4ea),LL(0xa4ad2849,0xdf84932f),LL(0x5f6f86a7,0x04b11335),LL(0x50cfcea7,0x2de6b29c),LL(0x9be6a3a3,0x46d8f68a)}, {LL(0xaf0204af,0xfb88cda7),LL(0x26029d72,0x3ece4491),LL(0x3f946dfd,0x69fef1e2),LL(0x01ef7bb5,0x708532fb),LL(0xeb3795a2,0x78d5053d),LL(0x6b36d57b,0x819a6320)}},
   {{LL(0xe509d19e,0xca07e0c1),LL(0x9f6281b1,0x6c7e42c3),LL(0x77b66728,0x0e2ff439),LL(0x80e76251,0x1d740e78),LL(0x31a0eb23,0x6bfae4c6),LL(0xaa9b0b3b,0xd78ca917)}, {LL(0x991e1781,0xe140c662),LL(0x0dd3cfee,0x6e396b5f),LL(0x6ce7f6c7,0xf0a1d197),LL(0xd5b01564,0xbe10f8ef),LL(0x101a5194,0x865cbd54),LL(0x66861ded,0xf6658852)}},
   {{LL(0x5b28f7da,0xe4e52e86),LL(0x9a58683f,0xeb43a680),LL(0xb49f2b38,0x73b951bf),LL(0x3f8097cf,0x7b6cb7db),LL(0x328fbf05,0x9dfb8d0b),LL(0xebce6211,0x491635a5)}, {LL(0x90fdd577,0xa31a1523),LL(0x1cd2f39c,0x334120df),LL(0x6b563876,0x1d22834e),LL(0x10ee5a28,0xfd91b30d),LL(0x59aee4ea,0x3d7a282d),LL(0x73300a76,0x36814c6b)}},
   {{LL(0x6621c251,0x7b584add),LL(0x4233aba3,0x98da669d),LL(0x33aa2065,0x4d652b79),LL(0xdf7b4ed4,0x901bcfb8),LL(0x48012f81,0xb2ce5879),LL(0x3cb71b88,0xc18e2cd6)}, {LL(0xff86279d,0xadb0f2bd),LL(0x5bd15866,0x46d9e5d6),LL(0xc635a4c0,0x11b1fb3e),LL(0x01b1006a,0x8bcd0ad2),LL(0xcbab210b,0x0f6f7502),LL(0x0d6b3995,0xd6cc3e56)}},
   {{LL(0x137264c5,0xa54a6420),LL(0xf9c2e45e,0xa6ef0e78),LL(0xd58d850c,0xba8b5a73),LL(0x6ef6fc3e,0xc0209ed8),LL(0x91f7518a,0xe39dd0f3),LL(0x42b3eda6,0x74697b89)}, {LL(0xabfc9150,0x2dccac36),LL(0x98b2f5a5,0x80e4fba2),LL(0x771018d5,0xe0e56fd2),LL(0x4c22bb94,0xa31fd168),LL(0x1a66ef21,0x8b0998f7),LL(0xb5a53ddb,0xed483e55)}},
   {{LL(0xf23978eb,0x95db1c0e),LL(0xf04011f4,0x80ad1612),LL(0x4d7ae83d,0xe76bd182),LL(0x8fc3bd60,0x841d6e66),LL(0x6875e2d0,0xb68e8079),LL(0xd5d9dee7,0xe3965efc)}, {LL(0x58930931,0xc488bb7e),LL(0xa907aa24,0x52f4de19),LL(0x321cc197,0x39aebbdd),LL(0x67de5c66,0xd2f5b1f9),LL(0x8efe3e76,0x60f1a8c2),LL(0xaf988831,0xf40604a0)}},
   {{LL(0x0acb5935,0x78b5c14c),LL(0x4311d3be,0xd9ec715c),LL(0x09e1759e,0xffa22ab2),LL(0xb4b2f68c,0x5a86263d),LL(0x6b5be7f4,0x71e77c51),LL(0x19844f6d,0xfb5bea3a)}, {LL(0x0890ffab,0x2519d006),LL(0xf0329ef0,0x426a03f0),LL(0x85b3c2a9,0x2c6d74a6),LL(0xc294f449,0x9306f68f),LL(0x2c69fb46,0x552e77c2),LL(0x10bb9886,0x7c7337ad)}},
},
/* digit=43 base_pwr=2^215 */
{
   {{LL(0x44b133d9,0x3313a9d5),LL(0x2da910dd,0xdb85c25d),LL(0x5e4dd5cd,0xc0fdef91),LL(0xc565dd67,0x902a2a93),LL(0x7fed05ac,0xd8eba4dc),LL(0xe157dae9,0xd453995c)}, {LL(0xf250cb55,0xd655d0b3),LL(0x86119222,0x4194a09e),LL(0x0652872b,0x5b7e525a),LL(0xe68c0ddb,0xaf7968ef),LL(0xf51cb31c,0x2ec02930),LL(0xf2be071e,0x237f3ae4)}},
   {{LL(0x94d0864f,0x9b23ab4e),LL(0x009c9fc1,0x46356266),LL(0xe798edf9,0xdbe99e51),LL(0x307675c7,0x38547449),LL(0x628c0fb6,0x23ffaf55),LL(0x1698c372,0x56ccd2a3)}, {LL(0x8347ce95,0x39f45a57),LL(0x4f2c6118,0xe0aaec74),LL(0x4af138fc,0x2a89079e),LL(0x2ee4ecc0,0xb86371ea),LL(0x06bbf92f,0x076d256a),LL(0xae3c4c51,0x9073adb8)}},
   {{LL(0x4c99252c,0xdaa77b43),LL(0x326cb0e1,0x59e38731),LL(0x03ca6c85,0x281a38cc),LL(0x433835a0,0x83565666),LL(0x30a928fd,0x3654ec9e),LL(0x7cb281f8,0x1c82abca)}, {LL(0x13fafa6f,0xbeba0fe5),LL(0x99440e63,0x67432292),LL(0x0034d0cf,0xd62777af),LL(0x9cde52a0,0xd42b95fe),LL(0x6a23630f,0xb5b891bc),LL(0x64594976,0x8e4d2984)}},
   {{LL(0x743c15d9,0xba2e9543),LL(0x1c99c984,0x7d5812db),LL(0x45bdc19e,0xf94db951),LL(0x382e77bd,0x951d00ae),LL(0xb220b29a,0x9940a5fb),LL(0x58fc91f1,0x6908d50e)}, {LL(0xdd0940fe,0x682e42ea),LL(0xa1d32009,0x2124e23a),LL(0x16294d05,0xbe158100),LL(0x2e326d68,0xaea13fe3),LL(0x15e64fce,0xc0dfe1ef),LL(0xb8237a8a,0x32dbc0b5)}},
   {{LL(0x663771f8,0xea6f1448),LL(0x9a0906cb,0xbf11f126),LL(0xd1a6a6a5,0x8c08219d),LL(0xdd56f277,0xf2af6e04),LL(0xd569188a,0x8ad26705),LL(0x6e071c9b,0xf0547631)}, {LL(0xfd4aa6ec,0x945f40bd),LL(0x8486987c,0xbd6a8a8d),LL(0xb947b6f2,0x4f577728),LL(0x7aab6bc1,0xe2754a8e),LL(0xfb48a9ab,0x918d02cd),LL(0xbf904200,0x7e3ddaea)}},
   {{LL(0x0bc28725,0xe59b8002),LL(0x149f8991,0xc6f8fa54),LL(0x5af5b47e,0x8799172a),LL(0x160d7e8d,0xf72c9780),LL(0xb2f9a549,0x1d1ce972),LL(0x4857b44a,0x8fce3f16)}, {LL(0x1ed5e0e5,0x2545bdd7),LL(0xc259176a,0x222c33a7),LL(0x4e23c064,0x5a60343a),LL(0x1d1fd9cd,0x986779f9),LL(0x3570b5b3,0x5bd5611b),LL(0xf9d765a3,0x2758caea)}},
   {{LL(0x3a16e352,0xd8135f00),LL(0x55d4e996,0xbfc784b2),LL(0x0a6874d0,0x5da46321),LL(0xe8e1e4d0,0xf1635286),LL(0xa0e9c4de,0xe71332b3),LL(0x60fc995e,0x5f076117)}, {LL(0xc19ebb59,0x1c1305d1),LL(0x4603baa2,0x1d5987f2),LL(0xea7c9f9c,0x6b7885b0),LL(0xfe9ba1fc,0x362734e8),LL(0xb3892110,0x49a3ff32),LL(0xd4997512,0x0e5b2166)}},
   {{LL(0xc36d3f25,0x6ee65a08),LL(0xe393e4d4,0x7b6c811f),LL(0x2876e523,0xc4a2cc38),LL(0xd3bf53aa,0xab7aba26),LL(0xdb7f290c,0x5bf00871),LL(0x1ee6d5bd,0x3cb1cd13)}, {LL(0xde998ada,0x4cafb218),LL(0xf6319101,0xa1ecf36a),LL(0x20b281cc,0xa1fe7855),LL(0x64d9c65e,0xe457198e),LL(0xc5a0e67b,0xa3d1a6d0),LL(0x90cc468a,0x69ddbc32)}},
   {{LL(0x53a38094,0x3f3cb2a5),LL(0x3b19448a,0xc9308152),LL(0x925ba579,0x9cf32819),LL(0x1801e686,0x44b9590a),LL(0xdf04be40,0xab6b284c),LL(0x2c216ae7,0xc40a58e6)}, {LL(0x018a60e8,0xa6ced619),LL(0x83e5cc65,0x83d4cde1),LL(0xb5a91945,0xe2559f51),LL(0xef0f53ee,0x53d9122b),LL(0x32e40a83,0x474c281b),LL(0x944dfe65,0x332324a7)}},
   {{LL(0xa1fd057d,0xcb0a3657),LL(0xb4aa013b,0xad79ae60),LL(0x97ed887a,0x0b852109),LL(0x30d9b297,0x5fec3e1f),LL(0x10fb9c74,0x4ce4149f),LL(0x8cbff785,0x4e08b4d9)}, {LL(0x2f07cbae,0x81b0d7b4),LL(0x495230ae,0x08bc321e),LL(0x2b841eb8,0xaec221de),LL(0xd83c22e6,0x62c7d86a),LL(0x85affe91,0x5504dcf8),LL(0xf445481d,0x785a06f1)}},
   {{LL(0x18670cf4,0x8b68acc4),LL(0xcba49dca,0x64073a95),LL(0x9f508534,0x633757b0),LL(0x916f3a09,0x78ad9df7),LL(0x08617468,0x46187c92),LL(0xc5f77b94,0x48f37eb6)}, {LL(0xd058bb27,0x333224f0),LL(0xd8852abb,0x7f28ace7),LL(0xa2e62327,0x8c9f634f),LL(0xc4116c1b,0x75212283),LL(0xcc0c0851,0xb7723ad0),LL(0x5b72e5aa,0xc8a4cff1)}},
   {{LL(0x6dadc46f,0xd4ee3f7f),LL(0x5d7febd6,0xa1f3dc92),LL(0x63ebab5b,0x4c0bee13),LL(0x005ec237,0x70e32d77),LL(0xc52fb006,0x302fc73d),LL(0x8f159899,0x1af84c0a)}, {LL(0x0686232a,0x42a5478f),LL(0x8a308687,0xb4fc5634),LL(0xc8378f0d,0x042c4970),LL(0x8e2c86c5,0x70c19575),LL(0x84c7c767,0x61a95e68),LL(0xd6fb43a8,0xd96a8216)}},
   {{LL(0x1a2db746,0x67f8fa3e),LL(0xba267cad,0x2f041d4a),LL(0x65bafbee,0xdf8126ea),LL(0xfad7b234,0xf63fc68f),LL(0x702db8ab,0x4280e1e1),LL(0x02468fc2,0xbe6122f2)}, {LL(0xc89b74de,0x06b076d0),LL(0x4bbdb7cc,0xc2515543),LL(0x50eb6108,0xae82a08f),LL(0x2f0f3f5e,0xf49da234),LL(0xeca6448e,0x7d1923b2),LL(0x70b144ab,0x9e01d58c)}},
   {{LL(0xee49a1bf,0xeaf237a8),LL(0x4ea1da86,0x78bf04bf),LL(0x3f251ad5,0x5fbffa47),LL(0xd828578e,0xc40570f6),LL(0xd4e118ad,0x5cc65c0d),LL(0x5da48548,0x9e18ff96)}, {LL(0xef7e714b,0xe27fc2e7),LL(0x13df7524,0x19ff3f7c),LL(0x5e27fb12,0x35a32fa5),LL(0x10003fae,0x7fcfd728),LL(0xf49800c1,0xc74b50a3),LL(0xdbafb2e6,0xbf0732eb)}},
   {{LL(0xc1238c95,0x1d1b02ab),LL(0xee8c4d2b,0x85b3878a),LL(0xc761afd0,0x23ba366c),LL(0x4f023bb7,0x47324d03),LL(0x388c8e5b,0x75902ccc),LL(0x86e2e6b6,0x374484b9)}, {LL(0x9f548719,0x38360f84),LL(0x41cbc7ef,0xdf999916),LL(0x5091ed27,0xa9cbe298),LL(0xc5f2cb5d,0xcf5a1440),LL(0xd413500d,0x5bdff729),LL(0x373f8b2d,0x55530d56)}},
   {{LL(0x543c1255,0x0c62fd2d),LL(0xef361a27,0x71ea9c6f),LL(0xcef3f9e3,0x76b0933d),LL(0x9889ffa2,0x51b1ec2d),LL(0x9a3c88d2,0x9e84b2ba),LL(0x1913e52f,0xc8996b96)}, {LL(0xcee43e36,0xbafc5e94),LL(0x70c658b7,0xd9898d24),LL(0xbed17108,0x4e9bcc41),LL(0x6c7a41c8,0x0db5b733),LL(0x795369cd,0xd4be07a7),LL(0x7bd3a934,0xb899f92f)}},
},
/* digit=44 base_pwr=2^220 */
{
   {{LL(0xde1e4e55,0x3f2eff53),LL(0xe4d3ecc4,0x6b749943),LL(0x0dde190d,0xaf10b18a),LL(0xa26b0409,0xf491b98d),LL(0xa2b1d944,0x66080782),LL(0x97e8c541,0x59277dc6)}, {LL(0x006f18aa,0xfdbfc5f6),LL(0xfadd8be1,0x435d165b),LL(0x57645ef4,0x8e5d2638),LL(0xa0258363,0x31bcfda6),LL(0xd35d2503,0xf5330ab8),LL(0xc7cab285,0xb71369f0)}},
   {{LL(0x14c5969b,0xf16938f5),LL(0x944b2271,0xde2e3cf0),LL(0x0b6490d6,0x2d509553),LL(0xa28a296a,0x8432fef1),LL(0x8d26415c,0x6f254dd0),LL(0xd50c2865,0x3780eead)}, {LL(0x665b8794,0x4f5bc455),LL(0x56cb7018,0xef31fb9e),LL(0x65e59340,0xbab8dd6e),LL(0xa56dc2ea,0x676baca2),LL(0xeaa90e05,0x38eea06b),LL(0x174bada0,0x26e64224)}},
   {{LL(0x40acc5a8,0xe6a19dcc),LL(0xdbc6dbf8,0x1c3a1ff1),LL(0xc6455613,0xb4d89b9f),LL(0xa7390d0e,0x6cb0fe44),LL(0x59ea135a,0xade197a4),LL(0x20680982,0xda6aa865)}, {LL(0x5a442c1b,0x03db9be9),LL(0x2bfb93f2,0x221a2d73),LL(0x753c196c,0x44dee8d4),LL(0x0b7c6ff5,0x59adcc70),LL(0x4ca1b142,0xc6260ec2),LL(0x46cbd4f2,0x4c3cb5c6)}},
   {{LL(0x2cb40964,0xa35e411c),LL(0xc331a3d6,0xdd7d4f4c),LL(0x89a66f2b,0x7c7c859e),LL(0x0def8ecd,0x9908c37e),LL(0x344947b7,0x8274124e),LL(0x568b0ce8,0x0d279f7b)}, {LL(0x866091ec,0xe5291961),LL(0x3a08acc7,0xb056e3bf),LL(0x56bd3a7d,0x60fb39e1),LL(0x268f8562,0xe56a34d6),LL(0x13fd8293,0xb3a1fe16),LL(0x67537fcb,0x6a41e1a9)}},
   {{LL(0xa417111f,0x8a15d6fe),LL(0x71d93fcc,0xfe4a16bd),LL(0x55bbe732,0x7a7ee38c),LL(0x1ff94a9d,0xeff146a5),LL(0xdd585ab5,0xe572d13e),LL(0x06491a5d,0xd879790e)}, {LL(0x2a58cb2e,0x9c84e1c5),LL(0x6c938630,0xd79d1374),LL(0x385f06c7,0xdb12cd9b),LL(0x7a7759c3,0x0c93eb97),LL(0x683bd706,0xf1f5b0fe),LL(0x85ec3d50,0x541e4f72)}},
   {{LL(0xc6bb5e5f,0x89e48d8b),LL(0x1ea95a10,0x0880ede0),LL(0x302c0daa,0x60f033d7),LL(0x048eefe3,0x15e4578a),LL(0xb0a72244,0xfd6dec89),LL(0x309489cd,0x1f7cd75e)}, {LL(0xe9aba7fd,0x7cdcc2a0),LL(0xf28ba00f,0xd18dc5c7),LL(0x5812b55f,0xa6300a45),LL(0x2ca31d8c,0x8fa5c415),LL(0x4f3a5b5a,0x36aa3c23),LL(0xc86cf4e0,0xd128739e)}},
   {{LL(0x81833608,0x9a0e1535),LL(0x6e2833ac,0x5cce871e),LL(0xfb29777c,0xc17059ea),LL(0xe354cafd,0x7e40e5fa),LL(0x4d07c371,0x9cf59405),LL(0xa71c3945,0x64ce36b2)}, {LL(0x56caf487,0x69309e96),LL(0x1ae3454b,0x3d719e9f),LL(0xe25823b6,0xf2164070),LL(0x0bc27359,0xead851bd),LL(0xb0925094,0x3d21bfe8),LL(0x34a97f4e,0xa783b1e9)}},
   {{LL(0x13575004,0xd9272830),LL(0x20b2275e,0x01a330d6),LL(0x450db713,0x58b9207f),LL(0x23e16d95,0xae953384),LL(0xe60e349c,0x4f10c6d4),LL(0xfeb122bc,0x541d03ec)}, {LL(0x2c648211,0x22548cd2),LL(0xd01354f5,0x5c2dc84c),LL(0xb6167b3c,0xa1c6f912),LL(0x7902d2ba,0x6967bab2),LL(0x36de34ba,0xebbe0b08),LL(0x4b79625e,0x6985b33a)}},
   {{LL(0x9546491a,0x406b0c26),LL(0xf293c4e5,0x9e5e15e2),LL(0x15b164db,0xc60d6413),LL(0x0c75a78e,0x0da46f53),LL(0xea0c656b,0x7c599bb7),LL(0x1b1a8122,0x0f07a512)}, {LL(0x15172686,0x14c7204a),LL(0x5165625d,0x8faedff8),LL(0x37aede40,0x20f260ce),LL(0x8f357ffe,0xc81f771e),LL(0xb0912557,0x25499197),LL(0x4c739c74,0x736197dc)}},
   {{LL(0xaefdd5e1,0xf2d015bd),LL(0x4bf9edae,0x33b4e21c),LL(0x64b35fcc,0x4860aadc),LL(0xf13f8112,0x77b657e9),LL(0x90dc84fe,0x65f28625),LL(0xd66be036,0x4eabfd22)}, {LL(0x6ff05dd0,0xb0213123),LL(0xa9fe5d11,0xa104d4aa),LL(0x9b8ae390,0xdc7efe3b),LL(0xeb87fb5c,0x46918b54),LL(0x72b7172a,0xf5771d71),LL(0x13587bf0,0x41e49e47)}},
   {{LL(0x381b3462,0x6151bab1),LL(0x43dbd344,0x27e5a078),LL(0xa1c3e9fb,0x2cb05bd6),LL(0x27cf2a11,0x2a759760),LL(0xff43e702,0x0adcf9db),LL(0x1f484146,0x4bbf03e2)}, {LL(0x55b6521a,0x0e74997f),LL(0xade17086,0x15629231),LL(0x7493fc58,0x7f143e86),LL(0xaf8b9670,0x60869095),LL(0x7e524869,0x482cfcd7),LL(0x1d454756,0x9e8060c3)}},
   {{LL(0x9004b845,0x3ec55ddc),LL(0xbe7b06eb,0x5101127c),LL(0xfc4176e9,0xdddafd57),LL(0xe4a31ddc,0xd8cb31c0),LL(0xd42feabb,0x94e83a89),LL(0x2f74ec68,0xd4401def)}, {LL(0x5adb654d,0x9c9defb6),LL(0x5053eeda,0x9a3513f0),LL(0x7cdb455e,0xceab2dec),LL(0x59d24f59,0x99542808),LL(0x31d30504,0x22ead452),LL(0x0521a229,0xe9df48f9)}},
   {{LL(0xc88b4d3b,0xe495747a),LL(0xae8a948f,0xb7559835),LL(0xdeb56853,0x67eef3a9),LL(0x9dee5adf,0x0e20e269),LL(0x61f0a1aa,0x9031af67),LL(0x683402bc,0x76669d32)}, {LL(0x06718b16,0x90bd2313),LL(0x864efdac,0xe1b22a21),LL(0x6620089f,0xe4ffe909),LL(0x3428e2d9,0xb84c842e),LL(0xfe3871fc,0x0e28c880),LL(0x3f21c200,0x8932f698)}},
   {{LL(0x441d0806,0x36dd28ea),LL(0x21518207,0x6680c72c),LL(0x0a484dbc,0xc5d40e28),LL(0xa3a2ba6e,0xdb1170c6),LL(0x40a91c7d,0x07290fd1),LL(0x95ee9ca2,0xdd125716)}, {LL(0x07876188,0x595dad63),LL(0x499d4827,0x6fcf18c0),LL(0x206e617d,0xdcd946a3),LL(0xe7bceaef,0x6cf08f51),LL(0xb19a06ac,0x7a85c02d),LL(0x7140a7df,0xf1365fc6)}},
   {{LL(0x6c90ea5d,0x603f00ce),LL(0x40a2f693,0x64739307),LL(0x2174e517,0xaf65148b),LL(0xf784ae74,0x162fc2ca),LL(0x4d5f6458,0x0d9a8825),LL(0x43aace93,0x0c2d5861)}, {LL(0x9f73cbfc,0xbf1eadde),LL(0x9c68bbca,0xde9c34c0),LL(0x67ef8a1a,0x6d95602d),LL(0xa791b241,0x0af2581b),LL(0x12cad604,0x14f77361),LL(0xe2acd1ad,0x19f2354d)}},
   {{LL(0x771560ab,0x9e857f33),LL(0x250f109d,0x4ae1ba22),LL(0xff4f6566,0xf8538d68),LL(0xac339148,0x35380f15),LL(0x5ddfc12f,0xfef0bfdd),LL(0x1387d93e,0xf706c6bf)}, {LL(0x5357e131,0x618ce77d),LL(0x236478c4,0xf0921744),LL(0x00dc0da5,0x24eaf06e),LL(0x07603cc7,0x049113be),LL(0x8f6963c7,0x5cf48908),LL(0xede4a300,0xbe5eb9e6)}},
},
/* digit=45 base_pwr=2^225 */
{
   {{LL(0x5d066c15,0x77e486f8),LL(0x4ed5307d,0x0c05b6c2),LL(0x7df36628,0x322b28ab),LL(0x6704dcd6,0x2d14d131),LL(0xf29a3567,0xd359977a),LL(0xec96d3b6,0xc29bb132)}, {LL(0xe6bfa701,0xfd6e400a),LL(0x4c7e5101,0x03db9924),LL(0x9b8533af,0x62d81c7d),LL(0x8de66eb8,0xefa638c2),LL(0xe86784ee,0x7405a9d7),LL(0xa6c22223,0xafaa74ef)}},
   {{LL(0xb9d36e91,0xf9b2dba4),LL(0xfda9b2c4,0x5fb4f6ce),LL(0x3b8104ee,0x7692a4f3),LL(0xe4e1896e,0x5da885b0),LL(0x73d2aa36,0xc2a30fec),LL(0x86f60bca,0x7d06e6af)}, {LL(0x87287887,0xbc8bf16d),LL(0x3d701bec,0x6c3dd86a),LL(0x7e35610a,0x8e79e2f3),LL(0x82f9d71c,0x981139f4),LL(0x24e62733,0xf8997ec4),LL(0xa3518061,0x330d989a)}},
   {{LL(0xca89fbad,0x4e6ef410),LL(0x53933b78,0xe0fc53ba),LL(0xfd41d143,0xa4f03403),LL(0xe0774c37,0x3a507177),LL(0x8ec7484a,0x078e8c56),LL(0xfbb3f66b,0xfb73c6b6)}, {LL(0x3bfbdff6,0x169c9475),LL(0x0a232243,0x44d28606),LL(0x08303114,0x3e8e9685),LL(0xfad0def2,0x7a9797b8),LL(0xefc1c8da,0x0ad14404),LL(0x21ced721,0x6daae4e9)}},
   {{LL(0x88dd2dc2,0xc7e9ddef),LL(0x19a0c0b5,0x2c21a998),LL(0xb239bb82,0x6bc0746d),LL(0x28ea1341,0xc811a8eb),LL(0x1d1309b0,0x5f714ca7),LL(0xd4eb9b34,0x79eabd20)}, {LL(0xdf0fb30f,0xe0e5afdc),LL(0x8c0814c6,0x1b01a16d),LL(0x84334366,0x670e1e7b),LL(0x0eed1116,0xc8c38f9a),LL(0x619bbd50,0xf914fae2),LL(0x51c1995a,0x1ed062cb)}},
   {{LL(0xcb583422,0xd4e60e15),LL(0x320f296f,0xc6b1ef90),LL(0xd9bfc834,0x0714bad0),LL(0x9050e2c2,0x5ee2ca8c),LL(0x24f7cf1d,0x074a8ca8),LL(0x10df8516,0xb9750249)}, {LL(0xc2636d2c,0xecee8ab7),LL(0x3b4b7bbd,0x308e5af1),LL(0xee2ae021,0xfed4f27e),LL(0x2065253b,0x7cd4bb19),LL(0x4de525b4,0x6b21a3f8),LL(0xac27fddb,0x0f10e7bd)}},
   {{LL(0x870e29cb,0xd5068487),LL(0xfc52d5cb,0xf9420b85),LL(0x496d000d,0x50c3265a),LL(0x166bd6b4,0xe605414a),LL(0xc62b2a6c,0x4de8d724),LL(0xa1a11048,0x16af06f2)}, {LL(0x45f43c4c,0x5406bde9),LL(0x751ad18e,0x5e15bf6c),LL(0xb6a59587,0xa846e665),LL(0x1816ac55,0xcdb28a7d),LL(0x819b73f8,0x899b3551),LL(0xbc848d08,0x2d46297b)}},
   {{LL(0x299127be,0xdc4cc720),LL(0xfaab8165,0x5b34e762),LL(0xb39c120d,0x2289b2f7),LL(0x6e52b913,0x687a78d0),LL(0x2a3ea6a5,0xd2a091dd),LL(0x38eab329,0xc61eced6)}, {LL(0x7887ff2b,0x652231ea),LL(0x0479db4e,0x77a56875),LL(0xd43c5722,0x1ef471c8),LL(0xf3764c34,0xf82bf436),LL(0x0445cafe,0x962af405),LL(0x5ff47259,0xed8b227f)}},
   {{LL(0xd89594ab,0xde849cd1),LL(0x0ec4fb3a,0x00e2d2b1),LL(0xabe92fba,0x3fbd9e3d),LL(0x3324900a,0x785414d4),LL(0xde20904e,0xdaead1ab),LL(0xaa5f1ba8,0xb493e121)}, {LL(0x6eaea0dc,0xd60a4f2d),LL(0x6fca8596,0x394746b5),LL(0x34efa243,0x163dc789),LL(0x216a8d8c,0x3067dccf),LL(0xa901617b,0x116b6534),LL(0xbbabe51e,0x8c4bd099)}},
   {{LL(0xac3a082c,0xc8c2df45),LL(0xc8d4c40e,0xc353d074),LL(0x5a3c2de7,0xb214f9c0),LL(0xf86b0214,0x504bc42c),LL(0xd1922a58,0xc82df5cb),LL(0xa5bc3267,0x40887948)}, {LL(0x88ba8bb2,0x04bcd217),LL(0x046fd401,0xe21b3e7f),LL(0x616af5cf,0x8419c338),LL(0xaedfce9d,0x7f24760b),LL(0xddbd519a,0xded8035b),LL(0x1693faab,0x1f1fb0d7)}},
   {{LL(0xd02ffcf6,0xbb067b49),LL(0x3e657299,0x7cedf8f9),LL(0x406bbfe3,0xc3829961),LL(0x37c12472,0xefe4b5aa),LL(0xfec7dee8,0x7dc01cf9),LL(0x89472f50,0x70a9db23)}, {LL(0xb31bf737,0x29c269f8),LL(0xae3fa7db,0xa26deac3),LL(0x33caca41,0x0046e912),LL(0xb6e78b55,0x3bf4bc8a),LL(0xd9eb5ef1,0xca83bc6c),LL(0xc0c5deff,0x73f25c62)}},
   {{LL(0x44b4aae8,0x697dc47d),LL(0x782c331c,0xb3525cc0),LL(0x0bd7c78c,0xff71cca4),LL(0x10c0ab69,0x5f3d7766),LL(0xe2ba07e3,0xbdc10267),LL(0xe6373f6e,0xc656f75c)}, {LL(0xb5607b62,0x9e2938b4),LL(0x10b0a0f7,0xa65017d4),LL(0x5cc6ac25,0x8dad3119),LL(0x8ba5d1e6,0x00f8f2d1),LL(0x43305aef,0x608137bc),LL(0xdcb81cb1,0xddad34bb)}},
   {{LL(0xb1f82ca5,0xe133d941),LL(0xfdf115bf,0x2af8b98b),LL(0x57aaa6f3,0xdc6179c8),LL(0x130ade06,0xabaa83e9),LL(0x0e8bffd1,0x7836b6fb),LL(0xfa103703,0xc479751f)}, {LL(0x9c89963b,0x0ff3c129),LL(0x0b84c24f,0xe6407256),LL(0xf34f6bc9,0xa92a4ea2),LL(0x3197989b,0xba45b305),LL(0x99243aab,0xd12b5a01),LL(0x442af625,0x3015772c)}},
   {{LL(0xe6f065c5,0xd95fca81),LL(0xfc8655de,0x45e886d5),LL(0x27cff79e,0x35809577),LL(0x625877d9,0x92a39a34),LL(0xdfee17ee,0xdda02684),LL(0x986f635b,0x6354f871)}, {LL(0xd409c182,0xb3a6e9ed),LL(0xc4fbbb3a,0xf0b1c8d9),LL(0x9b77aded,0x28721c01),LL(0xbf94f028,0x3c356df1),LL(0x29a81f1a,0xff221bd2),LL(0x56b20b0d,0x20edf2e8)}},
   {{LL(0x835fda9e,0x97fff124),LL(0x0bc68512,0xa79ceb2f),LL(0xa2fc3995,0x70ba93d1),LL(0x9e51c5ee,0x62bd28ab),LL(0xd5bbbaa9,0xb95fa624),LL(0x8c1f571e,0x0654dc45)}, {LL(0x65a45ed6,0xb9a4edc6),LL(0x21ad0612,0xbf5ed1bc),LL(0xb1a3551b,0x74adc1a1),LL(0xdbbd6cef,0x3dfa3dc8),LL(0x2fa3afd2,0xce5dd40b),LL(0x30a746ca,0x14894e0f)}},
   {{LL(0xb8ca2a2c,0x7e729c58),LL(0xcaac04af,0x0f32ea1e),LL(0xbdd549e3,0x47267f13),LL(0x90be3b50,0x35b94406),LL(0x4b27f670,0xad0f2bb1),LL(0x92341803,0xd7e5874e)}, {LL(0x1f9ec462,0x7dc841cf),LL(0x512b2a42,0xebeff994),LL(0x320dc858,0x22998a7f),LL(0x19946f59,0xf08eb5c7),LL(0xa68ea75e,0x228c8dcd),LL(0x7b20dee5,0x40dc6dc3)}},
   {{LL(0xb3952db4,0x929454f6),LL(0x4d3f69f5,0x412142ec),LL(0xee25c0b0,0xf5b0a7c5),LL(0x2e752295,0x7d3372ff),LL(0x6eacac68,0xd6dadc7d),LL(0xa96a8e3c,0x5f0076cc)}, {LL(0x71725b3a,0xea831db6),LL(0xc29ab454,0x4a286c89),LL(0x72e3c00c,0x5ff817e5),LL(0x2a5fb6ba,0xb022e25d),LL(0xbb392476,0xb611c5bc),LL(0x190485a0,0x062c14dc)}},
},
/* digit=46 base_pwr=2^230 */
{
   {{LL(0xc419b0aa,0x0372678d),LL(0xc13fdf17,0xf95031d8),LL(0xb79594c3,0xebaebca4),LL(0xaf3b75cf,0xe587850b),LL(0x2c1e09c6,0x534183ac),LL(0xc08204cd,0x3f5b0bfd)}, {LL(0xe297cc77,0xdac2cf06),LL(0xd0487084,0x5e47d9c6),LL(0x90b0f6c2,0xf6f509f4),LL(0xc2c62207,0x3ffc3cd6),LL(0x32ff1887,0xbb21eb11),LL(0xe62ccc6f,0x2116a023)}},
   {{LL(0x16960728,0x406a7e21),LL(0x5597d8c4,0xd03923f8),LL(0x020748ee,0xd4402eff),LL(0xf39b58db,0x7827442a),LL(0x8d8cfb04,0x77e3f276),LL(0xe45a978f,0xf6eb49c8)}, {LL(0x49247f6a,0x9db08299),LL(0x06669fe5,0xce71a747),LL(0xb82775f5,0xe434ce47),LL(0x63910016,0xe84995ef),LL(0x1e47792f,0xa35e8b97),LL(0x7c6aaeb9,0xc779cb3d)}},
   {{LL(0xff381db7,0x66428800),LL(0x55574ac6,0xa9b9d019),LL(0xbdf4a86c,0x30cdc21f),LL(0x741c4a26,0x2ec38d35),LL(0x0b6be057,0x35496c23),LL(0x01656b1f,0xaecc67e6)}, {LL(0xf7d70324,0x781af00f),LL(0x5d7ee71e,0xac0e6579),LL(0xa6b14e3a,0x60a35c6c),LL(0x0e6c1c3a,0xacd6813b),LL(0x1faeef73,0xd7f77024),LL(0x23eddf05,0xd2254b8f)}},
   {{LL(0xaf31ea1d,0x1fa064cf),LL(0x48e8d974,0x2a9547a8),LL(0xfa9d9453,0xda8102a1),LL(0xdc6bd7ea,0x786aecab),LL(0xca2f6044,0xcaf91e3b),LL(0x8573f208,0x67d86ea7)}, {LL(0xc505ae24,0xd309fce9),LL(0x7f86eb8e,0x67ddc5b1),LL(0xf3d53056,0x57791ae0),LL(0x0d1fd61e,0x26b053f0),LL(0x045ebfa6,0x91c962c0),LL(0x076ed979,0xe95246de)}},
   {{LL(0x5f6e9ea4,0xc49c9989),LL(0xe16ec8e4,0x4a91578b),LL(0x0aeb5ac5,0xa1c54e89),LL(0xee09b9a0,0xa9094b07),LL(0x09a74b27,0x3587752f),LL(0x44bbfed0,0x973bf8b0)}, {LL(0x5636a52e,0x91c26f23),LL(0xeb7e3b41,0x8ac948b8),LL(0xfc457d56,0x14234675),LL(0xc76398a3,0xde98e4fa),LL(0x0f4a46e0,0xa80f7311),LL(0x22b66fa3,0xcba089bd)}},
   {{LL(0x0746d174,0x156eaf57),LL(0xcda35250,0xa2d4a83d),LL(0x0290fa02,0x60a9f48c),LL(0x5c33b4ac,0x9855d26d),LL(0x97eb1c30,0x06e379c6),LL(0x6e219664,0x4f2e2dbe)}, {LL(0x29006065,0x6b7448f8),LL(0x115062a9,0x237a1f31),LL(0xad92cb24,0x5c635a90),LL(0x2eed977e,0x2e857f8c),LL(0x856dc88a,0x3d512df7),LL(0xe597a27b,0xbde85263)}},
   {{LL(0x10a98e42,0x0b114aea),LL(0xae19dd14,0x6133aa52),LL(0xa99eb2c5,0x0c235df2),LL(0x7f59582c,0x0085a619),LL(0x8cf7feba,0xf9002bba),LL(0xaf6a3261,0x275742d7)}, {LL(0x249e8e9e,0x302b4823),LL(0xce696f91,0xa142aba9),LL(0x64c37b14,0xdeb28c44),LL(0x0766002f,0x14bb8f23),LL(0xc52fe891,0xafeff88b),LL(0x46faeb2d,0xb9d493ae)}},
   {{LL(0xe7c03ce4,0x49f24994),LL(0x2aed9ba3,0x274a8c13),LL(0xd5e91bc0,0x897b9103),LL(0xcb404f68,0x63db1efb),LL(0x42f7fc02,0x70efd9d8),LL(0xc6a230af,0xd6e02921)}, {LL(0x11ae0a56,0x8d5b199f),LL(0xce33da6a,0xc98287de),LL(0x504dd889,0xde583d34),LL(0xf823686f,0x03756001),LL(0x95fc73dc,0xf19ab86f),LL(0x93f12f42,0x300406c6)}},
   {{LL(0xa427d89c,0x68fdb78a),LL(0xa3944c0d,0x84e9ba49),LL(0xc1833422,0x1c3569b1),LL(0x1a01f4b9,0x30773fc0),LL(0x7da01321,0x18b8f17d),LL(0x7198c85e,0x8370fb0f)}, {LL(0x99a898e3,0xda12c8d4),LL(0x8ba82ded,0x7667b46a),LL(0x77e1e31f,0x2aab259b),LL(0xbe71c9bc,0xd03f7708),LL(0x8e43eb38,0x9f784cce),LL(0x21c1208c,0x7ddedc8b)}},
   {{LL(0x7759701c,0x2f73595f),LL(0x6fe0e0d9,0x8dc2069a),LL(0xc286a65d,0xb7de7114),LL(0x84c0e487,0xfecc429e),LL(0x14344c07,0x51061a2c),LL(0x96869e37,0x4d709725)}, {LL(0x2be9403b,0x8b02781f),LL(0xde3ab5d9,0x6cb6aa02),LL(0xff6bdc9a,0xb013508e),LL(0xe5438c58,0x568d2e84),LL(0xe4206c3b,0x7b35a979),LL(0xb17a8bc7,0x0bb793c0)}},
   {{LL(0x57ed2360,0x41248052),LL(0x6ba9bd95,0xfc0cb1b6),LL(0x2337a8fd,0x342f16db),LL(0x88099007,0xe1417411),LL(0x8cd74752,0xc96c29ee),LL(0xedf5fb4a,0x376047cc)}, {LL(0x439546c1,0x5f40ce08),LL(0x1a235de8,0x14d2c666),LL(0x98e355f0,0x9b66892d),LL(0xa4bb19e9,0x8a65f6dd),LL(0x046a2581,0xf72848f8),LL(0x8373b2b6,0xfed74b3b)}},
   {{LL(0xfa4dd561,0x3f896ca9),LL(0xd2de2ecb,0x4b9a98ab),LL(0x600e4e2a,0xd0741632),LL(0x69e702d5,0x87c7db5f),LL(0x53e0df2b,0x1f5a3b80),LL(0xf443dfba,0xe1e24b49)}, {LL(0x5eef3a1d,0xeb90e230),LL(0xd38f73fc,0x8f3fc8a6),LL(0xa5aa335a,0xfb1e8299),LL(0x4197b32a,0xd78504cd),LL(0x6755918e,0x0e7a79cc),LL(0x883b1c72,0xc7c98ae2)}},
   {{LL(0x969088c1,0x4b74fcbd),LL(0x361a8c96,0x4d16f895),LL(0x760d61fa,0xefcb6ced),LL(0xcc3e8808,0x3f14a7cb),LL(0x664ea335,0x51f5fd2c),LL(0x3a65d305,0xe0cad090)}, {LL(0x031a6911,0x86409de9),LL(0xe5f9715b,0x23ea4aed),LL(0x1f3532c6,0x6e5b8cea),LL(0x11271ead,0x33fc873e),LL(0x5b8131d7,0x842b59a6),LL(0x61b7bf60,0xbd95818a)}},
   {{LL(0x03d2becc,0x1a12727c),LL(0xc6741372,0x810a37df),LL(0xb7049f39,0x44ac483f),LL(0xa36fc614,0xab73e5e7),LL(0xeeff8aeb,0x298d453f),LL(0x7e1b586b,0x2127dd16)}, {LL(0xe07bd60c,0xeadc5c54),LL(0xf5e2d2e2,0x67cdae00),LL(0xc9d2f10a,0x03fe0446),LL(0x95e38ed2,0x07840987),LL(0xe1a6306e,0x5d348a7c),LL(0x562f5463,0x4903f1b6)}},
   {{LL(0x4a3862e8,0x906ab8a8),LL(0xb2f5c878,0x8fc76114),LL(0x2035287f,0x2dac3952),LL(0x18af4378,0xaa8372f3),LL(0xdbf64476,0x915050c9),LL(0xe992d0c8,0x896f734d)}, {LL(0x3a35846c,0x5c3e36da),LL(0xac8f4fc3,0xfe774b4a),LL(0xaadd8a59,0x66347050),LL(0x2cd12be8,0xea94ebda),LL(0xdab94de0,0x45b1e7e2),LL(0x264b508c,0x539d580e)}},
   {{LL(0xb44b1d0c,0xbf66baa5),LL(0xa44f8eda,0xbbed18ed),LL(0xeaaa466c,0x80bc32ab),LL(0xe5f2733b,0x605b7897),LL(0xa2531afa,0xe9e7e3a1),LL(0x3deb8369,0x25d66db3)}, {LL(0xb2f25d10,0x36212ea3),LL(0xa08d303e,0x52d6b3f4),LL(0x444e9e9f,0xefa54b31),LL(0x69530c1b,0x9c2229a1),LL(0x4b79bdd1,0x68feb985),LL(0x8b984cc3,0xd570e84f)}},
},
/* digit=47 base_pwr=2^235 */
{
   {{LL(0x845f26e4,0xd6a4d25c),LL(0x1b039dff,0x71e554ce),LL(0x1cdedfc0,0x94205973),LL(0x03d6502f,0x0c4e3856),LL(0xe15ce8c8,0x981a4fc5),LL(0x7aca30b7,0x85d1b0f1)}, {LL(0x77bb9e43,0xf2037ef7),LL(0xe87ae187,0xc52804f4),LL(0x71f3e4e3,0x9c98a23c),LL(0xf47b504b,0xa73c8b89),LL(0x023233aa,0xb9e33f54),LL(0xf92c9f68,0xf2bcfc17)}},
   {{LL(0x7b3b336d,0xba03ba3b),LL(0x28c9c55d,0xe57ce509),LL(0x4f0f60b2,0xf96b8cfe),LL(0x6fcccd96,0xb908d77e),LL(0xe79dd17a,0x7208ef7d),LL(0x3ec3d048,0x73909533)}, {LL(0x1163fe78,0x9c5ad2da),LL(0xcd4a15c2,0x4e2a8685),LL(0x470eb938,0xac999449),LL(0xee7d772f,0xfaaf27fb),LL(0xd0b7ad09,0xfbe402ab),LL(0x57db00a9,0x704d4f0e)}},
   {{LL(0xe12b4e64,0xe93ee31a),LL(0x662d17f4,0x2ab8e378),LL(0x69516582,0x2544bd99),LL(0x2e1e5485,0x7bf80e4b),LL(0x729d9361,0xf30f0b14),LL(0x8268d40f,0xb3ffb5d1)}, {LL(0xac193a63,0x34605055),LL(0xf8e04d69,0x9e5ca9a9),LL(0x085ecbb2,0xcbbeebc1),LL(0xf340eac4,0xda03b75b),LL(0x84436462,0x3bf9468a),LL(0x0f26f20c,0xdfa8b4c8)}},
   {{LL(0x3fc14a85,0x10c082a6),LL(0x1c0b14c4,0x59389ebc),LL(0x4cb291a7,0x785d935b),LL(0x13e9ce08,0xfc2ae153),LL(0x4df6f1c4,0x3146fabf),LL(0xc87dd24c,0xa2a4a457)}, {LL(0x1deb49bb,0x85fdd877),LL(0x9b055934,0x2b784370),LL(0x3e7e0297,0xc81d0501),LL(0xb92df904,0xb56ddd1f),LL(0x295ddccb,0x4612df9f),LL(0x0e27cf1d,0xc24bd4cf)}},
   {{LL(0x422ea2c0,0x5564875b),LL(0x8285b03f,0xabc2e7de),LL(0x733e9f1f,0xfd662091),LL(0x68465da3,0x68f16745),LL(0x38fa6f63,0x965a0a05),LL(0xdfae710d,0x0ed70fee)}, {LL(0x153b24da,0x56c6227f),LL(0x01470f6e,0xf1dcf574),LL(0xf51771cf,0x9992caa8),LL(0x14a9b029,0xa884b481),LL(0x7b9a4062,0xad11cbaa),LL(0xe55533e0,0x60ec99d8)}},
   {{LL(0x18674f49,0x86d9b060),LL(0x97bb1a7d,0x81b06486),LL(0x27d9d64c,0x9b6e8e7c),LL(0xba04e6ad,0x79bd66ba),LL(0x828abbbe,0x77e4d0b0),LL(0xcc540d04,0xae7548ac)}, {LL(0x869cbebf,0xf5d8a46f),LL(0x38a6cc83,0x99fb1a63),LL(0x563fe6b4,0xb93bb852),LL(0x97cdd04f,0x06bb3ae9),LL(0x8b7de47d,0x07f011d4),LL(0xde78f61c,0x8d90e2e4)}},
   {{LL(0xd33cb6e2,0x731c6dd6),LL(0x8cce0290,0xa3ea317e),LL(0x1c42206e,0xdca9b2f9),LL(0xd1e5dfd9,0x6acbbce9),LL(0x2fc948cd,0x40745846),LL(0xa82f9cec,0xc7a50d91)}, {LL(0x4c1aa161,0xb906d69e),LL(0x0ebe948e,0x3a9b14be),LL(0xb63aeb70,0x11a9f12c),LL(0xead745f2,0x0365b4cc),LL(0xf9f16c17,0x5f6c2bbd),LL(0xa03e558e,0x89131238)}},
   {{LL(0xd1944d1b,0x7830460f),LL(0x84350af2,0xc56f08e7),LL(0x307d9c78,0x73bee2aa),LL(0x5aad8b6c,0x1b02af1b),LL(0x03848db5,0x5e318827),LL(0xf230f476,0x4785958b)}, {LL(0x4f80e25d,0x4ea6535d),LL(0xd23c7f72,0x9958c9c7),LL(0x2fd33cab,0x4c197b33),LL(0xc566914f,0x24c7b0b1),LL(0x71952d3b,0x956ce3c3),LL(0xfabae5f2,0x8735694b)}},
   {{LL(0x40ec913c,0xde37ae28),LL(0x056685b3,0xdc915f83),LL(0xf66a4501,0xf7bc3488),LL(0x6a900e5f,0x30e61042),LL(0xca3cf645,0x505525c1),LL(0xb1f3ed40,0x35338c53)}, {LL(0xd70b7c41,0x6823159a),LL(0x7384ba7c,0x660f518a),LL(0x2482056b,0xc6cf6a4b),LL(0x1df15990,0xb308b215),LL(0xfb5c130b,0xba63b2e3),LL(0x277b7515,0x1c660db8)}},
   {{LL(0xfddc9fdd,0x8a95e5c1),LL(0x5adab0c3,0x679d4e0f),LL(0xcda40bc0,0x1859df6a),LL(0x8234471d,0xf9097aa5),LL(0x783c0100,0xaddc0c9b),LL(0xfeb7067c,0x55388dc8)}, {LL(0x80a2eac7,0xe3805fd1),LL(0xec886879,0xf800a75c),LL(0x1943a0a0,0xa4599992),LL(0x1dfe627e,0xb47f0619),LL(0xda06515f,0x313d4f09),LL(0x1f54a73e,0xde26052d)}},
   {{LL(0x6827365d,0x85a1b879),LL(0x667debe8,0x595a6915),LL(0x93a3d50f,0x214670fb),LL(0xb37de08e,0x9dfb028e),LL(0x9c6cf2a9,0xdd077e2f),LL(0xc9b96e8e,0x96897d8f)}, {LL(0xefd39543,0xe6e93c07),LL(0x4454e73d,0x19dcdaa4),LL(0xd7b8c758,0x4a67424f),LL(0xb1e91e2a,0x03d4de0d),LL(0xe887b6b6,0x7c843988),LL(0x6ffdcfff,0x7db4f3da)}},
   {{LL(0xd8bb43ec,0x0db87bc3),LL(0x13a7b669,0xbf71d27c),LL(0xa3fd2a60,0xea81e9c4),LL(0xc9f017e5,0x190c9c71),LL(0xbcc75768,0x21864180),LL(0x43dbcde6,0x137bd615)}, {LL(0x5d468ff5,0x5b090c71),LL(0x126c6bc6,0x3a622b60),LL(0xa918bf24,0xfef3d268),LL(0xae204f49,0xe10c52c8),LL(0x86d7c356,0xf4be898d),LL(0x2fdb5a17,0x8276da18)}},
   {{LL(0x7f2ad562,0x89c992af),LL(0xafc83ad3,0x7e1459d9),LL(0x9278dd04,0x4c0d9681),LL(0xd8eebe36,0x4496d9a7),LL(0xb8d4b1a2,0x7c037261),LL(0xbad3d6d7,0x827c49a5)}, {LL(0x836926ae,0xf4d94deb),LL(0x4064af58,0x65417bf6),LL(0xfcdafc9f,0xa79471ac),LL(0x8123312d,0x3f85ccb4),LL(0xa3360be4,0x4d374cad),LL(0xee325a2b,0x56b476d6)}},
   {{LL(0x41af8c08,0x0079c69d),LL(0xacbe515e,0x7dcfa4f4),LL(0x01396859,0xb8d18666),LL(0x946fbedd,0x08590ca4),LL(0x641aace4,0x7fecd9b9),LL(0x2936a1b6,0xaad5cc44)}, {LL(0xf92c5958,0x925b6235),LL(0x82d6231e,0x7b1442f1),LL(0x8c6fb34a,0x971e663d),LL(0x2fc1c10d,0x543146dc),LL(0x0642b822,0x6e4053c7),LL(0x492e524d,0x4a49f247)}},
   {{LL(0x17ba53f1,0x1b51f7b4),LL(0x3d5c43bf,0x170ff1eb),LL(0x681f7ee7,0xc2f160f8),LL(0x47814310,0x4c0a54d0),LL(0xa83d061c,0xfc689a13),LL(0x7ff6333d,0x1cbc99b4)}, {LL(0x6581cd16,0xe19fd790),LL(0x9ca37b0c,0x67da79c7),LL(0x63bd0b5c,0x2507d167),LL(0x1befb82b,0x4449985b),LL(0x914699ec,0x6bea3969),LL(0xef202abb,0x9f606dd4)}},
   {{LL(0x597bd10e,0xca9872e1),LL(0x4aed951f,0x6725cc9a),LL(0x4e05b280,0x96b17cb8),LL(0xfa234d45,0x97987146),LL(0xbb35a7d8,0xba78949e),LL(0x6fc59384,0xb82e9b9f)}, {LL(0x70f165c7,0xa303e54a),LL(0xb9c2cad9,0xfd6bb0dc),LL(0xee722045,0xe57e2de8),LL(0x63e27035,0xa05c1065),LL(0x02d2fe6f,0xaa38e866),LL(0xee2f6aad,0x78e02fa8)}},
},
/* digit=48 base_pwr=2^240 */
{
   {{LL(0x5e3c647b,0xc0426b77),LL(0x8cf05348,0xbfcbd939),LL(0x172c0d3d,0x31d312e3),LL(0xee754737,0x5f49fde6),LL(0x6da7ee61,0x895530f0),LL(0xe8b3a5fb,0xcf281b0a)}, {LL(0x41b8a543,0xfd149735),LL(0x3080dd30,0x41a625a7),LL(0x653908cf,0xe2baae07),LL(0xba02a278,0xc3d01436),LL(0x7b21b8f8,0xa0d0222e),LL(0xd7ec1297,0xfdc270e9)}},
   {{LL(0xe2a07891,0x4f120aa7),LL(0xa25d3225,0x9158bab3),LL(0xcfe5f7a8,0xc96bac5e),LL(0xbbf3cec6,0xd4e73d59),LL(0x60361cd5,0xed8d2335),LL(0x562f444c,0x9b1a252c)}, {LL(0xc70f23c2,0xbd37d3cf),LL(0xa52ea19e,0xf13b3b6e),LL(0x3d2f41ed,0x7e35535a),LL(0xe8b1743e,0x0353b52e),LL(0x7b5a2765,0x31d89dfd),LL(0x8d9ea8b8,0x2b7ac684)}},
   {{LL(0x9f101e64,0x06a67bd2),LL(0xe1733a4a,0xcb6e0ac7),LL(0x97bc62d2,0xee0b5d51),LL(0x24c51874,0x52b17039),LL(0x82a1a0d5,0xfed1f423),LL(0xdb6270ac,0x55d90569)}, {LL(0x5d73d533,0x36be4a9c),LL(0x976ed4d5,0xbe9266d6),LL(0xb8f8074b,0xc17436d3),LL(0x718545c6,0x3bb4d399),LL(0x5c757d21,0x8e1ea355),LL(0x8c474366,0xf7edbc97)}},
   {{LL(0xc46db855,0x73457010),LL(0xdd579fb8,0xccb68c43),LL(0x9c25fe5b,0x705b0e8c),LL(0x82dd0485,0x40f36ea1),LL(0x27ac2805,0x3d55bc85),LL(0xad921b92,0x15177c6f)}, {LL(0x5ab18cab,0x51586cd5),LL(0xcbb4488c,0xf51b5296),LL(0x84f0abca,0xbb4e605e),LL(0x772dd0da,0x354ef8e3),LL(0x5e4e1d41,0x7f1a8f79),LL(0xde5d8491,0x93461f09)}},
   {{LL(0x6ea83242,0xec72c650),LL(0x1b2d237f,0xf7de7be5),LL(0x1819efb0,0x3c5e2200),LL(0x8cdde870,0xdf5ab6d6),LL(0x92a87aee,0x75a44e9d),LL(0xbcf77f19,0xbddc46f4)}, {LL(0x669b674d,0x8191efbd),LL(0xed71768f,0x52884df9),LL(0x65cf242c,0xe62be582),LL(0x80b1d17b,0xae99a3b1),LL(0x92de59a9,0x48cbb446),LL(0x2dcb3ce2,0xd3c226cf)}},
   {{LL(0x9311182c,0xf3899558),LL(0xb657a7b7,0x1bee4c4b),LL(0x2df8d1a7,0x0b1c4fd3),LL(0x76d3fbbf,0xf16bcc23),LL(0xf4fd52bc,0xd5888916),LL(0xd5cde1f0,0x3de6cfb4)}, {LL(0xd4a07dfd,0x764ffffd),LL(0xe2642182,0x5e674426),LL(0xccd57b85,0x34f64762),LL(0x29351062,0x2233a4c3),LL(0xd9c642f3,0xdf076095),LL(0x59f0df34,0xac917a2c)}},
   {{LL(0x9fd94ec4,0x9580cdfb),LL(0x28631ad9,0xed273a6c),LL(0xc327f3e7,0x5d3d5f77),LL(0x35353c5f,0x05d5339c),LL(0x5c258eb1,0xc56fb5fe),LL(0xedce1f79,0xeff8425e)}, {LL(0xcf83cf9c,0xab7aa141),LL(0x207d6d4f,0xbd2a690a),LL(0x458d9e52,0xe1241491),LL(0xaa7f0f31,0xdd2448cc),LL(0xf0fda7ab,0xec58d3c7),LL(0xc91bba4d,0x7b6e122d)}},
   {{LL(0x775f516f,0x3bd258d8),LL(0xc715927f,0x4bedebd5),LL(0xe3f966a0,0x5b432512),LL(0x709d0c2d,0x338bfca7),LL(0x49658259,0xd142cc10),LL(0x636b8023,0xfabc6138)}, {LL(0x4d4ef14d,0xa9ef9401),LL(0xc54c570c,0xd5917ac1),LL(0x5cb64487,0xfd2f63c5),LL(0x1cea475b,0xbae949b1),LL(0x1e67a25f,0xa4544603),LL(0xdc6a7a6a,0xa547abc1)}},
   {{LL(0xb1b48156,0x2a2dedaf),LL(0xbb93db87,0xa0a2c63a),LL(0x08acd99e,0xc6559078),LL(0xfe4ac331,0x03ea42af),LL(0xeb180ed6,0x43d2c14a),LL(0xb1156a1a,0xc2f293dd)}, {LL(0xa9d81249,0x1fafabf5),LL(0x9a8eee87,0x39addead),LL(0x119e2e92,0x21e206f2),LL(0xd74dceb6,0xbc5dcc2e),LL(0x0a73a358,0x86647fa3),LL(0x2f53f642,0xead8bea4)}},
   {{LL(0xb12c4bb1,0x158d814d),LL(0x2f0cf4fa,0xe52f75d2),LL(0x6141b59c,0xf106023e),LL(0xbeb9d941,0x5eb8b8eb),LL(0x90cf579c,0x1dd39729),LL(0x69ee6efa,0xb273252e)}, {LL(0x3e9947a0,0xe43a3c59),LL(0x6c19dd01,0xd605124f),LL(0x05c578b0,0x8090fdbd),LL(0x622ff18c,0x8e6c535a),LL(0x57d12071,0x3600b0c2),LL(0x78d001d7,0x6d026e5c)}},
   {{LL(0x91c09091,0x636225f5),LL(0x71bdcfdf,0xccf5070a),LL(0xb9668ee2,0x0ef8d625),LL(0xb5e04e4f,0x57bdf6cd),LL(0x7c75ea43,0xfc6ab0a6),LL(0xf7fd6ef3,0xeb6b8afb)}, {LL(0x2a3df404,0x5b2aeef0),LL(0xb9823197,0x31fd3b48),LL(0x83a7eb23,0x56226db6),LL(0x5bb1ed2f,0x3772c21e),LL(0xcd1aba6a,0x3e833624),LL(0xac672dad,0xbae58ffa)}},
   {{LL(0xdaff1807,0x00e0a003),LL(0x92c94fd0,0xcb9d1559),LL(0xcebbf905,0x3c2b5c3d),LL(0xd338afa9,0x9c799ec7),LL(0x4e2cfccc,0x60b9908c),LL(0xae3c6f92,0x4bfe1a57)}, {LL(0xfb116150,0x480d310e),LL(0xe3e7888e,0xa1ed6c31),LL(0x720b5196,0x841a11d9),LL(0x8adff37d,0xcc337d17),LL(0x5faa86c5,0x08c66826),LL(0x9dfcc7ad,0x945c90d4)}},
   {{LL(0x31ba1705,0xce92224d),LL(0xf0197f63,0x022c6ed2),LL(0xa4dc1113,0x21f18d99),LL(0x03616bf1,0x5cd04de8),LL(0x9ff12e08,0x6f900679),LL(0x48e61ddf,0xf59a3315)}, {LL(0xb51bd024,0x9474d42c),LL(0x9051e49d,0x11a0a413),LL(0xdce70edb,0x79c92705),LL(0x34198426,0x113ce278),LL(0xea8616d2,0x8978396f),LL(0xea894c36,0x9a2a14d0)}},
   {{LL(0x8e2941a6,0x9f9ac960),LL(0x2fc4fe1e,0x43e7ff90),LL(0x6033e041,0x5ec41359),LL(0x6f6ff0f3,0x5ce791c4),LL(0x9d907343,0x8d134b89),LL(0x86304df2,0x7bd15c77)}, {LL(0x77c4a913,0x2cd2ebc7),LL(0x45f07153,0xcd86a39d),LL(0x88bc423b,0xe7e12d2e),LL(0x0b3163f4,0x478e814b),LL(0xbe8ec766,0x78bd9c8a),LL(0x7709ce48,0x6a5763e8)}},
   {{LL(0x604f6e4a,0x4f1e1254),LL(0x0187d585,0x4513b088),LL(0x19e0f482,0x9022f257),LL(0xe2239dbf,0x51fb2a80),LL(0x998ed9d5,0x49940d9e),LL(0x6c932c5d,0x0583d241)}, {LL(0xf25b73f7,0x1188cec8),LL(0x3b3d06cd,0xa28788cb),LL(0xa083db5a,0xdea194ec),LL(0x22df4272,0xd93a4f7e),LL(0x6a009c49,0x8d84e4bf),LL(0x3e3e4a9e,0x893d8dd9)}},
   {{LL(0x8d095606,0xd699ea2d),LL(0x1e0ddd3a,0x3cd080c5),LL(0x66a8b35b,0x46604bad),LL(0x4233fccb,0x0c779b62),LL(0xbfd3cf0c,0x578458ac),LL(0x96bf57af,0x6820f665)}, {LL(0xbf1f302c,0xa9724245),LL(0x277a6c3e,0xbbde24da),LL(0xc6be8c14,0x0980a5b8),LL(0x774d62c4,0x6230e3ec),LL(0x4fbde24b,0xda1467d8),LL(0xcc862204,0xd9d68d07)}},
},
/* digit=49 base_pwr=2^245 */
{
   {{LL(0x7378f90d,0x67c51634),LL(0x66647082,0xbc201a79),LL(0x9ee450cf,0x77fcc8dc),LL(0xb41a3e2f,0x8dd2b318),LL(0x93bf0689,0xdf6a935e),LL(0xa92e5464,0x75edabf3)}, {LL(0x604d208a,0x49afcd9f),LL(0xd465ca48,0x372f0ea7),LL(0xc7ea7810,0xcdbd8ad2),LL(0x550822b2,0xfe61571e),LL(0x86606adc,0x744a4f93),LL(0xd9d4e110,0x6beb3c9c)}},
   {{LL(0xe700b9f2,0x1fef389c),LL(0x425bc8ab,0x63029466),LL(0x37f04a33,0xbd770a14),LL(0xd0169369,0xc7438e29),LL(0xe2377cc3,0x6b265742),LL(0xc369fa4f,0xdf24bf96)}, {LL(0x0ad94e08,0xdfdbcf47),LL(0x7f75a7dc,0xd101b861),LL(0x2a9c483c,0x5574a0b8),LL(0x2de43228,0x0563fe94),LL(0xead1fabe,0x58ca0e8a),LL(0x66023966,0xdc3d9a84)}},
   {{LL(0xc3fd20e5,0x383bda07),LL(0x5c29449b,0x9619b1df),LL(0x369f39bf,0x6f3c717d),LL(0x1a5a3900,0x1bb593d1),LL(0x2aec6c2b,0xd0f07ecc),LL(0x4240b202,0x9d72eb2a)}, {LL(0xc50e4a0c,0x35342f6c),LL(0x6b93bf61,0x701b4662),LL(0xccb6a888,0xfcd6eb09),LL(0x85aa42c5,0xabb7a6f7),LL(0xaa4e5895,0x952f8824),LL(0x5c406582,0x49860db8)}},
   {{LL(0x3955812b,0x3667a720),LL(0x284d1dac,0x0d73483b),LL(0xfc62f791,0xe084535e),LL(0x389faf7f,0x5bc1652b),LL(0x3a71b7f6,0x40cf5168),LL(0xd4f39703,0x8a4b19fa)}, {LL(0x2a8eff13,0x823e754a),LL(0xbffa5afc,0xf01b2021),LL(0x7225b319,0x5639ee02),LL(0xfc282f16,0x7533bc86),LL(0xc69f61ae,0x710009d2),LL(0xbf65e803,0xe30c499d)}},
   {{LL(0x734b4ec3,0x0da7ac1b),LL(0x12a2afbe,0xf47fc1d0),LL(0x87dce4a2,0xbbbc99be),LL(0xdd5c6378,0xf7264b4e),LL(0xf618ffdc,0xe9409305),LL(0xd1846ac1,0xafadda9b)}, {LL(0xa21850d4,0xe734f9d0),LL(0x8722a316,0x199cb44f),LL(0x38cae89f,0xcfe8704b),LL(0x6b151b57,0x2db1e56b),LL(0x69ce7b2c,0x116ca5cf),LL(0x57de97c8,0xe9b8625f)}},
   {{LL(0xaf247c49,0x18811bd5),LL(0xe124dbda,0xbc180793),LL(0x21234fc4,0xed978d3a),LL(0x0616ae15,0x516dd9a7),LL(0x74e430b8,0x8f806777),LL(0x06e8fc49,0x90942569)}, {LL(0xa4e61235,0x4ca03fb5),LL(0xb617f361,0xb91de709),LL(0x0ed08bc3,0x0898d82d),LL(0x8cb08146,0x2bd71236),LL(0xe213176d,0x45b92d45),LL(0xf2bf5b9c,0x05894791)}},
   {{LL(0x2695ea2b,0x0d79cb89),LL(0xc88e538a,0x2cb0f8df),LL(0xa80f36fe,0xc1b8dc3d),LL(0x84f00cc2,0xd756fa66),LL(0x9cb9efb2,0xa6f1cdec),LL(0xa6a21818,0x5c3f15a8)}, {LL(0x6995d09f,0x9a7ee351),LL(0xd70434bf,0x88885463),LL(0x4f7d5d33,0x18cecc6d),LL(0x6b353bd1,0x3f013886),LL(0x0d9ad368,0x53bf798b),LL(0x28dbc3ee,0xeffd465a)}},
   {{LL(0xb5d98ac1,0xeb29e44c),LL(0x0e227a4f,0xe47e57f8),LL(0x3d2bf688,0xd09c0494),LL(0x47428dd2,0x3ab7799a),LL(0xe9aafac8,0xdc558d6b),LL(0x87f9f6e0,0xc042c4cd)}, {LL(0x89fb4693,0x93842bcd),LL(0x7068fbf7,0x62dbc82f),LL(0x7e6d47b5,0x16455268),LL(0x4c37eeee,0xab304b7a),LL(0x3fc412ce,0xdbb3d4e1),LL(0xa726a2c8,0x4f65dad0)}},
   {{LL(0x605cdaee,0xb25e01b2),LL(0xbc57969d,0x74abec55),LL(0xcdd9d41a,0x9c57bfab),LL(0x4a9e32a3,0xa3330e3f),LL(0xe5792fd8,0x5929a0d8),LL(0x71ea2cde,0x830b4ea2)}, {LL(0xfd06d246,0x80065ac1),LL(0x32e64a25,0xa2b416e6),LL(0xc0c927a9,0x3950bde7),LL(0x679d9b8c,0x9951f3bd),LL(0x651b6855,0xc235a274),LL(0x5ad97bc1,0xbfe5e08e)}},
   {{LL(0x744ae145,0x4409a5b6),LL(0x7f620908,0x5e83fa0b),LL(0x2e140aa0,0xfc489bec),LL(0xe3cae337,0x5805a462),LL(0xc2211c21,0xe56e9ff7),LL(0x0c955362,0xb722f2b4)}, {LL(0x41371f33,0xb098a32f),LL(0xbb4923d6,0xe6ccecea),LL(0xd82a311c,0x1cfbe2b3),LL(0x6b98f917,0xcf942893),LL(0x92ef848c,0xd60dc624),LL(0x5adb5228,0x34af446e)}},
   {{LL(0x796ce1ca,0x0eb7e743),LL(0xd851377c,0x138653e5),LL(0x2b11c8e0,0x69c7c86f),LL(0xcdf2b205,0x878ec1de),LL(0xae0e8562,0x03e6688a),LL(0x935a36a8,0x20810666)}, {LL(0x26635c50,0xc8ab7c7f),LL(0x744a21db,0xe75cdb06),LL(0xd720e198,0x4e26f32f),LL(0xd8cded81,0xa1c6395a),LL(0x6ce4fc04,0xb75dc6ea),LL(0x004623b5,0x71750b33)}},
   {{LL(0x7e60c447,0xbdef8407),LL(0x2a65acca,0x88570f71),LL(0x0bb6aa79,0xef3d4a40),LL(0x60212976,0x5c9d1890),LL(0x1d96c43c,0x80179ea2),LL(0x53d2948e,0x3f002e6d)}, {LL(0x49d78183,0x14b2cc91),LL(0xb496c279,0x7a549c71),LL(0x44995f6d,0xf4beac3f),LL(0x00bc78fe,0x5a342398),LL(0x60e42da0,0xa874dc1b),LL(0xcf5824d5,0x3a984010)}},
   {{LL(0xdfb9760c,0xe514ee06),LL(0x77b8951f,0xb8862d75),LL(0xf8ee1141,0x0144676e),LL(0x02eb3e82,0x49561a30),LL(0x4ff9f897,0xb3541c15),LL(0xa7a99791,0x1670edf0)}, {LL(0x64aea7f9,0xd41d6035),LL(0x2b3463b4,0xf66ffd09),LL(0xc3b26fb6,0x0784e015),LL(0xec46f8c8,0x88edce33),LL(0xb6381011,0x1b1e25a3),LL(0xff95ab97,0xbfaadc03)}},
   {{LL(0x0c7be4e1,0x727a59fe),LL(0xf58ced15,0x75a7d5e3),LL(0x90f569e7,0x146fc0d9),LL(0xb7f1dc54,0x94dbccd2),LL(0xb75bf232,0x0df1ef90),LL(0xa2568190,0x2943a082)}, {LL(0x67837b06,0x75f2f80d),LL(0x24b44b6e,0x07e3506f),LL(0xd0d2231b,0x7c30829a),LL(0x93277abf,0x9ce577ca),LL(0xb17549ec,0xa19d1868),LL(0x25e8c4d7,0x0ad6ff55)}},
   {{LL(0x1c24d075,0x16b38dfe),LL(0x992959f6,0x3acd4c36),LL(0xac2da7ab,0xdaf2fe88),LL(0x89644935,0x76e8ff0e),LL(0xe85f7076,0xb8547c26),LL(0x1cdea7ce,0x9f149faa)}, {LL(0x9e125d84,0x181a6072),LL(0x18751ce6,0xc4aef9fa),LL(0x0e00f00a,0x451c8466),LL(0xc4e3e6b8,0x662b3e7a),LL(0xc6b64507,0x57b7114e),LL(0x0b37fb70,0x07aeb198)}},
   {{LL(0x4516234a,0x79d88e00),LL(0x31f9ceda,0x98dd3cb9),LL(0xce7d606e,0xb528000f),LL(0x2fa27fd3,0xc773557e),LL(0xe19436af,0x55b53dd3),LL(0xe10b64c7,0x675084b3)}, {LL(0x56d56374,0xe5832665),LL(0x307e2e60,0xf8f7fd2a),LL(0x7af3e3dc,0x7b93bf53),LL(0xf47d298c,0x94fafa2c),LL(0x21121369,0x94c2ff9a),LL(0x33468ff6,0xa41de95f)}},
},
/* digit=50 base_pwr=2^250 */
{
   {{LL(0x1b270599,0xbe45d81a),LL(0x97d6c603,0x50696e7d),LL(0xb078ea89,0x63c5a516),LL(0xb4464764,0x9f3efe41),LL(0x101e5232,0x84580e24),LL(0xc8ae8220,0x00850a1a)}, {LL(0xed55c404,0xbff4077d),LL(0xf2e7bf50,0xd74de734),LL(0x07e1c03d,0x4df4eef2),LL(0x6e654d58,0x4ab3d039),LL(0x086f1596,0xb20056cd),LL(0x8acd7cd5,0xe4d08a27)}},
   {{LL(0xc90b13f5,0x8cd6c9f7),LL(0x52a9d469,0xec0c187d),LL(0x89b8ad2b,0x9c0db0f5),LL(0x0d9c999d,0x692a8db7),LL(0xc9f84ab4,0xa407fd03),LL(0xcc9a906c,0xa5742fd1)}, {LL(0xc8e72867,0x4813a765),LL(0xe2e9a10f,0x9c65943d),LL(0x4fa0a23e,0xca6bf293),LL(0xcb1f8d7a,0x1dfa3af7),LL(0x98d10c53,0x28036f54),LL(0x0e012c13,0x7bfbcaf2)}},
   {{LL(0x893e8032,0x2d9513a9),LL(0xf4688db4,0x49257f7a),LL(0x3af4d9ac,0x73d8b12c),LL(0x48a13c4d,0x903dc9fa),LL(0x60709433,0x6190753e),LL(0x49387d24,0xa093364e)}, {LL(0x3b261e16,0x0436949e),LL(0x4a3055b6,0x96db3b27),LL(0xe85dfe23,0x514eacc7),LL(0x5d8805c9,0x1538b25c),LL(0x664a20f6,0xd4c6b75b),LL(0x4753292d,0xd1984f21)}},
   {{LL(0x4ec177f6,0xa53f1a10),LL(0x3faa5ca8,0x4a2ef9aa),LL(0x32976d13,0x30efed85),LL(0x5ee692d1,0xcf5ada16),LL(0x259e7cc1,0x3ceda69d),LL(0x9baab472,0x2aae29e9)}, {LL(0x737cc8bc,0x7ee5baef),LL(0x7fe68ded,0x1717af74),LL(0xcfdaff63,0x9e5c8dda),LL(0xcec07693,0x575c8db9),LL(0xfdfb509d,0x9afc8ae0),LL(0x85651697,0x27836d36)}},
   {{LL(0x85e79c26,0x9d152839),LL(0xd36ab6f7,0x8d87faad),LL(0xa87afe15,0x08d4fe09),LL(0xfab81fd4,0x1e10b10f),LL(0x93c98a84,0x3cbd17d3),LL(0x246ceed3,0xbd40a4e8)}, {LL(0x9a09d134,0x582d9de0),LL(0x4ee37dec,0x120440c2),LL(0x2f1073e4,0x4d4ff934),LL(0xf45e648d,0x7da76757),LL(0xebcbb80b,0x4427a608),LL(0x30e661a7,0xc6bccb54)}},
   {{LL(0x135e2a30,0xb7b7628f),LL(0xb526fa7b,0x863552c4),LL(0xe0e30451,0x6ec18d05),LL(0x5769db60,0x1b36c93a),LL(0x170c236c,0xf0fe0007),LL(0x66130046,0xeceb540c)}, {LL(0x3fc4bdbe,0x86a7a74d),LL(0xfbae3320,0x066b097b),LL(0x6e5c21ae,0x78fb5247),LL(0xe1adf398,0x3e19e9fc),LL(0x1a32a745,0x429b9cbd),LL(0x36d1a2b8,0xedd2c40f)}},
   {{LL(0xceeaeed7,0x993ca8a9),LL(0x62545429,0xb2d28681),LL(0x91cf32d4,0x24003737),LL(0xd88bd4f0,0xda9ef96f),LL(0xe1b2d52a,0x916a4947),LL(0xf31b107c,0x5ef7f9d0)}, {LL(0x62d3d3cb,0x3c424ba8),LL(0x7c3c3bba,0xab155cc2),LL(0x4bae8070,0x6a0404b2),LL(0xdf36677c,0x5f3d0592),LL(0x9a0e4800,0x7c9f2bef),LL(0x97959a09,0x6babbcb4)}},
   {{LL(0x17eb9264,0xa7342f95),LL(0x0a8a6eef,0x9264a6a0),LL(0x7471c384,0x50e48bf0),LL(0x30827f34,0x729e5ab1),LL(0xea779c23,0x17199191),LL(0x9fa9fd58,0xd13ab853)}, {LL(0x3b1d773e,0x7d579937),LL(0xd196c3df,0x65f8e7c6),LL(0xe8541725,0x253f7d51),LL(0xec720355,0x107a793d),LL(0x6aa16268,0x1c14d056),LL(0x8bbb231b,0x9dc5fca3)}},
   {{LL(0x897a05a4,0x69935431),LL(0x51eaa290,0x34397b68),LL(0xf58fb7a5,0x90ec1a37),LL(0x50c4d76b,0xf10d0783),LL(0xcc47f990,0xeb9db48c),LL(0x0cea5865,0xef0b97a2)}, {LL(0xd9f94396,0x04708d6f),LL(0x41c21452,0x82ff5771),LL(0xdbb65bdd,0x772d8493),LL(0xddf73c8c,0x561abc8b),LL(0x9830ff05,0x98a56463),LL(0x9f0d4cad,0x73e28296)}},
   {{LL(0x589cb234,0x7fa7064e),LL(0x65ca4f3c,0xdfa4e846),LL(0x792d5254,0x476b6618),LL(0x583bdaeb,0xc0ce93bc),LL(0xe4ab5dc9,0x30b11dac),LL(0xf5c89e2f,0x237a64e5)}, {LL(0x7dba60b7,0x54339fad),LL(0x084b09ed,0x0072505a),LL(0xb140717a,0x5e89c81f),LL(0xb407595d,0xd56de3a2),LL(0x0aab0d25,0x9f3a6c42),LL(0xa685543d,0x6c2aa69d)}},
   {{LL(0x6cd00d89,0xda5fcddc),LL(0xac9ef99d,0x0b8c3feb),LL(0xa19298e3,0x16569ca8),LL(0x2783ac1d,0x5d998d56),LL(0xc18ecbdf,0x298c681c),LL(0xe6e13de4,0x209323d1)}, {LL(0xbd75118d,0xe064255e),LL(0x43f945ef,0x78f69fd3),LL(0x95e6ff8f,0x9a4d591f),LL(0x3b378848,0xfa621d46),LL(0xe4d12bb8,0x4c951f31),LL(0x00ebfea6,0xefe8d01b)}},
   {{LL(0x12b09f53,0xf5689c5e),LL(0x9e87ff7d,0xc1da32e1),LL(0x12eaa533,0x1af879d0),LL(0xd9271e94,0xdba775e6),LL(0x10e63c34,0x60f85073),LL(0xa686a24b,0x445f3e21)}, {LL(0x15bc277e,0xed5ca8fa),LL(0x364ab7ab,0x9839198a),LL(0x6d90a7d4,0xe2ee3942),LL(0xccd37e76,0xe5b3e4cb),LL(0xf1412e0f,0x9013bd08),LL(0xce999048,0x82f5c532)}},
   {{LL(0xf4bbf123,0xdd7ff816),LL(0xde3e9923,0xbf1a5fd0),LL(0x68d10f94,0x8c388f8c),LL(0x45057388,0x4e9cf0c4),LL(0x8d010855,0xcff64aa3),LL(0x7d8f55fc,0x4b639596)}, {LL(0xbddc00c5,0xca314522),LL(0x3da89cfa,0x95482d72),LL(0x9eb9a710,0xb9bf18b8),LL(0x4651cc5d,0x0936a88e),LL(0xf59d0f45,0xec20ac01),LL(0x8ba74374,0x011a4868)}},
   {{LL(0xf9c4caf8,0x07252272),LL(0x290ab63f,0x9b016799),LL(0x558d649d,0xeaa616bb),LL(0xa66d8089,0x00f2ef38),LL(0xf72863ae,0x284b0146),LL(0x1968cf45,0x9a207d77)}, {LL(0xbdcbb689,0x33d7bac8),LL(0xe5348dae,0x393f34d5),LL(0x6f524620,0xeb86c8f1),LL(0x610689f8,0x62500c62),LL(0x6b7fa65e,0x66febc05),LL(0xfb836b3e,0x39c8a70b)}},
   {{LL(0x5e4d0351,0xb91977b5),LL(0x9e8dddf7,0xd8ed39aa),LL(0x9d1b25c3,0x9ae994c2),LL(0x6ca7b19b,0x7369e189),LL(0xec0d7c2c,0x33deb695),LL(0xede6435b,0xddcc6250)}, {LL(0x145a654d,0x44b7ba23),LL(0xd280567e,0x653ee81a),LL(0x0a39d324,0x7694c972),LL(0xe97e1710,0xf0af25b1),LL(0x3ee1a076,0x6e154646),LL(0x7ccfde8c,0x062ce983)}},
   {{LL(0xf3fe3441,0x61d0e01b),LL(0x2af47609,0x674e5233),LL(0xb362902d,0xd4a4e224),LL(0x9e0a5d16,0x45923c12),LL(0x95e580e9,0x4fc2bdd4),LL(0xa8c3d954,0x6d1d974c)}, {LL(0xd0bbeaaa,0xaeff1135),LL(0x1baafc9e,0x013ab5b3),LL(0xab8f9f31,0x80907d3e),LL(0x6d566c15,0xaf2c1216),LL(0x952e6fa7,0x0082daba),LL(0x2df9e03a,0xa4671003)}},
},
/* digit=51 base_pwr=2^255 */
{
   {{LL(0x1fd64063,0x5f79f0df),LL(0x81e118ec,0xd2d39dd3),LL(0x11571c5b,0xd631a68e),LL(0x2474faf7,0x6d072b4e),LL(0x862a924a,0x5e043a6d),LL(0xb0fc8d7a,0xcae58bd8)}, {LL(0xb1351f28,0xf54bb7f3),LL(0x0413275e,0x4588b628),LL(0x5909ec04,0x81459f4c),LL(0xabd16460,0xd28cda25),LL(0x8db1c69e,0xbb676d01),LL(0xac5036f4,0xc0056e2d)}},
   {{LL(0x44ce3ad8,0xdbe04c30),LL(0x4ce8aad5,0x995fbb1b),LL(0x70911457,0xdbf8b546),LL(0x3f7a1757,0x9e683b5b),LL(0x9c7bd62c,0x7b89a08a),LL(0x0b3fc97e,0x448865a4)}, {LL(0x3bb01e94,0x0ac9abfc),LL(0x1e756124,0xa0776042),LL(0xd9deed97,0x0aa6c335),LL(0x72603e08,0xe270580f),LL(0x6c783bb2,0x70857a94),LL(0xcaa929ae,0xa0047774)}},
   {{LL(0xb25c4d4a,0x292f8874),LL(0x7e79f526,0x54961fd8),LL(0x008c6ec9,0x949a1fae),LL(0x525524fd,0x6ae82f0d),LL(0x2edbcb1a,0xd1f6f4ef),LL(0x977ddffb,0x41617a6d)}, {LL(0x1baf0668,0x6ae38fb7),LL(0xd538ab3c,0xa79ea228),LL(0xfc44e273,0x70babb05),LL(0xbca85910,0x247384fb),LL(0x6a564959,0xdc0e069b),LL(0x1a7438ad,0x37a9c552)}},
   {{LL(0x8fa06859,0xf071c987),LL(0x1a52390b,0x0083e531),LL(0x61483bc2,0x845eb12a),LL(0x1caf6dd6,0x17471d80),LL(0xddc21b92,0x7b603616),LL(0xb992536d,0xd38fe0f6)}, {LL(0x297c25a4,0x433f0652),LL(0xb1c4bf41,0x03d4d8fc),LL(0xa9adf49b,0xdf617386),LL(0x2cb2944f,0x4bfeb399),LL(0xb3d9c076,0xbf288427),LL(0x965b4576,0x17818c3e)}},
   {{LL(0x689f43ef,0x2bb798ff),LL(0x5f26ec54,0x5813e441),LL(0x5005c929,0x51f64c49),LL(0x4b42e417,0x60e213a5),LL(0x62cc3734,0xc1528442),LL(0x09d994e1,0x6ecd6c3b)}, {LL(0x83dd047f,0xa6e72f71),LL(0xb0019803,0x3836f663),LL(0x257493cb,0xbcf1265b),LL(0x9e62d78b,0x59b15ff0),LL(0x6cb92ecf,0xaac5ed5b),LL(0x9662651c,0x37e6ad7d)}},
   {{LL(0xa17e560c,0xce23a19a),LL(0x62550e2b,0x6491b95f),LL(0x1d15a005,0xc7200012),LL(0xf4355a1f,0x15fde735),LL(0x607f7807,0x3849761f),LL(0x18204691,0xcbe322d0)}, {LL(0xa95e8e91,0x75756e4e),LL(0x817a9b8e,0x365959fe),LL(0x3d4ce3dc,0x63123276),LL(0xf1d66e00,0xa769d2fe),LL(0xc28829e6,0x8624ddba),LL(0xd2df06ef,0x03274297)}},
   {{LL(0x16f01d7b,0x52fced31),LL(0xdaf046aa,0x88c6b172),LL(0xfe7a338e,0x1a189403),LL(0x61798b1e,0x39741ecd),LL(0x2934b879,0x6a47b071),LL(0x828d1e9d,0x3b1a5dd1)}, {LL(0x7f35a7ef,0xd4bd4848),LL(0xc1eebaf8,0x71774b5b),LL(0xd55344ba,0xa86471e5),LL(0x7b8a483a,0xfbf145f1),LL(0xaa53802c,0x70f9b214),LL(0x10b066e1,0x995af930)}},
   {{LL(0x27a28f9d,0x9ec11597),LL(0xb847cd83,0x96f2c44b),LL(0x31fca111,0xacf794e1),LL(0x96076f45,0x438b9178),LL(0x51732588,0xad71035b),LL(0xa5d910da,0x2db32f32)}, {LL(0xfe1cc184,0xefaad0e8),LL(0x2e00bbed,0x6f0360b5),LL(0x474ce326,0x99402426),LL(0x2aa270da,0xd53b687a),LL(0xd78fa6eb,0x96c8bb78),LL(0x6e699411,0xd07f3bba)}},
   {{LL(0x361dae4c,0x07276886),LL(0x84c57896,0xb26f2579),LL(0x5f0c2bf4,0xe8b59ef5),LL(0xdd64aaac,0xf544c145),LL(0x9fdcb039,0xa9dc03bb),LL(0x58c44b72,0xeb1e9aeb)}, {LL(0x1d6bf26e,0x39a4fe8e),LL(0xeaf8e241,0x2c4285cc),LL(0x14770f31,0x647ce584),LL(0x062f0912,0xe5bfeb70),LL(0xe8314467,0x7031db80),LL(0xd658d2cd,0xac970937)}},
   {{LL(0xefe1757b,0x2dfc39e6),LL(0xb5d2cb93,0xab52fbf3),LL(0x313aa477,0x1cf12123),LL(0x9c6acbd1,0x785b025b),LL(0xe4d54177,0xb4aeb5b8),LL(0xd943c1ea,0xde3d28f8)}, {LL(0x5b0b1921,0x7892db85),LL(0x58caff2d,0xc09ff903),LL(0x1cbd3231,0xbdee13c6),LL(0x00b6c34e,0x873e0a77),LL(0xe23de32b,0xb279505a),LL(0xc5a03302,0xf056ffdc)}},
   {{LL(0x6ba563ee,0x0ac76959),LL(0xb9a51868,0x2053a5fb),LL(0x06178e3c,0x60e2555f),LL(0xc0933775,0x1bc99e72),LL(0xb7d5160e,0x4a2d31b2),LL(0x895d1db8,0x7cc03ff3)}, {LL(0xee53e79f,0x98fb331e),LL(0x8583b893,0xc2f0a93f),LL(0x49802eb4,0x8fa79ed8),LL(0x55ce8dcd,0xe9548175),LL(0x1d4d44f1,0xb5155a21),LL(0x89570a82,0xbb044d28)}},
   {{LL(0xe1f833e0,0x430b669f),LL(0x6d8127bd,0x1797ac3a),LL(0x4c33493e,0x01ad730c),LL(0xca00ed39,0x8c882c1f),LL(0xd24a5516,0xab2e9c89),LL(0xf5d0327d,0x21a49e0a)}, {LL(0x58a280d1,0x46488bd8),LL(0x772ed759,0xfc4a1e8a),LL(0xaecfac7c,0xf9f60e90),LL(0xac6a9e8c,0x4afd1f5f),LL(0xc98bcda8,0x57a20bf8),LL(0x8b46b998,0xa1107ea0)}},
   {{LL(0x45fc08c4,0xe931c21b),LL(0x3aca653f,0x13fad41f),LL(0xec1fe395,0xf28a3515),LL(0xcf55cc91,0x191a4f2d),LL(0x90535f59,0x46fb07b7),LL(0xd69686a9,0x76c60cc1)}, {LL(0xc8122c04,0x1e7dec15),LL(0x1b72798c,0x0743fcf0),LL(0x1e5939fd,0xb10c96bd),LL(0x6e8338e5,0x261e6c0a),LL(0xe5cdaa43,0x56148d3c),LL(0xa031239a,0x3f004371)}},
   {{LL(0x0db0c9c0,0x1f2f6c1f),LL(0xac770b5f,0xca4e1964),LL(0x569c090b,0x20dce4a6),LL(0xdb3986d0,0xe4f7401a),LL(0x85f553ea,0x13897994),LL(0x09026bff,0x119ad50d)}, {LL(0xe265a0a9,0x88067aa7),LL(0x7f749167,0x209c62b4),LL(0x459bf269,0x84165019),LL(0x8b6e76f6,0xec125bd5),LL(0x1f0d2434,0xd636f932),LL(0x8e05e0b0,0xf6023a4c)}},
   {{LL(0x65ea2b23,0x84c14e31),LL(0x19e93301,0xb68121be),LL(0xb25d9a83,0x9c1a873f),LL(0xd13773f0,0xafb9d04c),LL(0x51c32d57,0xd05014da),LL(0x0904efdc,0xff2e350b)}, {LL(0xa0a01069,0x20bd7d22),LL(0x6c3b3fd8,0x3e74eb0f),LL(0xb743b72c,0x643ea531),LL(0x8e3ae785,0xa2fe7414),LL(0x92e8d320,0xb8fe89bf),LL(0x495f6d28,0x58985dc1)}},
   {{LL(0x389283ba,0xfe1f11ad),LL(0x0cd91b22,0xc87e20b6),LL(0x3c5babf8,0x99d0015a),LL(0x5929ea0a,0x7e795b4d),LL(0x1dfb7b7e,0xc9cf6833),LL(0xa64992e8,0xc1c07346)}, {LL(0x9889746d,0x0b7e0dd8),LL(0x1c43ea4a,0xa89d7b46),LL(0x34f02b96,0x64023cf0),LL(0x5662f0c8,0xf7dd410a),LL(0xa1058cca,0xa3bb6088),LL(0x4e7801ed,0xedb25dc3)}},
},
/* digit=52 base_pwr=2^260 */
{
   {{LL(0xdd93d50a,0x140a0f9f),LL(0x83b7abac,0x4799ffde),LL(0x04a1f742,0x78ff7c23),LL(0x195ba34e,0xc0568f51),LL(0x3b7f78b4,0xe9718360),LL(0xf9efaa53,0x9cfd1ff1)}, {LL(0xbb06022e,0xe924d2c5),LL(0xfaa2af6d,0x9987fa86),LL(0x6ee37e0f,0x4b12e73f),LL(0x5e5a1dde,0x1836fdfa),LL(0x9dcd6416,0x7f1b9225),LL(0x677544d8,0xcb2c1b4d)}},
   {{LL(0xda1c29ab,0x279fd119),LL(0x2b30d40c,0xbd068802),LL(0xda44105d,0xd8f57da4),LL(0x28223fe1,0xb1814b7a),LL(0xe06f2d2e,0xcf2fd241),LL(0x01dfde06,0x99003a02)}, {LL(0xfded7e4b,0x876a31af),LL(0x2f725094,0x1efaf827),LL(0x493a6a0a,0x5117d608),LL(0xa88c03e7,0xdcec8088),LL(0xea916897,0xeae1d352),LL(0x6e8b2c57,0x8cdc2810)}},
   {{LL(0x9c213d95,0x0254486d),LL(0xcb2f6e94,0x68a9db56),LL(0x000f5491,0xfb5858ba),LL(0x34009fb6,0x1315bdd9),LL(0xc42bde30,0xb18a8e0a),LL(0xf1070358,0xfdcf93d1)}, {LL(0x3022937e,0xbeb1db75),LL(0xcac20db4,0x9b9eca7a),LL(0xe4122b20,0x152214d4),LL(0xaabccc7b,0xd3e673f2),LL(0xaed07571,0x94c50f64),LL(0xe66b4f17,0xd767059a)}},
   {{LL(0x54e93c1e,0x09f8bb06),LL(0xad81e27c,0xb0045884),LL(0x076e13eb,0x26ebc7b6),LL(0x5d5ac07f,0xbda0b553),LL(0x48ab69e6,0xbcb81322),LL(0x1c0f21fa,0xd3847d2e)}, {LL(0xc834d740,0x7a466528),LL(0xe0823ff2,0x6c67a79a),LL(0x4c1d7cb8,0x85dd1186),LL(0x2d081301,0x096f849f),LL(0x8a5ea0f0,0xb4f503dd),LL(0xd1bf69b2,0x71ee0889)}},
   {{LL(0xdcd6d14b,0x40336b12),LL(0xe3b4919c,0xf6bcff5d),LL(0x9c841f0c,0xc337048d),LL(0x1d617f50,0x4ce6d025),LL(0x8117d379,0x00fef219),LL(0xf95be243,0x18b7c4e9)}, {LL(0x38df08ff,0x98de119e),LL(0x8d772d20,0xdfd803bd),LL(0x0f9678bd,0x94125b72),LL(0x334ace30,0xfc5b57cd),LL(0xb7e86e04,0x09486527),LL(0x6e552039,0xfe9f8bcc)}},
   {{LL(0x02a2d1e6,0x4ab7a22c),LL(0x1371d5a4,0x967e19a3),LL(0x078de336,0x20f59f95),LL(0xf7869245,0xfd28fa36),LL(0xcbf1d96f,0x1de42581),LL(0x366e1f0f,0x2e0127d7)}, {LL(0x2258c741,0xbc65fa9d),LL(0xdd6d65f8,0x1f2f3356),LL(0x4a0822a9,0x06384f3a),LL(0xfd05a0aa,0x1c81332b),LL(0xd95ee3ce,0xbfb12361),LL(0x42016d00,0x180aaf06)}},
   {{LL(0xd6f5a10e,0x3b75c45b),LL(0xc1c35f38,0xfd4680f4),LL(0xf8e0a113,0x5450227d),LL(0x73ddba24,0x5e69f1ae),LL(0x57f24645,0x2007b80e),LL(0x3d159741,0xc63695dc)}, {LL(0x4530f623,0xcbe54d29),LL(0x2869586b,0x986ad573),LL(0x4cc39f73,0xe19f7059),LL(0x2b1b8da9,0x80f00ab3),LL(0x73f68d26,0xb765aaf9),LL(0xe993f829,0xbc79a394)}},
   {{LL(0x95b3287d,0x0a159f62),LL(0x48cecad0,0xb18f8759),LL(0x1661a23f,0x6d1ab8ee),LL(0xc95c41b3,0xcae7f40e),LL(0x7c51eb56,0xbc3d2040),LL(0xe8754250,0xa7527283)}, {LL(0x1f9e668a,0x81561056),LL(0x900f5912,0xb8aa7296),LL(0x6af2a00c,0xabdbc1bf),LL(0x2d0a56c0,0xe9a94254),LL(0x7bc8959e,0x4774a7b7),LL(0x19cef2f3,0x0a837ff0)}},
   {{LL(0xf310d2a0,0x9c441043),LL(0xdc5eb106,0x2865ee58),LL(0x9cb8065c,0x71a95922),LL(0xa052af0f,0x8eb3a733),LL(0xb09d716e,0x56009f42),LL(0xabcbe6ad,0xa7f923c5)}, {LL(0xfa375c01,0x263b7669),LL(0x21ef27a2,0x641c47e5),LL(0xb08ffd25,0xa89b474e),LL(0xf0a239f3,0x5be8ec3f),LL(0x242a6c5a,0x0e79957a),LL(0x0c6c75f5,0x1dfb26d0)}},
   {{LL(0xa084fae6,0x36f3a3d8),LL(0x9a9b0d95,0x75983589),LL(0xcc80fcb6,0x70722186),LL(0x96d84c04,0xf28ed0c7),LL(0xffb63f90,0x95a32263),LL(0x98766034,0xdd7d60a0)}, {LL(0x1d5c387c,0xe193a31f),LL(0xb8310f8b,0x6c5eca7e),LL(0xc083ff47,0xfe61d523),LL(0xcb2944e9,0x90c832db),LL(0x593334b7,0xa9f3f293),LL(0x2d7d1c33,0xe6cde2e1)}},
   {{LL(0x9dfbf22a,0x2fd97b9b),LL(0x5643532d,0xdec16cc8),LL(0x60fee7c3,0xdf0e6e39),LL(0x545860c8,0xd09ad7b6),LL(0x73fc3b7c,0xcc16e984),LL(0x0d4e1555,0x6ce734c1)}, {LL(0x4b5f6032,0xc6efe68b),LL(0x14f54073,0x3a64f34c),LL(0xac44dc95,0x25da689c),LL(0x5358ad8a,0x990c477e),LL(0xf36da7de,0x00e958a5),LL(0xc9b6f161,0x902b7360)}},
   {{LL(0xf144b6cc,0xbd079cf1),LL(0xb4f4a764,0x7f86e29b),LL(0xf21f9cbf,0x5b08b290),LL(0x75e3aeb9,0xada0c85b),LL(0x6666c2df,0xd0789f8b),LL(0xd71ec2ec,0xcf5d8a8c)}, {LL(0xe7e4364b,0x6f7780c3),LL(0x85d2eb75,0xdd9a6529),LL(0xd952a38e,0x8222f66b),LL(0x27260a29,0x9dd5f7eb),LL(0x57947178,0xce49b344),LL(0xcdda7e39,0xaa215f82)}},
   {{LL(0x9347b90a,0x454ab42c),LL(0xa698b02b,0xcaebe64a),LL(0xfb86fa40,0x119cdc69),LL(0xc3109281,0x2e5cb7ad),LL(0xcd0c3d00,0x67bb1ec5),LL(0x83f25bbf,0x5d430bc7)}, {LL(0x5cde0abb,0x69fd84a8),LL(0x9816b688,0x69da263e),LL(0x0e53cbb8,0xe52d93df),LL(0xadd2d5a7,0x42cf6f25),LL(0xc87ca88f,0x227ba59d),LL(0xda738554,0x7a1ca876)}},
   {{LL(0x3004db31,0xaa44b286),LL(0xd43e4430,0x86f43d7a),LL(0xb0b0240d,0xdc4874cd),LL(0xadc45a06,0x79986a23),LL(0x3cee4631,0xbb275b44),LL(0x63a217aa,0x21daee8a)}, {LL(0xd7b25c02,0x1e7c5397),LL(0xc5e668fa,0xe677d3cb),LL(0xed51b4bf,0xc7c84e28),LL(0x923e5408,0x7ca19e99),LL(0xc3f832e7,0xc6f8a595),LL(0x5fb049a3,0x2d0a789c)}},
   {{LL(0x1cac82c4,0x3fa5c105),LL(0x8a78c9be,0x23c76087),LL(0x1c5cfa42,0xe98cdad6),LL(0x0a6c0421,0x09c30252),LL(0x42fc61b9,0x149bac7c),LL(0x3004a3e2,0x3a1c22ac)}, {LL(0x202c7fed,0xde6b0d6e),LL(0xe7e63052,0xb2457377),LL(0x3706b3ef,0x31725fd4),LL(0x2b1afdbf,0xe16a347d),LL(0x8c29cf66,0xbe4850c4),LL(0x2939f23c,0x8f51cc4d)}},
   {{LL(0x44922386,0x114a25c8),LL(0x6d4e8b57,0xdd084d44),LL(0x1e7bd7de,0xc49b6841),LL(0xd6da54db,0x5b0359fa),LL(0x3f0da321,0xa6e6e5f9),LL(0xd640a87e,0xb65ec55c)}, {LL(0xae64020e,0xc1a4f6ce),LL(0x088e1337,0x91e29cd2),LL(0x3c0a631c,0xf44ceb8e),LL(0xb756445f,0x0205b11d),LL(0x5bc8880e,0x04844e84),LL(0xb85e00d3,0xb630ddc0)}},
},
/* digit=53 base_pwr=2^265 */
{
   {{LL(0xae7ce296,0xe76894c3),LL(0xa6cafc34,0x87737ee2),LL(0xe55cd1e6,0x566dfcfb),LL(0x3a7ad5b9,0x5421a9f2),LL(0x4687a4ef,0xa005838a),LL(0x23a2c423,0x3837219a)}, {LL(0x8a82cd1b,0x4b780012),LL(0xc728b588,0x401c07be),LL(0x37ced8f3,0x2b5f69e9),LL(0x8c1e1eaa,0x306b621d),LL(0xd389cc4d,0x8acbbe71),LL(0xf4ab7774,0x922fa665)}},
   {{LL(0xd35c2d80,0x2df6f242),LL(0x3493ce97,0xf65a99a9),LL(0x372bcc87,0x9e80232b),LL(0x6e428cc5,0x26ba13b8),LL(0x13a1b763,0x2526ef1f),LL(0xdc97c5f3,0xcef3edcd)}, {LL(0xbde16b73,0x4954867f),LL(0x368ff6cb,0x9817813d),LL(0xbe143027,0x7e39fa69),LL(0xcf54f28b,0x12329463),LL(0x7597c2da,0xcf0991dc),LL(0x52e07099,0x0cda3969)}},
   {{LL(0xee993749,0xdc9daeef),LL(0x7ff3d775,0xd40c8a54),LL(0xca53d53d,0x206d2271),LL(0x4d1aa50b,0xb0546335),LL(0x88001e99,0x52bca910),LL(0x17fb7aa4,0x25c117c2)}, {LL(0x42685945,0xfff3af95),LL(0x8f4ce0fb,0x083fd4de),LL(0x24753989,0xaac004be),LL(0xe90950c6,0x4a5de2b6),LL(0x738efe5b,0xb46af0ab),LL(0xdb4459f9,0xcf80a17f)}},
   {{LL(0xf303955d,0x412f64a3),LL(0xbd692593,0xe92bdca9),LL(0xc2e964e0,0xfbe6cdc2),LL(0x0011cb01,0xe9a3b1fd),LL(0xcf228f23,0x6c30762d),LL(0xbe9199a1,0x1270b84a)}, {LL(0xe3c9cbb1,0x732711df),LL(0xd91d9513,0xa3aabe37),LL(0xc6eceba7,0x8ee08ba0),LL(0xf3c3d31d,0xb1711531),LL(0x3c716948,0x65060b63),LL(0x2ff2cadd,0x046b4ea1)}},
   {{LL(0x961719fe,0xdeb7a462),LL(0x76ba6ed7,0x5e22796d),LL(0x25d22208,0x5907daf2),LL(0x2c21c04c,0xd98260a7),LL(0xe090f349,0x56b24923),LL(0xa31a8f95,0xb5960ad9)}, {LL(0xf8b1da62,0xb6ac57ce),LL(0x9f14f70f,0xe47d0995),LL(0x04dba20b,0x0ba6eb01),LL(0x46b60e27,0x41ca71c8),LL(0x6bf5eeef,0x2bbfdb30),LL(0xf075b238,0x58e16788)}},
   {{LL(0xbab220c6,0x25d1124f),LL(0x61524e3e,0xcd1423c8),LL(0x0434fb51,0x75e4f45f),LL(0x5180ab2b,0xb5180a8f),LL(0x5b22e388,0x144e214e),LL(0x92263054,0x6b16dad1)}, {LL(0x40863566,0x3ea75907),LL(0xdada3b46,0x372d5abd),LL(0x893d210f,0xb3ff5a3a),LL(0x5e29f3dc,0x39f8d1ce),LL(0x68200e82,0x559186ce),LL(0x1202cb66,0xf4876454)}},
   {{LL(0xeaf4f2a1,0x699c2db9),LL(0x61c0c17c,0x8cd33227),LL(0x64f16a56,0x971b50d4),LL(0x102bbe10,0xcd00d42b),LL(0xb05f3cac,0x928d0ae2),LL(0x14bcf472,0x245dbe38)}, {LL(0x947c0184,0x43d29526),LL(0x4612a4ea,0x24089968),LL(0x6c2b4541,0xbc763fcd),LL(0xf82448fa,0xb4e7ae0f),LL(0xb02b6459,0x94f1fa15),LL(0x67d39bbd,0xbcafa1ec)}},
   {{LL(0x8b540904,0x6f178dbf),LL(0x8720472a,0x0264bccd),LL(0x59b46611,0xa6e8b4b4),LL(0xc72b4a58,0xafce8267),LL(0xa45985ad,0x21142175),LL(0xe649d733,0xd23401df)}, {LL(0x85dc7720,0x6bf42fe0),LL(0x40e3f2f5,0xc5c8ab94),LL(0xcd029197,0xb0c8a58a),LL(0x215492e1,0xa73ff329),LL(0xb1b5a5f0,0x895c545e),LL(0x6fcaf49b,0x6dbc2445)}},
   {{LL(0x44cc852a,0x705ad3d8),LL(0xb80518e0,0x10bb5add),LL(0x0de9f160,0xa34905bb),LL(0x94b1aacd,0x7d45aae3),LL(0x7fd5de7a,0xd30411d0),LL(0x4d0167cd,0xe2fc6206)}, {LL(0x10ce71cf,0xd3b19612),LL(0xb28225fd,0x8e8096b6),LL(0x64a1c849,0x4b46fa80),LL(0xa51364e9,0x160479ac),LL(0xfaa3f0f2,0x9ebb6fdb),LL(0xf1511754,0x9ce029b9)}},
   {{LL(0x0f2d76a3,0x25ef32d6),LL(0xaf4a7d46,0x540650b9),LL(0xd991d7f4,0x8979a4b8),LL(0x99202400,0xdaa706c2),LL(0xf19d281d,0x8a729680),LL(0x4ec44de2,0xde25bdc4)}, {LL(0xc2054496,0x0fc50832),LL(0x0aaf2941,0xfee72fb6),LL(0xb82ed4f0,0xc8263e64),LL(0x6f49055c,0x91a8cb73),LL(0xf2bb515c,0xb7585458),LL(0xb855e6c6,0x03d2b23a)}},
   {{LL(0xa0879d68,0x8e11e8e8),LL(0xbf7a84ac,0x2bea77a3),LL(0xa74b45d0,0x98140930),LL(0x810e587c,0x1ce28654),LL(0x8869daac,0x0a30756b),LL(0x39d2fe12,0xbf5e8245)}, {LL(0xcac16a87,0xe6414992),LL(0x437aff7b,0xd2fa182d),LL(0xb6146094,0x4e61412d),LL(0x2f31bb4b,0x30a949ec),LL(0x22dc8ac6,0xf254c71f),LL(0x1ab2a0bd,0x1d9ed85c)}},
   {{LL(0xcfd3f182,0x09ec1e3d),LL(0xadae7af9,0x1f1c30b5),LL(0x6b454164,0xf3a33f7c),LL(0x94647c4f,0x0318926f),LL(0x87db14ec,0x8e37bdd7),LL(0x2ab364d3,0x811cbd34)}, {LL(0x7c2b369d,0x1dd1e507),LL(0xa28056bd,0x7a57bc46),LL(0x089efe44,0xfca5be4b),LL(0x6dc1290e,0xb3bd84d7),LL(0x8793e6ae,0x40d7af09),LL(0xa3723942,0x4e08e11f)}},
   {{LL(0xeb73ec7d,0x649eeabb),LL(0x26a9aaf9,0x10983304),LL(0x4e296235,0xf22e4514),LL(0x7b85f801,0x695c8df4),LL(0x3ae7caaa,0xd4553344),LL(0xffbba90e,0x3e35bd47)}, {LL(0x5d13b9ec,0xdd04f7c9),LL(0xe259f70d,0xe39a5d12),LL(0x201ae17c,0x39073063),LL(0x6a85435f,0xdba3eda0),LL(0xe948924a,0x6df48093),LL(0x00e3394f,0x9ffc4dcf)}},
   {{LL(0x899ffebb,0xca3709ad),LL(0x77c00602,0x1a873778),LL(0xa99b4af0,0x5ff40c2a),LL(0xa80e870c,0x680464e5),LL(0x94e10b1d,0xd2f7f044),LL(0x4e9aa1a7,0xee9b206c)}, {LL(0x96cbe950,0xb536d675),LL(0x9e8305f3,0x84185689),LL(0x369fa255,0xae1b669c),LL(0x7233e1ea,0x62e26026),LL(0x6aa60c24,0xac05c513),LL(0xd2691677,0xdfc6814f)}},
   {{LL(0x397147d6,0xbe414528),LL(0x42592203,0x74851314),LL(0x0364b0eb,0x9084d330),LL(0x6ad70814,0xf8e5d6d5),LL(0xffb4ac5f,0xfcd4e0e3),LL(0x1fbf8899,0xf652417a)}, {LL(0xccbd7eaa,0xb1165da7),LL(0x6e2d4e8c,0xf5dbd11a),LL(0x32ddcea8,0x5dab120a),LL(0x9892f728,0x30aaa56f),LL(0xd3d73838,0x71c2412d),LL(0xd2e2becb,0xbc0253d1)}},
   {{LL(0x0a02b0fb,0x8baef5df),LL(0xc2b92b02,0x58a2b06b),LL(0x54c8267a,0x268558d7),LL(0xccf70393,0xf924f795),LL(0xf68ee021,0xe3763f30),LL(0x5c01ba4b,0xc1e856f0)}, {LL(0x722b6bff,0xcc01a3e9),LL(0xed5b3b02,0xd2be4623),LL(0x6c45e33f,0x1ab3512e),LL(0x4ef433f6,0xa978fe48),LL(0x8e21f5af,0x23e2ea01),LL(0x11524a40,0x49647d88)}},
},
/* digit=54 base_pwr=2^270 */
{
   {{LL(0xd087b788,0xe1d42d94),LL(0xba0e176a,0xfbfb221a),LL(0x83686966,0x5f6698e7),LL(0x74a30dbf,0xbb5e1594),LL(0xcfd20230,0xef86bb5b),LL(0x403b8f8b,0xf055a1c5)}, {LL(0xd9d85ea7,0xf249aac8),LL(0x3d200198,0x7318f7bc),LL(0xefca9a90,0x3b80960c),LL(0x8f449c4b,0xf28e3388),LL(0xf0cfe09e,0x0cdfc61b),LL(0x8b22cd26,0x3b169c63)}},
   {{LL(0x203bb368,0x923e948c),LL(0x231a80e0,0x58e37a2b),LL(0x6df27deb,0x345a011a),LL(0xd57f4ca2,0xba6784c1),LL(0x114196e9,0xf01b3703),LL(0x1aab426d,0x981a63eb)}, {LL(0x51770c1c,0x2ffdc978),LL(0xefa722fc,0xddd19da6),LL(0x16f09c1e,0x5ca1c012),LL(0x5b9cc0b6,0x612021de),LL(0x5e150569,0x910e10e9),LL(0xe2ab93ea,0xacace9dc)}},
   {{LL(0x5070e0a9,0x0cdd8372),LL(0xec550783,0x7c5ad562),LL(0x4f3b8d2b,0x9652b847),LL(0xe6e98d73,0xfdd60d93),LL(0xa3479d0b,0xd51cae2c),LL(0xee05c006,0x11b93b6d)}, {LL(0x8a3b40d5,0x9d72b82d),LL(0xa7d24855,0xc6e996fe),LL(0x398603de,0x420672f7),LL(0x9a1af2ce,0xd551b34a),LL(0x13bdce0c,0xdeb8c1d9),LL(0xebbeba7a,0x56ca926d)}},
   {{LL(0x9db9ca19,0xac58c9e0),LL(0x390054d0,0xd308ea5d),LL(0x2cc42529,0x32ef4afc),LL(0x97c2bdf9,0x08bd48b3),LL(0xa849e19a,0xac8a7803),LL(0x75c31496,0xcd51c0da)}, {LL(0xf0e2d49f,0x733dc7de),LL(0xb44b8cc5,0x7c9caad1),LL(0x47be21a8,0x6d9c5b08),LL(0x5ebf426f,0xfab0fdc5),LL(0xf94e9e5b,0xd60748ca),LL(0x69366089,0x3072e592)}},
   {{LL(0x007d3d4d,0x8f24cfd9),LL(0xcb27ae66,0x2f4f7195),LL(0x9c1ed35e,0x8166162e),LL(0x4f4bdfd8,0xfdcf7666),LL(0x3b5373ca,0x6dc993bc),LL(0xa3ec42d5,0x2faa34d6)}, {LL(0xc6b6109e,0x1cce84cb),LL(0x43632b7e,0x7a4178ad),LL(0xb2aab55b,0x6899adb5),LL(0x683c1397,0x080e42bd),LL(0x10960bfe,0x34d5e192),LL(0x7223406a,0x65415d21)}},
   {{LL(0x6254954d,0x8acd0a3e),LL(0x1d8c2442,0x708f9359),LL(0x6dab2ff9,0x85f59135),LL(0x33a9d96a,0x9df92007),LL(0xcaa797ad,0xac9849a7),LL(0xfe95aa38,0xcea2067f)}, {LL(0x995cb879,0x5a3c21b4),LL(0x69d4f07d,0x5c981a5d),LL(0xf0dc9aa6,0x028cf95d),LL(0xc68fd96a,0xcc5cee93),LL(0x0b69676f,0x8a5e73ee),LL(0x832d230b,0x190a7229)}},
   {{LL(0xd40c5d9f,0xc0ab244b),LL(0xf3ef03ed,0xc05a11cf),LL(0x650ea714,0x1593d8ad),LL(0xfc6a1235,0x1edb6cbd),LL(0x4e451d0b,0xead76e1e),LL(0xae7ee558,0x4759e3c0)}, {LL(0x7c8145e2,0x480627be),LL(0x88a339a2,0x82cec2c7),LL(0x752b0f58,0x17e887b6),LL(0xaa6c9df1,0x91c866d0),LL(0x29b3b1d3,0xd9a54848),LL(0xf641bbc1,0x17ae47ae)}},
   {{LL(0xf0ef77c5,0xb06c17a3),LL(0x6df6bf59,0xc144e784),LL(0x0038aeb2,0x2440ae99),LL(0x58b402ca,0x83bf711b),LL(0xb577732a,0xb8763e00),LL(0xf651a932,0x509e91ef)}, {LL(0x00ac109e,0xbe02ab9d),LL(0x8dfd78f1,0xfbcb426c),LL(0x4283f80f,0x7ed272f6),LL(0x2365da5e,0x098cf057),LL(0x05dc6beb,0xd90e6f18),LL(0xcf7b9d72,0x09ef177f)}},
   {{LL(0x199785d5,0xebee13d1),LL(0x92f8f141,0x3d370648),LL(0x15e3a026,0x56595c83),LL(0xe6e5c659,0xa0f4960e),LL(0xc052f5d6,0x27d5bc54),LL(0x8677ef24,0x9c5437dc)}, {LL(0xdb9b78fc,0x28ae53a3),LL(0xd91113f9,0x025b81a7),LL(0x6ac445f9,0xfb423fb3),LL(0x575649cf,0xb555a8b5),LL(0x06ae0cb6,0x3ac93b08),LL(0x8d2e3be1,0x7a706bb5)}},
   {{LL(0x97c0cb32,0xcabd7301),LL(0x65f83e1d,0x1698cfd3),LL(0x62043a8b,0x44833254),LL(0x3b4c8bac,0x43d1641f),LL(0xe344a05a,0xcc394a5b),LL(0x853c1f6d,0x9e085b71)}, {LL(0xcf627733,0x35a67867),LL(0xeab971dc,0x9d668c2b),LL(0x46c5f3bd,0x458a801d),LL(0x446c5e3b,0xb7b08696),LL(0xc15fc828,0x468ae44a),LL(0xab2761ff,0x9503fd49)}},
   {{LL(0xce8d4427,0x7278f12f),LL(0xeb15f81f,0x036aef18),LL(0xa58e95ea,0xcbe64b86),LL(0x59b2428c,0xf933a850),LL(0x52f1b0fb,0x8d117bfc),LL(0x0bed33e2,0xba9dbc6f)}, {LL(0x6ece7dc5,0x3e152fb5),LL(0xeb80d7ff,0x90e88871),LL(0xddf09489,0x0aa265aa),LL(0x2476414f,0xa17584cc),LL(0xcb0418ea,0x2d241d33),LL(0xc3a7529b,0x1c40e835)}},
   {{LL(0xe3e1aee9,0x89b2edba),LL(0xe55e4aca,0x33533137),LL(0x8943fce4,0x1dded9ce),LL(0x5a2ff996,0xaaf07ff4),LL(0xc96e87fe,0x69e60f92),LL(0xb9ee808c,0xc2e5c1d5)}, {LL(0xbe466616,0x79d6d8c6),LL(0xdffe4e28,0x897f6c2f),LL(0x350b7fc9,0xc8a65267),LL(0x8a2abab0,0xe9fcb46d),LL(0x2c4faf7c,0x57c3bfc6),LL(0xa8207c8a,0x41a8cc2a)}},
   {{LL(0xa74858fb,0xb60e8daa),LL(0x50c7c2f8,0xaa4c8ca5),LL(0x31a39837,0x9eb37096),LL(0x69f44aa5,0x5742fb32),LL(0x51e793a0,0x086aa479),LL(0x55e7c373,0x9bcaf657)}, {LL(0x1b13101d,0x63cee90c),LL(0xe7d42d91,0xbc410452),LL(0x913aa0fb,0x3041007f),LL(0x502be876,0x9a00ad14),LL(0xb8677568,0xa137a443),LL(0x46ca4efa,0x5b26b995)}},
   {{LL(0xaed60a78,0xe33585b7),LL(0x3cbb2e24,0xd208deea),LL(0x1aca34e8,0xff32cccb),LL(0x7b64d250,0x52ebce07),LL(0x84d8b907,0x93a4d0a1),LL(0x77de8d42,0xd759bef0)}, {LL(0x5d07487d,0xbd60c87a),LL(0xc74f79df,0x63620986),LL(0x4a52f837,0x54f5f2c8),LL(0xf3cec619,0x340a1f53),LL(0xfc85bf85,0x1e922bc4),LL(0xc3bfdf09,0xc92e7b85)}},
   {{LL(0x0eb9e6bc,0xeb1a37c3),LL(0x4fbf1cb1,0x8c393d89),LL(0xd1f10389,0x6ac9427e),LL(0x7c8caefc,0x77b8cfc1),LL(0x1a61bdd5,0x554ab4af),LL(0xad75e46b,0x661a99f1)}, {LL(0x439a0456,0x11fc01bb),LL(0x8d0046c9,0x5b7fedd2),LL(0xfbece9e3,0xcfded2ef),LL(0x11440124,0x4e4dcb0b),LL(0xdee23c7d,0x2a5fd0fa),LL(0xf662402c,0xaa5f345a)}},
   {{LL(0xeeb791c4,0x6f101762),LL(0xdf261eff,0x0d942184),LL(0xac1dc827,0x2c58e2aa),LL(0xf835a1b6,0x51410e89),LL(0x629915a4,0x981333a7),LL(0x0c14148d,0x371891b6)}, {LL(0xc0904446,0x4d20b3d3),LL(0x949776d8,0xdda7ecc8),LL(0x2a2645f7,0xa664b68c),LL(0xadd082ea,0x7a6bc857),LL(0x3e5ff206,0xe7467dc6),LL(0x04e2dfcc,0x40a6c340)}},
},
/* digit=55 base_pwr=2^275 */
{
   {{LL(0x03744726,0xa2354569),LL(0xd8d275ac,0xd2169e6d),LL(0x132c5689,0xab0c247b),LL(0xcc4760bb,0x129a5c9d),LL(0x26ae821b,0x03eba467),LL(0x3df1cf83,0x67a33fda)}, {LL(0xb8421b7a,0x010813cf),LL(0x98cd6d76,0x7b0f5070),LL(0x1fe4b600,0x907320b3),LL(0x98dd3239,0xda3bfeb3),LL(0x41abb34c,0x23f1ed16),LL(0x946f85f1,0x01b30f29)}},
   {{LL(0x9666f8ea,0x28524830),LL(0xeff6502b,0xd579b3df),LL(0x00e4f78a,0x3a66fa99),LL(0x54a3f7a3,0xfd8a65bb),LL(0x1965a79a,0x505d3f63),LL(0x1891469a,0x9524972c)}, {LL(0x3354da3a,0x78367cbc),LL(0xe4941c6e,0xbfe1fe3d),LL(0xf5af173e,0xe41bb3f6),LL(0x5ca36597,0x57cb03ca),LL(0x7b99f795,0x27f86cb8),LL(0xad4dcef1,0x5cae6514)}},
   {{LL(0xd87a86c9,0x4952a350),LL(0x034f45f9,0x08ed7da7),LL(0x2bd716d0,0x1e9ff827),LL(0xf1d9331f,0x2471fd15),LL(0xd7370b75,0x0c708362),LL(0xfc1a1051,0xaddedde6)}, {LL(0xdb27b186,0xf4475288),LL(0x3760bc11,0x5be4d46b),LL(0x06d47ee1,0xe44435d9),LL(0xd0b7c8a2,0x865cf7c8),LL(0x8d31a252,0xdb412be0),LL(0x2f24d71c,0x4b90a932)}},
   {{LL(0x55f0ad9f,0x38a49bf5),LL(0xf3618639,0x1a84c6b8),LL(0x01b2f7c2,0x5f709eca),LL(0x5be8359f,0xc479a650),LL(0xd6646b3f,0x6b6a22bf),LL(0xcc5b711b,0xcce78878)}, {LL(0xb446cc63,0x8e7dbc63),LL(0x218f800f,0x231bd027),LL(0x030271eb,0x2d3a7e04),LL(0xe22fb3c7,0xb08b5978),LL(0x9be0d46c,0x860d6278),LL(0x1d49a915,0x253a31c2)}},
   {{LL(0x376bdc3f,0xbad6b669),LL(0x23a9ff38,0xc4a8e7bc),LL(0x555fb0a3,0x3f54d8c4),LL(0x2b23db1b,0xfb3d5e1d),LL(0xf0d7eba9,0x6379f78a),LL(0xfa0beffa,0x36004feb)}, {LL(0x334ff01a,0xdf0a373c),LL(0xdff12a1c,0x10314749),LL(0xf184c1b3,0x1d52ddc7),LL(0xab02d404,0x79431663),LL(0x7f4d3795,0x1a6488c1),LL(0x7cca9102,0x3363660f)}},
   {{LL(0x04dc109c,0x56d65156),LL(0x09c1d307,0xcd740cc8),LL(0x10dfaead,0xef9e049f),LL(0xe30b70b8,0x19750b3a),LL(0x15c6a562,0x11ed8600),LL(0x12097026,0x53bdf97e)}, {LL(0x6c0d908f,0x79559d05),LL(0xb506d527,0x8f1d75ba),LL(0xae8fb3c4,0xd6fd7323),LL(0xa4111f88,0x834639c9),LL(0xa310a683,0xfc69a029),LL(0x255f2e9a,0xa4467bbb)}},
   {{LL(0xa3526fec,0x5f0a58a8),LL(0x02f028c1,0x849c171b),LL(0x56a5d3b5,0x34d77ce8),LL(0x54d5a92b,0x97016217),LL(0x2cc5b70b,0x0cce35c1),LL(0xe83f1f4a,0xd9d5a00a)}, {LL(0x26a0368b,0x064223f8),LL(0x40e16452,0x328a9f69),LL(0xd305ad2c,0x3a6ac093),LL(0xbeba7c44,0x759d9a16),LL(0x637ce7c8,0x86021de7),LL(0xcc80c1cc,0x276bed61)}},
   {{LL(0xbbe4bdda,0x66797f56),LL(0xda51b1a2,0xb92a369e),LL(0x31adb034,0x18eef4a5),LL(0x5d185cfc,0xcf1cb5ee),LL(0xbd53c27b,0xf596a59b),LL(0x69002569,0x1e1dd6f5)}, {LL(0x7687e48a,0xd9433e79),LL(0x0cbcb9ce,0x7d8d24c2),LL(0x65d68ecd,0x233cd7ed),LL(0xfb2aded8,0x201bbe09),LL(0xac9b750e,0x987f4975),LL(0x337f7f25,0x949da385)}},
   {{LL(0x9c7de99f,0x00c2ee1c),LL(0x15e50391,0x28a7461d),LL(0xa1c77952,0x1bdc0e32),LL(0xd53d640d,0xe98242c4),LL(0xcf153c7d,0x1a4724d2),LL(0xba477d46,0x194e5dcc)}, {LL(0x3a0d4ccd,0x871c8cfe),LL(0x9af451fb,0x62010af0),LL(0x6ddec75c,0x9b354f9e),LL(0x680e3511,0xe5db0a5d),LL(0xd247745b,0x183d1270),LL(0xeecf52dc,0x9910867a)}},
   {{LL(0x0410ca0b,0x6fb6b7bc),LL(0xb3c13935,0x0e16eed2),LL(0x316ff531,0x98ad89d8),LL(0x9894d65f,0x4800ee17),LL(0x48280170,0x034ea3c4),LL(0xc30be537,0x8126d12d)}, {LL(0x5120e525,0x43c2d27e),LL(0xee65df90,0x96a5d498),LL(0x5eaef29a,0x65454010),LL(0x7b678fc7,0x1d8f07a1),LL(0x7b301270,0x54bc6f73),LL(0xe9473365,0xe58a8102)}},
   {{LL(0x4ac6cf02,0x460c2990),LL(0x420a35b7,0x01482cfa),LL(0x34680972,0xf793933a),LL(0x17e2367b,0x2cd1f500),LL(0x3944f060,0x2411c352),LL(0x11c06b05,0x3d58b974)}, {LL(0xcddebb3b,0x4552e369),LL(0x009aeab9,0xe1c38aec),LL(0x353b6e4f,0x9d34737c),LL(0xb16d7b0c,0xf2c99e2c),LL(0x7bbba6a2,0x57029fa4),LL(0xd13ef64d,0x0565d1bc)}},
   {{LL(0xa329deb1,0x3561b3fe),LL(0xf1c3c3e4,0xfdca0e34),LL(0x47fb79d6,0x43748313),LL(0xc48002ed,0xa7f497e1),LL(0x2c44dcb0,0x86221cce),LL(0x43785e06,0x65e3f046)}, {LL(0xdf4cf461,0x9ee9061f),LL(0xf022d27a,0xc7479e8c),LL(0x76f7f52b,0x1d8de85b),LL(0x0fd6c65f,0x39a713c9),LL(0x711f8a39,0xf74ca067),LL(0x8ebc640a,0xad1119ad)}},
   {{LL(0x8ba85dff,0x59753bf6),LL(0xd1c89bce,0xec8b82ef),LL(0xb8b6a683,0xd7f1a651),LL(0x6f84416d,0x9c329bf3),LL(0xe68db225,0xaecbf4b9),LL(0x5a614d23,0x94ec3b0f)}, {LL(0x93a9543d,0xbcb66725),LL(0xf19132ed,0x90c46c46),LL(0x950b080f,0x4767c73c),LL(0x971fd9e5,0x0b9b143e),LL(0x8ec8c68d,0xbce6886f),LL(0xd47f512e,0x167b0f8a)}},
   {{LL(0xcd74009d,0x1942c2ff),LL(0xe9c286a4,0x71c4d5f5),LL(0x771a5972,0xf3c152b5),LL(0x363c2048,0x4cfb1e74),LL(0x9ddb8da2,0xcd2ce824),LL(0xa5ee443d,0x5d97c8e0)}, {LL(0x68d7b3d5,0x6fa84b3d),LL(0x9ce14ec3,0x97eaa76d),LL(0x8e13869d,0x2e457136),LL(0x96f0f8a1,0x39ac6a0c),LL(0x42d93dc0,0xe24458ac),LL(0x5f60bec9,0x7eb3689d)}},
   {{LL(0x588360ec,0x29544fe8),LL(0xffb550ea,0xa1aa9b9f),LL(0x4af4d28d,0xe1f6cf99),LL(0x0c6fd477,0x723d48b0),LL(0x5c81b252,0xf51f1b2f),LL(0x4f5a33ee,0x88ec11c0)}, {LL(0x2cd72de4,0x7747f043),LL(0xd71c92c1,0xcca69b0a),LL(0x4e8cc763,0x9455d86e),LL(0xc08444e0,0xc9e0aa1b),LL(0xe8fffa63,0x93803b68),LL(0x2d781b7d,0xc296af29)}},
   {{LL(0xb08d2d0e,0x514cc1dc),LL(0x30e93536,0x4e6b379e),LL(0x2fc9230f,0xf0e422ac),LL(0x92e23e21,0xaa50a1ad),LL(0x676d1ac0,0x70ac46d8),LL(0xf9f54493,0x698b9991)}, {LL(0x8649519f,0x59a6b86a),LL(0xe3511da4,0xc1f11ad6),LL(0x3192968c,0xd3d9cff1),LL(0x0b342dd8,0x13e700b4),LL(0x3b1da441,0xfd5dc7bb),LL(0x2c883760,0x02426e7c)}},
},
/* digit=56 base_pwr=2^280 */
{
   {{LL(0x1c4c9d90,0x9e9af315),LL(0xd12e0a89,0x8665c5a9),LL(0x58286493,0x204abd92),LL(0xb2e09205,0x79959889),LL(0xfe56b101,0x0c727a3d),LL(0x8b657f26,0xf366244c)}, {LL(0xcca65be2,0xde35d954),LL(0xb0fd41ce,0x52ee1230),LL(0x36019fee,0xfa03261f),LL(0x66511d8f,0xafda42d9),LL(0x821148b9,0xf63211dd),LL(0x6f13a3e1,0x7b56af7e)}},
   {{LL(0xcc8998d6,0x3997900e),LL(0xbaa60da1,0x8fa564b7),LL(0x661f3c57,0x71bf5b0a),LL(0xaab1292b,0x44b13388),LL(0xd4d993f2,0xcbe80cb9),LL(0x2203f966,0x0b19b4c9)}, {LL(0x0080f259,0xbc82a652),LL(0xad96dea3,0x870ebc08),LL(0x502f0003,0xa388c7e7),LL(0x56a38f73,0x9c704ef0),LL(0x3487d9b0,0x93cde8a7),LL(0xec11a1f3,0x5e9148b0)}},
   {{LL(0x5913e184,0x47fe4799),LL(0x82145900,0x5bbe584c),LL(0x9a867173,0xb76cfa8b),LL(0x514bf471,0x9bc87bf0),LL(0x71dcf1fc,0x37392dce),LL(0x3ad1efa8,0xec3efae0)}, {LL(0x14876451,0xbbea5a34),LL(0x6217090f,0x96e5f543),LL(0x9b1665a9,0x5b3d4ecd),LL(0xe329df22,0xe7b0df26),LL(0x0baa808d,0x18fb438e),LL(0xdd516faf,0x90757ebf)}},
   {{LL(0xa748b8f5,0x63f27a25),LL(0x2cd246c4,0x68c8f3ec),LL(0x65f9ce38,0x5d317cd9),LL(0x635ba300,0x162c92e0),LL(0xfe343662,0x5259f64f),LL(0x8e614ac8,0x4a6b2b66)}, {LL(0x01177c3b,0x97fb55bb),LL(0xa705cb01,0xfb586c21),LL(0x78061824,0xa57e7325),LL(0x6c1e6306,0x892f6b38),LL(0x2367b14c,0xf12e4c07),LL(0xc83a48c5,0x580d5fe2)}},
   {{LL(0xd5a98d68,0x1e6f9a95),LL(0x849da828,0x759ea7df),LL(0x6e8b4198,0x365d5625),LL(0x7a4a53f9,0xe1b9c53b),LL(0xe32b9b16,0x55dc1d50),LL(0xbb6d5701,0xa4657ebb)}, {LL(0xeacc76e2,0x4c270249),LL(0x162b1cc7,0xbe49ec75),LL(0x0689902b,0x19a95b61),LL(0xa4cfc5a8,0xdd5706bf),LL(0x14e5b424,0xd33bdb73),LL(0xe69eba87,0x21311bd1)}},
   {{LL(0xea2bafb3,0x6897401c),LL(0x15c56fe4,0x7b96ecc2),LL(0x39e2b43b,0xe511b329),LL(0xbf809331,0x39522861),LL(0xc958f8f4,0x815f6c1d),LL(0xc213e727,0x2abbdf6b)}, {LL(0xc39bc01f,0xeb09ae59),LL(0x676b56a5,0xffe3b831),LL(0xa20f86c6,0x8f4815a2),LL(0x9aa30807,0x748a1766),LL(0x1b758878,0xf1f46a21),LL(0x6f6fc3d7,0xbd421fe7)}},
   {{LL(0x72a21acc,0x75ba2f9b),LL(0xa28edb4c,0x356688d4),LL(0x610d080f,0x3c339e0b),LL(0x33a99c2f,0x614ac293),LL(0xaa580aff,0xa5e23af2),LL(0xe1fdba3a,0xa6bcb860)}, {LL(0xb43f9425,0xaa603365),LL(0xf7ee4635,0xae8d7126),LL(0x56330a32,0xa2b25244),LL(0x9e025aa3,0xc396b5bb),LL(0xf8a0d5cf,0xabbf77fa),LL(0xea31c83b,0xb322ee30)}},
   {{LL(0x0d6ded89,0x300b0484),LL(0xc3ab55ed,0x0b1092cb),LL(0x0cc10a74,0x17d9c542),LL(0xeff9d010,0x7f637e84),LL(0x27aa1285,0xd732aa1e),LL(0xe2a77114,0xedb97340)}, {LL(0x5ef4dfb0,0x62acf158),LL(0xba1d7b81,0x1e94fc6e),LL(0x2e6eb2db,0x88bec5d2),LL(0x8d18263d,0xaec27202),LL(0xe4bbd6ac,0x4b687353),LL(0x0ff7e4c0,0x031be351)}},
   {{LL(0x7890e234,0x04881384),LL(0x672e70c6,0x387f1159),LL(0x7b307f75,0x1468a614),LL(0xed85ec96,0x56335b52),LL(0xd45bcae9,0xda1bb60f),LL(0xf9faeadd,0x4d94f3f0)}, {LL(0xfc78d86b,0x6c6a7183),LL(0x3018dec6,0xa425b5c7),LL(0x2d877399,0xb1549c33),LL(0x92b2bc37,0x6c41c50c),LL(0x83ee0ddb,0x3a9f380c),LL(0xc4599e73,0xded5feb6)}},
   {{LL(0xf086d06c,0x6c00f388),LL(0x5add0cf4,0x17ee4503),LL(0x07caf89c,0xf96984c7),LL(0x648ed5e9,0x9d49d667),LL(0xa0164881,0x3ef95015),LL(0x7d9c651f,0x39e28e44)}, {LL(0x59f37780,0xb13ad240),LL(0xb9522225,0x08cee349),LL(0x2ba1b214,0x9245ee6f),LL(0xa886d8d2,0x12bedaa9),LL(0xfcb8186f,0xe139ae08),LL(0xfc2ef864,0x99203fb6)}},
   {{LL(0x0b7f8354,0x14d34c21),LL(0x9177ce45,0x1475a1cd),LL(0x9b926e4b,0x9f5f764a),LL(0x05dd21fe,0x77260d1e),LL(0xc4b937f7,0x3c882480),LL(0x722372f2,0xc92dcd39)}, {LL(0xec6f657e,0xf636a1be),LL(0x1d30dd35,0xb0e6c312),LL(0xe4654efe,0xfe4b0528),LL(0x21d230d2,0x1c4a6820),LL(0x98fa45ab,0x615d2e48),LL(0x01fdbabf,0x1f35d6d8)}},
   {{LL(0x64c9323d,0x3c292847),LL(0x0491f77d,0x40115a89),LL(0x2d7a05f5,0xec141ade),LL(0x222a5f9f,0x0c35e4d9),LL(0x442a3e9b,0x5ea51791),LL(0xe51b841e,0x17e68ece)}, {LL(0xd6ae9174,0x415c0f6c),LL(0x9ffd7595,0xe6df85f8),LL(0x8dedf59c,0x65fc694f),LL(0xfee92718,0xc609503e),LL(0x97d565ae,0x57d2592e),LL(0x7e20862b,0xb761bf15)}},
   {{LL(0x3a7b10d1,0xa636eeb8),LL(0xf4a29e73,0x4e1ae352),LL(0xe6bb1ec7,0x01704f5f),LL(0x0ef020ae,0x75c04f72),LL(0x5a31e6a6,0x448d8cee),LL(0x208f994b,0xe40a9c29)}, {LL(0xfd8f9d5d,0x69e09a30),LL(0x449bab7e,0xe6a5f7eb),LL(0x2aa1768b,0xf25bc18a),LL(0x3c841234,0x9449e404),LL(0x016a7bef,0x7a3bf43e),LL(0x2a150b60,0xf25803e8)}},
   {{LL(0x82376117,0xd443b265),LL(0x1a1beb0d,0xb91087c1),LL(0x45cc5951,0x3fe62a65),LL(0xe6e472d5,0x49c754bc),LL(0x77c424eb,0x7e60bb81),LL(0x830cbb97,0xbcd4088e)}, {LL(0xba26df7b,0x3da5c94e),LL(0xf72b4338,0x508b4f55),LL(0x69ad7784,0x409c5c74),LL(0xfdf44d6a,0x82e5f1b0),LL(0xeed2766f,0x10654a1c),LL(0xa6e83f4a,0xef1e65fa)}},
   {{LL(0xb215f9e0,0xe44a2a57),LL(0x19066f0a,0x38b34dce),LL(0x40bb1bfb,0x8bb91dad),LL(0xe67735fc,0x64c9f775),LL(0x88d613cd,0xde142417),LL(0x1901d88d,0xc5014ff5)}, {LL(0xf38116b0,0xa250341d),LL(0x9d6cbcb2,0xf96b9dd4),LL(0x76b3fac2,0x15ec6c72),LL(0x8124c1e9,0x88f1952f),LL(0x975be4f5,0x6b72f8ea),LL(0x061f7530,0x23d288ff)}},
   {{LL(0x5f56dc3c,0xa6e19d0a),LL(0x0b88326a,0xe387e269),LL(0x0ee527a4,0xef738095),LL(0x7c4278a6,0x78b7174b),LL(0xe70798ff,0xc133d867),LL(0x9e9230ca,0x9d0fef75)}, {LL(0x1a955ab9,0x7431eef0),LL(0x8868d922,0x3772e703),LL(0x8d6af3f7,0xf7a4306a),LL(0xbbec076a,0x633bb5a0),LL(0x7a257ca3,0x6d07623e),LL(0x21c00663,0xffb5e165)}},
},
/* digit=57 base_pwr=2^285 */
{
   {{LL(0xbb71e9b0,0x35ae891b),LL(0x522b77f0,0x1f6ce6ca),LL(0xe63745c4,0xc2dab3ca),LL(0xf218d139,0x55b8c185),LL(0x89f3b0e2,0x6ab039c8),LL(0xc644c3fa,0xd9e25bfd)}, {LL(0x3e2ed47b,0xc8496f20),LL(0x8d67e17c,0xc395ec02),LL(0x92114918,0x5c678392),LL(0xef73f345,0xe962e52f),LL(0x54fcfb22,0x3818baf3),LL(0x9d4bc911,0x4d75d65d)}},
   {{LL(0x022ea83a,0x7f1f985c),LL(0xa7584e7f,0x90a22662),LL(0x5188fcf6,0xb40a930a),LL(0xa3a82904,0x3fad79ab),LL(0xf3151027,0x7bee8d22),LL(0xc2c3e17b,0x79a1a838)}, {LL(0x33cc3509,0x1fbe06e9),LL(0x9abd5cca,0x629c56aa),LL(0x2d9cf7a5,0xfff290ec),LL(0x9bd062c5,0x5d0dedaa),LL(0xd7d35381,0x080344ab),LL(0xf5cf9eda,0x0848373a)}},
   {{LL(0xa1b31832,0x170098de),LL(0x2cd8c540,0xc6876bf9),LL(0x8660d773,0x35b1b04b),LL(0x5152ad1e,0x8b4b6c4b),LL(0x4a8e92aa,0x0b0bd1c2),LL(0x88172f4e,0xd814f6f4)}, {LL(0x41ac83d1,0x13a76899),LL(0xf1d80357,0x355933b4),LL(0x3dc17e72,0x4b394f97),LL(0x0a7124b2,0x1b0cd166),LL(0xce8a372c,0x7549880c),LL(0xe712603b,0x79f53f7e)}},
   {{LL(0x7a0c0bc0,0x31d31f7a),LL(0xb251d2bf,0x7a37a84a),LL(0x52f04d67,0x1793362e),LL(0x21c7b651,0x5808e709),LL(0xed6f47f6,0x33fe9123),LL(0x58f71405,0xdeb1dde9)}, {LL(0xae56b472,0x821d3045),LL(0xe02043ad,0x9f61f761),LL(0x5b2048a9,0x932ddb14),LL(0xd7811330,0x17d989fe),LL(0x128fd85f,0x032ae4cb),LL(0x7d1ef434,0x8f1956b4)}},
   {{LL(0xce1d819c,0x30cc44e0),LL(0x15a3a414,0x3c2c4f04),LL(0x665b634b,0x7f06d2c2),LL(0x7df0bd69,0xf609fe64),LL(0x6a0dbf94,0x4b0c8c36),LL(0x46e9b487,0xd0a3a752)}, {LL(0x9578a83f,0x7407e157),LL(0xfe0f975f,0xc46ffa4f),LL(0x1efaeccf,0xb086f199),LL(0x64796c18,0xf984b69d),LL(0x8ebfb97d,0x14de44fa),LL(0xf5245ba6,0x47bcb675)}},
   {{LL(0x0dbafc49,0xed58c1ab),LL(0x6fa8b473,0x41995a5a),LL(0x85450ccb,0x3af3853d),LL(0x601fe529,0x18d1a7be),LL(0xdab2d926,0xd196520f),LL(0x56213ae9,0xd1f2c1e9)}, {LL(0x77ded01d,0x37218255),LL(0xa7905433,0x366c552d),LL(0x7d9430d1,0x023d0373),LL(0x9ddf69ea,0x65d70cb6),LL(0xd4d3fa24,0x6321df7a),LL(0xb91084c3,0x77b1c2bf)}},
   {{LL(0x0aef0ff7,0x2434dbdf),LL(0x9cc3d2ba,0xca74413e),LL(0x3dcbaeb9,0x0aee0d65),LL(0x4184a72e,0x89568076),LL(0xbe7d4e0f,0x3d7e4f61),LL(0xae441b4c,0x284a8608)}, {LL(0x6a67283c,0x7274fd92),LL(0xc06e0b84,0xa222be15),LL(0xe4623e88,0xa0713cd7),LL(0x4710aa33,0x0ff4a317),LL(0x044dcecd,0xc9a35a65),LL(0xde77a24b,0x7bae1eca)}},
   {{LL(0x16973cf4,0x070d34e1),LL(0x7e4f34f7,0x20aee08b),LL(0x5eb8ad29,0x269af9b9),LL(0xa6a45dda,0xdde0a036),LL(0x63df41e0,0xa18b528e),LL(0xa260df2a,0x03cc71b2)}, {LL(0xa06b1dd7,0x24a6770a),LL(0x9d2675d3,0x5bfa9c11),LL(0x96844432,0x73c1e2a1),LL(0x131a6cf0,0x3660558d),LL(0x2ee79454,0xb0289c83),LL(0xc6d8ddcd,0xa6aefb01)}},
   {{LL(0x81069df0,0xda10f552),LL(0x560ea55a,0x70088f4e),LL(0xda8cbaa5,0xdcee31af),LL(0xbe16a7b7,0x1213b76a),LL(0x781114d1,0x4fffa388),LL(0x904479d2,0x2cc19aa9)}, {LL(0x85eec3cd,0xc4ebf9be),LL(0xbaff7431,0x48ea527a),LL(0x52d3ce22,0xa5b6fdca),LL(0x7e139d7a,0xbcb07835),LL(0x5de5de75,0xc7f17551),LL(0x15bcf7aa,0x005ab4d9)}},
   {{LL(0x41383866,0x041511b0),LL(0xdb9c2b4c,0x7c90692a),LL(0x61deb106,0x736cc6af),LL(0xb5b84abe,0x57d428cb),LL(0x35dbcca9,0x14bc4981),LL(0x37f3ef2e,0xece90041)}, {LL(0x85066bd0,0xe2c33711),LL(0xabef34c2,0x335bd50b),LL(0xb017a337,0x755f513d),LL(0x284445ad,0x28d263e2),LL(0xecb8436e,0x83402ba9),LL(0xf31d28d4,0x66357324)}},
   {{LL(0xbce3cea0,0x05fb9811),LL(0xdf2b07df,0x74adb5b5),LL(0x0c4e5e2a,0x870e63b8),LL(0xe5fec889,0xe873cba7),LL(0x19cd8a3e,0x28b0e005),LL(0x48112c8c,0xbb16491b)}, {LL(0x5df42faa,0xddb8cfb5),LL(0x2ab1a097,0x35f952ae),LL(0xd2dfa18f,0x35e885e1),LL(0x5b4a0277,0xc3ec2325),LL(0xc5aecee0,0xdbe2e40c),LL(0x0690080a,0x133383c9)}},
   {{LL(0x7521f457,0xe5d473dc),LL(0xa00be577,0xe9ef09bd),LL(0xb6eaa640,0xf6d0965f),LL(0x75726560,0xeb494868),LL(0x28817302,0x452116d5),LL(0xfbde3597,0xf0424fdb)}, {LL(0xbb454915,0xd6096da3),LL(0x41422141,0xde482808),LL(0x2d19fac0,0x7a135197),LL(0x21393f6f,0xdc9a5ec4),LL(0xeb2c8ada,0xcabcc1e3),LL(0x42d8c4f2,0xd4366431)}},
   {{LL(0x34246834,0x46c7b438),LL(0x2eb9f9bc,0xa0a570c3),LL(0x586a0fdd,0x3e01db21),LL(0x9d21192c,0x15732b07),LL(0xfd747dd8,0xe544b5c9),LL(0x7ed374fb,0xeee7e104)}, {LL(0x3b75a586,0xfa619086),LL(0xe8a4cfd4,0x9d0909f4),LL(0xccd93fa4,0x30e0e476),LL(0xa14595d4,0x5bdc5c02),LL(0xe245400f,0xf6f8ab6a),LL(0xa4198419,0x7bd194ce)}},
   {{LL(0x68e190a6,0x713a654c),LL(0x3037a8e6,0x6317f6ab),LL(0x6b6bafef,0x005f412a),LL(0xb53eb41a,0xf67f8201),LL(0xf51ccf93,0x96585fcb),LL(0x045104e0,0x68ed706e)}, {LL(0x40db03d9,0x535dc2a7),LL(0xda583d56,0xb9f0b69d),LL(0x414dc76a,0xeed71cb1),LL(0xc5e7698e,0x7368aabd),LL(0x4689da93,0x3eb6a4f1),LL(0xfcb73b42,0x753fa65a)}},
   {{LL(0x9b860b0f,0xf5131aad),LL(0xd9bcf9c9,0x19af93ce),LL(0x307ca1ca,0x69b9765d),LL(0x87936c0b,0x0b735808),LL(0x1107f1cb,0x10eaf7ae),LL(0x16dfe03b,0x15211ad5)}, {LL(0xb11c8af6,0x9bde40d5),LL(0xed1fc8be,0x513ea01a),LL(0x5412ab86,0x2f04be27),LL(0x9de582fc,0x2a1adc4a),LL(0x00de0043,0xb0c43235),LL(0x006efb33,0x689e15e4)}},
   {{LL(0x89e4e449,0x0ed1082f),LL(0x833f2378,0xdb1fb471),LL(0xece77352,0xa35fef0e),LL(0x4bf0c426,0x76adaa46),LL(0xa011b2fb,0xfbab929a),LL(0x9d8cc4d3,0x6f475d5b)}, {LL(0x74351480,0xbe6d7f21),LL(0x93e4a7ae,0x2d1362d1),LL(0x106ceaab,0xc7e2cba5),LL(0x45258697,0xfe94528a),LL(0x075945b0,0x7109b17d),LL(0xcae17f7a,0xfd395b2c)}},
},
/* digit=58 base_pwr=2^290 */
{
   {{LL(0x310404a8,0xfc1faedc),LL(0xd3bcb128,0xea339148),LL(0x6416defd,0xf0048545),LL(0xc58653e7,0x75de7770),LL(0xe2f6f99e,0xdd2dcbeb),LL(0xd159ac07,0xa4380ef4)}, {LL(0xe4173608,0x45dd713c),LL(0x446a6789,0x44919b61),LL(0x6b962b38,0x3f73756b),LL(0xbffd3f0c,0x3cb9f53b),LL(0x7f08ebae,0xd723c40b),LL(0x0c3cddba,0x998a9b17)}},
   {{LL(0xf6ba1469,0xf6377e3b),LL(0x09c832d3,0xc334fb6c),LL(0xc21c0cf1,0x7f85ac42),LL(0x857d8edb,0x7a3e31c9),LL(0x27b77ed6,0x2eb10763),LL(0x38dae10b,0x2bfbbdbc)}, {LL(0x7bae3b4f,0xed7c6fb1),LL(0x36d04e6f,0xc5911d9f),LL(0x4569e72f,0x4dc43550),LL(0xbedae3ab,0xaa82fb97),LL(0x4f27e463,0x06d37bef),LL(0xf0c35a11,0xd0dbce6d)}},
   {{LL(0x661019ec,0x43c78835),LL(0x24e66d29,0x68e916b1),LL(0x24094671,0x02c0f3a2),LL(0xd0f17d86,0xab6f1c05),LL(0xa22d4264,0x6d3bac72),LL(0xf6e5fafe,0xd7b8f152)}, {LL(0x39447eb3,0x95627c63),LL(0x79e1ff93,0xfd159018),LL(0x5ad80806,0x39277c83),LL(0x0d7c7b74,0x758aafc9),LL(0x4cb8bec9,0x605ad8ca),LL(0x5741828a,0x6a90085c)}},
   {{LL(0xff8b1fbe,0x6edf5561),LL(0xf6eac0c0,0x614b788e),LL(0xd8d66d29,0x7699ae56),LL(0xe9d58eb2,0x5f81602a),LL(0xfaf9176d,0xd0c04874),LL(0x523153b1,0x4b3a0046)}, {LL(0xf6315883,0x9690930f),LL(0xa60ca92d,0xa81c0b44),LL(0x73bcba90,0x2d0e7258),LL(0xe902e329,0x57efe72d),LL(0x76bc27b9,0x3fcd5986),LL(0x93940c09,0x492adf03)}},
   {{LL(0xf820c195,0x6895dbe2),LL(0x3f6c7b40,0x3787a500),LL(0xac1e90f3,0xdc718243),LL(0xba5d0870,0x352f8c91),LL(0xec0112b5,0xf3d1c53e),LL(0x6b84f64a,0x08a0782f)}, {LL(0x8eedd5d4,0xd659e635),LL(0x29537276,0xfc30df6c),LL(0xa1755ce0,0xbfb09978),LL(0xaa2b4187,0x227f7b12),LL(0x226539d2,0x828730b9),LL(0xb2472c95,0x9051a37c)}},
   {{LL(0xb39317b8,0x7de743e2),LL(0x2d372acf,0x9205d447),LL(0x3eeb0012,0x8226fc30),LL(0x2af74be6,0xab2a3e05),LL(0x4af91ac0,0xbe476780),LL(0x0ca36bf4,0x98497c71)}, {LL(0x8d6dedb4,0x74fdf7cd),LL(0xa0fc5919,0xb50778ee),LL(0x2fcd7c63,0x5d5ec33f),LL(0x7f33cde0,0x667b8193),LL(0x38364d44,0xce48ae4b),LL(0x223ed67e,0xb8578963)}},
   {{LL(0x9567f5f4,0x4b20d182),LL(0x18f02b34,0xde7e8149),LL(0xecff9dd7,0xc9a4be7b),LL(0x9812fd3f,0xe2f70bbe),LL(0x9c889263,0x471bf90c),LL(0x3e61f5bf,0xb60d01b5)}, {LL(0xd22d855b,0x258c7f89),LL(0xb75a7d4f,0x35ef5c15),LL(0xb247f27d,0x26d8e1da),LL(0x8d0f7757,0xcf136199),LL(0x3f8e894d,0x31244780),LL(0xe1a3d47d,0x8d2a20ba)}},
   {{LL(0xc795a2c7,0xc68b641e),LL(0x5a4d6647,0x4fe559b1),LL(0xd89ce668,0xeda98cba),LL(0x6c269d8e,0x15f84dc0),LL(0xcbf34023,0xf0eb685e),LL(0xc032634a,0x3668c530)}, {LL(0xe4531f59,0x2e3d7fff),LL(0x85494d06,0xe6270306),LL(0xa3e050df,0xf02cabcf),LL(0xc001dcd9,0xccd2da67),LL(0x066d2d52,0x50aa3723),LL(0x7224a41f,0xdb075650)}},
   {{LL(0x43c1e526,0xeaf0b05d),LL(0x5c673e17,0x7eddb25a),LL(0xc1f9bda7,0xfce700de),LL(0x646a1550,0x4e6f4c70),LL(0x8ba60c35,0xae06c3e5),LL(0xc0c89c50,0x1a57101f)}, {LL(0x60a36192,0xdb35b1f3),LL(0x265c3ed1,0xf740a467),LL(0x6175e771,0xeab65a88),LL(0xcbe28ed4,0xd94c8cc6),LL(0x465dd7a8,0xac347e7b),LL(0xca9b899d,0xca927337)}},
   {{LL(0xdd045c9e,0x3499422e),LL(0x727f1f25,0x5119c675),LL(0xd76e7077,0xf44a89ae),LL(0x8eb17f39,0x9d6967b5),LL(0x2bbed398,0x49b39b81),LL(0x07deb5e9,0x05b3a455)}, {LL(0x62039ba0,0x8fa1ebf3),LL(0x0b650544,0x52c3d14b),LL(0x37d3fd20,0x2603aaf7),LL(0xec70bfc0,0xb6fe1148),LL(0xfeb74df8,0xe68b8475),LL(0xe1f4aba2,0x56fafdcb)}},
   {{LL(0xa05b2adf,0xd061bce6),LL(0x8b7cf4c6,0xdb6e9324),LL(0x189f7ce1,0x6344c3b6),LL(0x3cfe678d,0x06993dc9),LL(0xb6e22f1b,0xb00b5227),LL(0x780137c1,0x8f836e4c)}, {LL(0x1e9ad145,0xaf822953),LL(0xdfaaa159,0xb438997c),LL(0x904396e6,0x03e27459),LL(0xc8259d86,0xd06a0fdb),LL(0xe040907b,0x8afe766b),LL(0x038fbdba,0x0f5ff265)}},
   {{LL(0xe1aa5953,0x4b1f8698),LL(0xbf0a531c,0x94af5c0c),LL(0x6536e156,0xba1c49f5),LL(0x09c95a75,0x31c31631),LL(0x0a6d9af1,0x268a308f),LL(0xfa9ca8b7,0x185b9762)}, {LL(0x65035840,0xc3b1cbcd),LL(0xb429c82a,0x17b276c6),LL(0x34396850,0x1b4ec7b3),LL(0x048a6943,0x42f21d5a),LL(0x5b5820bf,0x0bf948f5),LL(0xb48dca58,0xac944487)}},
   {{LL(0xac575242,0xefa91b2e),LL(0xcdaca77a,0x58280b99),LL(0xcf39d117,0x5c3f8382),LL(0x70e5d17c,0x4d460dba),LL(0x12f6d6ad,0x440b4db8),LL(0x7f3a6bb8,0x14b9b335)}, {LL(0x837833cf,0xfbb3746a),LL(0xda482df0,0xbbf4735b),LL(0x9fbf146f,0x4efe32c0),LL(0x346e3b9e,0x65e5d7c2),LL(0x434ce5d5,0x742e81ce),LL(0x3bb80bd5,0x87e2b772)}},
   {{LL(0xe02f8540,0x18d0983b),LL(0xe8058c66,0x708654c7),LL(0x92894720,0x0b137f68),LL(0x4345444d,0x0b8b62e9),LL(0xbfd043d0,0x426807d3),LL(0x2a1ac390,0x17ac2e64)}, {LL(0xf2076839,0xd037e79c),LL(0x3ba4d3f2,0x991d4ff0),LL(0xd3724bfa,0x3937247e),LL(0x218413d1,0xb5b50d34),LL(0x28157b81,0x2accc172),LL(0xc472e8b2,0xd07f18b7)}},
   {{LL(0xca12be15,0x0df64ab2),LL(0xc2f0d0f9,0xdbac68f8),LL(0xa2107d3e,0x5772ba8b),LL(0x76724cfe,0x32c43ac9),LL(0x166229b7,0x6f4a367e),LL(0x230b3527,0x9c8bdb42)}, {LL(0xe71f261d,0x343d644d),LL(0x26283547,0x45c849bf),LL(0xd8e4c7b0,0x3a42b9b8),LL(0x9f8af5c7,0x0aff84ad),LL(0x418cfee8,0x453c3a30),LL(0xa241633a,0x6be57cdb)}},
   {{LL(0x65b55050,0x04418b59),LL(0xd324aa67,0xa8a797c3),LL(0x7c65a6d9,0x5f87e22c),LL(0x1dbeffe4,0xaac71065),LL(0xbd3cc05c,0xff619d64),LL(0xe65c92c4,0x9a29c966)}, {LL(0xdad7fcbd,0x23af2b21),LL(0x153b817f,0x4950a767),LL(0xc6478c55,0xc34a7efa),LL(0xf6cd140e,0x57cde95a),LL(0xf5a0db2e,0x64b74575),LL(0x75d7fb76,0xd4b5ea52)}},
},
/* digit=59 base_pwr=2^295 */
{
   {{LL(0x8e72aafb,0x28405062),LL(0x8ea8bf00,0x655bf353),LL(0x05547f7b,0x789d9444),LL(0x3441e472,0x7fa445ed),LL(0x4a44ce87,0xfeb19825),LL(0x129aed14,0xccb5f12c)}, {LL(0xaf94fb34,0x22b05de3),LL(0xd3f03199,0x7422a040),LL(0xa83f7f08,0xfba252ca),LL(0xcefaa757,0x0f6ad6e6),LL(0x6517d806,0xe1ad1871),LL(0x8e9d97ad,0xd16dc8ed)}},
   {{LL(0xa3736eb6,0xcbb99194),LL(0x36dcf470,0xdd5161cd),LL(0xb6ab6c03,0xd50b24aa),LL(0xbc41f4b7,0x419d2810),LL(0x295496cf,0xe2e88d7a),LL(0xf2457ac0,0x350713f2)}, {LL(0x0427e2ae,0x838e4a36),LL(0x4d974e5a,0x7631472a),LL(0x7a5c5fdc,0x9fa3ab1c),LL(0xde34cb8d,0x324798cd),LL(0x889105fe,0xbfa5a9d0),LL(0xfd0df249,0xd05dad34)}},
   {{LL(0xcb419fc3,0x536e5657),LL(0x1d271dd1,0xe8c208bc),LL(0x22d2b9ad,0x6a3713bd),LL(0x471d808d,0xa4c761a7),LL(0x7e6dca35,0xd93aafb6),LL(0x8f55ca32,0xc46c0ae3)}, {LL(0xa78bfca0,0x55dc0de7),LL(0x3407d0ca,0xe9cfb301),LL(0xb3256c14,0x777e2a60),LL(0x6d8fee02,0x32b2238c),LL(0x46e43ee8,0xe8b35396),LL(0x247985dd,0x310bc1ba)}},
   {{LL(0x581f9d3c,0x9974759c),LL(0xe5cb1973,0x9e76a970),LL(0xc64941ca,0x8afec58e),LL(0x01d05575,0x2d7c57fa),LL(0x5c448db5,0xc07c74cc),LL(0x01bb1440,0xa52474ce)}, {LL(0x00115bbb,0x93162d97),LL(0xfd7480f9,0x483b6147),LL(0x6af18ded,0x4f28c57e),LL(0x174a3089,0x36faed8f),LL(0xa3dd6265,0x702dbd64),LL(0x6adc0d7b,0x86a9c43f)}},
   {{LL(0xb9de7b63,0xa3510710),LL(0x4019c9df,0x9f364ad1),LL(0x9b5bdce3,0x5b66a5d7),LL(0x78b1b385,0x2b2f6951),LL(0x3cfa9f99,0x3e4783d3),LL(0x6bd6bcf4,0x1af51750)}, {LL(0x81d8d7ef,0xf9c0686a),LL(0x37c068d3,0xdc0f22ec),LL(0x93545faf,0xe1b86653),LL(0xa8a52881,0x37ca8501),LL(0x5603e359,0x07ac5c8a),LL(0x542cc937,0x98fb2bab)}},
   {{LL(0x0c5bbd3e,0x4981be69),LL(0x185fdb55,0xb047df0a),LL(0x74cff00c,0x3168e050),LL(0xb52c7f9c,0x111150a1),LL(0xa51c7986,0x0db2ed84),LL(0xe61272ad,0x7d991630)}, {LL(0x28de14dd,0x7443d936),LL(0xa5daed5f,0xfdf31f41),LL(0x866b5e40,0x71e0ef4e),LL(0xb965a670,0x05c57a45),LL(0x70e1aa77,0x85bdb58c),LL(0x9df3ce32,0xe4d1fe2a)}},
   {{LL(0x2f722563,0x5d461898),LL(0x567db14c,0x11d22b39),LL(0x6779cd40,0x9a8f004e),LL(0x5473ecd5,0x0812ae3d),LL(0x4e6c296a,0x4ed82862),LL(0x064ee61f,0x2d9ce40c)}, {LL(0xd8a9eb1e,0x4856d586),LL(0x5d1b5e3a,0x2ddd6b12),LL(0x382fba3c,0x0ab5eec0),LL(0xfcf4a9c8,0x302018df),LL(0xab3cdedb,0x7b4e6fd2),LL(0x8f64cb1d,0x266c246a)}},
   {{LL(0xf5c3af59,0xc41e4aed),LL(0x6de9a78a,0xa0284ad0),LL(0x8ed812d2,0xf5eaab7b),LL(0x1afb58b8,0x7801fbb8),LL(0x71efcc3a,0xbe5cdba6),LL(0xcd10cb91,0xe31a0e3c)}, {LL(0x85dc0bc6,0x882e821e),LL(0xbb32e506,0xd3ad070f),LL(0xd8a0f038,0x3afede2b),LL(0x857fd3a0,0xe20a117c),LL(0x3060f767,0xebaa2aa4),LL(0x2b9d1da1,0x6524aa0d)}},
   {{LL(0x88cffe33,0x9cc5847c),LL(0x6e8eb6c1,0xff726b0f),LL(0x1bc45d8e,0x9bb2ca16),LL(0xa6d8a5a6,0xe7903009),LL(0x47db2201,0x4f089cc0),LL(0xe6b5928a,0x4135884d)}, {LL(0xe5c017cf,0xb1a86a0a),LL(0xb0a393df,0xb1d9bf6d),LL(0x28bb3277,0x33d9c1c6),LL(0x45b582ce,0xcb05b67b),LL(0xf33792c5,0xa72585fc),LL(0xa7d1ed84,0x78b7c5e8)}},
   {{LL(0x9346df25,0x70e478fb),LL(0xb4a4ada5,0x01dc0c2e),LL(0x5be36ea7,0xaec82b00),LL(0x6717e06a,0x82618b8f),LL(0x008f1977,0x2db1f6d4),LL(0x16b390d1,0x4e493f3b)}, {LL(0x990a75eb,0xfe86fd4d),LL(0x783f6076,0xa1cf7f99),LL(0x0c049158,0x6cbb23e7),LL(0xed456235,0xd05be7e5),LL(0x9bd836ef,0x60137406),LL(0x32e5f604,0x94ec9644)}},
   {{LL(0x3d87bfa8,0x3361e1ce),LL(0x8dcca4f0,0x92f235e7),LL(0xbe323fd1,0xc8084cb4),LL(0xc24c6d16,0x3fd481a5),LL(0x2cea81ba,0x9b1bd940),LL(0x0c5aa59f,0xf5091191)}, {LL(0xf81d5e2a,0x4cd8c9ef),LL(0x1550bff4,0x5ad00013),LL(0x8cc32e55,0x29d47b9f),LL(0x11694ece,0x66e3e6f1),LL(0x7950dd7e,0xd5edf701),LL(0x0f6350c6,0x9ccb1096)}},
   {{LL(0x18f2470c,0x09db138d),LL(0xf613658f,0x63bd2290),LL(0x4feebab9,0x0bb64779),LL(0x7fdb1e71,0xfce4aee1),LL(0xa7f1f65d,0x7d5c0c61),LL(0x8d02d6cd,0x46405b61)}, {LL(0x6fdcb0d0,0x7cac0485),LL(0x2f8ec5af,0x85224c4b),LL(0xdb0aa864,0xb5879a59),LL(0xff94f8b5,0x75f391b8),LL(0x49c97f8e,0xa6c994ae),LL(0xd690b232,0x4d968fad)}},
   {{LL(0x67e0b4e7,0xf5cd290a),LL(0x7c1594b6,0xaa6fa680),LL(0xb63270be,0xebedfbd7),LL(0xa369bfee,0x574b410b),LL(0x020ea888,0x431cba5a),LL(0x56c71d47,0xd3a3102f)}, {LL(0xa90a853a,0x4894bfe0),LL(0x5f9c4b6b,0xd78bd98b),LL(0xd900c5c1,0x9b1324f6),LL(0x718c2147,0xc65c944d),LL(0xa987f634,0xf661de6b),LL(0x172628d8,0x0315e69f)}},
   {{LL(0x22ea5f1c,0x07c60c75),LL(0xa36bee4f,0x35beae34),LL(0xdcba8997,0xa8b00a09),LL(0x802ce50c,0xa77f1f3a),LL(0x2a2144b0,0x6c4050df),LL(0xab1b10db,0xf79bfa96)}, {LL(0x433a9b1c,0x9025d470),LL(0x90d9eec8,0xaf3e3917),LL(0x9ae2d535,0xbcad2d62),LL(0xeff0f6a9,0x7a152829),LL(0x925fa5a0,0xe87345cd),LL(0x0e84039c,0x6ce00720)}},
   {{LL(0x07f6a05a,0xdae449c0),LL(0x5bf26c9e,0xbc1b84f5),LL(0xb1c13820,0xe3b3f9ed),LL(0x4090598b,0x5442ad5b),LL(0x13749e4d,0x794ef656),LL(0x948b71c5,0xde809180)}, {LL(0xe203c5b5,0x4c72dc7d),LL(0x1b349fc4,0x8902b097),LL(0x225a1569,0xa899bedb),LL(0xe6ff3f53,0xeb7da73d),LL(0x7c0be37b,0x6ee8e160),LL(0xa31bf943,0x9ee667d2)}},
   {{LL(0xdb81146d,0x5017e145),LL(0x45c54db8,0xc7d2086d),LL(0xfa98234a,0x2541059d),LL(0x9985af98,0x4bf344d9),LL(0x7b5b7b1c,0x39737ed6),LL(0x87c411ad,0x8e246919)}, {LL(0xb877a75f,0x2fad8ced),LL(0x17e60ee2,0xe42352df),LL(0x404043f7,0x1a53d856),LL(0x863927a1,0x6c1f07a5),LL(0xb6892121,0x38d3a4f4),LL(0x01976c8f,0xf4c10920)}},
},
/* digit=60 base_pwr=2^300 */
{
   {{LL(0xc2ec3731,0xbcc88422),LL(0x10dc4ec2,0x78a3e4d4),LL(0x2571d6b1,0x745da1ef),LL(0x739a956e,0xf01c2921),LL(0xe4bffc16,0xeffd8065),LL(0xf36fe72c,0x6efe62a1)}, {LL(0x0f4629a4,0xf49e90d2),LL(0x8ce646f4,0xadd1dcc7),LL(0xb7240d91,0xcb78b583),LL(0x03f8387f,0x2e1a7c3c),LL(0x3200f2d9,0x16566c22),LL(0xaaf80a84,0x2361b14b)}},
   {{LL(0x246dc690,0x7a1a522b),LL(0x4b61ab70,0xb563cbe1),LL(0x3d4ac4ab,0x41bb4abe),LL(0x37f996e8,0xc52950b3),LL(0x79727761,0x01d991e6),LL(0x978fd7d2,0x35de93bd)}, {LL(0x5706d336,0x86bad5e6),LL(0xe7f26c20,0x10844155),LL(0x05757453,0x58ffeb77),LL(0x3939df77,0xbb186129),LL(0x6a78ea0f,0xbfdd394a),LL(0x6e33e1d3,0x907ff054)}},
   {{LL(0xb5733309,0xdb1cffd2),LL(0x0f9dd939,0x24bc250b),LL(0xa3c1db85,0xa4181e5a),LL(0xac55d391,0xe5183e51),LL(0xefd270d0,0x2793d5ef),LL(0xc0631546,0x7d56f63d)}, {LL(0x0c1ee59d,0xecb40a59),LL(0xbb5bfa2c,0xe613a9e4),LL(0x6c5830f9,0xa89b14ab),LL(0xa03f201e,0x4dc477dc),LL(0xc88c54f6,0x5604f5da),LL(0x2acfc66e,0xd49264dc)}},
   {{LL(0x0df93b34,0xa7f29532),LL(0x5c14df30,0x855934f2),LL(0xefae348c,0xd2f54ce9),LL(0xac52758d,0x5acb931c),LL(0xd22961a4,0x287b3e18),LL(0x748f8fe1,0x42a5516d)}, {LL(0x877224ca,0x1b62b341),LL(0xd30a4aa7,0xaff58db3),LL(0xbe8da847,0xbad78dad),LL(0x54f18276,0x85fa7109),LL(0x7c4bfdad,0xe2cc9d28),LL(0x2c75f237,0xbb131f76)}},
   {{LL(0x1c4dfa95,0x283dd7f0),LL(0x62c0b160,0xb898cc2c),LL(0x870282aa,0xba08c095),LL(0xf4e36324,0xb02b00d8),LL(0x604cecf2,0x53aaddc0),LL(0x84ddd24e,0xf1f927d3)}, {LL(0xe2abc9e1,0x34bc00a0),LL(0x60289f88,0x2da1227d),LL(0xcef68f74,0x5228eaaa),LL(0x3c029351,0x40a790d2),LL(0x8442e3b7,0xe0e9af5c),LL(0xa9f141e0,0xa3214142)}},
   {{LL(0x03844670,0xcdcdd7d7),LL(0xb4a23f91,0x79ec59af),LL(0xc00ce5c3,0x5923c569),LL(0xc589d0c7,0x099c17ff),LL(0x89fa6fe6,0x0335eeea),LL(0xa4e868c4,0x916bcaca)}, {LL(0xfb687bd5,0xb7037325),LL(0x9853b564,0x57d6bca7),LL(0xd5e26d28,0xdf3132ef),LL(0xde919cbe,0x7ed994b8),LL(0x6fbbb18d,0x12df67cd),LL(0x6baff508,0x516e07c0)}},
   {{LL(0xf9a58e3d,0x72f4949e),LL(0xa48660a6,0x738c700b),LL(0x092a5805,0x71b04726),LL(0x0f5cdb72,0xad5c3c11),LL(0x554bfc49,0xd4951f9e),LL(0x6131ebe7,0xee594ee5)}, {LL(0x3c1af0a9,0x37da59f3),LL(0xcb040a63,0xd7afc73b),LL(0x4d89fa65,0xd020962a),LL(0x71d824f5,0x2610c61e),LL(0x3c050e31,0x9c917da7),LL(0xe6e7ebfb,0x3840f92f)}},
   {{LL(0x97e833e4,0xf2ec9ef5),LL(0x34ec7e41,0x97bdef97),LL(0x7d2ac6e3,0x90e2b238),LL(0x0318a3b7,0xcf682b12),LL(0xea84a7a0,0x7fe76089),LL(0x16546d05,0x85c489f9)}, {LL(0x6abdda05,0xf987118f),LL(0xaa4b95fc,0x675cf998),LL(0x888a7e8c,0x544c7774),LL(0x63ec5831,0xbd2647ba),LL(0xfd2fe985,0xb479cea3),LL(0x28d163e8,0xa0421345)}},
   {{LL(0x8d8b8ced,0x50fbd7fe),LL(0x47d240ae,0xc7282f75),LL(0x1930ff73,0x79646a47),LL(0x2f7f5a77,0x2e0bac4e),LL(0x26127e0b,0x0ee44fa5),LL(0x82bc2aa7,0x678881b7)}, {LL(0x67f5f497,0xb9e5d384),LL(0xa9b7106b,0x8f94a7d4),LL(0x9d329f68,0xbf7e0b07),LL(0x45d192fb,0x169b93ea),LL(0x20dbe8c0,0xccaa9467),LL(0x938f9574,0xd4513a50)}},
   {{LL(0xe5947c6f,0xd93506a4),LL(0x39b81d08,0x4340d76a),LL(0x17930d30,0x741aee59),LL(0x18fdb81c,0xfea3d99a),LL(0x289bcb07,0x1088ff6b),LL(0xb7c082c6,0xc6b45602)}, {LL(0x453d8d69,0x50e2baab),LL(0xe893e183,0xda9bf561),LL(0xb29a284d,0x0af25f86),LL(0x73e01380,0x0e92e674),LL(0x2be00e59,0xe173a0e3),LL(0xada8954a,0x402d2f3d)}},
   {{LL(0x054cb874,0x841c96b4),LL(0xa3c26834,0xd75b1af1),LL(0xee6575f0,0x7237169d),LL(0x0322aadc,0xd71fc7e5),LL(0x949e3a8e,0xd7a23f1e),LL(0xdd31d8c7,0x77e2d102)}, {LL(0xd10f5a1f,0x5ad69d09),LL(0xb99d9a0b,0x526c9cb4),LL(0x972b237d,0x521bb10b),LL(0xa326f342,0x1e4cd42f),LL(0xf0f126ca,0x5bb6db27),LL(0xa4a515ad,0x587af22c)}},
   {{LL(0x0399721d,0xca9cb389),LL(0xa3291479,0x03ad9f4a),LL(0x6dee003d,0xd85b5df5),LL(0x64a4f83a,0xe1fa7b02),LL(0xb73f7324,0x01c4cbfd),LL(0x5cf2ddf4,0x707010d4)}, {LL(0xb12e02f8,0x3c6df430),LL(0x85531489,0x921a2901),LL(0x91d1022c,0x302fc77c),LL(0x342d8f3f,0xc3733ec0),LL(0x6195a665,0xb83bc75f),LL(0xa79f8027,0x4a14b9e7)}},
   {{LL(0xb12e542f,0x1123a531),LL(0xb9eb2811,0x1d01a64d),LL(0xf2d70f87,0xa4a3515b),LL(0xb4bd0270,0xfa205234),LL(0x5eda26b9,0x74b81830),LL(0x56578e75,0x9305d6e6)}, {LL(0x9f11be19,0xf38e69de),LL(0x44dbe89f,0x1e2a5c23),LL(0xfd286654,0x1077e7bc),LL(0x0fca4741,0xd3669894),LL(0x278f8497,0x893bf904),LL(0xeb3e14f4,0xd6ac5f83)}},
   {{LL(0xe2a57359,0x9f0e5428),LL(0x14998c16,0xc690a3c7),LL(0xde37e07e,0xd73c3ca2),LL(0xdba0bc0d,0x2ddf91b8),LL(0x7570ae71,0x69d834b2),LL(0x735195a6,0x2ac8bed4)}, {LL(0x3b1fcc5c,0xcd8c51ff),LL(0x1ba6863f,0x7aa8cf4e),LL(0xae70f428,0xebb69e72),LL(0xaa9e936d,0xa29409df),LL(0x5a332b9b,0x43f6ee80),LL(0xc2eab0a9,0x0de49efa)}},
   {{LL(0x488f5f74,0x327b9dab),LL(0xcab7364f,0x2b44f4b8),LL(0x19b6c6bd,0xb4a6d22d),LL(0xfc77cd3e,0xa087e613),LL(0xb0b49bc7,0x4558e327),LL(0xcd835d35,0x188805be)}, {LL(0xc1dc1007,0x592f293c),LL(0x6af02b44,0xfaee660f),LL(0x904035f2,0x5bfbb3bf),LL(0x79c07e70,0xd7c9ae60),LL(0x234896c2,0xc5287dd4),LL(0xcb0e4121,0xc4ce4523)}},
   {{LL(0x2310333e,0x04baa176),LL(0x7b9bad46,0xdc75e35f),LL(0xc6cd6108,0xc4a6031d),LL(0x30bf87a5,0xba2534d0),LL(0x31e497cc,0x7ebc6e21),LL(0x851fd665,0x8a2a82b4)}, {LL(0x6d5faf40,0x9ecae011),LL(0x96956ecb,0xfa3a6d7f),LL(0x2fa52782,0x39e8a9c2),LL(0x236d442e,0x74c93801),LL(0xb1c289ce,0x8b21ba23),LL(0x25c769cf,0x7f3e221b)}},
},
/* digit=61 base_pwr=2^305 */
{
   {{LL(0xca114c4a,0x761e10e2),LL(0x894301b3,0xe39d121d),LL(0x3dbc6fca,0xa0870ff4),LL(0xcbe0ba8a,0x97651286),LL(0xc0f1ff6a,0x47d46075),LL(0x3abeb5b6,0x18669c84)}, {LL(0xad8d9309,0x1234c80e),LL(0x1f6f97ff,0x1ccbe4d5),LL(0xd82ab780,0x399a2d41),LL(0xde426e50,0x8a03afaf),LL(0xca6dde77,0xa2bcb109),LL(0x0618f5ec,0x840e13b0)}},
   {{LL(0xec645a62,0x15d47e52),LL(0x8d6d4423,0xabe0ddb3),LL(0x70cddb11,0x51226a30),LL(0x2b5a8db7,0x63a253d3),LL(0xbef37d65,0xe8be4d1f),LL(0xc0920b91,0x41e625d9)}, {LL(0xd9d040ec,0x08b713a8),LL(0xc450cdba,0x467fb08d),LL(0x917ee393,0xa8975877),LL(0x1528cd12,0x294792e9),LL(0x37daf6aa,0x4512dc8c),LL(0x197a99b9,0xa83becc9)}},
   {{LL(0x538d92d8,0x3b21dc1f),LL(0xc005aa86,0xc80b22b3),LL(0x0da87d65,0xf536e5d3),LL(0x0cd999a0,0x4ce10edf),LL(0x50e08f5d,0x89491814),LL(0x526647e6,0x77fd8f2e)}, {LL(0x250099fd,0xcb207ee9),LL(0xfd6aa078,0x03c7d1ab),LL(0x25e0cf15,0x7d4940d2),LL(0x067fa052,0xb688b311),LL(0xa98b2e21,0x89308326),LL(0x72311eab,0x3ee4cc2b)}},
   {{LL(0x68d7dfcf,0x37be5d3f),LL(0xb945e6f2,0x97bdbd49),LL(0x9d1569e7,0x165a24b5),LL(0xb4e293ab,0x254aaf59),LL(0x6fb7c0a4,0x3c751fbd),LL(0x5018cb18,0x14eda4ba)}, {LL(0x1b5f6aed,0xacb3b897),LL(0x1e4b6b78,0x6d10be44),LL(0x621df6d7,0x245d7258),LL(0x185f0e2a,0x2af0e283),LL(0x8fddbd81,0x1e7edc81),LL(0xc538d02a,0xbd1e6c72)}},
   {{LL(0xaa006a4f,0x12014812),LL(0x83374604,0xf84aa1a2),LL(0x0ee4a8ae,0x9f8475d9),LL(0x135811df,0x37a1b21d),LL(0x1166af52,0x34143171),LL(0xf5a33016,0x204dd449)}, {LL(0x372b6edd,0xc838d3d5),LL(0x3987611c,0x314f3053),LL(0xd112605c,0x819adbe0),LL(0xf6b32c5c,0x45da01ae),LL(0x7e3b13a9,0x3deb3018),LL(0x79058926,0x0e78a3a2)}},
   {{LL(0x42adface,0xaab5f0c7),LL(0xcb580132,0x50d9f53f),LL(0xb5fd6ebf,0x68a3c689),LL(0x9cde184a,0xdea2f2ee),LL(0x13a98466,0x8c174c44),LL(0xcb4d921b,0x8c4e2ae1)}, {LL(0x2d4c6d5a,0xba973c6c),LL(0x61d2ec5f,0xc6150714),LL(0x666d8bf9,0x2aba1375),LL(0xd41272cd,0x2fa2768f),LL(0xa0bc34bf,0x49f3b8d7),LL(0xca45e5fc,0x61118166)}},
   {{LL(0x2665ae2e,0x92f565ea),LL(0x1aefd472,0xaaa91acf),LL(0xfbb062aa,0x3878c718),LL(0x7de3c64d,0xde46e7d8),LL(0xbd506a76,0xff9900a4),LL(0x3daa73b9,0x4e30ed72)}, {LL(0x7cbaff42,0xd6be9446),LL(0x5dd691f4,0x26fe6305),LL(0x6d393800,0x64ef093c),LL(0x7bb8f155,0x2448c67a),LL(0xe5d732d7,0x9da6e75f),LL(0x50b080e0,0xe837a602)}},
   {{LL(0x844626a2,0x6858b674),LL(0x0cbba6a6,0x610cd40f),LL(0x29d9194d,0x324e674e),LL(0xdcb30a51,0x2dc6fdf6),LL(0x528aa549,0x3f3ecb77),LL(0x23ffaa92,0x0721f8f9)}, {LL(0x27a77538,0xd8efcbd6),LL(0xd6162c9c,0xf4e642bf),LL(0x4cf4a16f,0x04f2b0b7),LL(0xbbf335fd,0xbc0bb49f),LL(0x5a928c36,0xc6b6e5bd),LL(0xd893dd45,0x981b01f4)}},
   {{LL(0x518f01dc,0x2836b977),LL(0xa06c7362,0x117e833e),LL(0x31152b22,0xcda89f2c),LL(0x34be0102,0x2084c5c4),LL(0x2bfac8c2,0x6478de75),LL(0xebda5ede,0x9f7e901a)}, {LL(0x5aa25b6f,0x5f43adfc),LL(0xcdd0eee3,0xfb719dca),LL(0xf9b16d84,0x14431b2d),LL(0x97f04b2b,0x846261af),LL(0x1edd7d4d,0xcc6100b8),LL(0x87cde5f4,0x6197c87f)}},
   {{LL(0xa01cb6d9,0x4a9e6281),LL(0x87065307,0x699c0719),LL(0xdffa58be,0xa8ca4971),LL(0x89efaadd,0x8adc304f),LL(0x1f3c79d2,0xeef0af15),LL(0x581587e8,0xb3be9c6c)}, {LL(0xda0be326,0x79010ad2),LL(0x4be00f8b,0x4f361e1e),LL(0xc53f3c74,0x180e66ba),LL(0xb2521c2d,0xa668c3f3),LL(0xabb73a09,0x60bc2fa6),LL(0x4392692f,0x0bda0ff6)}},
   {{LL(0x9aa349d0,0x2e88f308),LL(0xa7cf751f,0x9f19df26),LL(0x4885be75,0x1e0229c7),LL(0xa32fce2e,0x6770eee2),LL(0x562d99d7,0x448366e4),LL(0x8bc7484a,0x7670bd68)}, {LL(0x92d83c6b,0x15374837),LL(0x14f7c403,0xe0f499f8),LL(0x8a6b78f6,0x4cf02671),LL(0x9849e689,0x75f6e30a),LL(0xcad5065a,0xaf6fe2a5),LL(0x43ba98aa,0x6378401b)}},
   {{LL(0xd2446552,0x26d6d225),LL(0xd2600e42,0x9b74929e),LL(0x447126d2,0xcef3a052),LL(0x4d1c7e0a,0xcba2f70c),LL(0x020d33a3,0x0250a96f),LL(0xa5e587d9,0x9c946f94)}, {LL(0x86653ae6,0xc7d4343f),LL(0x8884e9c6,0x9c832859),LL(0x1c234f88,0xa44fa8c5),LL(0x987f04dc,0x7193e6db),LL(0x6a25ff37,0x24efebac),LL(0x953b3db8,0xe23f0a14)}},
   {{LL(0x491d9ba9,0xb13db9bb),LL(0x5556de42,0x7105da91),LL(0xbd48b3d9,0xafa75d3b),LL(0xed6c519c,0x10246797),LL(0x1aa866f5,0x83b27882),LL(0x54f64e96,0x66f4ceaa)}, {LL(0x43c07b18,0xf4f03faf),LL(0xdd18ddec,0x97eed374),LL(0x6c454f23,0x43b702f1),LL(0x3cb61e31,0x3c53810c),LL(0xedcaea49,0x8a50cfe9),LL(0x89a6acd9,0xfb70772b)}},
   {{LL(0xa01edb12,0xedc09655),LL(0x7ad80675,0xdaa9f823),LL(0xf6a1052d,0xb7f23b6c),LL(0x697dc18b,0x22dc809f),LL(0x6c8bcd69,0x0453593f),LL(0xc9b43cf0,0x80f76d8b)}, {LL(0x8c5b3ba9,0x8e781e17),LL(0x66a0b318,0x66544c7c),LL(0x9d95b620,0xedd99cbd),LL(0xbdf0933e,0xc36c4334),LL(0x8b59acf3,0x3d550b68),LL(0x6ca6a2b7,0xcd7d1701)}},
   {{LL(0xc3d5ad9a,0x94457602),LL(0x3a7a1abc,0xa1a9608e),LL(0x31a107cb,0x16eb2310),LL(0x987bf106,0x5d921026),LL(0xd2ccc296,0xec2e5789),LL(0xb8f2fbd4,0x87b86d1b)}, {LL(0x6da88d7d,0x939b5802),LL(0xe19707e2,0x22fa6ef8),LL(0xc547ce83,0x8b95bd5d),LL(0xd16fb119,0x91268688),LL(0x6e3627f2,0xbf199148),LL(0xc31ab346,0xd072bdf2)}},
   {{LL(0xf810465b,0xfb083c2a),LL(0x02ce0dee,0xb66a8de9),LL(0x47a81b95,0x6e4130e7),LL(0x58a98737,0xcd704dc6),LL(0x592829c9,0x842ae329),LL(0xbe20dd63,0x99bedc34)}, {LL(0xd53b2df4,0xabee8e55),LL(0x6010b37c,0x6ce65758),LL(0x467112b9,0x781f39b2),LL(0xbe341038,0x6f06058f),LL(0x12a2f8be,0x5effdca5),LL(0xaf34466e,0xaa9bdad7)}},
},
/* digit=62 base_pwr=2^310 */
{
   {{LL(0x9c216137,0xab86639e),LL(0x82b18d64,0x45a12fb8),LL(0xd763f0bb,0xb5734418),LL(0x11a9802c,0xd2cc3322),LL(0x81269b8a,0xe41d7db8),LL(0x2ecfa355,0x91072fc1)}, {LL(0x04ce306f,0x59d69125),LL(0xa131b86d,0x916d9d4d),LL(0x8a739738,0x84478b6b),LL(0x1cc83ae3,0xe86ad7d9),LL(0x797ccd97,0xbc9b2084),LL(0x694944c6,0xc1e94af4)}},
   {{LL(0x895c0318,0x585edee3),LL(0x45e8205b,0x775e142f),LL(0xd85ad31f,0x3bd7924f),LL(0x9124bffa,0x2e7d8f91),LL(0x44c62868,0x885397c0),LL(0x7fda9f5d,0xc0c2dff4)}, {LL(0xc14e693d,0xd302582e),LL(0x6cec31ba,0x53d6e33a),LL(0x63653c06,0xb0216b5b),LL(0x9c70dad4,0x8f08a1ad),LL(0xffbba93d,0xccf014aa),LL(0xa33f12b7,0x900b0d2c)}},
   {{LL(0x9b8cfa41,0x0dd2395e),LL(0xd4f92a44,0x50e203ab),LL(0x6630023b,0x7280aff1),LL(0x07de820e,0xfcce59dc),LL(0xbc8189ad,0xa686be05),LL(0xaac70b7e,0xac4b59bf)}, {LL(0x7a3c71ac,0xd2c0070e),LL(0x35ac1c47,0x1d550add),LL(0xfb881c1b,0xd42b6389),LL(0xd0dafd42,0x57ca3fcc),LL(0xbe26ccc9,0x909e8284),LL(0xa002235c,0x1abe7595)}},
   {{LL(0xa6a1913d,0x1e34781a),LL(0x7d0adc38,0x9a8f3228),LL(0x28af85ba,0xfc185ccc),LL(0x3ae9ba11,0xc923d78b),LL(0xa7bdb313,0x7d494d7e),LL(0xaf8f8b87,0xf774dfa5)}, {LL(0x16e863b8,0xc178ccc1),LL(0xa8899691,0x2d472f2a),LL(0x80a50372,0x608747cc),LL(0xe6f90197,0x8147aa90),LL(0x78c2f216,0x4683d4c9),LL(0x552f3b51,0x8323652c)}},
   {{LL(0xcc2c9a2a,0xa5c08e8b),LL(0x8baaf0fc,0x70e1b405),LL(0x9e36e50c,0xf29e1e5c),LL(0x80f258c5,0xa3d90800),LL(0xecad4498,0xc9ceac25),LL(0xca32f3fe,0xcb73130f)}, {LL(0x48b3863d,0x2dbe620c),LL(0x14ff53bf,0x8c52727f),LL(0x6b45e9b8,0xb60b22a8),LL(0xf6483c5d,0x81e05bc0),LL(0xcd542972,0x217caa6b),LL(0xfa780778,0xffab716a)}},
   {{LL(0x0d7410d6,0xe5d3e0d8),LL(0x2be432c9,0xcfa9ed74),LL(0xa85a0686,0x60044434),LL(0x6ad6918a,0x93b35716),LL(0x051762be,0x1a3c3e6d),LL(0xb0ab32d6,0x80813589)}, {LL(0xaad403fd,0x64214b92),LL(0x4d3fb746,0x684befc1),LL(0x79515046,0xaca5a514),LL(0x72e84485,0xacdba034),LL(0x287d9e97,0x61aa2834),LL(0xcad222e7,0x07a515a5)}},
   {{LL(0x8af19670,0xb0309306),LL(0x34c6bf0f,0xd784125a),LL(0x255a8396,0x0b425ee0),LL(0xfb541162,0x91076433),LL(0x86f47a0e,0xc4d81885),LL(0xfd7bc7c1,0x3b767d54)}, {LL(0xbee196e9,0x98b405d3),LL(0xedaccf4a,0x4ef9c511),LL(0x03f4f1a6,0x5a6deb65),LL(0x1b4c5104,0x4a22ca64),LL(0x9145ce41,0x2cce3667),LL(0x3206810d,0xd0518752)}},
   {{LL(0x037bebad,0x29d81538),LL(0xd9e0b78c,0x76e52c73),LL(0x8783d1fd,0xaa4ace6e),LL(0xf0e3c126,0x9c14ebdd),LL(0x6eca4b71,0x0eb1c08d),LL(0x1c91df35,0xd10c6b96)}, {LL(0xe81bb84a,0xdb8119bb),LL(0x17e3ceef,0xf784d3c1),LL(0x35436f81,0x053c9168),LL(0x9b18d212,0xeb41ccbb),LL(0xb1bc3497,0x93b3fb43),LL(0x8c1ced81,0xd85a7c75)}},
   {{LL(0x811af84a,0x004105c3),LL(0xa7934a0f,0x01307934),LL(0x9b3226a1,0x179fd49b),LL(0xde6834b4,0x195d9e5c),LL(0x0e6051bd,0xfbb79dc0),LL(0x367f4487,0x354273ed)}, {LL(0x74fb892d,0x4afa9d45),LL(0xa1b7f3bb,0x03ae905e),LL(0x592f6122,0xea32cd5d),LL(0xf1103301,0xa758eed2),LL(0xc59d1cc8,0x9dde4238),LL(0x51022a42,0xe2760bcc)}},
   {{LL(0xd377d7b0,0x54f84d70),LL(0x3344bc4e,0xb745d190),LL(0x8f33aa53,0x1c693ed0),LL(0x8bfbee7f,0x990ed45f),LL(0xe9b258fb,0xad620c9f),LL(0x1a54bf46,0x465ccb10)}, {LL(0xebc40951,0x5330a0d3),LL(0xa405da61,0x34423e8c),LL(0xb83043b6,0xeef1ce78),LL(0xac06d182,0x99678f22),LL(0x1802f14c,0x9213f57d),LL(0xadf11fda,0xf8549616)}},
   {{LL(0xb6e392e1,0xf31796d2),LL(0x93b3395e,0x199d6248),LL(0x12f9b763,0xef14c7c2),LL(0x43edb7a5,0x721ebf21),LL(0x5e96f3ba,0xa40b8894),LL(0x4cff8394,0x8770608c)}, {LL(0x8d0def0e,0x990c99ae),LL(0xa15a5649,0x292b26df),LL(0x91ca89d7,0xa98fda2c),LL(0x973e5f5f,0x916cb1b4),LL(0xa72de0bb,0xa2823f13),LL(0x8cd3219d,0x415f7bd2)}},
   {{LL(0x3ed03d5f,0x521f4af7),LL(0xeaf9064c,0xe3461f66),LL(0xae03777f,0xad099ab7),LL(0xb65f73ff,0x541cadcd),LL(0xa86059b9,0x53430463),LL(0x043e9f82,0x8ff88fe5)}, {LL(0xe42cde45,0xd515f4c7),LL(0xf41c3269,0xf7f3dec3),LL(0x7ef1b8ff,0x7bed5356),LL(0x1295b5fe,0x8782b45f),LL(0x03917627,0xab54ebaa),LL(0x8787ed9f,0x8516beb2)}},
   {{LL(0x65b68624,0xba7df5a9),LL(0x6e7d58bb,0x30b4d6ed),LL(0xdbb81762,0x67e52341),LL(0x0deeac1d,0xd697ab1b),LL(0x5577ea92,0x01d15e8e),LL(0x98fb38da,0xbb12d724)}, {LL(0x4e04908d,0x302faa6d),LL(0x09b90a9c,0x66cf6cb9),LL(0x98d96736,0xcd665dbd),LL(0xb86f3af4,0xf7d3c528),LL(0x1d8b07f4,0x4844c754),LL(0x1eaf7dc0,0x2a77d7b9)}},
   {{LL(0x1d70eb73,0x53e59f25),LL(0xe69d0525,0x8aed17af),LL(0x64413768,0x26ddc178),LL(0x5e48c349,0xa7c8d40f),LL(0x87ff01fb,0x29ad92d1),LL(0x965b2de1,0x8f4e1b3b)}, {LL(0x1446eca2,0xb83cfadf),LL(0xe609d416,0x7432bda4),LL(0xf1c7de69,0xcf97e8a3),LL(0x32f55f07,0x45899bd8),LL(0x51175738,0x41a68117),LL(0xb8efff21,0x89eeb115)}},
   {{LL(0x936c3eba,0x54a01e60),LL(0xec01b12c,0xf326fe96),LL(0x66e4de2e,0xcdfaf003),LL(0x392fd0a8,0xc53dba07),LL(0x6ec46004,0x00d9b80f),LL(0x3ba63f8d,0x84d59be8)}, {LL(0x9dea6062,0xbac4ea43),LL(0xadd568ca,0xb4b4845b),LL(0xd225e2d0,0xa6ca3d34),LL(0xb50070a2,0xce72955d),LL(0x21c78b68,0x56e5c913),LL(0x999488a3,0x888eb198)}},
   {{LL(0x11c92f34,0x5255508c),LL(0xa294d382,0x9a346cf3),LL(0x3095205b,0xd9765eea),LL(0x2c470ef7,0xfea2ed70),LL(0x9c40bf0a,0xf5e8a0fc),LL(0xe4137a16,0xb572390e)}, {LL(0x2bf2f545,0xb9175371),LL(0x58cd9cc7,0x2c2d0f4c),LL(0x02385486,0xbea6bce9),LL(0xa8bc3a94,0x46208408),LL(0x3ac45044,0x64a87a2a),LL(0x7df70151,0xe40da33c)}},
},
/* digit=63 base_pwr=2^315 */
{
   {{LL(0x39161b8a,0xee9e25d9),LL(0xe2eead91,0x8763f2a2),LL(0xd2d91300,0xd2fc1157),LL(0xffcbe50f,0xe7597e2f),LL(0xe11d376e,0x4be3814f),LL(0xdbf14562,0x1eab3d7e)}, {LL(0xc0ad183a,0x38a107c0),LL(0x7c753bbf,0x82976626),LL(0xcaebd481,0x18014e09),LL(0xf9ace60d,0xb28c331b),LL(0x211cb8e7,0xe8fba04f),LL(0xe42dc65e,0x41c4b797)}},
   {{LL(0xc3e88580,0x009dc2f4),LL(0x99db1fb5,0x4a405be8),LL(0xec5d91fe,0xc89bfaa2),LL(0xf160afcb,0x461be9a0),LL(0x7d7566b5,0xfdd084bb),LL(0xe48099a2,0x795275e8)}, {LL(0xfe9815db,0x1b461fc9),LL(0x73627bbc,0x576214cd),LL(0x9f09a206,0x3246332d),LL(0x6941d6ef,0xbde4c0c3),LL(0xf387f5f6,0x44ef03fd),LL(0x57b63400,0x99c8ac01)}},
   {{LL(0x2f6e4301,0xaa512f20),LL(0xbf94a1cc,0xef668a5f),LL(0x15861b88,0x08713c30),LL(0xc99bb2b6,0x49d47551),LL(0xe2f0258e,0x6db5f812),LL(0x998d7435,0x70c9b299)}, {LL(0x5d176ae0,0x46168e1c),LL(0xf730ec30,0xec3306e4),LL(0xab69c15d,0x49439df3),LL(0xea0143e4,0x1040408b),LL(0xbc549b0a,0xb48ab8ea),LL(0x10f89223,0x4aa38bbf)}},
   {{LL(0x9598f49a,0x7e485159),LL(0x9629305b,0xbdac3d5e),LL(0xa6fbabfb,0x20de0daf),LL(0x8f09fff7,0x04f01583),LL(0x6a06994f,0x5a056297),LL(0x6e3ccd33,0xf51dac8f)}, {LL(0x3af507b8,0xc087ef9c),LL(0x6a5c6663,0x525ab76e),LL(0xd916ee93,0x4fc04814),LL(0xd23d140b,0x3369c978),LL(0x1662028f,0xb0fcd70f),LL(0xe1e28adb,0x2ca77de2)}},
   {{LL(0xc512bc71,0x838acd1b),LL(0xdc18afd0,0xac06d6bd),LL(0x9ec45f4b,0xc991c1e3),LL(0xcc27c68e,0x667c5e89),LL(0xed07f829,0x0e059b04),LL(0xcec4b3a7,0xceccf1d4)}, {LL(0xb953f9a1,0x3d9c2dc9),LL(0x2d599b16,0x4be2f7e7),LL(0x97256c26,0x1a2054b1),LL(0x8b4fdfeb,0xcf66fa47),LL(0x8134d7ef,0x896cc1b3),LL(0xd41dadbc,0xa17264ca)}},
   {{LL(0x37627e56,0xe3ccfe8e),LL(0x7b6b21a6,0x00733a86),LL(0xb605c427,0x3f13e2cb),LL(0xb0d80992,0x5ee12395),LL(0xb9991381,0x4dcaea94),LL(0x8c4c4b6e,0x4cfed7ee)}, {LL(0x7f7f45df,0xd7aad54b),LL(0xb3809bf8,0x2229407f),LL(0x68048fd9,0x6eb31eee),LL(0xd57225fc,0x693842df),LL(0xa88dfd3f,0x3e62cd85),LL(0xd5462cf1,0xc6307d53)}},
   {{LL(0xf344f5fd,0x2d15615e),LL(0xa7f23989,0xe0ba6a8a),LL(0x1c84e3f2,0xbbfc5804),LL(0x6f4ba826,0x22ffeaae),LL(0x94292682,0x1e9bf274),LL(0x46c02af9,0xc768f891)}, {LL(0x177cdafa,0x894127d6),LL(0x2acdc791,0x8d0523da),LL(0xdc78c3c4,0x71ada9ae),LL(0x2c532a01,0xf21dbbb9),LL(0xacb20fda,0x0c797d5e),LL(0x16cf57b0,0x1ff99d76)}},
   {{LL(0x493c1d64,0x99b5f150),LL(0xfb74075e,0x3422b656),LL(0xff19bf24,0xe7493900),LL(0x260925ed,0xc82e5b80),LL(0xc0ea1eaf,0x3398d340),LL(0x1287121e,0xe7de2ba1)}, {LL(0x87847031,0xea6dfb0b),LL(0x566af2f2,0x73bed0a1),LL(0x12012999,0xe26678bf),LL(0x32e5cebc,0xb5369e4d),LL(0x6d181e32,0x2304eac8),LL(0x3d364add,0xafdbd954)}},
   {{LL(0x75da4189,0x5b1a53ca),LL(0x2eb4862b,0xa9048580),LL(0x2783ad6a,0x31942409),LL(0x1a9e025e,0x15a4c5e1),LL(0x13837199,0x841bc533),LL(0xe642954a,0x6e9d3e14)}, {LL(0xd436ec5c,0xf4a02bbd),LL(0xc6d6ad53,0x62fe177b),LL(0xac86425a,0xedbf1e4e),LL(0xd9f752f5,0xff9359c8),LL(0x2d7ad656,0x79c685d9),LL(0xfdde9052,0x8d82c0c4)}},
   {{LL(0x702f640d,0xf55f868e),LL(0x1dedda11,0xe459aa9b),LL(0xbb5ba193,0xbec0ff9b),LL(0x57724703,0xf7325c49),LL(0x23e0e4fb,0x5ab8f063),LL(0xecb0fd7c,0xfbf02e91)}, {LL(0xa2e5fa31,0xcc72e8da),LL(0x32cb53cf,0x47de2528),LL(0x4252763c,0xbfa646e6),LL(0xb8d81de3,0x7a769efe),LL(0x1e772f00,0xf5ec7003),LL(0x2729aa5e,0x049bea9a)}},
   {{LL(0x759090d6,0xe987ba54),LL(0x619ef08b,0x904d6901),LL(0x2024a6fc,0x9e16d138),LL(0xa9f3b7e4,0xb6f0459b),LL(0x17ee069a,0x1f2a5308),LL(0x2be31049,0x99403b2e)}, {LL(0xbfb2f288,0xba1663c6),LL(0xc7a92b41,0xf829195c),LL(0x8ae621b1,0x89b915ee),LL(0x50f8ea92,0x3fbbb1e1),LL(0x8c901ddc,0xb1fe7f97),LL(0xbbc69ca4,0x16d1f62c)}},
   {{LL(0xfda072db,0x51f19bb3),LL(0xe3f7e0a2,0xa815459f),LL(0x987112ca,0x5f7cde2f),LL(0x759de2cb,0xdc51d948),LL(0xed49bd98,0x9d05c410),LL(0x364341fd,0xf063ab99)}, {LL(0xd1aa0a11,0xd7869d68),LL(0x5d862d01,0xc2029106),LL(0xc2591073,0x7f258180),LL(0x6ebc4ebc,0x7b90fc7a),LL(0x3dda1d68,0x5565390f),LL(0xa44e4493,0xae77fca8)}},
   {{LL(0x47c49ee8,0x97564e48),LL(0xab4ebef5,0xc56bb5a9),LL(0x7b4f86bc,0x80d96941),LL(0x41026cf0,0xa594b4e5),LL(0x5a89ece9,0xd56c8996),LL(0x6a0f922b,0xbcf60931)}, {LL(0x1103475c,0x70259616),LL(0x8a2a2abb,0xb1224fb5),LL(0x715cd61b,0x0a437a03),LL(0x739921ed,0xcbe2d2b2),LL(0x385541c4,0xf3b1b5e9),LL(0xae010520,0x5d0984f4)}},
   {{LL(0xcfd9295e,0xb4a2742d),LL(0xae929cd0,0x9cd36774),LL(0xdd7fcf4f,0xb15fadcc),LL(0x37d4fcc6,0x0b1fa2b3),LL(0xf01c7ab7,0x242c7b26),LL(0x50574cc9,0x2be8131b)}, {LL(0xbd89a03c,0x6ee50f42),LL(0x005e7765,0xc7f6ff8f),LL(0x8420501b,0x04d13af1),LL(0x1b6e7d2a,0xc22e092b),LL(0xe9516f80,0xa393be7e),LL(0xb80bb5b5,0xa2593652)}},
   {{LL(0x8b23bebb,0x5caa5da6),LL(0x1fdbbdf4,0xa1ad33e8),LL(0x4e5c1de0,0x18dc93cf),LL(0x5bd9e178,0xc3e6addb),LL(0x7cb8cd03,0xf30d517e),LL(0xf1abc417,0xbb84ce54)}, {LL(0x67699659,0x0774b64c),LL(0xb7d4a401,0x228005b9),LL(0x80b2d3d2,0xd8c2ec5b),LL(0x3450ba7f,0x419c4cd9),LL(0x789286a6,0x520ae681),LL(0xaa8bcfbb,0x24b67ea9)}},
   {{LL(0x0f74808f,0x9e41b9b7),LL(0x0c061bdb,0x2d835dae),LL(0xf272346c,0x67e50c8c),LL(0xdef57493,0xc98a5ef5),LL(0xa02676fc,0xc2dea8af),LL(0x6ace4659,0x59508de2)}, {LL(0xda6cd733,0xc2b707aa),LL(0x4be7bfb9,0x6c1f226a),LL(0xa778c20b,0x5b580fa2),LL(0x57af166b,0x272c3a1d),LL(0xca78ce62,0xe47a64a9),LL(0x71d35087,0xd12db7d7)}},
},
/* digit=64 base_pwr=2^320 */
{
   {{LL(0x1a42e5e7,0xc20fb911),LL(0x81d12863,0x075a678b),LL(0x5cc0aa89,0x12bcbc6a),LL(0x4fb9f01e,0x5279c6ab),LL(0x11ae1b89,0xbc8e1789),LL(0xc290003c,0xae74a706)}, {LL(0x79df3f45,0x9949d6ec),LL(0x96c8d37f,0xba18e262),LL(0xdd2275bf,0x68de6ee2),LL(0xc419f1d5,0xa9e4fff8),LL(0xa52b5a40,0xbc759ca4),LL(0x63b0996d,0xff18cbd8)}},
   {{LL(0x844eefc5,0xf6827150),LL(0x4515ef68,0x002e82c4),LL(0xc51916c4,0xa46c8f55),LL(0x61ee081f,0x98c3524b),LL(0xad64872a,0x5ab7f2c2),LL(0x7e555faa,0x0b503ff0)}, {LL(0xb4c58d29,0x802e0d23),LL(0x2fd917fe,0x12289040),LL(0x7af20d26,0xb56d1908),LL(0x6be50784,0x8d619e21),LL(0x1372b851,0x10fdbb72),LL(0x4935576e,0xf2c1673e)}},
   {{LL(0xd7dd47e5,0x73c57fde),LL(0xd49a7f5d,0xb0fe5479),LL(0xcfb9821e,0xd25c71f1),LL(0xcf6a1d68,0x9427e209),LL(0xacd24e64,0xbf3c3916),LL(0xbda7b8b5,0x7e9f5583)}, {LL(0xcf971e11,0xe7c5f7c8),LL(0x3c7f035e,0xec16d5d7),LL(0xe66b277c,0x818dc472),LL(0xb2816f1e,0x4413fd47),LL(0x48383c6d,0x40f262af),LL(0x4f190537,0xfb057584)}},
   {{LL(0x51a135f6,0xd97a9b14),LL(0x97b4df14,0x6d16aaf5),LL(0x54818818,0xc57160c2),LL(0x1d59be44,0x4dbdeab6),LL(0x81f2b247,0xb93a9dad),LL(0xecbcab33,0xe2868cf5)}, {LL(0x83a86711,0x5e1ce828),LL(0x29c55428,0x29a9ca2f),LL(0x2d82b0df,0xe716273a),LL(0xac8ff52f,0xb017f5f6),LL(0x70ea7ccd,0x7563e799),LL(0x3f0e674b,0x5fedf0a6)}},
   {{LL(0x08962f6b,0x487edc07),LL(0x190a7e55,0x6002f1e7),LL(0x10fdba0c,0x7fc62bea),LL(0x2c3dbf33,0xc836bbc5),LL(0x4f7d2a46,0x4fdfb5c3),LL(0xdca0df71,0x824654de)}, {LL(0x0c23902b,0x30a07676),LL(0x77fbbf37,0x7f1ebb93),LL(0xfacc13db,0xd307d49d),LL(0xae1a261a,0x148d673a),LL(0x52d98650,0xe008f95b),LL(0x9f558fde,0xc7614440)}},
   {{LL(0xaf907da2,0xd084564b),LL(0x51d4997a,0x5b2ae487),LL(0x3bc7206d,0x24bd4bf6),LL(0xfc3d5772,0xdd37b4ef),LL(0x35c4924f,0x8156d6f6),LL(0x1d1d396e,0x21e067c3)}, {LL(0xd40c7db8,0x977b3b39),LL(0xf5ad63bc,0x7ea4ecb4),LL(0xae811d70,0xe581f9c4),LL(0xa06c7f0d,0xe5441d5c),LL(0x1949d87f,0x0275c92b),LL(0x780469bb,0x511fd3e1)}},
   {{LL(0x9cb16650,0x17cd6af6),LL(0x69f4eebe,0x86cc27c1),LL(0x78822432,0x7e495b1d),LL(0x1b974525,0xfed338e3),LL(0x86f3ce21,0x527743d3),LL(0xb515c896,0x87948ad3)}, {LL(0xb17f2fb8,0x9fde7039),LL(0xd9b89d96,0xa2fa9a5f),LL(0x36ff74dc,0x5d46600b),LL(0x8302c3c9,0x8ea74b04),LL(0xf744b5eb,0xd560f570),LL(0xfe762402,0xc921023b)}},
   {{LL(0x88d7b3fb,0xa7f85014),LL(0xec78386e,0x3b5ec513),LL(0x2ad5053d,0xc6586b8a),LL(0xfbcebe43,0x88c09a43),LL(0x20054f16,0xde7f2a4a),LL(0xbbbb147f,0x63daba80)}, {LL(0x7d352b55,0x087e48f3),LL(0x8317ab79,0x997e32a0),LL(0x7f27cac7,0x8ae802ff),LL(0x37b1f6e1,0xb01a131c),LL(0x9a6d1dea,0x3f0d4c2e),LL(0xe7ceef80,0xe06114fc)}},
   {{LL(0xfff4c8ed,0xa35ab657),LL(0x8a5fabd7,0x017c6124),LL(0x09acda28,0x56463025),LL(0x14cf238a,0x6038d361),LL(0xaf1b9f07,0x1428b1b6),LL(0x7482e95c,0x5827ff44)}, {LL(0x780ff362,0xcb997e18),LL(0xe0bcac1e,0x2b89d702),LL(0xa837ddc8,0xc632a0b5),LL(0x59762647,0xf3efcf1f),LL(0x38b0d60a,0xe9ba309a),LL(0x20b5fb37,0x05deabdd)}},
   {{LL(0xe5ea795b,0x1e2a4fb1),LL(0x89ef6c3d,0x862616a2),LL(0xf69e2f1d,0x24617f4f),LL(0xffa0eb28,0xe0be24fe),LL(0x4b76a8ad,0x0ffb092f),LL(0x3a0dc9e8,0x21549090)}, {LL(0xe9080e04,0xf255fe06),LL(0x39228e7f,0xd270d25e),LL(0x5d6c6f1c,0x198e45db),LL(0x4373044c,0x6c9060ad),LL(0x61a8cc25,0x3af93464),LL(0xf22bbaa3,0x1945bf59)}},
   {{LL(0xcb8af047,0xd44e5dba),LL(0x943cfe82,0x15400cb4),LL(0x9df88b67,0xdbd69575),LL(0xb2405a7d,0x8299db2b),LL(0x0b1d80cd,0x46e3bf77),LL(0xe82ba3d9,0xc50cf66c)}, {LL(0xf2f747a9,0xb2910a07),LL(0x5adc89c1,0xf6b669db),LL(0x9052b081,0x3b5ef1a0),LL(0xb594ace2,0x0f5d5ed3),LL(0xd5f01320,0xda30b8d5),LL(0xaafcd58f,0x0d688c5e)}},
   {{LL(0x359590bf,0x311df84c),LL(0xdf6ca4b4,0xf907d69d),LL(0x82f22c64,0x876fd367),LL(0x9713e68c,0x64c4d14d),LL(0x6b07f539,0xd431858d),LL(0x84990283,0x39dfea33)}, {LL(0x80cf6498,0x6afb8cf0),LL(0xde060e9e,0x327056bc),LL(0x49a71086,0x5103ce4a),LL(0xcdf853ab,0xfc94be75),LL(0x8ca579cd,0x2bfb105f),LL(0x50454b41,0x02d19c3a)}},
   {{LL(0x2a161074,0x5eee3a31),LL(0xefe2be37,0x6baaae56),LL(0xe3d78698,0xf9787f61),LL(0x50630a30,0xc6836b26),LL(0x1445def1,0x7445b85d),LL(0xd568a6a5,0xd72016a2)}, {LL(0xe355614f,0x9dd6f533),LL(0x91e04588,0x637e7e5f),LL(0xb9fb1391,0x42e142f3),LL(0x41afe5da,0x0d07c05c),LL(0x1394edf1,0xd7cd25c8),LL(0xb99288ee,0xebe6a0fc)}},
   {{LL(0x7d4867b7,0xd9e2c5ee),LL(0x87873152,0x2c5602e0),LL(0x2c06b73a,0xb3358aa6),LL(0x09063076,0x967aec39),LL(0xd2f654fc,0x94dec534),LL(0xd69f485e,0x119aa4ed)}, {LL(0x35bc085d,0xb7c597b8),LL(0xbdbe0d0c,0x8781396d),LL(0x22f92ef5,0xba688929),LL(0xd438f5ec,0xeece3d4e),LL(0x44faac8b,0x4ead06f8),LL(0x9ef7c5f1,0x86a01ba9)}},
   {{LL(0xbabbad86,0xb8e63b7b),LL(0x90d66766,0x63226a9f),LL(0x5cf26666,0x26381836),LL(0x4cadd0bf,0xccbd142d),LL(0x9ac29470,0xa070965e),LL(0x25ff23ed,0x6bdca260)}, {LL(0x87dca7b3,0xd4e00fd4),LL(0x9e0e8734,0xa5097833),LL(0x048173a4,0xf73f162e),LL(0x9c3c2fa2,0xd23f9196),LL(0xe4ac397a,0x9ab98b45),LL(0x543f2d4b,0x2baa0300)}},
   {{LL(0xcde121c7,0xaa03b474),LL(0x55e52c76,0x74a648cb),LL(0xf37b57bc,0xb286ef86),LL(0x2a6371d2,0x95b797eb),LL(0x4077ccbd,0xa489ef89),LL(0x8e99ca6d,0xf46ade04)}, {LL(0x23242d03,0x5cf9e237),LL(0xcb708390,0x33c7d32a),LL(0xba7ba477,0x329523b6),LL(0x57de30bf,0xd406ab87),LL(0x1536ca01,0xaa10e4a2),LL(0xdfa7aac5,0xdcec94f4)}},
},
/* digit=65 base_pwr=2^325 */
{
   {{LL(0x1f5a9609,0x762a5eec),LL(0x765b337f,0xfe4f5f6a),LL(0xaa4f964a,0x0fd534ae),LL(0xd6526f01,0xcf46648e),LL(0x18d71d72,0xbc62a54a),LL(0x4f8488ea,0x48d94f2a)}, {LL(0xa0c72a86,0x62c40de7),LL(0x725dd2ef,0xd73ac51a),LL(0x6ab19096,0x3a51d746),LL(0x2dd1ad3d,0xf07bea4b),LL(0x2ef88078,0x2a0ec467),LL(0x664e435d,0x92598cb3)}},
   {{LL(0xb515fff5,0xee6e7006),LL(0x13258ed5,0xaedf6e39),LL(0xfc45111b,0x373adf7d),LL(0x875c23c8,0x0c7535b1),LL(0x97039d49,0x2a7e04f8),LL(0x9afd1a06,0xd76787ae)}, {LL(0x91b6dc89,0x049dd385),LL(0x932dab78,0x8f0c8ad0),LL(0x925498c2,0xdce057b9),LL(0xda25daa3,0x7b9c9bd2),LL(0xd4decb7d,0x6d0b70a3),LL(0x03df76ef,0x099a2183)}},
   {{LL(0xd8948f95,0x779905b6),LL(0x91cd0206,0x3c7085b5),LL(0x679096db,0xce9af0aa),LL(0xf558913c,0xfdf04f10),LL(0x6f24a2e2,0x05300cb0),LL(0x5d581b35,0xf9d9a2f2)}, {LL(0x6a713751,0x855c8de9),LL(0x0e0c0dfb,0xc9ac24bc),LL(0x97740d65,0x67612a41),LL(0x44c9360d,0x7588a527),LL(0x325cc880,0x928ac910),LL(0xacdd3188,0xa74abdaf)}},
   {{LL(0xb9d926c2,0x3c6c5618),LL(0x4a9099f9,0x7e14c3ae),LL(0xae2fb830,0xb3259c90),LL(0xec31a504,0xf7cc6e43),LL(0x126230bd,0x83bb13c6),LL(0xff1dae3a,0x5a1f4313)}, {LL(0x49b0b65b,0x0cc6c1a5),LL(0x274a84c2,0x67fa836a),LL(0xe604a58d,0xd454c75f),LL(0x2491f872,0xceadfd91),LL(0x9ce116a5,0x6c5575da),LL(0xb24a4481,0xfaa4903f)}},
   {{LL(0x5a4703aa,0x7a8a898d),LL(0x1cd6f9d6,0xc59933ea),LL(0xd28124cd,0x703265f5),LL(0x0178d1fe,0xe1c1bee1),LL(0x241262e9,0x9ff31cd4),LL(0xa3c9f80f,0x9174a939)}, {LL(0xbc2a62ee,0x0f7a3d2d),LL(0x62f1b3ac,0x0454051c),LL(0xa2421254,0x83502c9e),LL(0xb684199b,0xb4fa51fc),LL(0xc5e36a44,0x257e9e2b),LL(0x97d8647f,0x14efeed5)}},
   {{LL(0x5cecb21f,0x6e96a819),LL(0xd8beecae,0x3a58d8b2),LL(0xc0c715a8,0x93c3cbb0),LL(0x541759b7,0xfb06f977),LL(0x771c3d2c,0xf25ba095),LL(0xa3bfd322,0x7560446e)}, {LL(0xa015cb4f,0x7cd99f35),LL(0x0786f235,0xa0e54196),LL(0x8b8e291a,0x0f868f76),LL(0x2f95050b,0xc8260b0b),LL(0xf4c0a462,0xaf38376e),LL(0x98a3395d,0x2b3c0f3b)}},
   {{LL(0xb9d0bdf0,0x0952b888),LL(0x8ce32fb7,0x3973763c),LL(0x6dd860c7,0x221f0ba5),LL(0xb16ac501,0xbb7a27fe),LL(0xbc8fe58f,0xf113b194),LL(0x65839ffb,0x18f3297a)}, {LL(0x8dc30003,0xa2d4eb7c),LL(0x8e334479,0x3fb4b487),LL(0x1a8310e3,0xa4f32c65),LL(0xf78f46ac,0x944cd644),LL(0xf96fb91f,0x14e40c4a),LL(0x4ddf6e72,0xc31402bd)}},
   {{LL(0xb45a8002,0xc5ad791e),LL(0xba2d7a40,0x4a23fd68),LL(0x98544bc4,0x673b9e49),LL(0xd273c360,0x934d8f55),LL(0x68a75a8c,0x7fb48d07),LL(0x5e0fac97,0x2e620105)}, {LL(0xf10ed580,0xbe01655f),LL(0x9e96731f,0xd21d52ae),LL(0x53325138,0x74f830de),LL(0xde9f3fc5,0xa7240331),LL(0xa7e01fa5,0x96b25206),LL(0x07eda4b4,0x3fcfedee)}},
   {{LL(0x9336b8bb,0xf1b08a42),LL(0x9a768ca6,0xbaaa5571),LL(0x2c0a2938,0x70180b4c),LL(0x92dd3c07,0x8e735436),LL(0xe3fd5b1c,0x2dd3af0b),LL(0x1f1af8b8,0x3cf3d179)}, {LL(0xd558c174,0xaff210c6),LL(0xe2560d5e,0x1007938a),LL(0x8f99a78f,0xa625558c),LL(0xc1b91dea,0xa61d5edf),LL(0xdab80815,0xa86a4e5b),LL(0x78283ea3,0x88944bfe)}},
   {{LL(0x0ee6d492,0xeec9b118),LL(0xb7ef00fc,0xf2bd388f),LL(0x3191c902,0x5c6c0bbf),LL(0xd42366de,0x6796929a),LL(0x3285710d,0xeecb5b3f),LL(0x41bad8a3,0x58d6773d)}, {LL(0xd0f05005,0x7cdbade6),LL(0x0e25444b,0xb117e1ba),LL(0xfe4dc071,0xa52b4926),LL(0xe0669f10,0xce8a1b69),LL(0x0db21d46,0x5e765439),LL(0xd929bfab,0xc5dcdea4)}},
   {{LL(0xa36aea84,0x67832f48),LL(0xfc78df1d,0x25256118),LL(0x03b8f04d,0x5a085d70),LL(0x108969c2,0x19a25d9c),LL(0xb70b14b7,0xb6fe713b),LL(0xfa6b89d8,0xba23ac4a)}, {LL(0xc2684b8c,0x6a88e4e7),LL(0x6ab59297,0x3e816609),LL(0x30151aaf,0xdb7927fe),LL(0xaa7d95da,0x0fd67681),LL(0x17432b4c,0xc60e5dd5),LL(0xed48ccda,0x3192dc27)}},
   {{LL(0x8af859e2,0x2116a017),LL(0xbd9f8800,0x8a77a7a2),LL(0xf1f2034d,0xc78836fb),LL(0x3864566b,0x8fd4299a),LL(0x29f9deca,0x0386eead),LL(0x2f1a677c,0x2042ef18)}, {LL(0x2af95cc8,0xed4511c0),LL(0xc93dfbc5,0xdb0a334d),LL(0xb64ab345,0x0d788ac9),LL(0xb20638f2,0xd8410402),LL(0x592448e6,0x8aca5131),LL(0x000de69b,0x3ac508d2)}},
   {{LL(0x4f9b2400,0x1ee6d3b5),LL(0x00c9c182,0x4c71999a),LL(0x35fc481e,0xd6b1c470),LL(0xf8b5d59c,0xf41ef454),LL(0x2824a13d,0x7edbc3c9),LL(0xb7aa0ade,0xa3baba91)}, {LL(0x2b97b392,0x75c77e71),LL(0x9cc2d53c,0x8ec107d6),LL(0x29322233,0x652146fe),LL(0xe679e990,0x710578df),LL(0x260547db,0x47f838ba),LL(0x23a78365,0xa4e57bec)}},
   {{LL(0xefb058db,0xe4320313),LL(0x3d04e752,0xaad2377c),LL(0x9865c63e,0x3f8cbca9),LL(0x3009e55d,0xe89238a5),LL(0x12519936,0x58fad5fe),LL(0xb024b695,0x03b16a00)}, {LL(0xf8f3b8c7,0xa556d096),LL(0xd8a599c4,0xce6df197),LL(0x6a13b031,0x45ac8a25),LL(0xca6819e8,0x31af2dcc),LL(0x390418e9,0x7a4dce86),LL(0x4600d7f4,0xdd24b0bd)}},
   {{LL(0xb96e667f,0x88a8aa87),LL(0x4e704eaa,0xb9a76c18),LL(0x72c924b7,0x25d4edc1),LL(0x16b67f80,0xe56872a8),LL(0xc464e4ff,0xda0dbb11),LL(0xc8cec410,0x0435f391)}, {LL(0x0983fd7a,0x8d9b4043),LL(0xe7ff343c,0x51ec5bc1),LL(0x85994bc9,0x8bc85fc4),LL(0x8806c150,0x69c78834),LL(0x3db3665b,0x542cec89),LL(0xfd720bde,0x931bc4b0)}},
   {{LL(0x8d5c039f,0x3e6e9381),LL(0x8a8d2cc9,0x80949422),LL(0xb843ec06,0xf2d7c8b4),LL(0xaf8a23f1,0x0055d882),LL(0xd3792335,0xe848010e),LL(0x55e08e74,0x9b41a55f)}, {LL(0x5de83059,0x956ea8e9),LL(0x3263678e,0xf159a997),LL(0xcca1b548,0x5f7b9271),LL(0xf1d0b7f1,0xd41d2281),LL(0x5c9963fb,0xb187047b),LL(0x02536cd8,0x213ff6af)}},
},
/* digit=66 base_pwr=2^330 */
{
   {{LL(0x0d0fa76c,0xe51a9570),LL(0x4d2e9c8e,0x67c7890e),LL(0xf974d2cb,0xc6160fa2),LL(0x4c6a78de,0xe00474f7),LL(0x0ac89d11,0xee916e51),LL(0xf826f133,0x1adad97a)}, {LL(0x8d2d77f3,0x3fc65d3f),LL(0x0ba6c300,0xda942075),LL(0x0b9196b1,0x5237a82e),LL(0xa572b6f5,0x4975e680),LL(0xb9bed2bc,0x41ea8b92),LL(0x9826825e,0xbe0ad710)}},
   {{LL(0x9fcaba39,0x8a390dca),LL(0x278d22b3,0x3879f0b4),LL(0xbc5e82f1,0x77bbea69),LL(0x4628d6f1,0x71f02e2d),LL(0xf968e240,0x6260790c),LL(0x665270ee,0x1c7f3df5)}, {LL(0x1a87b1c8,0x33639545),LL(0x4ffd9fb8,0x2011fd21),LL(0x7807ed55,0x69060f86),LL(0x9dfa452c,0x1b0ac011),LL(0x06d27c0d,0xbbdb25fe),LL(0xa60ef90c,0x5c25d23a)}},
   {{LL(0x734b2e9c,0xd3eb69e4),LL(0xc35ff1b3,0x1c2754e2),LL(0x9f3e8c51,0xa26681e6),LL(0xa2cae737,0x7892ad11),LL(0xcbd8bda6,0x88b1da43),LL(0x419d21c4,0x8a576942)}, {LL(0xc90f4545,0x7c124343),LL(0x26453bae,0xa5a8d93b),LL(0x76ae72e8,0x9a4c08fd),LL(0x7b064e94,0xa08b82d9),LL(0x83725330,0x4f803ba0),LL(0x865235f3,0x33672455)}},
   {{LL(0xc8110f1f,0x1877dfd0),LL(0x18db27c2,0xea88f59d),LL(0xc78e295e,0x9d089536),LL(0x74a04cc5,0xcbb5d553),LL(0x827f75ed,0xe3666006),LL(0x61e7378c,0x8557b81a)}, {LL(0xed223f48,0x74170170),LL(0xd86ee829,0x84197a6e),LL(0xac1c4a0f,0xd75a30f8),LL(0x3cd92824,0xd7e7be0d),LL(0x1b5e86d4,0x5ea0abdb),LL(0xb3b615ef,0x41146ae1)}},
   {{LL(0x1ae5e9da,0x570deceb),LL(0xb73ead01,0x5c079c70),LL(0xd2ce6639,0x522a30a6),LL(0xf4056ac9,0x71dc5c3f),LL(0xbaac149f,0xd93c7a2d),LL(0xf1844ceb,0x5c3298b8)}, {LL(0x8c23c0dc,0x282adf40),LL(0x9b646f86,0xbe498189),LL(0x628da9e5,0xe77c1950),LL(0xa1fd5a18,0x38cc27ba),LL(0xaacdca52,0xb5579728),LL(0xc8e1ecbd,0x8d34fdb4)}},
   {{LL(0xadea7d6a,0x323f2102),LL(0xb694b253,0x035b354e),LL(0x5b8a36c2,0x66dc4e4a),LL(0x71795ca4,0xb6092224),LL(0xd300d80f,0xd8c6d7ee),LL(0xb3b94954,0xf31f258d)}, {LL(0x277ced5b,0x0f2eb679),LL(0xeba40e3f,0x0b16efa3),LL(0x0dca4f36,0x40003507),LL(0x59a9a3a1,0xd34c91cb),LL(0x86da6408,0x5e8fea32),LL(0x03f31728,0xf237959f)}},
   {{LL(0xbbaaedc6,0x1b5bc630),LL(0xb49cbb3b,0xe7d25088),LL(0x0deb8cf0,0x5622cbf7),LL(0xd309c3ba,0x3b20803c),LL(0xff45e2fc,0x64c2e7de),LL(0x9aab84a5,0xfa730ffb)}, {LL(0x4edfb52e,0xba83cc51),LL(0x748bbd69,0xe05c0140),LL(0x2254ec43,0x27bbb5f5),LL(0x324c8c40,0xca740989),LL(0xd26491b4,0xa21488b1),LL(0x69d8626b,0xe2753a1f)}},
   {{LL(0x8956146c,0x618ca06b),LL(0x552cdecf,0xd51f1e6f),LL(0xa3b6ce7d,0x981372cc),LL(0x5f14bb57,0xb44a68ee),LL(0x6373abbb,0xfc1167e9),LL(0x767d4c0a,0x3d621f8b)}, {LL(0xf6ecc778,0xc6dcdfeb),LL(0x82d1fbdd,0xddda9262),LL(0xbfcbf2f7,0x477501aa),LL(0x67aa8277,0x0be7228a),LL(0x1daab9cd,0x5de7b833),LL(0x262feb4a,0xb88a4f9a)}},
   {{LL(0x936fb33f,0x28f18586),LL(0x381bf7bb,0x9809b2ab),LL(0xeac3c252,0xf6e16931),LL(0x5e151187,0x366d1833),LL(0x7a3b6460,0xe5b4c235),LL(0x0a68bc91,0x693a9fa5)}, {LL(0x6a7f8b6e,0xa35f104a),LL(0x688676c4,0x3e5d6981),LL(0x0651a609,0xc0c081b1),LL(0xd77057c9,0x6df5da2d),LL(0xc4602847,0x8bb271bb),LL(0xc4bd07d8,0x322547b3)}},
   {{LL(0x233d39e4,0x8c283529),LL(0xc6092096,0x96300796),LL(0x5dde766c,0x2c549de5),LL(0xb4151002,0x27e0b444),LL(0xf2f88f1b,0x802e5fc3),LL(0x8ba1956d,0x2af579c2)}, {LL(0xd68196c7,0x52edd04e),LL(0x74a202b0,0x2e22e714),LL(0x8bf66459,0x33894824),LL(0x9e39df55,0x8f0d8c25),LL(0x6c5276d9,0xee4f109e),LL(0xc5dc0bf0,0xc0c893f0)}},
   {{LL(0xf8482849,0x06054c76),LL(0x5fcca65d,0xc24b4a6a),LL(0xa17ebda3,0x71c27e01),LL(0x1be9dfb8,0x1ffce028),LL(0xebc43854,0x3784c950),LL(0xd5086510,0xcf0ecc2d)}, {LL(0xbe24d8eb,0x86d0fc3c),LL(0x1f21788e,0x5bad0191),LL(0xc49b3a12,0xe2c3bcb9),LL(0xf7d5992d,0x66f82433),LL(0x13969246,0xf7cc5eb9),LL(0x8660a6da,0xe52defd4)}},
   {{LL(0x53fd1a04,0x7a9623b6),LL(0x3a3b8500,0x13bd35bf),LL(0xe0f8e530,0xf8a5dec9),LL(0x1d65dcd4,0x88bcbe29),LL(0x6739541c,0x09fe3803),LL(0xe716a97a,0xebd04b7f)}, {LL(0x1e5ef7cb,0xbd8e34df),LL(0xd7c4fd6c,0xddfc4243),LL(0x3519411f,0x0183d905),LL(0xf7a3c483,0x63450996),LL(0x01355739,0x18283cea),LL(0x9aaa72f7,0x8c1d72cf)}},
   {{LL(0x8dc72468,0x551e1b4e),LL(0xa7b2f1ac,0x8a926cb2),LL(0x0fd12fad,0xb873e83b),LL(0xa4e7fb13,0xb6cde14f),LL(0x5befc256,0x81ae4141),LL(0xb4c7631c,0xffb0c636)}, {LL(0x8a2478fe,0x80f1408f),LL(0x44fa7605,0xde6d051d),LL(0x4d44a1e4,0x5a15b1f8),LL(0xa0daafe3,0x1a0031c5),LL(0x597652a7,0x304338dd),LL(0xf257f17a,0x6830dcc7)}},
   {{LL(0x193aabbc,0x4a67ec76),LL(0xd74761f9,0x3da6dec6),LL(0x0b35bb70,0x751720c9),LL(0x8d9e0f8b,0xe5e04905),LL(0x0858f29c,0x3cd37c84),LL(0xb881733e,0x7ff1abfb)}, {LL(0x0c4f7694,0xa0c2698b),LL(0x96b95e4e,0xc7364192),LL(0x37ece651,0xcfa55c55),LL(0x7cb1e9e1,0xa2bbd6ae),LL(0xa0eb0e8a,0xcd2292b9),LL(0x8d5030d0,0x8aba99e1)}},
   {{LL(0xdc4a1d3e,0xb7c74c58),LL(0x0331ea39,0xe3ec3016),LL(0x023c8712,0x83afb271),LL(0xc9c82680,0xc2670d56),LL(0xfeca1061,0xd426f350),LL(0xba6edc01,0xe8aee692)}, {LL(0x46e801d9,0xc916fbe5),LL(0x7097286e,0xcb001c37),LL(0x78ee1328,0xfcf79d26),LL(0xb6a4afb3,0xb05b0634),LL(0x306da14f,0x2ab327bb),LL(0xba5ff534,0xc11a0294)}},
   {{LL(0xe19763dd,0x7b7da028),LL(0x8b98ff78,0x662f54df),LL(0x51f3dbd9,0xc056d83c),LL(0xa91d085a,0xe2f4d46f),LL(0xeb35262d,0x31759c9c),LL(0x0c9dd29e,0x624d0cf2)}, {LL(0x1624b02d,0x108cf9bb),LL(0x345531d6,0xa241444e),LL(0x73d372b2,0xf69816b2),LL(0xd5415e53,0x126575a7),LL(0x306b8b0e,0x546bb4c1),LL(0x4d54ea5e,0x82bb0c12)}},
},
/* digit=67 base_pwr=2^335 */
{
   {{LL(0xf9474bb7,0xdaa7fcc9),LL(0xafa5db2a,0x3c82e74b),LL(0x9894edce,0xfbf918c5),LL(0xa9ac29a7,0x470c45ed),LL(0xbc372f2c,0xdfd44f6f),LL(0xa1e38d3f,0x73a4790a)}, {LL(0xa9674837,0x23d2400b),LL(0x136a92da,0x3dad71bc),LL(0x48baa4ab,0xc76a4881),LL(0xbc26e6b0,0x73227e4e),LL(0xe8ef5662,0xe732edcf),LL(0x0c5662bb,0xfe96aa5f)}},
   {{LL(0x139b3239,0x87c7dd7d),LL(0x4d833bae,0x8b57824e),LL(0x9fff0015,0xbcbc4878),LL(0x909eaf1a,0x8ffcef8b),LL(0xf1443a78,0x9905f4ee),LL(0xe15cbfed,0x020dd4a2)}, {LL(0xa306d695,0xca2969ec),LL(0xb93caf60,0xdf940cad),LL(0x87ea6e39,0x67f7fab7),LL(0xf98c4fe5,0x0d0ee10f),LL(0xc19cb91e,0xc646879a),LL(0x7d1d7ab4,0x4b4ea50c)}},
   {{LL(0xe2fdc88b,0x7e047d9c),LL(0x715be007,0x4f6166d9),LL(0xd0debb0a,0xd9661068),LL(0xc3dafce2,0x82f02fd3),LL(0x00f8df79,0x41a6b644),LL(0x6cedd3a8,0xccd5a798)}, {LL(0xf11431b2,0xb6617354),LL(0x8a677e83,0x116d0274),LL(0x89ef485a,0x2f399390),LL(0x5e2270d2,0x3ee06862),LL(0x06d6c72f,0x8c9a70de),LL(0x4e4497e3,0xd7e69177)}},
   {{LL(0x7db62b5a,0xcfbcbc4a),LL(0x4ab45dde,0x2919bf51),LL(0x22322f91,0x735de056),LL(0x7662ae23,0xd2590bda),LL(0xd82be7a6,0x63d468fe),LL(0x695ea172,0xc84d0435)}, {LL(0x20a6fccd,0xc50f4941),LL(0x620f44f1,0x2d613990),LL(0x1fd25778,0x680ccd04),LL(0x4a3d0808,0x25ddac44),LL(0xc4684cba,0x41d8b738),LL(0x53963888,0x2611645f)}},
   {{LL(0xe9f43747,0xffe6bb22),LL(0x22f6cd09,0xf387957b),LL(0x607a4892,0x3eb09302),LL(0xf3d2f552,0x52c733a8),LL(0x741bd215,0xcc935b2e),LL(0x1ae0b28e,0x5fff37e3)}, {LL(0xc2e9bc20,0x4234e33c),LL(0x39ea2555,0x4ee1488e),LL(0x17156a8a,0xb8821daf),LL(0x1af16ade,0xc7b45844),LL(0x5b4fa74d,0xc1009ec7),LL(0x5d7cf8bd,0xe0262e65)}},
   {{LL(0xb0279be5,0xb05cb834),LL(0xf08c5f93,0x2de7d0eb),LL(0xefa9e4f0,0xf023b5aa),LL(0x9bd075ec,0xb8061e5d),LL(0x1aa41bfb,0x7d2ba50f),LL(0x90865d96,0x8963f3e3)}, {LL(0x4713ec7a,0x7f221a79),LL(0x8500b4c3,0xc83bc517),LL(0xf6ab1540,0x085feb6a),LL(0xdc87cd93,0xfd141024),LL(0x3239dbf6,0x3e196fdb),LL(0xdbcd5364,0xb7cf3e16)}},
   {{LL(0x41a64252,0x72544edb),LL(0xa6fe493d,0x5f3d376f),LL(0xd635df1e,0x17ae424f),LL(0xdf598c63,0x69cb55a0),LL(0x6ce0f1d5,0x31297f4a),LL(0x1bd11b61,0x4573bb7d)}, {LL(0x45a1e918,0x1d8a65c1),LL(0xe5dc2e63,0x2729ab25),LL(0x3ecc307b,0xa3dd0df0),LL(0x952019dd,0x4856546f),LL(0xc784e4fe,0xf8d39888),LL(0x0043b09e,0xdc6732c7)}},
   {{LL(0xe03a2fb4,0x1466c9f5),LL(0x862a58a2,0xb866c006),LL(0xb5865550,0x291e8c75),LL(0xe65862cc,0x1ddb7154),LL(0x2b997167,0x285153bc),LL(0x954b6c19,0xe2fce0e7)}, {LL(0x16dc2937,0x985d4506),LL(0xee41d9c3,0xf7f14216),LL(0xfa5fe5e5,0x39e098da),LL(0xf90f231d,0x3fc26046),LL(0x32afd0b5,0xde5d5ced),LL(0x60c09c18,0xad688b1d)}},
   {{LL(0x8f84e987,0xefd9aed0),LL(0xae8c8308,0x5ee0c707),LL(0x2aafc403,0x4c8a7653),LL(0xa232436a,0xfafb60ce),LL(0x9934f053,0xc641294d),LL(0x30310529,0xc673e5b2)}, {LL(0x9066469d,0x3c8fa99a),LL(0x7c09af37,0x5626038b),LL(0xabd66748,0x6ffd8f9b),LL(0xcea58a67,0x21ced048),LL(0x1496d048,0x31071213),LL(0xa9c28d0d,0xfa575242)}},
   {{LL(0x0f806b59,0x3720b172),LL(0xf224597b,0x1f696d47),LL(0x5b54eefc,0x03c46e31),LL(0x72b0edaa,0x387e4664),LL(0xee77476e,0xfc59b03d),LL(0x607a7259,0x86854e54)}, {LL(0x3e9320dc,0x1478bcee),LL(0x8c9d87e4,0x4aa825a8),LL(0xcf272ee0,0x71272f72),LL(0x8bd885cd,0x19e3a4a3),LL(0x376ba31c,0x9af6415b),LL(0x807b2b36,0x6394b5a7)}},
   {{LL(0x77460193,0x1bf6c56b),LL(0x5666ae6d,0xd6f7fabf),LL(0xe3e839d1,0xdf677909),LL(0x08cb9984,0x9124d341),LL(0x86a0c7c3,0xbb6b591d),LL(0x8f527a6a,0x4bf33423)}, {LL(0x26941bfe,0x7d137790),LL(0xcf6e4481,0x15a0b541),LL(0x124d5b9b,0xdf833b87),LL(0xa7fdf95d,0x85827dc5),LL(0x49e520af,0xf05a2c45),LL(0x91e0645a,0xfb506d53)}},
   {{LL(0xe572e06d,0xdbfcfa75),LL(0x8b7d5653,0xafa019d0),LL(0x67a19b60,0xcc6c851d),LL(0x31ae1a67,0xace88bf4),LL(0x93d1e135,0x74554a61),LL(0x4211890a,0x51ba2cdd)}, {LL(0x9e8d1f02,0x7cb32689),LL(0x8b66ab99,0x29a6b825),LL(0x766e72f3,0x0a672c21),LL(0x880642e3,0x24bb718a),LL(0x184d2b36,0x425dc41d),LL(0x891024ab,0x96a1468e)}},
   {{LL(0x665fe173,0xeff22b64),LL(0xef2eabea,0x38efdef6),LL(0x21a309df,0x8a1f3791),LL(0x2431e2ed,0xa9cf02cf),LL(0x1d939394,0xf38507dc),LL(0x82fc3178,0xf116b085)}, {LL(0xc7571366,0x4c5460dc),LL(0x978495fd,0x99efd9dd),LL(0xf26e347d,0x5159dd41),LL(0xe97ee9f1,0x692962ce),LL(0x6a288815,0x1e2f3af2),LL(0xa71ade78,0x03512344)}},
   {{LL(0x26df7050,0x3180789c),LL(0x96cdfd31,0xe375a43e),LL(0xe99e922d,0x7951b895),LL(0x3d0bbe80,0x987ea250),LL(0xe2fe79c0,0x6d2f49f0),LL(0xc2b18d2a,0xc9c2c636)}, {LL(0xd8c8620c,0x707798f3),LL(0xd5c6a0ee,0xc2d603da),LL(0xbc447940,0x46cf1e32),LL(0x38a845f3,0x4dfc1459),LL(0x455e5d92,0x210083fe),LL(0xa1fedc3f,0x6be989ea)}},
   {{LL(0x9f019162,0x55d4fdc3),LL(0xf1ec4585,0xa8222d08),LL(0x3a0ae4f9,0xd987e3eb),LL(0xa9c7a693,0x07deda59),LL(0xf04ee53f,0xc06567d9),LL(0x71364c4d,0x93945788)}, {LL(0xbaa5bc16,0xbb2a2422),LL(0xbfa3931a,0x89574a5d),LL(0xf300f081,0xf09b331e),LL(0x1a0ff82b,0x644de9b7),LL(0xa5ecdf9b,0xae023ce4),LL(0xc1907aac,0x5b67cf8b)}},
   {{LL(0xdacc038c,0x72fc8198),LL(0xf1077bbd,0x5fdae1d9),LL(0xd99e3036,0x369198bb),LL(0x0efddfca,0x6b68390a),LL(0xf0914741,0x8c35f3e4),LL(0xca7d7807,0xd2bc54ec)}, {LL(0x3a8695d1,0x564d991e),LL(0x1b0d937d,0x5e1e14c8),LL(0x5d635893,0x51f30dab),LL(0xf944e49a,0x0427e346),LL(0x6a233bc0,0x1e0bf1b5),LL(0x617bf93e,0x75b0ee6c)}},
},
/* digit=68 base_pwr=2^340 */
{
   {{LL(0xd2e15a8c,0xd23658c8),LL(0x16ba28ca,0x23f93df7),LL(0x082210f1,0x6dab10ec),LL(0xbfc36490,0xfb1add91),LL(0x9a4f2d14,0xeda8b02f),LL(0x56560443,0x9060318c)}, {LL(0x64711ab2,0x6c01479e),LL(0xe337eb85,0x41446fc7),LL(0x71888397,0x4dcf3c1d),LL(0x13c34fd2,0x87a9c04e),LL(0x510c15ac,0xfe0e08ec),LL(0xc0f495d2,0xfc0d0413)}},
   {{LL(0xc4e81268,0xb097b2c5),LL(0x1d50ca8c,0x7ef17552),LL(0x42099644,0x638266e9),LL(0xff729073,0x43d059de),LL(0x148c3940,0xeebb5fe1),LL(0xdaa8e925,0xb82e73d1)}, {LL(0x254380fd,0xf43c78d8),LL(0xfce37fa0,0x2beabc58),LL(0x6b636357,0xcdd5a7d6),LL(0xe096a954,0x8b70a2eb),LL(0xd0afa2fc,0x011d5419),LL(0x04fb095a,0x3e49eb67)}},
   {{LL(0x156636c2,0xeb05c516),LL(0x090e93fc,0x2f613aba),LL(0x489576f5,0xcfd573cd),LL(0x535a8d57,0xe6535380),LL(0x671436c4,0x13947314),LL(0x5f0a122d,0x1172fb0c)}, {LL(0xc12f58f6,0xaecc7ec1),LL(0x8e41afd2,0xfe42f957),LL(0x3d4221aa,0xdf96f652),LL(0x2851996b,0xfef5649f),LL(0xd5cfb67e,0x46fb9f26),LL(0xef5c4052,0xb047bfc7)}},
   {{LL(0x168d5e60,0x88e7ac8e),LL(0x6188a98f,0x53abd569),LL(0x18be419a,0x3b96d529),LL(0xc057621c,0x7e75e354),LL(0x5ce57e59,0xcb1b709f),LL(0x844f2463,0xe78befa2)}, {LL(0x3276d4a0,0x53608199),LL(0x157f2024,0x92636ade),LL(0xe0411414,0x6dd0d348),LL(0x4d73eeae,0x5b28e950),LL(0x690ed85e,0x08439232),LL(0x6da14b58,0xdde1a349)}},
   {{LL(0xf4484374,0x5cbdc442),LL(0xf92452ef,0x6b156957),LL(0xc118d02a,0x58a26886),LL(0x75aaf276,0x87ff74e6),LL(0xf65f6ec1,0xb133be95),LL(0x4b1b8d32,0xa89b6284)}, {LL(0x09c81004,0xdd8a8ef3),LL(0x0cf21991,0x7f8225db),LL(0x26623faf,0xd525a6db),LL(0xbae15453,0xf2368d40),LL(0x84f89fc9,0x55d6a84d),LL(0x86021a3e,0xaf38358a)}},
   {{LL(0x33c12f53,0x10520b27),LL(0x9286a1f5,0x97d4ef43),LL(0x8948f78b,0x12468ef8),LL(0x50ad452e,0x40a9d275),LL(0xc1c48470,0x5e382347),LL(0x33e73ace,0x5cd739fd)}, {LL(0x1041f8f3,0x7d83d02f),LL(0xe314ad92,0x32642eb0),LL(0x885679e6,0x6716d448),LL(0xfc95919c,0x0e12bdd5),LL(0x92c2194a,0x7da44204),LL(0x15ffcd2d,0x78956db1)}},
   {{LL(0xff52e280,0xbd048bdc),LL(0x526a1795,0x8a51d0b2),LL(0xa985ac0f,0x40aaa758),LL(0xf2c7ace9,0x6039bcdc),LL(0x6aec347d,0x712092cc),LL(0x6b5acab7,0x7976d090)}, {LL(0x6eed9617,0x1ebcf80d),LL(0xb0f404a4,0xb3a63149),LL(0xd0b610ef,0x3fdd3d1a),LL(0x98c28ac7,0xdd3f6f94),LL(0x3a59750f,0x650b7794),LL(0x2d3991ac,0xec59bab1)}},
   {{LL(0x39ed9ec9,0x6cbd2757),LL(0xfe5d4aa8,0x5db68a68),LL(0xe4c58c7b,0x177eaa0b),LL(0x0e488784,0x603551ef),LL(0xdf916b0f,0xc8eba131),LL(0x159732e2,0xd0dbceda)}, {LL(0xb0834afa,0x55acca84),LL(0xb59ffbf5,0xdbe98440),LL(0x3bd3b202,0x162a2c70),LL(0x6ddd8eba,0x4c5e5d25),LL(0x77b1d93d,0x66e7844a),LL(0x110b9dcf,0x1292bc0e)}},
   {{LL(0x2e552766,0x01f40e88),LL(0x66f5354f,0x1fe3d509),LL(0xb3a8ea7f,0x0e46d006),LL(0xf831cd6a,0xf75ab629),LL(0x91465119,0xdad808d7),LL(0x17ef9b10,0x442405af)}, {LL(0x672bdfcb,0xd5fe0a96),LL(0x355dbdec,0xa9dfa422),LL(0x79b25636,0xfdb79aa1),LL(0xeece8aec,0xe7f26ffd),LL(0x7edd5aa2,0xb5925550),LL(0x8eb3a6c2,0x2c8f6ff0)}},
   {{LL(0x9341a2e0,0x34a4632c),LL(0xc525bc5a,0x05ca421b),LL(0x4ae3284a,0x441cf2eb),LL(0x146012ab,0x1a57684e),LL(0x30acfd0e,0x23a52ee3),LL(0x7d29e389,0xc3f4d94a)}, {LL(0xb4154640,0xfbd4d48e),LL(0xaf9ec396,0x2352e791),LL(0x7327caee,0x45813e8e),LL(0xd9db7e81,0x977f7a08),LL(0x5f53d15d,0xbe55b630),LL(0x6a23f0dc,0xee182ac6)}},
   {{LL(0x757d6136,0x88887756),LL(0x88b92e72,0xad9ac183),LL(0x8785d3eb,0x92cb2fc4),LL(0x9319764b,0xd1a542fe),LL(0x626a62f8,0xaf4cc78f),LL(0x26bffaae,0x7f3f5fc9)}, {LL(0x40ae2231,0x0a203d43),LL(0x387898e8,0xa8bfd9e0),LL(0x474b7ddd,0x1a0c379c),LL(0x34fd49ea,0x03855e0a),LL(0xb3ef4ae1,0x02b26223),LL(0xe399e0a3,0x804bd8cf)}},
   {{LL(0x7cdc2211,0x22723fa3),LL(0x31ddb2bd,0x1d339232),LL(0x46626fe6,0x63f354c1),LL(0xf67a4257,0xd0f68526),LL(0x8d3d00b6,0x79aa889b),LL(0x0de4c413,0xca5fc8a7)}, {LL(0xd2879266,0xe311a966),LL(0x5f21e1dd,0xc8d982fe),LL(0xcbb159ff,0xc51f1604),LL(0x092d83ce,0xb449deb8),LL(0x644fd0be,0x4a5f68f8),LL(0xbffb0088,0xeef3fa4d)}},
   {{LL(0xde865713,0x11a9f3d0),LL(0xbde98821,0x81e36b6b),LL(0x6aa891d0,0x324996c8),LL(0x395682b5,0x7b95bdc1),LL(0xc1600563,0x47bf2219),LL(0x643e38b4,0x7a473f50)}, {LL(0xf5738288,0x0911f50a),LL(0x6f9c415b,0xdf947a70),LL(0x67a067f6,0xbdb994f2),LL(0x88be96cd,0x3f4bec1b),LL(0xe56dd6d9,0x9820e931),LL(0x0a80f419,0xb138f14f)}},
   {{LL(0x3b32932e,0x2b4056a4),LL(0xbd8a1cb8,0x1c74deb6),LL(0xb181b5a0,0x09601843),LL(0xdc6a92d8,0x50a92353),LL(0x2d6f4331,0xe86c022c),LL(0x3a3ae821,0x0d9671dc)}, {LL(0xc8228d82,0x3d618a20),LL(0xa5292169,0x20e809c1),LL(0x3803f840,0x3b2fe5e7),LL(0x416eb670,0x1f2978e9),LL(0xc35b795c,0x44dcc410),LL(0x503ce975,0xbf5065c0)}},
   {{LL(0x0429077a,0xa11a1a8f),LL(0x10351c68,0x2bb1e33d),LL(0x89459a27,0x3c25abfe),LL(0x6b8ac774,0x2d0091b8),LL(0x3b2415d9,0xdafc7853),LL(0x9201680d,0xde713cf1)}, {LL(0x68889d57,0x8e5f445d),LL(0x60eabf5b,0x608b209c),LL(0xf9cfa408,0x10ec0acc),LL(0x4d1ee754,0xd5256b9d),LL(0x0aa6c18d,0xff866bab),LL(0xacb90a45,0x9d196db8)}},
   {{LL(0xf9e89bea,0x1b66faab),LL(0x3a441284,0xc81c5ddc),LL(0xa675f7c8,0x1a82f3a0),LL(0x30313a71,0x82884a2f),LL(0x58aea9e6,0x7ac5d7b0),LL(0xcd5ff05d,0x1954f075)}, {LL(0x6178d270,0x7c29638d),LL(0x19381929,0x6af7f8ba),LL(0xa17ae3a7,0xe85e3c47),LL(0x7009e38a,0x91b107c7),LL(0xf1f9c52e,0xf3b777d8),LL(0x11b688a0,0x5b7b74ff)}},
},
/* digit=69 base_pwr=2^345 */
{
   {{LL(0xc0385874,0x4ae3d232),LL(0xcbf96d2a,0x83bda9e6),LL(0xec62fd6a,0xba73c769),LL(0x62a4720c,0xd586ba7f),LL(0x0cc1f491,0x6497cd14),LL(0x7b2ac571,0x8b012b70)}, {LL(0x268fd705,0xa65eabb6),LL(0x1431873d,0x8caf100a),LL(0x231457d7,0x25b31b84),LL(0x901645c5,0xcab62f75),LL(0xb2f7b656,0x2377d74d),LL(0x2d33c95c,0x4008277c)}},
   {{LL(0xed001c43,0xaae2f448),LL(0xcf4be493,0x08ad1d9b),LL(0x82c1f372,0x3262b2f4),LL(0x351a5f7f,0x5521febd),LL(0x916c75a8,0xf8ec9190),LL(0x2728dfb8,0xf3c258c7)}, {LL(0x8af19574,0x5dd4ff4f),LL(0x5d076b1c,0xefddf579),LL(0xba8b777a,0x318b5b98),LL(0xfb7f8409,0xd971d426),LL(0xb0fd31db,0xed1465e8),LL(0x00f66347,0x80d24d43)}},
   {{LL(0x1ae586a2,0x5ba5288c),LL(0x1b715821,0x044f1fc6),LL(0x602f3c65,0xc1a9a997),LL(0xe08c0223,0xc5c7512f),LL(0x367e6f1d,0x48a19c3c),LL(0xfb241597,0xa9f2195d)}, {LL(0xb5ba32a6,0x9f674a5f),LL(0x0a312742,0x275a060f),LL(0x03d6f13e,0x5aeb8c43),LL(0x917433fb,0x0fed575d),LL(0x59f53490,0xe4a5ef9a),LL(0xf315e616,0xa9f86145)}},
   {{LL(0x7e5a59f4,0x315ad7a4),LL(0x543c8b00,0x1c615bfc),LL(0xbaa56473,0xe12f97a8),LL(0x46edcfcb,0xf263db44),LL(0x3c1a968e,0x47cf91d5),LL(0xc15db875,0x1a1165b4)}, {LL(0x3479616a,0x5d35e53a),LL(0x5c59958f,0x649f87b4),LL(0x246da3d3,0x5d3d11ea),LL(0x53f06820,0xc1ddfcc6),LL(0x6610c00f,0x8169d711),LL(0x4bddc8c7,0x15f16ba5)}},
   {{LL(0x1e548ef0,0xcf19fb2a),LL(0xcc694171,0x8bb6dfa0),LL(0x5c5e390f,0xeb1668ca),LL(0xe1975263,0xf5a3485b),LL(0x442cc850,0x4edfc596),LL(0xf9627d74,0x9901f447)}, {LL(0x84d0413a,0x3a6b85c9),LL(0x67de639c,0x14663661),LL(0x11705bbb,0x9fc9fdcf),LL(0xbff2cf80,0x6d066e2b),LL(0xdc3026fd,0x38dedc2f),LL(0x1b828538,0xad533a98)}},
   {{LL(0xb7bfa29e,0x6c75bc93),LL(0x18ef6d69,0xf86f22b2),LL(0x36dcadf2,0x90ce6a15),LL(0x7ce50921,0xf11f711c),LL(0x38a479e3,0x0739ceda),LL(0x6ec3dbc6,0x840b825e)}, {LL(0x9fa23481,0x7c36c0a5),LL(0x70cb186d,0xceb61fd1),LL(0x26e4754d,0xac6f7d3c),LL(0xf317b385,0x4076d3b5),LL(0x3fd9e9c7,0x52f1bd72),LL(0xbf316043,0x6649d8b6)}},
   {{LL(0x243cfbd5,0xdc2870f8),LL(0x1ab496f3,0x000b71b3),LL(0x708f4507,0x53511a3f),LL(0x1949d835,0xbd7bd038),LL(0x938f4db6,0x723a007d),LL(0x2d04e9fd,0x5bc8679e)}, {LL(0x76ec7fc4,0x51ca5fd1),LL(0x988f354e,0x86c4205c),LL(0x2a0a4a90,0x9042e76b),LL(0x4ad44d2d,0x368f52a8),LL(0x912edfb7,0xddc2cab8),LL(0xcde80199,0xde74ccf5)}},
   {{LL(0x3e455a61,0x8002f458),LL(0x5bea205a,0xafbafd37),LL(0xfb93f735,0xa8ced112),LL(0x196e3084,0x27cb6292),LL(0x77e8c744,0x72395bdd),LL(0xee71f5ff,0x02e018d8)}, {LL(0xc1337a1d,0x7cfc14d9),LL(0xd7b4d86e,0x94e14c0a),LL(0xd213738e,0x66e50129),LL(0xbc0b5ea3,0x7a905d91),LL(0xfca06700,0x92cb630a),LL(0xbf3a0821,0x65e06d5c)}},
   {{LL(0x55c2369d,0xa81e4a4e),LL(0x60a0f544,0x394de01b),LL(0xa8906e17,0x22acfd07),LL(0xcc9bc4d0,0xf59b37a6),LL(0x7ffec12f,0xdd16a22c),LL(0xd5976455,0x07decc2a)}, {LL(0xabe1d122,0xc5019463),LL(0xe318c92c,0x2bf0ac0c),LL(0xb2bfc47b,0xfa50280a),LL(0xc7cf8bff,0x53354fc5),LL(0xe20ca341,0xaea1d293),LL(0x8b626244,0xec25ecda)}},
   {{LL(0xfd8ba33e,0x313b66ca),LL(0xfabe27dd,0x10bdb130),LL(0x125e2b8c,0x1181334c),LL(0xdb6f94ba,0x0f4f198f),LL(0xac3f5de9,0xf7000076),LL(0x9d6402ae,0x1a78813d)}, {LL(0xc8a9e758,0x3427f75d),LL(0xb01f791f,0xcdac8b34),LL(0x2a9ebaf5,0x922c36d1),LL(0xb0487cc4,0x195ea05f),LL(0xa808baec,0xe33de901),LL(0x57291d89,0x15e1d5ac)}},
   {{LL(0xc21cdd1f,0x4e2a05c1),LL(0xdd46e76a,0x8a232097),LL(0xd871b1d6,0x8b55313c),LL(0xaf396bc4,0x976ce5f6),LL(0xafd381b1,0xeb91527d),LL(0x14455ee2,0x6cfd4490)}, {LL(0x1f274d1e,0x8723be9e),LL(0x1999fa9f,0x1c63fd01),LL(0x8049b6f8,0x5f172625),LL(0x99a51b4d,0xe18a3ecd),LL(0xb13d4e65,0x329fc2c1),LL(0x0f18f300,0x94da252b)}},
   {{LL(0xe28fd10d,0x7b151b98),LL(0x1dd884cf,0x8fc01ce8),LL(0x98d56c2c,0x1f0ffb50),LL(0xb084606d,0xf9df1fa2),LL(0xdc7d2008,0xf86232bf),LL(0xd8751699,0xeae5cb8f)}, {LL(0x83ed54fd,0x70f02298),LL(0x86087697,0xb575283a),LL(0x0302e2c3,0xad219135),LL(0xc4b57e01,0x1c09a0d6),LL(0xc541b9fb,0x0f65e1e1),LL(0xf4fe76c0,0x85493d9b)}},
   {{LL(0x191c21cc,0x353718ce),LL(0x4ad6bd18,0x08e6edf6),LL(0x4dc5b572,0xc2bb0d6e),LL(0x88193daa,0x328e19df),LL(0x7211c958,0xccc9f6ab),LL(0x58aae5c5,0x377d99ef)}, {LL(0x1c823442,0x40e2ecc9),LL(0x8b0d36ab,0x036d6d52),LL(0xda4d0ad3,0x2fe0cd7e),LL(0xfc8af791,0xb8fc3c7f),LL(0x2b201b20,0xdb7e44a4),LL(0xebcf527d,0xa5176004)}},
   {{LL(0xfa5256d2,0xe19b7576),LL(0xdb3f8bfd,0x418d5425),LL(0x951a1719,0x00424869),LL(0x533b69b0,0x2383c7a8),LL(0xe67a86fd,0x166a38e2),LL(0x5876c435,0xa6baa01c)}, {LL(0x84a208f5,0x574ddc45),LL(0x26b18dbb,0x8cee30b8),LL(0xe9f6b30d,0xeced99c1),LL(0xa7d34bea,0xb638d88d),LL(0x069adedf,0xa4836806),LL(0x7a07c593,0x62beb7ee)}},
   {{LL(0x724fb94d,0x5093950b),LL(0x8117ff50,0x10782271),LL(0x9f5961d7,0xdc9e34b5),LL(0x2351a33e,0xfaa2fc01),LL(0xd5fc462e,0xb9e0f1d9),LL(0xdd9c6914,0x276a5b3b)}, {LL(0x75365ca5,0xe6136d17),LL(0xa91eed68,0x228b77e2),LL(0x411e4770,0x5cd6a269),LL(0x17590390,0xd8857b0e),LL(0xa0d45faf,0xe7094f3a),LL(0xf40693e4,0xe52d11dc)}},
   {{LL(0xe5f5b545,0xe96c4aeb),LL(0x2d4c43b0,0x10a85a00),LL(0x32f9151d,0xf86ad2f6),LL(0x302b99e2,0x05daf874),LL(0x14fd3171,0x4299dbfa),LL(0x812cfc62,0x27cbedd6)}, {LL(0xb8772164,0x42e61536),LL(0x6a5423ef,0x52eecef7),LL(0x548fffa3,0xc34c6c70),LL(0x7b6db825,0x1fbed777),LL(0x4ef2989e,0x850bded4),LL(0x815463ee,0x3b8a542c)}},
},
/* digit=70 base_pwr=2^350 */
{
   {{LL(0x3079fe2f,0x9decf217),LL(0x7c817f6d,0xc32ec570),LL(0xaeb36b92,0xd5649ce8),LL(0x58fb4dc8,0xab9f77d1),LL(0xb52d60cf,0x66b11fb6),LL(0xeaaa4619,0xe217941d)}, {LL(0x1607c412,0xf3522a9a),LL(0xc2a3d8c9,0xea2eba4f),LL(0x25e38722,0x267997c6),LL(0x2d4595ee,0xed5047b7),LL(0x55e5456c,0xaaa41e5f),LL(0x78cfc6fe,0x891e3d12)}},
   {{LL(0xd7357a51,0xa438634e),LL(0x18c04d59,0x918f14cd),LL(0xac40dd56,0x2ab4dedf),LL(0x4956a5de,0x758e95ee),LL(0x5113f84d,0xfc11e394),LL(0x6059f16c,0x6d71b673)}, {LL(0xfb357c3f,0xfd8e2236),LL(0x32dca873,0xd7c86812),LL(0x8ea13b44,0x02aeb153),LL(0x013d3827,0xde1275d3),LL(0x659ca201,0x470a7b7e),LL(0x5c77b351,0x862c83c5)}},
   {{LL(0x05084cfb,0xfc9b800f),LL(0x496f23fc,0x1c4d4510),LL(0xc1d08465,0xfea0003c),LL(0x9af48a41,0xf0281da0),LL(0x44d32eed,0xa5c0d971),LL(0x023a2e31,0x2613b73e)}, {LL(0x7dc8ac1a,0x455013c8),LL(0x5958b3da,0x581b1319),LL(0x2290aaea,0xd293f2f2),LL(0x96f6223b,0xa0682564),LL(0x69410ef6,0x38fd18fa),LL(0x2b2cf629,0x74eaf35f)}},
   {{LL(0xc7ff5b50,0x281f6e58),LL(0xcf9cd114,0xbc67791e),LL(0xfd89abd8,0xe29fa41a),LL(0x7984feef,0xfcb0b0b0),LL(0xd9d20a64,0x0b0928a6),LL(0x6979ccd5,0x2fd385c4)}, {LL(0x1fbe72e4,0xce9c34c8),LL(0xaad0135f,0x69364344),LL(0x50946a5b,0xd4646352),LL(0xf39f53b9,0xb09a97c6),LL(0xdcbc8b64,0x1d47bc20),LL(0xd458b0d6,0xcda5c7bd)}},
   {{LL(0x87eff3b3,0xad5b8c28),LL(0x9937833a,0xa8a3917d),LL(0x200c3b49,0xbafdc493),LL(0x972c6fbf,0x9e27aac5),LL(0x0518c97d,0xfd292bb2),LL(0x33515a63,0xa62126db)}, {LL(0x1bcfc875,0x9892a8bb),LL(0xe0b674d1,0x93b066b7),LL(0x7fd3d080,0xcde9b008),LL(0x59401ae8,0x1e285a88),LL(0x82cfea96,0x4679e329),LL(0x23e615d3,0x52406ea0)}},
   {{LL(0x8b6e9462,0x27de6113),LL(0x473464bf,0xb8ade1dc),LL(0x94dacc08,0x911ad493),LL(0x44252cb1,0xd036f28e),LL(0xd13dc20d,0x3865abf6),LL(0xd528f0ba,0xcea487cd)}, {LL(0x4fc290fe,0x14d77eaf),LL(0xc5084101,0x5106533b),LL(0xcda9eccd,0x11001dc7),LL(0x49fc4a78,0xb79ad4bc),LL(0x4567f8a9,0x4f6369f5),LL(0xdf7ab817,0x64050aa2)}},
   {{LL(0xde07f615,0xffe057aa),LL(0x342700bd,0xf3f91b55),LL(0x27a839f9,0x294761e1),LL(0x80eafe1c,0x6411a2b4),LL(0x0737b80a,0x4900eb12),LL(0xbb73264c,0xa1134d10)}, {LL(0x0ddbf7f1,0x0ebfad73),LL(0xcd1f73ec,0x57bbe692),LL(0xa20f8944,0x675931fc),LL(0xfad2ad19,0x1950eeff),LL(0x9cdf88a0,0x60d30402),LL(0x33fd2c6e,0x121af89e)}},
   {{LL(0x295c4db2,0x763e3664),LL(0xdbbaa92d,0x632fd676),LL(0xc66b40e9,0x62ab11a8),LL(0xf384b708,0x06244698),LL(0x69197876,0xe7cdf3bd),LL(0x064f8837,0x9cc79c48)}, {LL(0x9486589e,0x95900a22),LL(0x2ff01639,0x7953f6e7),LL(0xdd3e6e46,0x3f65fbbd),LL(0xbaa2e2a0,0x84f52e06),LL(0xe3852824,0x1dc462a8),LL(0x7e4c032c,0x9be69c3f)}},
   {{LL(0x70977e01,0xa40afc36),LL(0xa6c3b289,0x965f110d),LL(0x805a8491,0xc4952f87),LL(0x0b65e2d3,0xb36804b8),LL(0xe8cf2b2b,0xd3f6f5ac),LL(0xa4b71938,0x0f37a79d)}, {LL(0x489ef671,0xb2f810d9),LL(0x2df23cd8,0x1feae026),LL(0x21a14e4f,0x7412eee3),LL(0x179d51fa,0x1458b8ad),LL(0xe201509c,0x2156a90e),LL(0x72605867,0x39f29fca)}},
   {{LL(0xb2e066e3,0x231f70ad),LL(0xbb477a19,0xf09db880),LL(0x907e5c63,0xdfa0e503),LL(0xf97022ad,0x12fe09f4),LL(0x20bce7dd,0xdbf06f36),LL(0xf1371cba,0x0140e197)}, {LL(0x64b0b4b0,0x917b6da4),LL(0x20fe3320,0x9a6f4d9b),LL(0xd66bdf87,0x0981d60e),LL(0x62d3487c,0xb430e4e0),LL(0x34dc4a94,0xc3440fb9),LL(0x09a5e3c9,0xe7972dda)}},
   {{LL(0x93f47052,0x29d63940),LL(0x847e5937,0xadf04e70),LL(0x731bab6f,0xa0ef4fee),LL(0x6ee7d7bd,0x21de3195),LL(0xbd716777,0x99af4a8d),LL(0xdf4c569e,0x9e15c983)}, {LL(0xe94401ea,0x2ec7bc0c),LL(0x85727722,0xda1835ad),LL(0x5dad81da,0x2b5862ce),LL(0x88dddc2e,0xb2be5081),LL(0x1414286b,0xa0248210),LL(0x8ea33f3f,0xc52c436d)}},
   {{LL(0x3b24e776,0xcc580ea7),LL(0x9d721d6e,0x0f3a8b18),LL(0xb23480cf,0x8665604f),LL(0x34414689,0x95787cba),LL(0x4d10a945,0x425d7c6f),LL(0xb2f1cc78,0xb5ec2626)}, {LL(0x8658de6b,0x55da8885),LL(0xe9aba03e,0xb50919d1),LL(0xd99e417e,0xc64881d7),LL(0xbf28fba2,0x1eeba5aa),LL(0x504eff80,0x20feb7b3),LL(0x50debfb7,0x9f5f9db6)}},
   {{LL(0x230923db,0x4eb94584),LL(0x7b3a6929,0xba861128),LL(0xab1d6b31,0x5aa7faa3),LL(0x16ae0966,0x95c1e239),LL(0xa2fe2297,0x98674fd3),LL(0x3c42d488,0xa8da0ee5)}, {LL(0xe0740db0,0x103cabac),LL(0x5bf16882,0xf0b860d4),LL(0x289e48ce,0x03cb0cdc),LL(0x9e52c7d5,0x3c15d375),LL(0x98103ca2,0x524f7319),LL(0xc609ffeb,0x828ed65c)}},
   {{LL(0x83dfb993,0x518f231b),LL(0x37c0826c,0x4b0987db),LL(0xd5177ead,0x0c34961c),LL(0x452c92da,0x9d882d3e),LL(0x8765bced,0xbfeaf558),LL(0xb9962295,0x83957b62)}, {LL(0x7bb084cf,0x2d1d0175),LL(0xe8cffcfc,0x04c4cfcd),LL(0x8d4536c1,0x2f35e33d),LL(0xd83124cf,0xbebb31cb),LL(0xabb29019,0xe342bed2),LL(0x2692a0d3,0x2af0fcde)}},
   {{LL(0xc7e3b29f,0xece5d865),LL(0x622839dd,0xe58106a4),LL(0xf2969d76,0xf5272d43),LL(0x2a1a240f,0x90c72c1b),LL(0xaf15e14f,0x1e2aa0ac),LL(0xf1b6b5a0,0xfa2f1c7b)}, {LL(0x880224a5,0xfb5d343d),LL(0xf91881c5,0x47b88a84),LL(0xdd142fe7,0x140f5ee9),LL(0x24b37c44,0x4e76982e),LL(0x578b482b,0x6aaf61e9),LL(0x765bc4e2,0x01950e22)}},
   {{LL(0xe8a2e8f0,0x20ebf79c),LL(0xaca418a2,0xec040d0d),LL(0x8d630d2a,0x016c07e7),LL(0xfa605dcb,0x20021d57),LL(0x42d04705,0x6190f3e9),LL(0x8974b7e6,0x4e000df5)}, {LL(0x5abcedac,0x6710da6c),LL(0x5f95d37c,0xf31aa496),LL(0xa5830899,0x192c4b8b),LL(0xea7dbcdd,0x171ab8c4),LL(0x8cdf1097,0x715f6081),LL(0x205d10ed,0x0e0135bf)}},
},
/* digit=71 base_pwr=2^355 */
{
   {{LL(0xc8d2bf7b,0xd2db4d35),LL(0x81571d06,0x52105d09),LL(0x723a57bf,0x447565cc),LL(0xd8ded62c,0xd98c3597),LL(0xde2f1a9e,0x0aeac6d9),LL(0x0a98d3b2,0xd363b0b7)}, {LL(0x02ad9933,0xd9708f07),LL(0x64f5809d,0x93346775),LL(0x49cda010,0x499332cf),LL(0x858467e2,0x546df74a),LL(0x93748e8e,0x8b84a550),LL(0x06f09073,0x9e88ef97)}},
   {{LL(0x52133095,0xc0a40cac),LL(0x93c162bb,0xfe1b22fd),LL(0x34018741,0x8625898c),LL(0x36d9e57a,0x69c9f3f6),LL(0x378aa211,0x69d9d7f3),LL(0xe7dca168,0x6b03f897)}, {LL(0xf997a48f,0x24d49aeb),LL(0xc149ac40,0x1d984c67),LL(0x576f533f,0x667c1d01),LL(0x9ef82ece,0x372eee19),LL(0xc207c14d,0x577723c0),LL(0x0eed37f6,0x4225907a)}},
   {{LL(0xc5a5a001,0xb8623f36),LL(0x86878b74,0x21847b80),LL(0xd05ac443,0xd19f57f6),LL(0xb9f8acec,0x9bd22882),LL(0x2b41b60a,0x128186d8),LL(0x71980fd1,0x1772e6b0)}, {LL(0x812dfff5,0x22c5ee68),LL(0x7f952796,0xccbd2fe2),LL(0x7da6d35a,0x0d49bfde),LL(0xc249f319,0x348b48db),LL(0xc16a8c0f,0xdb376657),LL(0x002cf8b4,0x28ef362a)}},
   {{LL(0xbc0e0903,0xc61db977),LL(0x645c32fb,0xbaf6e4da),LL(0x060b1adb,0xce89b8ca),LL(0x88e2c178,0x41db4481),LL(0x923bdd3c,0xba6339f3),LL(0xd29db42c,0xff25b818)}, {LL(0xe6d6b35d,0x3521116e),LL(0xb22f16ac,0x4e1bd283),LL(0xbd79fe5d,0x9357c984),LL(0x9d45eee4,0x2eda73be),LL(0x6288e01f,0x1a50c59f),LL(0x75018873,0x37baf649)}},
   {{LL(0x751f66f6,0x431ea808),LL(0xf1b8e577,0x06feeefa),LL(0x488a8eee,0xd109d9f0),LL(0xc69843c5,0xeb826b96),LL(0x8b42da29,0x972272e9),LL(0xa137ee9c,0xa9ea9ad1)}, {LL(0x25163be5,0x0385aa95),LL(0x34c32f2b,0x963e9640),LL(0x4dfd935a,0x9067aa89),LL(0xbc3f5f3c,0x6ab23131),LL(0xf302e0e1,0xb96c3406),LL(0xd65f811a,0xc8caad4e)}},
   {{LL(0x750261dc,0xf323953b),LL(0x38552a8c,0xb8563bf1),LL(0xbd32cb8c,0x2937dfd9),LL(0x7ecf3538,0x07c4e563),LL(0xb6399415,0xb573960e),LL(0xdd1a4a06,0xc1ced6c1)}, {LL(0x787ddf7a,0xc625f400),LL(0x1ce6778f,0xd998c28b),LL(0x6220f24d,0x66c51e5b),LL(0x27d01c6f,0x9f97d758),LL(0x0412372d,0x60b15724),LL(0x04d55048,0x2253abc1)}},
   {{LL(0xfad177ed,0x15952eb8),LL(0x05058d9f,0x57aaf91d),LL(0x8ab1b9d9,0x268ba730),LL(0xd5b8f86a,0x7decfc47),LL(0x0879ab02,0x596353e7),LL(0x9a68d5a4,0xa3ff2311)}, {LL(0x7534fb5a,0x257c68a1),LL(0x160ec5ea,0x84f7c9de),LL(0x6754185f,0x1b2b770c),LL(0xa74562f7,0xf321ac71),LL(0x264a1961,0x28bf0a15),LL(0x86a033ae,0x0093db73)}},
   {{LL(0x097f322f,0xc552c6c6),LL(0x8bc06287,0xdf59a302),LL(0x19610b0c,0xc9ed375c),LL(0xb051dad5,0xf0e7b4eb),LL(0xc6556643,0x7267a304),LL(0xc96dc1d8,0x0044f6d8)}, {LL(0xf4fc3725,0xf0ed5f9a),LL(0x9de8e1ff,0xbbaf9f2c),LL(0xaf5a4b4b,0xef5d66f4),LL(0x20644cf2,0x0b5bed3d),LL(0x75ae23c0,0xf7e4543a),LL(0x41325b66,0x696f60dc)}},
   {{LL(0xcfc0dd38,0x4fd0b582),LL(0x55b7b68a,0x47fd984f),LL(0x699460ec,0x2722a0cb),LL(0xfa26d4c4,0x81b4afd8),LL(0x941c86e4,0xb921d0d5),LL(0xef4db114,0x7208f094)}, {LL(0x17ddadf8,0x2997f43c),LL(0x21bc290d,0xd1aabdea),LL(0x86182be1,0x64e20e00),LL(0xcbddb8eb,0x9bd11568),LL(0x3ba0e6b5,0x639db1d4),LL(0x429a6b4f,0xb99b11fb)}},
   {{LL(0xaaa48cd8,0x04ef7ad5),LL(0x8a8ac319,0xb97a6501),LL(0x47591d88,0x9ae38a6a),LL(0x6902edb4,0x27d91254),LL(0x812b143a,0x5dae3d83),LL(0x93a2fdf6,0x02ee1353)}, {LL(0x72410377,0x07a00389),LL(0x56e10c82,0xa2fbd343),LL(0x72b1bcb9,0x3fd6c171),LL(0x2d0033c7,0xa8d70f93),LL(0x2916c28a,0x9ea9eea3),LL(0x423edad7,0xecb7e486)}},
   {{LL(0xc515bacf,0x5525fbb0),LL(0xa2aa22ba,0x48a72394),LL(0x0b9e3fe3,0x9bcd64c9),LL(0x1975aa86,0xe9e11d17),LL(0xc9dbdaac,0xa435bbf0),LL(0xe30c8911,0xe8451f6c)}, {LL(0x5bc2d12f,0xa1706d89),LL(0x406d4883,0xfe73ff43),LL(0x49d5739c,0xb713efc6),LL(0x05c1ec9a,0xdfd0bbf8),LL(0x5e6b3bd0,0xde17c52c),LL(0x5be196af,0x57e06034)}},
   {{LL(0x2c20f868,0x9949b33d),LL(0xb5706250,0xdb3aa790),LL(0x88e17f2b,0x88ce71e7),LL(0xda9c0881,0xd851baf2),LL(0x86d8c9e9,0xe869c5ba),LL(0xa01425b6,0x1af68d65)}, {LL(0x9bbd3963,0xeae8b1c6),LL(0xec087425,0xf34900b1),LL(0xc374bb96,0x14942910),LL(0x05487483,0x3e13c457),LL(0x35bc6ee1,0xe0e6fad4),LL(0xb54d247b,0xc7c38dc7)}},
   {{LL(0x8bd92789,0x420dd8f6),LL(0x4ce541dd,0x3831b0e9),LL(0x31ed7f7f,0xbde477e4),LL(0x29c5557b,0xb46eb7f2),LL(0xc56940e0,0x00a07499),LL(0xabacf00e,0xabb9d567)}, {LL(0xc5ba9d0c,0x9f63b87d),LL(0x708d4f4c,0xf4c5c4a3),LL(0xde7fcf63,0xdbde9879),LL(0x616cf5d6,0xbe88f949),LL(0xd1aa38be,0x17560674),LL(0x9c436175,0x160ec365)}},
   {{LL(0xb9bd0afb,0x9c85e50a),LL(0xd26ac425,0x2b66a1c0),LL(0x807f33db,0xcb78ce81),LL(0x3db81e06,0x9d337d8a),LL(0x72638d70,0xe223eae4),LL(0x6bf2ebab,0x7f9ea217)}, {LL(0x9b634059,0x2804b59c),LL(0xf3dc8d46,0x1043fbf4),LL(0x7bc6949c,0x321eca1e),LL(0x8f051155,0xbf2906c0),LL(0x9c539f40,0xb802a328),LL(0x073cd808,0x08bcca20)}},
   {{LL(0xcea8bf63,0x08cb6315),LL(0xb59ee7fa,0x7ac2699d),LL(0x8d1601e0,0x4cd2c8a9),LL(0x7b90d9c3,0x6d7188e3),LL(0x6bfe73d3,0x63486716),LL(0xcf9f30b0,0x49ed7faa)}, {LL(0xc9515fd6,0xf7edf5d6),LL(0x8a8ae607,0xbb3ab848),LL(0x21c5c388,0xe6e6b209),LL(0x5b1e03bc,0x5dca0a1d),LL(0x94a8b174,0x15670f59),LL(0x68c27a97,0x6f79e381)}},
   {{LL(0xab463fa6,0x6d34bdf6),LL(0x0093b9cb,0x7bb127b6),LL(0x5a3bfdd0,0x61d05113),LL(0xf1296bdd,0x4abab575),LL(0x4d2e9a7c,0x72da6849),LL(0x8d11f03d,0x90267bca)}, {LL(0x3e9b310b,0x47811122),LL(0x1b1920cd,0x8ffe91d3),LL(0x7521898e,0xec293ec6),LL(0x96c1da75,0xf0cf0269),LL(0x80f2c7b3,0xb0dbd4c3),LL(0x34e4baf8,0xe5281755)}},
},
/* digit=72 base_pwr=2^360 */
{
   {{LL(0xdd14d47e,0xc32730e8),LL(0xc0f01e0f,0xcdc1fd42),LL(0x3f5cd846,0x2bacfdbf),LL(0x7272d4dd,0x45f36416),LL(0x5eb75776,0xdd813a79),LL(0x50997be2,0xb57885e4)}, {LL(0xdb8c9829,0xda054e2b),LL(0xaab5a594,0x4161d820),LL(0x026116a3,0x4c428f31),LL(0xdcd85e91,0x372af9a0),LL(0x673adc2d,0xfda6e903),LL(0xa8db59e6,0x4526b8ac)}},
   {{LL(0xcecb916f,0x51a033a5),LL(0x8d7de61c,0x2ac62f63),LL(0xa42a266e,0x92eece49),LL(0x82c4d11e,0x87e037db),LL(0x6fbae08a,0x875be141),LL(0xc539478c,0xf348fe26)}, {LL(0xff94c01e,0x51f8b907),LL(0x19695a9d,0xc46cc0e0),LL(0x6c51b9c2,0x2c74bd66),LL(0xee565de8,0x635d3d24),LL(0x8982c8c3,0x6bd65663),LL(0xdaf6a93c,0x5c345b79)}},
   {{LL(0xe23a8472,0x68fe359d),LL(0x4ce3c101,0x43eb12bd),LL(0xfc704935,0x0ec652c3),LL(0x52e4e22d,0x1eeff1f9),LL(0x083e3ada,0xba6777cb),LL(0x8befc871,0xab52d7dc)}, {LL(0x497cbd59,0x4ede689f),LL(0x27577dd9,0xc8ae42b9),LL(0x7ab83c27,0xe0f08051),LL(0x2c8c1f48,0x1f3d5f25),LL(0xaf241aac,0x57991607),LL(0xb8a337e0,0xc4458b0a)}},
   {{LL(0x210c3144,0x179c59cf),LL(0x33eebbc4,0xfb613c57),LL(0xba0cf384,0xdda75cfd),LL(0x3a8fbafa,0x94081a5b),LL(0x33384e0b,0xb91de90a),LL(0x27aa2a45,0x7d1f8f40)}, {LL(0x62031148,0x0747bcc1),LL(0xf324160b,0xd2db8e39),LL(0x722484f0,0x9c1ce3e9),LL(0xa62d1dda,0x13a7ee5d),LL(0x3a963bce,0x77fd7934),LL(0x83d2f21b,0xcd3d8717)}},
   {{LL(0x51dd1ba9,0x3dbb3fa6),LL(0x545e960b,0xe53c1c4d),LL(0x793ce803,0x35ac6574),LL(0x83dbce4f,0xb2697dc7),LL(0xe13cf6b0,0xe35c5bf2),LL(0xb0c4a164,0x35034280)}, {LL(0xd9c0d3c1,0xaa490908),LL(0xcb4d2e90,0x2cce614d),LL(0x54d504e4,0xf646e96c),LL(0xb73310a3,0xd74e7541),LL(0x18bde5da,0xead71596),LL(0xaa09aef7,0x96e7f4a8)}},
   {{LL(0x2d8bcd6e,0x609deb16),LL(0x2591750d,0xe42f23a9),LL(0xb378305c,0x4a9f3132),LL(0x69275f5e,0xf1017998),LL(0x61b089b5,0x14be7467),LL(0x0c81b0c5,0x05f620d2)}, {LL(0x6cb8412e,0xca90a9c0),LL(0x15b1b0d5,0xfe0f6a89),LL(0x20c71988,0x1b25ac96),LL(0x390aedd0,0xb971b61a),LL(0x79d8cd39,0x995214d7),LL(0x65c6e11a,0xd7fa135b)}},
   {{LL(0x5d6e5f48,0xa8393a24),LL(0xf9175ce8,0x2c8d7ea2),LL(0x55a20268,0xd8824e02),LL(0xa446bcc6,0x9dd9a272),LL(0x5351499b,0xc929cded),LL(0xcfe76535,0xea5ad9ec)}, {LL(0xdc32d001,0x26f3d7d9),LL(0x43eb9689,0x51c3be83),LL(0x759e6ddb,0x91fdcc06),LL(0xe302b891,0xac2e1904),LL(0xc207e1f7,0xad25c645),LL(0xab3deb4a,0x28a70f0d)}},
   {{LL(0x0f3ff12d,0xa13f19b4),LL(0x019564aa,0x57ee08b1),LL(0x7044a6f4,0x00ec0c99),LL(0xdca1075c,0xaf5665f8),LL(0x0620ab0c,0xded5ca3f),LL(0xa896deff,0x9b2cb8c7)}, {LL(0x07df2345,0x032ab2b3),LL(0xf1da3f88,0x964d109e),LL(0x25133304,0x2286b6f7),LL(0x977a4567,0x0d16d531),LL(0xf1abae4f,0x00a66036),LL(0x95f0103b,0x5debab1d)}},
   {{LL(0x03bea8f1,0x922d7f97),LL(0x584570be,0x3ad820d4),LL(0x3cd46b43,0x0ce0a850),LL(0xae66743d,0x4c07911f),LL(0xfda60023,0x66519eb9),LL(0xec2acd9c,0x7f83004b)}, {LL(0xc3117ead,0x001e0b80),LL(0x0722ba25,0xbb72d541),LL(0x6e9a5078,0x3af7db96),LL(0x701b6b4c,0x86c5774e),LL(0x37824db5,0xbd2c0e8e),LL(0xbfac286d,0x3ae3028c)}},
   {{LL(0x5fb7c5a8,0xfd1b413f),LL(0x0206cba1,0x6152b9de),LL(0xd8f51960,0x487f8e3a),LL(0x033ca1bc,0xac34a23c),LL(0x60258d55,0x90bba98f),LL(0xbd9098f2,0x30421acf)}, {LL(0x89c0ce44,0xd9c601f9),LL(0x2f2f1af1,0x621bda83),LL(0x38c45441,0x14fa7ef6),LL(0xe47faa31,0xbd5dc10f),LL(0x74eeb6a1,0x9dce0dcb),LL(0x06346849,0x2cca3e66)}},
   {{LL(0xa33e071b,0x83d4d4a8),LL(0x61444bb5,0x881c0a92),LL(0x520e3bc3,0xeea1e292),LL(0x2aaab729,0x5a5f4c3c),LL(0xe63c7c94,0x0b766c5e),LL(0xbb2cc79c,0x62bb8a9f)}, {LL(0xaa5dc49d,0x97adc7d2),LL(0x31718681,0x30cc26b3),LL(0x56e86ede,0xac86e6ff),LL(0xcd52f7f2,0x37bca7a2),LL(0x9ce6d87f,0x734d2c94),LL(0xc2f7e0ca,0x06a71d71)}},
   {{LL(0x20066faf,0xcdb67ea9),LL(0xc7fb7154,0x929b4d2a),LL(0x7fdeb411,0x5842d968),LL(0xafe55cb9,0x2ddf764a),LL(0x47df3cf4,0x608bf76f),LL(0x5984e339,0x1862463b)}, {LL(0x944d22a2,0x7feea86f),LL(0x281f2b84,0xf8562c30),LL(0xbd358ea4,0x332a54d2),LL(0xe7fe1ede,0xa54dec9f),LL(0x932264d4,0x9c8e52a4),LL(0x89817f5b,0x428acd1a)}},
   {{LL(0xc6357d33,0x559dcf75),LL(0x652517de,0x4616d940),LL(0x1ccf207b,0x3d576b98),LL(0x1979f631,0x51e2d1ef),LL(0x06ae8296,0x57517ddd),LL(0xd6e7151f,0x309a3d7f)}, {LL(0x0e3a6fe5,0xba2a23e6),LL(0xd28b22c3,0x76cf674a),LL(0xf8b808c3,0xd235ad07),LL(0x6b71213a,0x7bbf4c58),LL(0x93271ebb,0x0676792e),LL(0x05b1fc31,0x2cfd2c76)}},
   {{LL(0xeda2fa8f,0x42aeebc4),LL(0xe77a9c5b,0xda91ada3),LL(0xc585a572,0x29b9d55f),LL(0xa256353d,0xb0e52414),LL(0x29adbd21,0x1d0e7d5f),LL(0xd057d175,0x7ee5ff9c)}, {LL(0xc9097bf9,0x0bf76fcf),LL(0xfe09f5b3,0x023170f8),LL(0x0799f989,0x8a67c124),LL(0xc6a20819,0x4ce28eeb),LL(0x79502d13,0xfc1d7c91),LL(0xde43f895,0x7922d2d9)}},
   {{LL(0x37a450f5,0x4258e5c0),LL(0x52d2b118,0xc3245f1b),LL(0x82bc5963,0x6df7b484),LL(0x9c273d1e,0xe520da4d),LL(0x2c3010e5,0xed78e012),LL(0x3c1d4c05,0x11222948)}, {LL(0xc692b490,0xe3dae5af),LL(0xc197f793,0x3272bd10),LL(0xe709acaa,0xf7eae411),LL(0x778270a6,0x00b0c95f),LL(0x220d4350,0x4da76ee1),LL(0xab71e308,0x521e1461)}},
   {{LL(0x96230a58,0xf2cdae31),LL(0xf304e1ea,0x47cf36b4),LL(0xd750b29b,0x9d14f25a),LL(0xdba15f3e,0x931b9c65),LL(0xbf9fe2dd,0x34db169e),LL(0x52663433,0x8a89e47c)}, {LL(0x026ec31f,0x8859a4f8),LL(0xa913ceea,0xeeb703ea),LL(0x67ac4db7,0x74638d6a),LL(0xbe25d755,0x5c8ea7b2),LL(0x38db10ee,0x8a0f0a87),LL(0xe890bcd3,0x96a26bac)}},
},
/* digit=73 base_pwr=2^365 */
{
   {{LL(0x64a5e869,0x883533af),LL(0x6973ec23,0xaaa778c2),LL(0x46d0fcf3,0x8f0b5fb5),LL(0x4ab05da7,0x7e6b0a0a),LL(0xc67b6614,0xcd91a869),LL(0x6c6f7cf2,0x7de9f2ff)}, {LL(0xd1ec14c3,0xc072a106),LL(0x42a128ee,0x3f4b9606),LL(0x8f0ce946,0x7378192c),LL(0xd1149441,0xdf2e7b9f),LL(0x14ccf45a,0x4fa17cb6),LL(0x45f03568,0x575680e9)}},
   {{LL(0x8f458c68,0x8b70e94f),LL(0x4160ecc7,0x29272654),LL(0x4d3ef22f,0xe22219ba),LL(0x1999f948,0x7f8a712a),LL(0xabfe7302,0x25575e96),LL(0x564a1af0,0x21c6ffc6)}, {LL(0x7e8500da,0x045e9c66),LL(0x04ef8ea6,0xef7c3cf7),LL(0xc3db161a,0xdd23b825),LL(0xba33a906,0x05fb173a),LL(0x870e41f2,0x9a8b5ecb),LL(0xccc30d1d,0xf3d9db0b)}},
   {{LL(0xe873ff0f,0xd9243926),LL(0xf20b0e49,0x2e2a5ab6),LL(0x0e35f201,0xa1bcfeee),LL(0x196f23f3,0xd25be5f3),LL(0xffc1d068,0x298c67f2),LL(0x0c3d950b,0x77dae55c)}, {LL(0x8822c996,0x5e15ab99),LL(0x83f60a98,0x52de2e6d),LL(0x47a7e269,0xa9f82ec9),LL(0x2ac22e49,0xf02af9a2),LL(0xa706f090,0xdfb3103f),LL(0x3cf8dcb0,0x12559962)}},
   {{LL(0x88ad12b2,0xb7b85625),LL(0x1e44b254,0x81f5958b),LL(0xc91b8ffd,0xb4ebddd5),LL(0x55d38511,0xef815ae1),LL(0x1b0da525,0x98587d55),LL(0x34a9ebbd,0x1d418177)}, {LL(0x1e6057d7,0x844811fb),LL(0x76e5976d,0x0c169771),LL(0xf623789b,0x4b268bb4),LL(0x40498058,0xb26ae5be),LL(0x3c2b435a,0xb47a5ded),LL(0x8fceceb3,0xe15a684b)}},
   {{LL(0xd09dc3c8,0x122a3eed),LL(0xaefe0819,0x6a19907f),LL(0xda325339,0x057aafa1),LL(0xd42a5345,0x138033bd),LL(0x1a065ebe,0x8ac206af),LL(0x25c31ed6,0x0a46f5ae)}, {LL(0xd7e379db,0x7fc577a9),LL(0x69dcee54,0xc6df6943),LL(0xa8336bc1,0x4c167ba2),LL(0xf3a1339c,0x0fbd9708),LL(0x226f612f,0xc6b8c01f),LL(0xd51332e1,0x5d4ed789)}},
   {{LL(0x5a1abcd8,0x26aa2c2e),LL(0x9609d9d8,0x2b16a12e),LL(0xa2bee00c,0xe485a551),LL(0xf4f2292e,0xfa28c30b),LL(0xb7358f1b,0x99abef78),LL(0x10a276a1,0xda6b3cdf)}, {LL(0x47c03f71,0xbd3858b7),LL(0xb22d05d1,0x4f0bf5f0),LL(0x8250f760,0x2d80f5d2),LL(0x8cd9666c,0x060f9b27),LL(0xb1b014a9,0x6a6c40b0),LL(0x8c440a9e,0x44537af3)}},
   {{LL(0x76faaca5,0xb564cfd6),LL(0x920dd223,0x8a6e3925),LL(0xa590a383,0xee59a140),LL(0xa1922ad9,0x9e29b552),LL(0x60a0da63,0x604367de),LL(0x92c35fd0,0xc498aca5)}, {LL(0x250ed8a0,0x74135082),LL(0x6c7c3e77,0x5d109d1a),LL(0xc63dff94,0xf9e2d84d),LL(0xf7aa2b0e,0xca50f5e4),LL(0xd543d389,0x7cba9e87),LL(0xd8fd1292,0xaf5fbbef)}},
   {{LL(0x4fc11c3a,0x70765683),LL(0x66aac4d1,0x53a94031),LL(0xa6db6169,0x2a935ef0),LL(0x2032d599,0x00292761),LL(0x3a6f1316,0xb5babb2d),LL(0xdb26af51,0x601a7dfa)}, {LL(0x1322d983,0x00c34013),LL(0x2bb507c5,0x45b062ec),LL(0x0f9b3656,0xa1bbe2ed),LL(0x34031d18,0xe17a5d49),LL(0xf8fe1224,0xe3661047),LL(0x623c6cf5,0x0e4f3b3d)}},
   {{LL(0xca45cccf,0xffaac084),LL(0x061ffe3c,0xaea5cc3d),LL(0xb355f387,0x7c5d7c60),LL(0x99cba92d,0x4bbb2a0c),LL(0x2f7768d6,0x6b4ba3ef),LL(0xcc5f9c40,0xc7484ed2)}, {LL(0x52b57a7e,0x5d4e92fc),LL(0xca2c200b,0xba9f16c4),LL(0x3797ccba,0xebe02a8a),LL(0x38c4133a,0xb6b3f421),LL(0x8153d033,0xad5d85b6),LL(0x5714f269,0x782d6ee8)}},
   {{LL(0x9c0cf752,0x7845b696),LL(0x5a732acf,0xb82d052b),LL(0x1262877b,0x7760564c),LL(0x8ecc7aa5,0x29b3c57a),LL(0xdf1ebbed,0xb58eccb0),LL(0x3c3a3303,0x86fc1544)}, {LL(0x13060f0e,0x44761ddf),LL(0x7371a5a8,0x5a3dacfd),LL(0xf7cbc2bc,0x846f6faa),LL(0x368caabf,0xf5e098b0),LL(0x10c08769,0xe23ea107),LL(0x1563fcda,0xbc5df1db)}},
   {{LL(0x142d8955,0x65c3a54e),LL(0xe7814f25,0x5c6583cc),LL(0xd924dc7d,0xbd5a07d8),LL(0xc28f6e8e,0x9f717bd9),LL(0x3b6540a7,0xa0c4ec4e),LL(0x142b3283,0x3153bb2b)}, {LL(0x9b296946,0x53bf403c),LL(0xb1cdb6d2,0x659a828a),LL(0x1369af19,0xe9517d81),LL(0x8990e7a0,0xd8c7a099),LL(0xe535cd04,0xbaa9f59d),LL(0x0f269551,0xbb0cc68e)}},
   {{LL(0x6a78c6e5,0x3c00ac52),LL(0xdefaa52c,0x9c61aca6),LL(0x39794a09,0x00341289),LL(0x41cd7c0a,0xe08910d1),LL(0xa732e3bc,0xa58ffbb6),LL(0x91fe8fd8,0x87bf51ab)}, {LL(0x4a59e2be,0xc4f4f267),LL(0x438071c8,0xdeb512c7),LL(0xe9cd290b,0xddf82842),LL(0x6ae85fe0,0x3e17311d),LL(0xb41be881,0x6e9236a9),LL(0x53555ebf,0xbb9ddf98)}},
   {{LL(0x09f3f0be,0xccc163ea),LL(0x6a5b0a63,0x9932b56f),LL(0x9c69668e,0xf89fae91),LL(0x5ce13021,0x555f9821),LL(0x37037aa9,0x4b02693f),LL(0xbde50f35,0xc4afee79)}, {LL(0x02aa6c7a,0x4b0919c2),LL(0x991e15e9,0x3166de2a),LL(0x7077fb38,0x284baa3e),LL(0xa116ddec,0xbb7a6416),LL(0xb7636772,0xe8c89547),LL(0x0ef92c54,0xff940362)}},
   {{LL(0xe2ce6008,0xd5d81275),LL(0x0b3b9d10,0xc45bdf25),LL(0x6cbc83e2,0x15ab5da3),LL(0xc52a66cb,0x85a18cf8),LL(0xb042c047,0x77e202b8),LL(0xe7e7997e,0xc4dc3de2)}, {LL(0x995fa67a,0xfe9335b1),LL(0x75b96a00,0x809e161d),LL(0xa0c3baea,0xfb03c2a5),LL(0x888c2f77,0x5c7e0523),LL(0x87ad10e2,0xa8fda1c8),LL(0x858a3577,0x90484f78)}},
   {{LL(0xf9fde126,0x49e41f0a),LL(0x3613d3c2,0xec960044),LL(0x10421d3b,0x2c62a49d),LL(0x8131a0d8,0xe2402464),LL(0xbdf794fc,0x8a7ce188),LL(0x4665b1b6,0x704dea7d)}, {LL(0x4d57c6ba,0xbdb9c18e),LL(0xf669b3c0,0x5288a053),LL(0x78a5e252,0xbf7d01b8),LL(0x26b9cb7d,0xb26cccf9),LL(0x84326c47,0x14191a32),LL(0x91f8425b,0x460ff747)}},
   {{LL(0xbd27be7b,0x59367582),LL(0x1ab2c596,0x92bf5bbc),LL(0xf6a27741,0x5d96351a),LL(0x7f929e0d,0xeab94db8),LL(0x043f1afb,0x865ba011),LL(0x5fb631dd,0x43acea12)}, {LL(0xb2fd1436,0x192e0652),LL(0x7b38d121,0x44f22ff1),LL(0xb7cae5f6,0x7bcc228d),LL(0x6a828b03,0x02eaeccd),LL(0x91f301aa,0x7c48a2ea),LL(0xf5eb1a07,0x1e090717)}},
},
/* digit=74 base_pwr=2^370 */
{
   {{LL(0x941948e3,0xdf0ae8df),LL(0x1d010bcd,0x123fee90),LL(0x1dd28691,0xde3717ca),LL(0x709b678e,0x0c1db879),LL(0x400acdc6,0x0288959a),LL(0x5ca2d03a,0x66c69181)}, {LL(0xdbbb75de,0xe52534b3),LL(0x3de927cf,0xe914938c),LL(0x73eece30,0x1a9a34f8),LL(0x642a6799,0x0fb0c7bd),LL(0xeaa7e8a8,0x375cc0cf),LL(0xd00ec238,0x75fb9eb5)}},
   {{LL(0xb72958eb,0x9ca8cc9d),LL(0x1014f562,0x3c8cd0db),LL(0x059b2bba,0x72115d53),LL(0x730e5dc3,0x8fe7ac30),LL(0x841d8998,0x4e67ef69),LL(0xc8ed37a5,0xfb6439ff)}, {LL(0x26df84c4,0x48164b3e),LL(0x365bc99e,0x37d492ad),LL(0xbeed38ce,0xb7fd4643),LL(0xa3e30b3d,0x993cfa9f),LL(0x01ddd484,0xdcc5e7af),LL(0x6840175d,0x5edf3ac0)}},
   {{LL(0x0c19c625,0x3ee87e54),LL(0xf4a10f9a,0xe4ae611f),LL(0x0aec21e5,0x27d65512),LL(0x8bea1b16,0x737578f5),LL(0x5fb7a3a2,0x6e2b6bf7),LL(0x14f65000,0xecc59251)}, {LL(0xb8dd544c,0x53e1167c),LL(0xc5862fdc,0xee60e60f),LL(0x6cccabeb,0xc86582cf),LL(0x8b4d37c6,0x849fc15d),LL(0x8f4a87bc,0xaa7960ad),LL(0x3b7f0a6d,0x17fe1082)}},
   {{LL(0x51d33c11,0x79768e9f),LL(0xa4b24889,0xeec34505),LL(0xbe0c67d7,0xc194821b),LL(0x6909fdfc,0x537a6a4a),LL(0x95ccdda7,0xae6d7051),LL(0x92b3926c,0xed4b7222)}, {LL(0x6b24a3d4,0x2c5dd6af),LL(0xe4386095,0x9282ec39),LL(0x397a3bd3,0xdd3c7388),LL(0x8baf59c2,0x9d176c6a),LL(0x380ec958,0xd5c6219e),LL(0x54e8e315,0x194fc116)}},
   {{LL(0x79650de1,0x3c67c65c),LL(0x00bfb2ac,0x8e2a220b),LL(0x42f02a3d,0x59d225dc),LL(0x5d60e54f,0xc52ce4b2),LL(0x3f306112,0x8894e3cb),LL(0x50d85aec,0xb78a037b)}, {LL(0x2d85f328,0xeeeb3b40),LL(0xe0406ddd,0x3d391b37),LL(0x0502141b,0x5273ebe9),LL(0xe092bfeb,0xd17023e7),LL(0x282f5aed,0x04564385),LL(0x55d82356,0x7e55a4e2)}},
   {{LL(0x01ec1432,0x0b7576f9),LL(0xabc5f603,0x84b30eec),LL(0xf4a84b7a,0xdaaf7ba9),LL(0x3bb37a99,0x9e3a5daa),LL(0x80378cff,0x56bd9648),LL(0x8e6ed856,0x2fdeeeb7)}, {LL(0x3c81ac34,0x079014a7),LL(0xb4211c27,0xf8de4004),LL(0x7fe4391e,0x0cee3df9),LL(0x2fd2fc38,0x441aa7fb),LL(0x4d1b575a,0xeba7be86),LL(0x231c2d01,0xca2fb5b7)}},
   {{LL(0xd205869c,0x8307bc81),LL(0x9388e917,0xdd282809),LL(0xbb572a87,0xdd4cb2de),LL(0x14f0db39,0x5a41595d),LL(0xf4e5c71e,0xd5db5f59),LL(0x4a87f672,0xcc9c9f6f)}, {LL(0x012e65ce,0x4ddce8c3),LL(0xab3a94d3,0xaae28eaf),LL(0x0b333aba,0xd4b95011),LL(0x914eb476,0x3e171506),LL(0x69bbfa63,0x109ee9ab),LL(0x5dff5471,0xbddba098)}},
   {{LL(0x463a8251,0x0683349e),LL(0x5103e72c,0x97dc4f47),LL(0x50663b9e,0x47c71810),LL(0x9733dac6,0xf327d149),LL(0x292137c5,0x03f55e4a),LL(0xda59e1c7,0xccc6232a)}, {LL(0xadc59cb8,0xbaa8b4dd),LL(0xfe7486e4,0x45370d35),LL(0xb0322df9,0x99a88c1d),LL(0xfd69954f,0x394440a7),LL(0xa5a29889,0x9060473d),LL(0x2f04864a,0xc8ca43e3)}},
   {{LL(0x9451d422,0xf08c8efc),LL(0xba846ac3,0xc8beae41),LL(0x807df062,0x0157f46c),LL(0xdeaada6a,0x080eac20),LL(0xa3bfddd0,0xdf146f3d),LL(0x617d19d7,0x8ed7c2c6)}, {LL(0x40439488,0x314ea3e7),LL(0xb3e6806f,0x544132c7),LL(0xa215a3a2,0x9b9bb477),LL(0xe33f80b0,0x9310fee9),LL(0xe71769af,0xf277895c),LL(0x7d99d5b9,0xe8218e1a)}},
   {{LL(0xdd8b9644,0xd5c7241a),LL(0x993116d2,0x45a5f2d1),LL(0x823048df,0xbacacd4a),LL(0x2568547a,0xa278fa04),LL(0xbff72820,0x3a4f2482),LL(0x13e4314a,0x1305d1a7)}, {LL(0x34ba28e3,0x9d84c333),LL(0x6a32fb41,0x9995b5bb),LL(0x520946d8,0xb0f75f3c),LL(0xde98aa63,0xd7c4b8b7),LL(0xba856b6b,0xee5efcf3),LL(0x3324ed66,0x36af3368)}},
   {{LL(0xbb2af5fb,0x29c741c3),LL(0x89e6241e,0xb9c92da6),LL(0x474b7c0f,0x07dace3c),LL(0xd996b6a8,0xd0b3f9bc),LL(0x07c662de,0xe97e3847),LL(0x6c851030,0xbf6d1e05)}, {LL(0x99f1aade,0x150c5e93),LL(0x9bd848d5,0x9c1e2351),LL(0x09cab3db,0xeb808a55),LL(0x9a49916f,0x1bfbe08d),LL(0x64ab0e4e,0xc6a70ea7),LL(0x7823f505,0x77d189f2)}},
   {{LL(0x5827fe2c,0x90762c1f),LL(0xeaffda88,0x20160f7a),LL(0x5c47c645,0x7420849f),LL(0x6d72e748,0xb0823195),LL(0x8ee11773,0xaeac683b),LL(0xfb5c550e,0x8c2a0a79)}, {LL(0x6c07cc1e,0x6d986d69),LL(0xba8398b9,0x57269140),LL(0xd94d5223,0xd13e136f),LL(0xed5b01c6,0x1aa75419),LL(0x408fcdcc,0x7c2014b1),LL(0xcffde5ee,0x0680a985)}},
   {{LL(0x23133885,0xfbe88fce),LL(0x7c5a5c4b,0x3c2e3669),LL(0xbbacc6e1,0xe36cf261),LL(0x96ae3cad,0xa72c7bb1),LL(0x4cb1a375,0x08e37103),LL(0x2a02baee,0x5521f445)}, {LL(0xc157e471,0xda9329ba),LL(0x68470808,0x3f90cad1),LL(0xa657de60,0x24182208),LL(0x67c10d1b,0x17c082b7),LL(0x9928d6fd,0x9e222648),LL(0x36ced38f,0x1578c895)}},
   {{LL(0x42c5a7ee,0x92fc9a33),LL(0xf9f0ed71,0x8768614a),LL(0x87ebfb66,0x1ea5f7ed),LL(0xd361069c,0x296852de),LL(0x0192498e,0x1cec6f1a),LL(0xa9cca3aa,0xbfd4858f)}, {LL(0x2ef240e8,0xfba98c24),LL(0xab635d9f,0xc8b500e4),LL(0x913a3edd,0x9f49c572),LL(0xd42b2d4d,0xe6181f93),LL(0x6aa77fa3,0xf96b5db2),LL(0xe43558d8,0xdfb2241f)}},
   {{LL(0x7481cd31,0x8f8a1964),LL(0x17b37aa7,0xd5b8197e),LL(0x7ac0dbf6,0x7cfbcd19),LL(0x93662f46,0x4ecd8954),LL(0x0501365f,0x104a090d),LL(0x7f097083,0x828694cd)}, {LL(0x6dc105dd,0x60b865f3),LL(0x85cd4ed7,0x00549f1b),LL(0xcc197cc9,0xd262e38b),LL(0x9a262b4f,0x5d3271de),LL(0xa953d539,0xc7df47e9),LL(0x5b9a86a6,0xab8f1c8a)}},
   {{LL(0xedee15a5,0xc7d0abb2),LL(0x228cc4a1,0x72dc0105),LL(0xa80767de,0xeb67defc),LL(0x71820908,0x6fa174d8),LL(0x5674d19a,0x3215df48),LL(0x960a081a,0xf944531a)}, {LL(0xef2cce62,0x93ed7180),LL(0xc8bcfc0d,0xb318edbf),LL(0xfe787e58,0x0909d56e),LL(0x8fe8b96f,0x5ae74fc9),LL(0x35ab6811,0x8fc342c4),LL(0x0b991e0c,0x6fc6cc5c)}},
},
/* digit=75 base_pwr=2^375 */
{
   {{LL(0x542f4e90,0x55671129),LL(0x0623d4cd,0x43bedccf),LL(0xe99ca16b,0x7e21207c),LL(0x7c7a26b9,0x785fa105),LL(0xc2c3ab00,0x33c28658),LL(0xd79cd59f,0xcce42a48)}, {LL(0xb8c3bc75,0x9a674db4),LL(0x6904e3fe,0xea701d15),LL(0x66bf2c6c,0x990e7221),LL(0xbd4c3791,0xba29affa),LL(0x20696ee1,0xd98510cf),LL(0xf93d26a5,0x722ed471)}},
   {{LL(0xbc579793,0xe7cf5bac),LL(0xd73f881c,0x11db7ddf),LL(0x04fa8473,0x9c1a531d),LL(0x5780efda,0x399e8484),LL(0x4f62cb5a,0x6e9c12be),LL(0x94a5df3b,0xf21bdc49)}, {LL(0x11da2a4f,0x3c15fe12),LL(0x23e631d1,0xdea123bb),LL(0xbe294c90,0x3ef76da4),LL(0xa99b8398,0x5cf21d5a),LL(0x751b9f6a,0x50679cf8),LL(0x54d0b7bf,0x4b3f3b9c)}},
   {{LL(0x75a46271,0x6c8d97e0),LL(0x9dbed39f,0x0fa0c4cd),LL(0xde74ac6c,0xfb6da5e2),LL(0xc17c1ec5,0x041ce886),LL(0xd7419105,0xb42941a8),LL(0x002fdfd5,0x79768eee)}, {LL(0x88c8111f,0x64849afd),LL(0x814192d6,0xf425fe14),LL(0x0448fd7e,0xe916e864),LL(0x72ed351f,0x31e224ad),LL(0x7c0183c1,0x73e6e6ac),LL(0x21bf7ceb,0x375657c6)}},
   {{LL(0x03bdf6d6,0xc84f9172),LL(0x69f60e03,0xcfc47187),LL(0xa05068ea,0xcdc4753b),LL(0x077777ef,0xa177ad14),LL(0x7e4cf44a,0x0b7f54eb),LL(0x1860144e,0x4ee443f9)}, {LL(0x42bb6a93,0x1279ed4d),LL(0x436c1b54,0x511137d7),LL(0xb8cdb6ce,0xebc958fa),LL(0xa0c7614a,0xbc4f93f4),LL(0x7b2c6d8e,0xc5bd6cde),LL(0x8d65f38a,0xecff7dd7)}},
   {{LL(0xe45543d5,0x9a4af7af),LL(0x6a04c87b,0xb7478fe0),LL(0x66f72454,0x974eebba),LL(0x5901d1ec,0x682578fa),LL(0xe82b048b,0xc3595199),LL(0xbbc19ba2,0x83da52fb)}, {LL(0x90450b02,0x40f337af),LL(0x439c46d7,0xbd1ea60b),LL(0x00d0ed85,0x4f3e4818),LL(0x766d9e20,0x59d0a0a9),LL(0x56a16718,0xe81dc4d6),LL(0xdf3d3d98,0xac872a21)}},
   {{LL(0xbbaae09b,0x4da7d63a),LL(0x0783fab2,0xcae05f37),LL(0x68841d1e,0x1e8c0016),LL(0xb10366f6,0x0688f485),LL(0x05b121e9,0x38ee34b0),LL(0x14e0dc1e,0x2779f009)}, {LL(0xdbff60ce,0x83c1d44e),LL(0x4105c8c2,0x63fbcf82),LL(0x53715349,0x6b732744),LL(0xc5ca18f6,0x5065bdcd),LL(0x2def86e3,0x677313cf),LL(0x33ebff5d,0x6c54d224)}},
   {{LL(0x88b3fc7b,0x35361c91),LL(0xc22a219e,0x706cf8c5),LL(0x886348e9,0x545bb34b),LL(0x4594a530,0xf25eef25),LL(0x73843cbb,0x59427ed0),LL(0xbeea8c4d,0x39638f20)}, {LL(0x960e3c28,0x4e7d445f),LL(0xd383df18,0x989abc64),LL(0x04694ba7,0xb4238e02),LL(0x0203d67f,0xa3753137),LL(0x86d01fc1,0x404bc421),LL(0xb162392d,0x1af37190)}},
   {{LL(0xf65de0f5,0xffec6674),LL(0xd23ad193,0x4043079c),LL(0xee61bc95,0x31811365),LL(0x8948b6e2,0x358bbd6e),LL(0xe31644be,0x1cd9c342),LL(0x60a8a7a7,0xbab3aa8c)}, {LL(0xa375beb6,0xe065519f),LL(0x4439990c,0xf7d0b041),LL(0x8517ae8a,0x8957c03b),LL(0x73750d6e,0xc96a0401),LL(0xb2aee6d7,0x4eb2e364),LL(0xed099114,0x813054fe)}},
   {{LL(0xec3c19fe,0x79d98e89),LL(0x429341e7,0x860ec5a6),LL(0x25dd60c0,0x80ab8568),LL(0xe47973db,0x7d0b3f3f),LL(0xf7899dfb,0x654435bd),LL(0xe6542b1d,0x54c59689)}, {LL(0xbed69ab6,0x171c6842),LL(0x188a3126,0x82b8024d),LL(0x9c37538d,0x31bf057d),LL(0xcb0be742,0xe079326a),LL(0x29ace4e3,0x0232e981),LL(0xce8596ab,0x7b06fb86)}},
   {{LL(0xcaada268,0x71714896),LL(0xfd0e302c,0xbb3d05dc),LL(0xfe56d08f,0xb0785f33),LL(0x38a1b2ef,0xdd43e0f6),LL(0x360fc15a,0x2df35cfb),LL(0x90b3ed36,0x97173f0f)}, {LL(0xb720544e,0xd4970bdd),LL(0x94a01944,0xb6075f76),LL(0x4a43c4f3,0xc99e8a3e),LL(0xd9cb4808,0x8013609f),LL(0xecc3d094,0xf3fef0ea),LL(0x8642d223,0x3829fac7)}},
   {{LL(0x0402156b,0xb7613dab),LL(0x0b20ec7a,0x4ad70f1c),LL(0x9c46dd4d,0xd9189b20),LL(0x4bd5235a,0x4b22485f),LL(0x88822a0d,0x6e972031),LL(0x2c136807,0x3cf8d823)}, {LL(0x0884e550,0x5997fa64),LL(0x293aedb8,0x73110b25),LL(0x35319a22,0x7e820168),LL(0xa6c668ba,0x2222c809),LL(0x31e0bfb8,0x2f316be2),LL(0x0d832198,0x86cf3a2d)}},
   {{LL(0x684456b1,0xffe3104b),LL(0xf0f49278,0x37ba0db6),LL(0xcca2f150,0x15aaed42),LL(0xc421c694,0x8618aa02),LL(0xfab87b36,0x4d6a091d),LL(0x0e786d5f,0x304eaea0)}, {LL(0x2c114074,0x0be97747),LL(0xad387a8d,0xf57e3a19),LL(0x7a70d421,0x6094823b),LL(0x09de860f,0x59287918),LL(0x6dcf6020,0x7f7fca49),LL(0x5f46086e,0x57580c61)}},
   {{LL(0x5d89002f,0xeab34616),LL(0x72a01a68,0xb3115628),LL(0xdce0c191,0x2f0ced58),LL(0x1a895760,0xec08b09a),LL(0xae62153c,0x206faa7f),LL(0xcf2895bf,0xc31e3815)}, {LL(0x9ac88636,0xd57fbf57),LL(0xce91affd,0x966f5a84),LL(0x63620a73,0x092458b9),LL(0x50805fc1,0xda7b4910),LL(0x5c561649,0x1fc60a25),LL(0x4f899e20,0x8110a1a9)}},
   {{LL(0xf0cb1370,0xb509f702),LL(0xc658441f,0xbdfcf4a0),LL(0x7f07f328,0x853d832a),LL(0x8fbdcb83,0x074fdecd),LL(0xd6a4650d,0x80ed8de9),LL(0xa5d68720,0x61c39ce8)}, {LL(0x3177feb4,0xe66666d2),LL(0xafacf38b,0xdbf3fc57),LL(0x0da620fb,0x7e2d9951),LL(0xdf866f77,0x901145ff),LL(0xdb045beb,0x442a37e5),LL(0xbeb1b008,0x0cb0600f)}},
   {{LL(0x54d7a6af,0x1e6604d3),LL(0x07c97f80,0xba6ae4d0),LL(0x77c527fb,0x5e3d978b),LL(0xe93a0d78,0xd0642c72),LL(0xb3c3c215,0x06d8ae5c),LL(0x4eb9a4c9,0x8bf36e5a)}, {LL(0x4d505a53,0x1ca6403c),LL(0xd2f5c7a3,0x0187be5a),LL(0x9f850eac,0x68cce2ba),LL(0x5805353b,0x81055e4a),LL(0x89b4eb85,0x3c242c1d),LL(0xdda42eb8,0xee4a6691)}},
   {{LL(0x7c34f095,0xbb39a17a),LL(0x22fbbe61,0x7be330a8),LL(0xb91f1482,0x6be6abe3),LL(0xbd39a2bc,0xf972804f),LL(0xf91d813e,0x06737e54),LL(0x1a87cd4a,0xbd606668)}, {LL(0xf538d56e,0xbf88b2e5),LL(0x34afd68f,0xb8206a81),LL(0xa58af042,0x7a93aedf),LL(0xac0511b0,0x8853cdf6),LL(0x067e2c19,0x9d7f416d),LL(0xf9671d8a,0x5d0bc923)}},
},
/* digit=76 base_pwr=2^380 */
{
   {{LL(0x204be028,0x2e7d0a16),LL(0xd0e41851,0x4f1d082e),LL(0x3eb317f9,0x15f1ddc6),LL(0x5adf71d7,0xf0275071),LL(0xee858bc3,0x2ce33c2e),LL(0xda73b71a,0xa24c76d1)}, {LL(0x6c70c483,0x9ef6a70a),LL(0x05cf9612,0xefcf1705),LL(0x7502de64,0x9f5bf5a6),LL(0xa4701973,0xd11122a1),LL(0xa2ea7b24,0x82cfaac2),LL(0x0a4582e1,0x6cad67cc)}},
   {{LL(0x51e4de5e,0x96a1e74f),LL(0xe37f5006,0x72913696),LL(0xbe35109c,0x12449c4f),LL(0x4521d7e6,0x1fad8b30),LL(0x57d00293,0xc85eb23d),LL(0x35f68229,0x4ebd334b)}, {LL(0x2df5acf1,0x7c5b8668),LL(0x5463de2e,0xc2b4da6e),LL(0x757cd570,0x067b0456),LL(0x3a1c866b,0xeaab81be),LL(0xbbba88c0,0x72a6af75),LL(0x0ef567dc,0xaed4dbde)}},
   {{LL(0xb4dc8600,0x597a26ff),LL(0xf9288555,0x264a09f3),LL(0x5c27f5f6,0x0b06aff6),LL(0xd8d544e6,0xce5ab665),LL(0x99275c32,0x92f031be),LL(0xf42e0e7c,0xaf51c5bb)}, {LL(0x1e37b36d,0x5bb28b06),LL(0x8473543a,0x583fba6a),LL(0xf93fb7dc,0xe73fd299),LL(0x6e2ccad9,0xfcd999a8),LL(0x334d4f57,0xb8c8a6df),LL(0x9a2acc9b,0x5adb28dd)}},
   {{LL(0x7d221ab6,0x3afdee27),LL(0x47bb619e,0xecf10abc),LL(0xba4a3301,0x340c8ee3),LL(0x2a883b7f,0x1a6ea51a),LL(0xd5d7412b,0x64f27976),LL(0x91251b6e,0x7fcf0ecc)}, {LL(0x365b18b7,0x5f3f8f41),LL(0xe2e13e58,0x38e48b96),LL(0xad61b2cb,0xde3b73d6),LL(0xd542676d,0xf08398d5),LL(0x8e7d712b,0xd373931e),LL(0x7f96e023,0x89325d7a)}},
   {{LL(0x111792b9,0x5adf3d9a),LL(0x4f1e0d09,0x1c77a305),LL(0xa82d3736,0xf9fbce33),LL(0x718c8aa3,0xf307823e),LL(0x416ccf69,0x860578cf),LL(0x1ef8465b,0xb942add8)}, {LL(0xcd9472e1,0x9ee0cf97),LL(0xb01528a8,0xe6792eef),LL(0xc09da90b,0xf99b9a8d),LL(0xcbf3ccb8,0x1f521c2d),LL(0x91a62632,0x6bf66948),LL(0x854fe9da,0xcc7a9ceb)}},
   {{LL(0x4b3759cf,0xe44e3f86),LL(0x9d74e3f6,0x90cab0eb),LL(0x01c4e171,0x10042545),LL(0xce52defb,0xc12df68c),LL(0xf363100a,0xb1fae2fb),LL(0x3573235f,0x5016c853)}, {LL(0x1d922e9b,0x8d4deb66),LL(0x17f84ef2,0x8a20d423),LL(0x5a4e118e,0x32498583),LL(0x308772e9,0x5abfa961),LL(0xf54e4876,0x41c7611f),LL(0x1f5867b2,0xc1da40d3)}},
   {{LL(0x491ccb92,0x46303171),LL(0x2771235b,0xa80a8c0d),LL(0xf172c7cf,0xd8e497ff),LL(0x35b193cf,0x7f7009d7),LL(0xf19df4bc,0x6b9fd3f7),LL(0xb46f1e37,0xada548c3)}, {LL(0xc7a20270,0x87c6eaa9),LL(0xae78ef99,0xef2245d6),LL(0x539eab95,0x2a121042),LL(0x79b8f5cc,0x29a6d5d7),LL(0xb77840dc,0x33803a10),LL(0x11a6a30f,0xfedd3a70)}},
   {{LL(0x85adde98,0x3c90a59f),LL(0xe5269140,0x35414174),LL(0x1a0d58e2,0x9aca885c),LL(0x6816b009,0x77b9b6dd),LL(0x9ee4718f,0x8e5c1213),LL(0x4e4eac45,0x60ad991e)}, {LL(0x4d71f624,0xc00c3569),LL(0x5bc5fd2a,0xacbf4eb2),LL(0x5eaf3eaa,0xcba1ffc7),LL(0x42a87e32,0x5f99092d),LL(0x6f7a882f,0x2e7b49c7),LL(0x29040512,0x5e9bfc5c)}},
   {{LL(0x142403d1,0xfa070e22),LL(0x15c6f7f5,0x68ff3160),LL(0x223a0ce8,0xe09f04e6),LL(0x53e14183,0x22bbd018),LL(0xcf45b75b,0x35d9fafc),LL(0x7eceec88,0x3a34819d)}, {LL(0xd33262d2,0xd9cf7568),LL(0x841d1505,0x431036d5),LL(0x9eb2a79a,0x0c800565),LL(0x5f7edc6a,0x8e77d9f0),LL(0x65e800aa,0x19e12d05),LL(0xb7784e7c,0x335c8d36)}},
   {{LL(0x30a3ada1,0x75cba9d5),LL(0xf8ae9565,0xb69e308b),LL(0xca7b8369,0x990e3425),LL(0xe0a7ad0b,0x9f67567f),LL(0x18bd01b7,0x76ed6fe7),LL(0x2ff95cfe,0x282358aa)}, {LL(0x410f8841,0x28d2ea41),LL(0xccd67c81,0x89d1533f),LL(0xb6a7b8f9,0x969bb272),LL(0x26330782,0x54f8664c),LL(0x1dcd9164,0xb89f3ae8),LL(0x3d962c14,0x54d845b9)}},
   {{LL(0x6484fd40,0x8b2fc4e9),LL(0xa35d24ea,0xee702764),LL(0xb871c3f3,0x15b28ac7),LL(0xe097047f,0x805b4048),LL(0x647cad2f,0xd6f1b8df),LL(0xdc7dd67f,0xf1d5b458)}, {LL(0x25148803,0x324c529c),LL(0x21274faf,0xf6185ebe),LL(0x95148b55,0xaf14751e),LL(0x28f284f4,0x283ed89d),LL(0x4cbebf1a,0x93ad20e7),LL(0x882935e1,0x5f6ec65d)}},
   {{LL(0x6c0f3509,0xb3984b17),LL(0xd8b4d6bc,0xf9fa4483),LL(0x7dec20d2,0xf4ac2b67),LL(0xb3dbe034,0x67ef024e),LL(0x0f94f4d7,0x2dcc5118),LL(0x74a51393,0x024cdcfd)}, {LL(0x20e7abcb,0xf1c0fead),LL(0xd3a7414f,0xffc18f81),LL(0x7062cb0b,0xb00ce556),LL(0x817bc8d1,0xeccb0521),LL(0x40411c15,0xa0c0fe60),LL(0x1defbe00,0x05311322)}},
   {{LL(0xa4dcefe9,0xe222eba4),LL(0xec1ceb74,0x63ad235f),LL(0xe05b18e7,0x2e0bf749),LL(0xb48bdd87,0x547bd050),LL(0xf5aa2fc4,0x0490c970),LL(0x2b431390,0xced5e4cf)}, {LL(0x51d2898e,0x07d82704),LL(0x083b57d4,0x44b72442),LL(0x5037fce8,0xa4ada230),LL(0x50510da6,0x55f7905e),LL(0x8d890a98,0xd8ee724f),LL(0x11b85640,0x925a8e7c)}},
   {{LL(0x56467257,0xda828fe5),LL(0xd640c2a1,0x5e9abf67),LL(0xc25c696a,0x0eed233c),LL(0xb3e1d84f,0x72483dc5),LL(0x4f114abc,0x30bf1ee3),LL(0xd1f9bce8,0xf58b321e)}, {LL(0x97524f33,0xcb26564c),LL(0x1e453229,0xdc2f105e),LL(0x72a982dd,0x9da43ceb),LL(0xfeef8862,0xecf5649d),LL(0x1fa2f06d,0xd8afda34),LL(0x55035432,0xf0d0ced3)}},
   {{LL(0x1ca459ed,0x5bfa10cd),LL(0x6dcf56bf,0x593f085a),LL(0xc0579c3e,0xe6f0ad9b),LL(0x2527c1ad,0xc11c95a2),LL(0xcf1cb8b3,0x7cfa71e1),LL(0x1d6dc79d,0xedcff833)}, {LL(0x432521c9,0x581c4bbe),LL(0x144e11a0,0xbf620096),LL(0xbe3a107b,0x54c38b71),LL(0xe2606ec0,0xed555e37),LL(0xd721d034,0x3fb148b8),LL(0x0091bc90,0x79d53dad)}},
   {{LL(0x08d1be5d,0xcf17f9dc),LL(0xafdfeb23,0xb55de4c8),LL(0xe437b29c,0xa69454ff),LL(0xe27ee9e2,0x6628d789),LL(0xee3af03b,0x56e3b975),LL(0x2f532d62,0x0083fe9c)}, {LL(0xe63e7511,0xcae15213),LL(0x86ed849c,0xdb5384f3),LL(0xfa4d825f,0x902ba959),LL(0x5ae17566,0xbad700d5),LL(0x14c82eb4,0x16b2c5dc),LL(0x36708ea7,0xa4b057a7)}},
}
  /* clang-format on */
};
#endif /* _DISABLE_ECP_384R1_HARDCODED_BP_TBL_ */
#endif /* _IPP_DATA */


IPP_OWN_DEFN(const cpPrecompAP*, gfpec_precom_nistP384r1_fun, (void))
{
    static cpPrecompAP t = { /* w */ 5,
                             /* select function */ p384r1_select_ap_w5,
                             /* precomputed data */ (BNU_CHUNK_T*)ec_p384r1_precomputed };
    return &t;
}
