/****************************************************************
 *								*
 * Copyright (c) 2001-2024 Fidelity National Information	*
 * Services, Inc. and/or its subsidiaries. All rights reserved.	*
 *								*
 *	This source code contains the intellectual property	*
 *	of its copyright holder(s), and is made available	*
 *	under a license.  If you do not know the terms of	*
 *	the license, please stop and do not read further.	*
 *								*
 ****************************************************************/

/****************************************************************
*								*
*  This module provides translation functions between the	*
*  ASCII and EBCDIC code sets.					*
*								*
****************************************************************/

#include "mdef.h"
#include "ebc_xlat.h"
#include "gtm_string.h"
#include "gtm_stdio.h"
#include "stringpool.h"

GBLREF	spdesc		stringpool;
GBLREF	boolean_t	badchar_inhibit;

error_def(ERR_BADCHAR);

#define SIZE_OF_ASCII		0x80
#define MAX_UTF16_NON_ASCII	0x9f
#define PLACEHOLDER_CHAR	0x3f
#define CHAR_NOT_FOUND(IN, OUT, CHSET)											\
{															\
	char	ascii_code[THREE_BYTE_CHAR + 1];									\
	int	ascii_len;												\
															\
	if (!badchar_inhibit)												\
	{														\
		ascii_len = SNPRINTF(ascii_code, SIZEOF(ascii_code), "%u", IN);						\
		RTS_ERROR_CSA_ABT(NULL, VARLSTCNT(6) ERR_BADCHAR, 4, ascii_len, ascii_code, STRLEN(CHSET), CHSET);	\
	} else														\
		*(*OUT)++ = PLACEHOLDER_CHAR;										\
}

#define W1252_MATCH(I, CHAR_SIZE, STR, VAL, OUT, TABLE, CHSET)							\
{															\
	if (*(I) + CHAR_SIZE > (STR)->len)										\
		CHAR_NOT_FOUND(VAL, OUT, CHSET);									\
	if (!find_w1252(TABLE, ARRAYSIZE(TABLE), I, OUT, ((STR)->addr + *(I) + 1), (CHAR_SIZE - 1)))			\
		CHAR_NOT_FOUND(VAL, OUT, CHSET);									\
}

/* Translation tables */
/* These were generated using iconv between "ISO8859-1" (ASCII) */
/* and "IBM-1047" (EBCDIC));  */

/* EBCDIC to ASCII */
LITDEF unsigned char e2a[256] =
{
  /* 00 - 07: */ 0x00, 0x01, 0x02, 0x03, 0x9c, 0x09, 0x86, 0x7f,
  /* 08 - 0f: */ 0x97, 0x8d, 0x8e, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
  /* 10 - 17: */ 0x10, 0x11, 0x12, 0x13, 0x9d, 0x0a, 0x08, 0x87,
  /* 18 - 1f: */ 0x18, 0x19, 0x92, 0x8f, 0x1c, 0x1d, 0x1e, 0x1f,
  /* 20 - 27: */ 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x17, 0x1b,
  /* 28 - 2f: */ 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x05, 0x06, 0x07,
  /* 30 - 37: */ 0x90, 0x91, 0x16, 0x93, 0x94, 0x95, 0x96, 0x04,
  /* 38 - 3f: */ 0x98, 0x99, 0x9a, 0x9b, 0x14, 0x15, 0x9e, 0x1a,
  /* 40 - 47: */ 0x20, 0xa0, 0xe2, 0xe4, 0xe0, 0xe1, 0xe3, 0xe5,
  /* 48 - 4f: */ 0xe7, 0xf1, 0xa2, 0x2e, 0x3c, 0x28, 0x2b, 0x7c,
  /* 50 - 57: */ 0x26, 0xe9, 0xea, 0xeb, 0xe8, 0xed, 0xee, 0xef,
  /* 58 - 5f: */ 0xec, 0xdf, 0x21, 0x24, 0x2a, 0x29, 0x3b, 0x5e,
  /* 60 - 67: */ 0x2d, 0x2f, 0xc2, 0xc4, 0xc0, 0xc1, 0xc3, 0xc5,
  /* 68 - 6f: */ 0xc7, 0xd1, 0xa6, 0x2c, 0x25, 0x5f, 0x3e, 0x3f,
  /* 70 - 77: */ 0xf8, 0xc9, 0xca, 0xcb, 0xc8, 0xcd, 0xce, 0xcf,
  /* 78 - 7f: */ 0xcc, 0x60, 0x3a, 0x23, 0x40, 0x27, 0x3d, 0x22,
  /* 80 - 87: */ 0xd8, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
  /* 88 - 8f: */ 0x68, 0x69, 0xab, 0xbb, 0xf0, 0xfd, 0xfe, 0xb1,
  /* 90 - 97: */ 0xb0, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f, 0x70,
  /* 98 - 9f: */ 0x71, 0x72, 0xaa, 0xba, 0xe6, 0xb8, 0xc6, 0xa4,
  /* a0 - a7: */ 0xb5, 0x7e, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78,
  /* a8 - af: */ 0x79, 0x7a, 0xa1, 0xbf, 0xd0, 0x5b, 0xde, 0xae,
  /* b0 - b7: */ 0xac, 0xa3, 0xa5, 0xb7, 0xa9, 0xa7, 0xb6, 0xbc,
  /* b8 - bf: */ 0xbd, 0xbe, 0xdd, 0xa8, 0xaf, 0x5d, 0xb4, 0xd7,
  /* c0 - c7: */ 0x7b, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
  /* c8 - cf: */ 0x48, 0x49, 0xad, 0xf4, 0xf6, 0xf2, 0xf3, 0xf5,
  /* d0 - d7: */ 0x7d, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50,
  /* d8 - df: */ 0x51, 0x52, 0xb9, 0xfb, 0xfc, 0xf9, 0xfa, 0xff,
  /* e0 - e7: */ 0x5c, 0xf7, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58,
  /* e8 - ef: */ 0x59, 0x5a, 0xb2, 0xd4, 0xd6, 0xd2, 0xd3, 0xd5,
  /* f0 - f7: */ 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
  /* f8 - ff: */ 0x38, 0x39, 0xb3, 0xdb, 0xdc, 0xd9, 0xda, 0x9f
};

/* ASCII to EBCDIC */
LITDEF unsigned char a2e[256] =
{
  /* 00 - 07: */ 0x00, 0x01, 0x02, 0x03, 0x37, 0x2d, 0x2e, 0x2f,
  /* 08 - 0f: */ 0x16, 0x05, 0x15, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
  /* 10 - 17: */ 0x10, 0x11, 0x12, 0x13, 0x3c, 0x3d, 0x32, 0x26,
  /* 18 - 1f: */ 0x18, 0x19, 0x3f, 0x27, 0x1c, 0x1d, 0x1e, 0x1f,
  /* 20 - 27: */ 0x40, 0x5a, 0x7f, 0x7b, 0x5b, 0x6c, 0x50, 0x7d,
  /* 28 - 2f: */ 0x4d, 0x5d, 0x5c, 0x4e, 0x6b, 0x60, 0x4b, 0x61,
  /* 30 - 37: */ 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
  /* 38 - 3f: */ 0xf8, 0xf9, 0x7a, 0x5e, 0x4c, 0x7e, 0x6e, 0x6f,
  /* 40 - 47: */ 0x7c, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
  /* 48 - 4f: */ 0xc8, 0xc9, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6,
  /* 50 - 57: */ 0xd7, 0xd8, 0xd9, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6,
  /* 58 - 5f: */ 0xe7, 0xe8, 0xe9, 0xad, 0xe0, 0xbd, 0x5f, 0x6d,
  /* 60 - 67: */ 0x79, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
  /* 68 - 6f: */ 0x88, 0x89, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
  /* 70 - 77: */ 0x97, 0x98, 0x99, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6,
  /* 78 - 7f: */ 0xa7, 0xa8, 0xa9, 0xc0, 0x4f, 0xd0, 0xa1, 0x07,
  /* 80 - 87: */ 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x06, 0x17,
  /* 88 - 8f: */ 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x09, 0x0a, 0x1b,
  /* 90 - 97: */ 0x30, 0x31, 0x1a, 0x33, 0x34, 0x35, 0x36, 0x08,
  /* 98 - 9f: */ 0x38, 0x39, 0x3a, 0x3b, 0x04, 0x14, 0x3e, 0xff,
  /* a0 - a7: */ 0x41, 0xaa, 0x4a, 0xb1, 0x9f, 0xb2, 0x6a, 0xb5,
  /* a8 - af: */ 0xbb, 0xb4, 0x9a, 0x8a, 0xb0, 0xca, 0xaf, 0xbc,
  /* b0 - b7: */ 0x90, 0x8f, 0xea, 0xfa, 0xbe, 0xa0, 0xb6, 0xb3,
  /* b8 - bf: */ 0x9d, 0xda, 0x9b, 0x8b, 0xb7, 0xb8, 0xb9, 0xab,
  /* c0 - c7: */ 0x64, 0x65, 0x62, 0x66, 0x63, 0x67, 0x9e, 0x68,
  /* c8 - cf: */ 0x74, 0x71, 0x72, 0x73, 0x78, 0x75, 0x76, 0x77,
  /* d0 - d7: */ 0xac, 0x69, 0xed, 0xee, 0xeb, 0xef, 0xec, 0xbf,
  /* d8 - df: */ 0x80, 0xfd, 0xfe, 0xfb, 0xfc, 0xba, 0xae, 0x59,
  /* e0 - e7: */ 0x44, 0x45, 0x42, 0x46, 0x43, 0x47, 0x9c, 0x48,
  /* e8 - ef: */ 0x54, 0x51, 0x52, 0x53, 0x58, 0x55, 0x56, 0x57,
  /* f0 - f7: */ 0x8c, 0x49, 0xcd, 0xce, 0xcb, 0xcf, 0xcc, 0xe1,
  /* f8 - ff: */ 0x70, 0xdd, 0xde, 0xdb, 0xdc, 0x8d, 0x8e, 0xdf
};

/* W-1252 to UTF8 */
LITDEF unsigned char w_2_u8[128][THREE_BYTE_CHAR] =
{
  /* 80 - 83: */ {0xe2, 0x82, 0xac}, {0x81, 0x00, 0x00}, {0xe2, 0x80, 0x9a}, {0xc6, 0x92, 0x00},
  /* 84 - 87: */ {0xe2, 0x80, 0x9e}, {0xe2, 0x80, 0xa6}, {0xe2, 0x80, 0xa0}, {0xe2, 0x80, 0xa1},
  /* 88 - 8b: */ {0xcb, 0x86, 0x00}, {0xe2, 0x80, 0xb0}, {0xc5, 0xa0, 0x00}, {0xe2, 0x80, 0xb9},
  /* 8c - 8f: */ {0xc5, 0x92, 0x00}, {0x8d, 0x00, 0x00}, {0xc5, 0xbd, 0x00}, {0x8f, 0x00, 0x00},
  /* 90 - 93: */ {0x90, 0x00, 0x00}, {0xe2, 0x80, 0x98}, {0xe2, 0x80, 0x99}, {0xe2, 0x80, 0x9c},
  /* 94 - 97: */ {0xe2, 0x80, 0x9d}, {0xe2, 0x80, 0xa2}, {0xe2, 0x80, 0x93}, {0xe2, 0x80, 0x94},
  /* 98 - 9b: */ {0xcb, 0x9c, 0x00}, {0xe2, 0x84, 0xa2}, {0xc5, 0xa1, 0x00}, {0xe2, 0x80, 0xba},
  /* 9c - 9f: */ {0xc5, 0x93, 0x00}, {0x9d, 0x00, 0x00}, {0xc5, 0xbe, 0x00}, {0xc5, 0xb8, 0x00},
  /* a0 - a3: */ {0xc2, 0xa0, 0x00}, {0xc2, 0xa1, 0x00}, {0xc2, 0xa2, 0x00}, {0xc2, 0xa3, 0x00},
  /* a4 - a7: */ {0xc2, 0xa4, 0x00}, {0xc2, 0xa5, 0x00}, {0xc2, 0xa6, 0x00}, {0xc2, 0xa7, 0x00},
  /* a8 - ab: */ {0xc2, 0xa8, 0x00}, {0xc2, 0xa9, 0x00}, {0xc2, 0xaa, 0x00}, {0xc2, 0xab, 0x00},
  /* ac - af: */ {0xc2, 0xac, 0x00}, {0xc2, 0xad, 0x00}, {0xc2, 0xae, 0x00}, {0xc2, 0xaf, 0x00},
  /* b0 - b3: */ {0xc2, 0xb0, 0x00}, {0xc2, 0xb1, 0x00}, {0xc2, 0xb2, 0x00}, {0xc2, 0xb3, 0x00},
  /* b4 - b7: */ {0xc2, 0xb4, 0x00}, {0xc2, 0xb5, 0x00}, {0xc2, 0xb6, 0x00}, {0xc2, 0xb7, 0x00},
  /* b8 - bb: */ {0xc2, 0xb8, 0x00}, {0xc2, 0xb9, 0x00}, {0xc2, 0xba, 0x00}, {0xc2, 0xbb, 0x00},
  /* bc - bf: */ {0xc2, 0xbc, 0x00}, {0xc2, 0xbd, 0x00}, {0xc2, 0xbe, 0x00}, {0xc2, 0xbf, 0x00},
  /* c0 - c3: */ {0xc3, 0x80, 0x00}, {0xc3, 0x81, 0x00}, {0xc3, 0x82, 0x00}, {0xc3, 0x83, 0x00},
  /* c4 - c7: */ {0xc3, 0x84, 0x00}, {0xc3, 0x85, 0x00}, {0xc3, 0x86, 0x00}, {0xc3, 0x87, 0x00},
  /* c8 - cb: */ {0xc3, 0x88, 0x00}, {0xc3, 0x89, 0x00}, {0xc3, 0x8a, 0x00}, {0xc3, 0x8b, 0x00},
  /* cc - cf: */ {0xc3, 0x8c, 0x00}, {0xc3, 0x8d, 0x00}, {0xc3, 0x8e, 0x00}, {0xc3, 0x8f, 0x00},
  /* d0 - d3: */ {0xc3, 0x90, 0x00}, {0xc3, 0x91, 0x00}, {0xc3, 0x92, 0x00}, {0xc3, 0x93, 0x00},
  /* d4 - d7: */ {0xc3, 0x94, 0x00}, {0xc3, 0x95, 0x00}, {0xc3, 0x96, 0x00}, {0xc3, 0x97, 0x00},
  /* d8 - db: */ {0xc3, 0x98, 0x00}, {0xc3, 0x99, 0x00}, {0xc3, 0x9a, 0x00}, {0xc3, 0x9b, 0x00},
  /* dc - df: */ {0xc3, 0x9c, 0x00}, {0xc3, 0x9d, 0x00}, {0xc3, 0x9e, 0x00}, {0xc3, 0x9f, 0x00},
  /* e0 - e3: */ {0xc3, 0xa0, 0x00}, {0xc3, 0xa1, 0x00}, {0xc3, 0xa2, 0x00}, {0xc3, 0xa3, 0x00},
  /* e4 - e7: */ {0xc3, 0xa4, 0x00}, {0xc3, 0xa5, 0x00}, {0xc3, 0xa6, 0x00}, {0xc3, 0xa7, 0x00},
  /* e8 - eb: */ {0xc3, 0xa8, 0x00}, {0xc3, 0xa9, 0x00}, {0xc3, 0xaa, 0x00}, {0xc3, 0xab, 0x00},
  /* ec - ef: */ {0xc3, 0xac, 0x00}, {0xc3, 0xad, 0x00}, {0xc3, 0xae, 0x00}, {0xc3, 0xaf, 0x00},
  /* f0 - f3: */ {0xc3, 0xb0, 0x00}, {0xc3, 0xb1, 0x00}, {0xc3, 0xb2, 0x00}, {0xc3, 0xb3, 0x00},
  /* f4 - f7: */ {0xc3, 0xb4, 0x00}, {0xc3, 0xb5, 0x00}, {0xc3, 0xb6, 0x00}, {0xc3, 0xb7, 0x00},
  /* f8 - fb: */ {0xc3, 0xb8, 0x00}, {0xc3, 0xb9, 0x00}, {0xc3, 0xba, 0x00}, {0xc3, 0xbb, 0x00},
  /* fc - ff: */ {0xc3, 0xbc, 0x00}, {0xc3, 0xbd, 0x00}, {0xc3, 0xbe, 0x00}, {0xc3, 0xbf, 0x00}
};

/* UTF8 to W-1252 */
/* Use the first byte in the key to sort the u8_2_w_XX tables, find the values with find_w1252 */
LITDEF w1252_match u8_2_w_c2[32] =
{   /* First byte is 0xc2 */
    {{0xa0, 0x00}, 0xa0}, /* NO-BREAK SPACE */
    {{0xa1, 0x00}, 0xa1}, /* ¡ */
    {{0xa2, 0x00}, 0xa2}, /* ¢ */
    {{0xa3, 0x00}, 0xa3}, /* £ */
    {{0xa4, 0x00}, 0xa4}, /* ¤ */
    {{0xa5, 0x00}, 0xa5}, /* ¥ */
    {{0xa6, 0x00}, 0xa6}, /* ¦ */
    {{0xa7, 0x00}, 0xa7}, /* § */
    {{0xa8, 0x00}, 0xa8}, /* ¨ */
    {{0xa9, 0x00}, 0xa9}, /* © */
    {{0xaa, 0x00}, 0xaa}, /* ª */
    {{0xab, 0x00}, 0xab}, /* « */
    {{0xac, 0x00}, 0xac}, /* ¬ */
    {{0xad, 0x00}, 0xad}, /* ¬ */
    {{0xae, 0x00}, 0xae}, /* ® */
    {{0xaf, 0x00}, 0xaf}, /* ¯ */
    {{0xb0, 0x00}, 0xb0}, /* ° */
    {{0xb1, 0x00}, 0xb1}, /* ± */
    {{0xb2, 0x00}, 0xb2}, /* ² */
    {{0xb3, 0x00}, 0xb3}, /* ³ */
    {{0xb4, 0x00}, 0xb4}, /* ´ */
    {{0xb5, 0x00}, 0xb5}, /* µ */
    {{0xb6, 0x00}, 0xb6}, /* ¶ */
    {{0xb7, 0x00}, 0xb7}, /* · */
    {{0xb8, 0x00}, 0xb8}, /* ¸ */
    {{0xb9, 0x00}, 0xb9}, /* ¹ */
    {{0xba, 0x00}, 0xba}, /* º */
    {{0xbb, 0x00}, 0xbb}, /* » */
    {{0xbc, 0x00}, 0xbc}, /* ¼ */
    {{0xbd, 0x00}, 0xbd}, /* ½ */
    {{0xbe, 0x00}, 0xbe}, /* ¾ */
    {{0xbf, 0x00}, 0xbf}  /* ¿ */
};

LITDEF w1252_match u8_2_w_c3[64] =
{   /* First byte is 0xc3 */
    {{0x80, 0x00}, 0xc0}, /* À */
    {{0x81, 0x00}, 0xc1}, /* Á */
    {{0x82, 0x00}, 0xc2}, /* Â */
    {{0x83, 0x00}, 0xc3}, /* Ã */
    {{0x84, 0x00}, 0xc4}, /* Ä */
    {{0x85, 0x00}, 0xc5}, /* Å */
    {{0x86, 0x00}, 0xc6}, /* Æ */
    {{0x87, 0x00}, 0xc7}, /* Ç */
    {{0x88, 0x00}, 0xc8}, /* È */
    {{0x89, 0x00}, 0xc9}, /* É */
    {{0x8a, 0x00}, 0xca}, /* Ê */
    {{0x8b, 0x00}, 0xcb}, /* Ë */
    {{0x8c, 0x00}, 0xcc}, /* Ì */
    {{0x8d, 0x00}, 0xcd}, /* Í */
    {{0x8e, 0x00}, 0xce}, /* Î */
    {{0x8f, 0x00}, 0xcf}, /* Ï */
    {{0x90, 0x00}, 0xd0}, /* Ð */
    {{0x91, 0x00}, 0xd1}, /* Ñ */
    {{0x92, 0x00}, 0xd2}, /* Ò */
    {{0x93, 0x00}, 0xd3}, /* Ó */
    {{0x94, 0x00}, 0xd4}, /* Ô */
    {{0x95, 0x00}, 0xd5}, /* Õ */
    {{0x96, 0x00}, 0xd6}, /* Ö */
    {{0x97, 0x00}, 0xd7}, /* × */
    {{0x98, 0x00}, 0xd8}, /* Ø */
    {{0x99, 0x00}, 0xd9}, /* Ù */
    {{0x9a, 0x00}, 0xda}, /* Ú */
    {{0x9b, 0x00}, 0xdb}, /* Û */
    {{0x9c, 0x00}, 0xdc}, /* Ü */
    {{0x9d, 0x00}, 0xdd}, /* Ý */
    {{0x9e, 0x00}, 0xde}, /* Þ */
    {{0x9f, 0x00}, 0xdf}, /* ß */
    {{0xa0, 0x00}, 0xe0}, /* à */
    {{0xa1, 0x00}, 0xe1}, /* á */
    {{0xa2, 0x00}, 0xe2}, /* â */
    {{0xa3, 0x00}, 0xe3}, /* ã */
    {{0xa4, 0x00}, 0xe4}, /* ä */
    {{0xa5, 0x00}, 0xe5}, /* å */
    {{0xa6, 0x00}, 0xe6}, /* æ */
    {{0xa7, 0x00}, 0xe7}, /* ç */
    {{0xa8, 0x00}, 0xe8}, /* è */
    {{0xa9, 0x00}, 0xe9}, /* é */
    {{0xaa, 0x00}, 0xea}, /* ê */
    {{0xab, 0x00}, 0xeb}, /* ë */
    {{0xac, 0x00}, 0xec}, /* ì */
    {{0xad, 0x00}, 0xed}, /* í */
    {{0xae, 0x00}, 0xee}, /* î */
    {{0xaf, 0x00}, 0xef}, /* ï */
    {{0xb0, 0x00}, 0xf0}, /* ð */
    {{0xb1, 0x00}, 0xf1}, /* ñ */
    {{0xb2, 0x00}, 0xf2}, /* ò */
    {{0xb3, 0x00}, 0xf3}, /* ó */
    {{0xb4, 0x00}, 0xf4}, /* ô */
    {{0xb5, 0x00}, 0xf5}, /* õ */
    {{0xb6, 0x00}, 0xf6}, /* ö */
    {{0xb7, 0x00}, 0xf7}, /* ÷ */
    {{0xb8, 0x00}, 0xf8}, /* ø */
    {{0xb9, 0x00}, 0xf9}, /* ù */
    {{0xba, 0x00}, 0xfa}, /* ú */
    {{0xbb, 0x00}, 0xfb}, /* û */
    {{0xbc, 0x00}, 0xfc}, /* ü */
    {{0xbd, 0x00}, 0xfd}, /* ý */
    {{0xbe, 0x00}, 0xfe}, /* þ */
    {{0xbf, 0x00}, 0xff}  /* ÿ */
};

LITDEF w1252_match u8_2_w_c5[7] =
{   /* First byte is 0xc5 */
    {{0x92, 0x00}, 0x8c}, /* Œ */
    {{0x93, 0x00}, 0x9c}, /* œ */
    {{0xa0, 0x00}, 0x8a}, /* Š */
    {{0xa1, 0x00}, 0x9a}, /* š */
    {{0xb8, 0x00}, 0x9f}, /* Ÿ */
    {{0xbd, 0x00}, 0x8e}, /* Ž */
    {{0xbe, 0x00}, 0x9e}  /* ž */
};

LITDEF w1252_match u8_2_w_c6[1] =
{   /* First byte is 0xc6 */
    {{0x92, 0x00}, 0x83}  /* ƒ */
};

LITDEF w1252_match u8_2_w_cb[2] =
{   /* First byte is 0xcb */
    {{0x86, 0x00}, 0x88}, /* ˆ */
    {{0x9c, 0x00}, 0x98}  /* ˜ */
};

LITDEF w1252_match u8_2_w_e2[17] =
{   /* First byte is 0xe2 */
    {{0x80, 0x93}, 0x96}, /* – */
    {{0x80, 0x94}, 0x97}, /* — */
    {{0x80, 0x98}, 0x91}, /* ‘ */
    {{0x80, 0x99}, 0x92}, /* ’ */
    {{0x80, 0x9a}, 0x82}, /* ‚ */
    {{0x80, 0x9c}, 0x93}, /* “ */
    {{0x80, 0x9d}, 0x94}, /* ” */
    {{0x80, 0x9e}, 0x84}, /* „ */
    {{0x80, 0xa0}, 0x86}, /* † */
    {{0x80, 0xa1}, 0x87}, /* ‡ */
    {{0x80, 0xa2}, 0x95}, /* • */
    {{0x80, 0xa6}, 0x85}, /* … */
    {{0x80, 0xb0}, 0x89}, /* ‰ */
    {{0x80, 0xb9}, 0x8b}, /* ‹ */
    {{0x80, 0xba}, 0x9b}, /* › */
    {{0x82, 0xac}, 0x80}, /* € */
    {{0x84, 0xa2}, 0x99}  /* ™ */
};

/* W-1252 to UTF16 */
LITDEF unsigned char w_2_u16[32][TWO_BYTE_CHAR] =
{
  /* 80 - 87: */ {0x20, 0xac}, {0x81, 0x00}, {0x20, 0x1a}, {0x01, 0x92}, {0x20, 0x1e}, {0x20, 0x26}, {0x20, 0x20}, {0x20, 0x21},
  /* 88 - 8f: */ {0x02, 0xc6}, {0x20, 0x30}, {0x01, 0x60}, {0x20, 0x39}, {0x01, 0x52}, {0x8d, 0x00}, {0x01, 0x7d}, {0x8f, 0x00},
  /* 90 - 97: */ {0x90, 0x00}, {0x20, 0x18}, {0x20, 0x19}, {0x20, 0x1c}, {0x20, 0x1d}, {0x20, 0x22}, {0x20, 0x13}, {0x20, 0x15},
  /* 98 - 9f: */ {0x02, 0xdc}, {0x21, 0x22}, {0x01, 0x61}, {0x20, 0x3a}, {0x01, 0x53}, {0x9d, 0x00}, {0x01, 0x7e}, {0x01, 0x78}
};

/* UTF16 to W-1252 */
/* Use the first byte in the key to sort the u16_2_w_XX tables, find the values with find_w1252 */
LITDEF w1252_match u16_2_w_01[8] =
{   /* First byte is 0x01 */
    {{0x52, 0x00}, 0x8c}, /* Œ */
    {{0x53, 0x00}, 0x9c}, /* œ */
    {{0x60, 0x00}, 0x8a}, /* Š */
    {{0x61, 0x00}, 0x9a}, /* š */
    {{0x78, 0x00}, 0x9f}, /* Ÿ */
    {{0x7d, 0x00}, 0x8e}, /* Ž */
    {{0x7e, 0x00}, 0x9e}, /* ž */
    {{0x92, 0x00}, 0x83}  /* ƒ */
};

LITDEF w1252_match u16_2_w_02[2] =
{   /* First byte is 0x02 */
    {{0xc6, 0x00}, 0x88}, /* ˆ */
    {{0xdc, 0x00}, 0x98}  /* ˜ */
};

LITDEF w1252_match u16_2_w_20[16] =
{   /* First byte is 0x20 */
    {{0x13, 0x00}, 0x96}, /* – */
    {{0x15, 0x00}, 0x97}, /* — */
    {{0x18, 0x00}, 0x91}, /* ‘ */
    {{0x19, 0x00}, 0x92}, /* ’ */
    {{0x1a, 0x00}, 0x82}, /* ‚ */
    {{0x1c, 0x00}, 0x93}, /* “ */
    {{0x1d, 0x00}, 0x94}, /* ” */
    {{0x1e, 0x00}, 0x84}, /* „ */
    {{0x20, 0x00}, 0x86}, /* † */
    {{0x21, 0x00}, 0x87}, /* ‡ */
    {{0x22, 0x00}, 0x95}, /* • */
    {{0x26, 0x00}, 0x85}, /* … */
    {{0x30, 0x00}, 0x89}, /* ‰ */
    {{0x39, 0x00}, 0x8b}, /* ‹ */
    {{0x3a, 0x00}, 0x9b}, /* › */
    {{0xac, 0x00}, 0x80}  /* € */
};

LITDEF w1252_match u16_2_w_21[1] =
{   /* First byte is 0x21 */
    {{0x22, 0x00}, 0x99}  /* ™ */
};

void asc_to_ebc(unsigned char *estring_out, unsigned char *astring_in, int len)
{
	int			i;
	unsigned char		*in_ptr, *out_ptr;

	for (i = 0, in_ptr = astring_in, out_ptr = estring_out; i < len; i++, in_ptr++, out_ptr++)
		*out_ptr = a2e[*in_ptr];
}

void ebc_to_asc(unsigned char *astring_out, unsigned char *estring_in, int len)
{
	int			i;
	unsigned char		*in_ptr, *out_ptr;

	for (i = 0, in_ptr = estring_in, out_ptr = astring_out; i < len; i++, in_ptr++, out_ptr++)
		*out_ptr = e2a[*in_ptr];
}

boolean_t find_w1252(const w1252_match *table, size_t size, int *i, unsigned char **out_ptr ,char *in_ptr,
												unsigned short bytes_to_cmp)
{
	int			cmp, high, low, mid;

	assert(0 < size);
	for (low = 0, high = (size - 1); low <= high; )
	{
		mid = (low + high) / 2;
		cmp = memcmp(in_ptr, table[mid].key, bytes_to_cmp);
		if (0 == cmp)
		{
			(*i) += bytes_to_cmp;
			*(*out_ptr)++ = table[mid].value;
			return TRUE;
		} else if (0 > cmp)
			high = mid - 1;
		else
			low = mid + 1;
	}
	return FALSE;
}

int no_conversion(mstr *w1252_in)
{
	ENSURE_STP_FREE_SPACE(w1252_in->len);
	memcpy(stringpool.free, w1252_in->addr, w1252_in->len);
	return w1252_in->len;
}

int w1252_to_utf8(mstr *w1252_in)
{
	int			i, j, out_len;
	const unsigned char	*utf8_bytes;
	unsigned char		unsigned_val, *utf8_out;

	/* Ensure maximum space for the stringpool.
	 * e.g. EURO sign(€) is 1-byte(0x80) in w1252, but 3-bytes(0xe2 0x82 0xac) in utf8
	 */
	ENSURE_STP_FREE_SPACE(w1252_in->len * THREE_BYTE_CHAR);
	for (i = 0, out_len = 0, utf8_out = (unsigned char *)stringpool.free; i < w1252_in->len; i++)
	{
		unsigned_val = (unsigned char)(w1252_in->addr[i]);
		if (SIZE_OF_ASCII > unsigned_val)
		{
			out_len++;
			*utf8_out++ = unsigned_val;
		} else
		{
			utf8_bytes = w_2_u8[unsigned_val - SIZE_OF_ASCII];
			for (j = 0; j < THREE_BYTE_CHAR; j++)
			{
				if (0 == (*utf8_bytes))
					break;
				out_len++;
				*utf8_out++ = *utf8_bytes++;
			}
		}
	}
	return out_len;
}

int utf8_to_w1252(mstr *utf8_in)
{
	int			i, out_len;
	unsigned char		unsigned_val, *w1252_out;

	ENSURE_STP_FREE_SPACE(utf8_in->len);
	for (i = 0, out_len = 0, w1252_out = (unsigned char *)stringpool.free; i < utf8_in->len; i++, out_len++)
	{
		unsigned_val = (unsigned char)(utf8_in->addr[i]);
		if (SIZE_OF_ASCII > unsigned_val)
		{
			*w1252_out++ = unsigned_val;
			continue;
		}
		switch (unsigned_val)
		{
		case 0xc2:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf8_in, unsigned_val, &w1252_out, u8_2_w_c2, "UTF-8");
			break;
		case 0xc3:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf8_in, unsigned_val, &w1252_out, u8_2_w_c3, "UTF-8");
			break;
		case 0xc5:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf8_in, unsigned_val, &w1252_out, u8_2_w_c5, "UTF-8");
			break;
		case 0xc6:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf8_in, unsigned_val, &w1252_out, u8_2_w_c6, "UTF-8");
			break;
		case 0xcb:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf8_in, unsigned_val, &w1252_out, u8_2_w_cb, "UTF-8");
			break;
		case 0xe2:
			W1252_MATCH(&i, THREE_BYTE_CHAR, utf8_in, unsigned_val, &w1252_out, u8_2_w_e2, "UTF-8");
			break;
		default:
			CHAR_NOT_FOUND(unsigned_val, &w1252_out, "UTF-8");
			break;
		}
	}
	return out_len;
}

int w1252_to_utf16(mstr *w1252_in)
{
	int			i, j, out_len;
	const unsigned char	*utf16_bytes;
	unsigned char		unsigned_val, *utf16_out;

	/* Ensure maximum space for the stringpool.
	 * e.g. EURO sign(€) is 1-byte(0x80) in w1252, but 2-bytes(0x20 0xac) in utf8
	 */
	ENSURE_STP_FREE_SPACE(w1252_in->len * TWO_BYTE_CHAR);
	for (i = 0, out_len = 0, utf16_out = (unsigned char *)stringpool.free; i < w1252_in->len; i++)
	{
		unsigned_val = (unsigned char)(w1252_in->addr[i]);
		if ((SIZE_OF_ASCII > unsigned_val) || (MAX_UTF16_NON_ASCII < unsigned_val))
		{	/* UTF-16 has a 2-byte fixed length so front pad all characters that
			 * take 1-byte with a <NUL> byte
			 */
			out_len += TWO_BYTE_CHAR;
			*utf16_out++ = 0;
			*utf16_out++ = unsigned_val;
		} else
		{
			utf16_bytes = w_2_u16[unsigned_val - SIZE_OF_ASCII];
			for (j = 0; j < TWO_BYTE_CHAR; j++)
			{
				if (0 == (*utf16_bytes))
					break;
				out_len++;
				*utf16_out++ = *utf16_bytes++;
			}
		}
	}
	return out_len;
}

int utf16_to_w1252(mstr *utf16_in)
{
	int			i, out_len;
	unsigned char		unsigned_val, *w1252_out;

	ENSURE_STP_FREE_SPACE(utf16_in->len);
	for (i = 0, out_len = 0, w1252_out = (unsigned char *)stringpool.free; i < utf16_in->len; i++, out_len++)
	{
		unsigned_val = (unsigned char)(utf16_in->addr[i]);
		switch (unsigned_val)
		{
		case 0x00:
			unsigned_val = (unsigned char)(utf16_in->addr[++i]);
			if ((SIZE_OF_ASCII <= unsigned_val) && (MAX_UTF16_NON_ASCII >= unsigned_val))
				CHAR_NOT_FOUND(unsigned_val, &w1252_out, "UTF-16");
			*w1252_out++ = unsigned_val;
			break;
		case 0x01:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf16_in, unsigned_val, &w1252_out, u16_2_w_01, "UTF-16");
			break;
		case 0x02:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf16_in, unsigned_val, &w1252_out, u16_2_w_02, "UTF-16");
			break;
		case 0x20:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf16_in, unsigned_val, &w1252_out, u16_2_w_20, "UTF-16");
			break;
		case 0x21:
			W1252_MATCH(&i, TWO_BYTE_CHAR, utf16_in, unsigned_val, &w1252_out, u16_2_w_21, "UTF-16");
			break;
		default:
			CHAR_NOT_FOUND(unsigned_val, &w1252_out, "UTF-16");
			break;
		}
	}
	return out_len;
}
